/*
 * Type definitions for GoJS v2.0.16
 * Project: https://gojs.net
 * Definitions by: Northwoods Software <https://github.com/NorthwoodsSoftware>
 * Definitions: https://github.com/NorthwoodsSoftware/GoJS
 * Copyright (C) 1998-2019 by Northwoods Software Corporation.
 * This requires TypeScript v2.8 or later.
 */


// UMD. For ES6, use go-module.d.ts

export as namespace go;

/**
 * The ObjectData type is the same as `{ [index: string]: any; }`.
 * This is to ease writing `someDataObject.anyPropertyName`,
 * when dealing with arbitrary JavaScript Objects used as model data.
 * @category Type
 */
export interface ObjectData {
    [index: string]: any;
}
/**
 * Undocumented
 * @unrestricted
 */
export class EnumValue {
    /**
     * @param {function()} classfunc
     * @param {string} name
     * @param {number} val
     */
    constructor(classfunc: Function, name: string, val: number);
    /**
     */
    readonly classType: Function;
    /**
     */
    readonly name: string;
}
/**
 * This interface is implemented by the List, Set, and Map
 * classes; it provides the #iterator read-only property that returns an Iterator.
 *
 * Typical usage is:
 * ```js
 * var it = anIterableCollection.iterator;
 * while (it.next()) {
 *   var item = it.value;
 * }
 * ```
 * @interface
 * @template T
 * @category Collection
 */
export interface Iterable<T> {
    /**
     * Gets an Iterator that can iterate over the items in the collection.
     *
     * Typical usage is:
     * ```js
     *  var it = anIterableCollection.iterator;
     *  while (it.next()) {
     *    var item = it.value;
     *  }
     * ```
     * @return {Iterator.<T>}
     */
    iterator: Iterator<T>;
    /**
     * Returns the first item in the list, or null if there is none.
     * @return {T|null} This returns null if there are no items in the list.
     */
    first(): T | null;
    /**
     * This read-only property is the number of elements in the collection.
     */
    readonly count: number;
}
/**
 * This interface defines properties and methods for iterating over a collection;
 * it provides the #next predicate and the #value read-only property.
 * Some Iterators also provide `key` property values along with each `value`.
 *
 * Typical usage is:
 * ```js
 *  var it = anIterableCollection.iterator;
 *  while (it.next()) {
 *    var item = it.value;
 *  }
 * ```
 *
 * Many iterators will signal an error if #next is called
 * after the underlying collection has been modified.
 *
 * To avoid confusion when dealing with Iterables,
 * iterators implement the Iterable#iterator property
 * by just returning themselves.
 * @interface
 * @template T
 * @extends {Iterable.<T>}
 * @category Collection
 */
export interface Iterator<T> extends Iterable<T> {
    /**
     * Returns itself, which is convenient for code that expects an Iterable
     * instead of an Iterator.
     * @return {Iterator.<T>}
     */
    iterator: Iterator<T>;
    /**
     * Call this method to advance the iterator to the next item in the collection.
     * This should be called before accessing any #value.
     * @return {boolean} whether another item is available; when true the value of #value will be that item.
     */
    next(): boolean;
    /**
     * Call this method to advance the iterator to the next item in the collection.
     * This should be called before accessing any #value.
     * @return {boolean} whether another item is available; when true the value of #value will be that item.
     */
    hasNext(): boolean;
    /**
     * Advance if needed to the first item in the collection and return it, or return null if there is none.
     *
     * Caution: this returns a *key/value pair*, not a *value*, for Map iterators.
     * @return {T|null}
     * @since 1.1
     */
    first(): T | null;
    /**
     * Start this iterator all over again.
     */
    reset(): void;
    /**
     * This is true if any invocation of the given predicate on items in the collection is true.
     *
     * Call the given predicate on each item in the collection.
     * As soon as a call returns true, this returns true.
     * Otherwise this returns false.
     * For an empty collection this returns false.
     *
     * This automatically #reset's itself when it is called.
     * @expose
     * @param {function(T):boolean} pred This function must not have any side-effects.
     * @return {boolean} True if any predicate call is true; false otherwise.
     * @since 1.4
     */
    any(pred: (x: T) => boolean): boolean;
    /**
     * This is true if all invocations of the given predicate on items in the collection are true.
     *
     * Call the given predicate on each item in the collection.
     * As soon as a call returns false, this returns false.
     * Otherwise this returns true.
     * For an empty collection this returns true.
     *
     * This automatically #reset's itself when it is called.
     * @expose
     * @param {function(T):boolean} pred This function must not have any side-effects.
     * @return {boolean} True if all predicate calls are true; false otherwise.
     * @since 1.4
     */
    all(pred: (x: T) => boolean): boolean;
    /**
     * Call the given function on each item in the collection.
     *
     * This automatically #reset's itself when it is called.
     * @expose
     * @param {function(T)} func This function must not modify the collection.
     * @return {Iterator} this iterator itself
     * @since 1.4
     */
    each(func: (x: T) => void): void;
    /**
     * Call the given function on each item in the collection and present the results in an iterator.
     *
     * This automatically #reset's itself when it is called.
     * @expose
     * @param {function(T)} func This function must not modify the collection.
     * @return {Iterator} this
     */
    map<S>(func: (x: T) => S): Iterator<S>;
    /**
     * Call the given predicate on each item in the collection and for each item that it returns true, present the item in an iterator.
     *
     * This automatically #reset's itself when it is called.
     * @expose
     * @param {function(T):boolean} pred This function must not have any side-effects.
     * @return {Iterator.<T>} this
     */
    filter(pred: (x: T) => boolean): Iterator<T>;
    /**
     * Gets the current item in the collection, assuming #next has just returned true.
     */
    readonly value: T;
    /**
     * Gets the current index to the item in the collection, assuming #next has just returned true.
     */
    readonly key: any;
    /**
     * This read-only property is the total number of items in the iterated collection.
     */
    readonly count: number;
}
/**
 * Undocumented interface used for both the SVGSurface and the CanvasSurface, which are undocumented classes.
 */
export interface ISurface {
    domElement: Element;
    context: IContext;
    width: number;
    height: number;
    ownerDocument: Document;
    resize(pixelWidth: number, pixelHeight: number, width: number, height: number): boolean;
    elementFinished: ((a: GraphObject, b: SVGElement) => void) | null;
    getBoundingClientRect(): ClientRect | DOMRect;
    focus(): void;
    dispose(): void;
    style: CSSStyleDeclaration;
}
/**
 * Undocumented interface used for both the SVGContext and the CanvasSurfaceContext, which are undocumented classes.
 */
export interface IContext {
    fillStyle: string | CanvasGradient | CanvasPattern | SGradient;
    font: string;
    globalAlpha: number;
    lineCap: string;
    lineDashOffset: number;
    lineJoin: string;
    lineWidth: number;
    miterLimit: number;
    shadowBlur: number;
    shadowColor: string;
    shadowOffsetX: number;
    shadowOffsetY: number;
    strokeStyle: string | CanvasGradient | CanvasPattern | SGradient;
    textAlign: string;
    imageSmoothingEnabled: boolean;
    clipInsteadOfFill: boolean;
    setImageSmoothingEnabled(smooth: boolean): void;
    arc(x: number, y: number, radius: number, startAngle: number, endAngle: number, counterclockwise: boolean, lx?: number, ly?: number): void;
    beginPath(): void;
    bezierCurveTo(a: number, b: number, c: number, d: number, e: number, f: number): void;
    clearRect(x: number, y: number, w: number, h: number): void;
    clip(): void;
    closePath(): void;
    createLinearGradient(aX0: number, aY0: number, aX1: number, aY1: number): CanvasGradient | SGradient;
    createPattern(image: HTMLCanvasElement | HTMLImageElement, repetition: string): CanvasPattern | string;
    createRadialGradient(aX0: number, aY0: number, aR0: number, aX1: number, aY1: number, aR1: number): CanvasGradient | SGradient;
    drawImage(src: HTMLCanvasElement | HTMLImageElement | HTMLVideoElement, sx: number, sy: number, sw?: number, sh?: number, dx?: number, dy?: number, dw?: number, dh?: number): void;
    fill(): void;
    fillRect(x: number, y: number, w: number, h: number): void;
    fillText(str: string, x: number, y: number): void;
    getImageData(x: number, y: number, w: number, h: number): ImageData;
    lineTo(x: number, y: number): void;
    measureText(text: string): TextMetrics;
    moveTo(x: number, y: number): void;
    quadraticCurveTo(a: number, b: number, c: number, d: number): void;
    rect(x: number, y: number, w: number, h: number): void;
    restore(): void;
    rotate(angle: number): void;
    save(): void;
    setTransform(a: number, b: number, c: number, d: number, e: number, f: number): void;
    scale(x: number, y: number): void;
    stroke(): void;
    transform(a: number, b: number, c: number, d: number, e: number, f: number): void;
    translate(x: number, y: number): void;
    fillContext(brush: BrushLike): void;
    strokeContext(): void;
    shadowsSet(x: number, y: number, blur: number): void;
    shadowsOff(): void;
    shadowsOn(): void;
    enableDash(strokeDashArray: Array<number>, strokeDashOffset: number): void;
    disableDash(): void;
    clearContextCache(clearFont: boolean): void;
}

/**
 * <span style="color: red; font-weight: bold;">NOTE: For 2.0 the #constructor argument has changed.
 * List now optionally accepts a collection, and only checks types in TypeScript.</span>
 *
 * An ordered iterable collection.
 * In TypeScript it is a generic class that enforces at compile-time the type of elements that may be added to the List.
 *
 * An example usage:
 * ```js
 *   var list = new go.List();  // or in TypeScript: new go.List<go.Point>();
 *   list.add(new go.Point(0, 0));
 *   list.add(new go.Point(20, 10));
 *   list.add(new go.Point(10, 20));
 *   // now list.length === 3
 *   // and list.elt(1) instanceof go.Point
 * ```
 *
 * You can iterate over the items in a List:
 * ```js
 *   var it = aList.iterator;
 *   while (it.next()) {
 *     console.log("#" + it.key + " is " + it.value);
 *   }
 * ```
 * Or:
 * ```js
 *   aList.each(function(val) {
 *     console.log(val);
 *   });
 * ```
 * The key will range from zero to #count-1.
 *
 * For convenience this **GoJS** List class has synonyms for the following methods and property:
 *   - **get(idx)**: #elt
 *   - **set(idx,val)**: #setElt
 *   - **has(val)**: #contains
 *   - **delete(val)**: #remove
 *   - **clear()**: #clear
 *   - **size**: #count
 *
 * The constructor now takes an optional Iterable or Array argument that provides the initial elements for the new List.
 *
 * Note that GoJS iteration is quite different than ES6 iteration, so that functionality has not been made somewhat compatible.
 * These collection classes were defined in GoJS before the ES6 collection classes were proposed.
 * @template T
 * @implements {Iterable.<T>}
 * @unrestricted
 * @category Collection
 */
export class List<T> implements Iterable<T> {
    /**
     * There are two possible constructors:
     *
     * `new go.List()`, for JavaScript
     *
     * `new go.List<type>()` for TypeScript, to enforce type checking.
     *
     * Typical usage would be something like:
     * ```js
     *   var list = new go.List();  // keep a list of GraphObjects
     * ```
     * @param {Iterable.<T>|Array.<T>=} coll an optional collection of items to add.
     */
    constructor(coll?: Iterable<T> | Array<T>);
    /**
     * @return {string}
     */
    toString(): string;
    /**
     * Adds a given value to the end of the List.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {*} val
     * @return {List.<T>} This modified List.
     */
    add(val: T): List<T>;
    /**
     * Adds a given value to the end of the List.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {*} val
     */
    push(val: T): void;
    /**
     * Adds all of the values of a collection to the end of this List.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {Iterable.<T>|Array.<T>} coll the collection of items to add.
     * @return {List.<T>} This modified List.
     */
    addAll(coll: Iterable<T> | Array<T>): List<T>;
    /**
     * Clears the List.
     * This sets the #count to zero.
     *
     * Be careful not to call this method while iterating over the collection.
     */
    clear(): void;
    /**
     * Returns whether the given value is in this List.
     * @param {T} val The value to check.
     * @return {boolean} Whether or not the value is contained within the List.
     */
    contains(val: T): boolean;
    /**
     * Returns whether the given value is in this List.
     * @param {T} val The value to check.
     * @return {boolean} Whether or not the value is contained within the List.
     */
    has(val: T): boolean;
    /**
     * Returns the index of the given value if it is in this List.
     * @param {T} val The value to check.
     * @return {number} int returns -1 if the value is not in this list.
     */
    indexOf(val: T): number;
    /**
     * Returns the element at the given index.
     * @param {number} i int The index of the element to return.
     * @return {T} the value at the given index.
     */
    elt(i: number): T;
    /**
     * Returns the element at the given index.
     * @param {number} i int The index of the element to return.
     * @return {T} the value at the given index.
     */
    get(i: number): T;
    /**
     * Set the element at the given index to a given value.
     * @param {number} i int The index of the element to set.
     * @param {T} val The value to set at the index.
     */
    setElt(i: number, val: T): void;
    /**
     * Set the element at the given index to a given value.
     * @param {number} i int The index of the element to set.
     * @param {T} val The value to set at the index.
     */
    set(i: number, val: T): void;
    /**
     * Returns the first item in the list, or null if there is none.
     * @return {T|null} This returns null if there are no items in the list.
     */
    first(): T | null;
    /**
     * Returns the last item in the list, or null if these is none.
     * @return {T|null} This returns null if there are no items in the list.
     * @since 1.5
     */
    last(): T | null;
    /**
     * Returns the last item in the list and removes it from the list, or just return null if these is none.
     * Use #add to push an item onto the end of the list.
     * Use #last to get the last item without reducing the length of the list.
     * @return {T|null} This returns null if there are no items in the list.
     * @since 1.5
     */
    pop(): T | null;
    /**
     * This is true if any invocation of the given predicate on items in the collection is true.
     *
     * Call the given predicate on each item in the collection.
     * As soon as a call returns true, this returns true.
     * Otherwise this returns false.
     * For an empty collection this returns false.
     * @expose
     * @param {function(T):boolean} pred This function must not have any side-effects.
     * @return {boolean} True if any predicate call is true; false otherwise.
     * @since 1.4
     */
    any(pred: ((a: T) => boolean)): boolean;
    /**
     * This is true if all invocations of the given predicate on items in the collection are true.
     *
     * Call the given predicate on each item in the collection.
     * As soon as a call returns false, this returns false.
     * Otherwise this returns true.
     * For an empty collection this returns true.
     * @expose
     * @param {function(T):boolean} pred This function must not have any side-effects.
     * @return {boolean} True if all predicate calls are true; false otherwise.
     * @since 1.4
     */
    all(pred: ((a: T) => boolean)): boolean;
    /**
     * Call the given function on each item in the collection.
     * @expose
     * @param {function(T)} func This function must not modify the collection.
     * @return {List.<T>} This List itself
     * @since 1.4
     */
    each(func: ((a: T) => void)): List<T>;
    /**
     * Call the given function on each item in the collection and collect the results in a new List.
     *
     * Unlike Iterator#map, this returns a List, not an Iterator.
     * @expose
     * @param {function(T):*} func This function must not modify the collection.
     * @return {List.<S>}
     */
    map<S>(func: ((a: T) => S)): List<S>;
    /**
     * Call the given predicate on each item in the collection and for each item that it returns true, collect the item in a new List.
     *
     * Unlike Iterator#filter, this returns a List, not an Iterator.
     * @expose
     * @param {function(T):boolean} pred This function must not have any side-effects.
     * @return {List.<T>}
     */
    filter(pred: ((a: T) => boolean)): List<T>;
    /**
     * Insert a value before the index i.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {number} i int The index to insert before.
     * @param {T} val The value to insert.
     */
    insertAt(i: number, val: T): void;
    /**
     * Removes a given value (if found) from the List.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {T} val The value to remove.
     * @return {boolean} true if the value was found and removed, false otherwise.
     */
    remove(val: T): boolean;
    /**
     * Removes a given value (if found) from the List.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {T} val The value to remove.
     * @return {boolean} true if the value was found and removed, false otherwise.
     */
    delete(val: T): boolean;
    /**
     * Removes a value at a given index from the List.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {number} i int The index to remove.
     */
    removeAt(i: number): void;
    /**
     * Removes a range of values from the List, given both the starting and the ending zero-based indexes.
     * For example,
     * ```js
     * list.removeRange(2, 4)
     * ```
     * will remove elements 2, 3, and 4 from the list.
     * If there were two or fewer elements in the list to begin with, the list is unchanged.
     * If *from* is greater than *to*, the list is unchanged.
     * If *from* is greater than or equal to the length, the list is unchanged.
     * If *to* is less than zero, the list is unchanged.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {number} from int The starting index of the range to remove, inclusive; negative values are treated as zero
     * @param {number} to int The ending index of the range to remove, inclusive; values greater than the length of the list are treated as referring to the last element
     * @return {List.<T>} This modified List
     */
    removeRange(from: number, to: number): List<T>;
    /**
     * Makes a shallow copy of this List.
     * The values are not copied,
     * so if they are objects they may continue to be shared with the original List.
     * @expose
     * @return {List.<T>} The new List with the same elements.
     */
    copy(): List<T>;
    /**
     * Produces a JavaScript Array from the contents of this List.
     * @return {Array.<T>} A copy of the List in Array form.
     */
    toArray(): Array<T>;
    /**
     * Converts the List to a Set.
     * The count of the resulting Set may be less than the count of this List
     * if any duplicates were removed.
     * @return {Set.<T>} A copy of the contents of this List,
     * but with duplicates removed and ordering lost.
     */
    toSet(): Set<T>;
    /**
     * Sort the List according to a comparison function.
     * @param {function(T,T):number} sortfunc  This function is passed two items in the list.
     * It should return zero if they are equal,
     * less than zero if the first value should come before the second value,
     * or greater than zero if the first value should come after the second value.
     * @return {List.<T>} This modified List.
     */
    sort(sortfunc: ((a: T, b: T) => number)): List<T>;
    /**
     * Undocumented
     * @param {function(*,*):number} sortfunc  This function is passed two elements in the list.
     * It should return zero if they are equal,
     * less than zero if the first value should come before the second value,
     * or greater than zero if the first value should come after the second value.
     * @param {number=} from int The optional index at which to start the sort, including that element;
     * default to zero, the first element of the list.
     * @param {number=} to int The optional index at which to end the sort, excluding that element;
     * defaults to the end of the list.
     * @return {List.<T>} This modified List.
     */
    sortRange(sortfunc: ((a: T, b: T) => number), from?: number, to?: number): List<T>;
    /**
     * Reverse the order of items in this List.
     * @return {List.<T>} This modified List.
     */
    reverse(): List<T>;
    /**
     * This read-only property is the length of the List.
     */
    readonly count: number;
    /**
     * This read-only property is the length of the List.
     */
    readonly size: number;
    /**
     * This read-only property is the length of the List, a synonym for the #count property.
     */
    readonly length: number;
    /**
     * Gets an object that you can use for iterating over the List.
     * The key will be an integer from zero to the count-1.
     * The value will be the item at that index in the list.
     * Typical usage:
     * ```js
     *   var it = aList.iterator;
     *   while (it.next()) {
     *     . . . "index: " + it.key + " value: " + it.value . . .
     *   }
     * ```
     */
    readonly iterator: Iterator<T>;
    /**
     * Gets an object that you can use for iterating over the List in backwards order.
     * The key will be an integer from count-1 to zero.
     * The value will be the item at that index in the list.
     * The list is not modified by traversing in reverse order.
     * Typical usage:
     * ```js
     *   var it = aList.iteratorBackwards;
     *   while (it.next()) {
     *     . . . 'key: ' + it.key + ' value: ' + it.value . . .
     *   }
     * ```
     */
    readonly iteratorBackwards: Iterator<T>;
}
/**
 * <span style="color: red; font-weight: bold;">NOTE: For 2.0 the #constructor argument has changed.
 * Set now optionally accepts a collection, and only checks types in TypeScript.</span>
 *
 * An unordered iterable collection that cannot contain two instances of the same value.
 * In TypeScript it is a generic class that enforces at compile-time the type of elements that may be added to the Set.
 *
 * An example usage:
 * ```js
 *   var set = new go.Set();  // In TypeScript: new go.Set<string>();
 *   set.add("orange");
 *   set.add("apple");
 *   set.add("orange");
 *   // now set.count === 2
 *   // and set.contains("orange") === true
 *   // and set.contains("banana") === false
 * ```
 *
 * You can iterate over the items in a Set:
 * ```js
 *   var it = aSet.iterator;
 *   while (it.next()) {
 *     . . . it.value . . .
 *   }
 * ```
 * Or:
 * ```js
 *   aSet.each(function(val) {
 *       . . . val . . .
 *     });
 * ```
 *
 * Although not precisely implementing the features of the EcmaScript 6 **Set** class,
 * this **GoJS** Set class has synonyms for the following methods and property:
 *   - **add(val)**: #add
 *   - **delete(val)**: #remove
 *   - **has(val)**: #contains
 *   - **clear()**: #clear
 *   - **size**: #count
 *
 * The constructor now takes an optional Iterable or Array argument that provides the initial elements for the new Set.
 *
 * Note that GoJS iteration is quite different than ES6 iteration, so that functionality has not been made somewhat compatible.
 * These collection classes were defined in GoJS before the ES6 collection classes were proposed.
 * @template T
 * @implements {Iterable.<T>}
 * @unrestricted
 * @category Collection
 */
export class Set<T> implements Iterable<T> {
    /**
     * There are two possible constructors:
     *
     * `new go.Set()`, for JavaScript
     *
     * `new go.Set<T>()` for TypeScript
     *
     * In TypeScript, the optional generic argument describes the type of values
     * that this Set may hold.
     *
     * For example, the expression:
     * ```js
     * // TypeScript:
     * new go.Set<go.Point>()
     * ```
     *
     * Creates a new Set that may only contain Points.
     *
     * @param {Iterable.<T>|Array.<T>=} coll an optional collection of items to add.
     */
    constructor(coll?: Iterable<T> | Array<T>);
    /**
     * @return {string}
     */
    toString(): string;
    /**
     * Adds a given value to the Set, if not already present.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {T} val The value to add to the Set; must not be null.
     * @return {Set.<T>} This modified Set.
     */
    add(val: T): Set<T>;
    /**
     * Adds all of the values of a collection to this Set.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {Iterable.<T>|Array.<T>} coll the collection of items to add.
     * @return {Set.<T>} This modified Set.
     */
    addAll(coll: Iterable<T> | Array<T>): Set<T>;
    /**
     * Returns whether the given value is in this Set.
     * @param {T} val The value to check.
     * @return {boolean} Whether or not the value is contained within the Set.
     */
    contains(val: T): boolean;
    /**
     * Returns whether the given value is in this Set.
     * @param {T} val The value to check.
     * @return {boolean} Whether or not the value is contained within the Set.
     */
    has(val: T): boolean;
    /**
     * Returns true if all of the values of a given collection are in this Set.
     * @param {Iterable.<T>} coll the collection of items to check for.
     * @return {boolean}
     */
    containsAll(coll: Iterable<T>): boolean;
    /**
     * Returns true if any of the values of a given collection are in this Set.
     * @param {Iterable.<T>} coll the collection of items to check for.
     * @return {boolean}
     */
    containsAny(coll: Iterable<T>): boolean;
    /**
     * Returns the first item in the collection, or null if there is none.
     * @return {T|null} This returns null if there are no items in the collection.
     */
    first(): T | null;
    /**
     * This is true if any invocation of the given predicate on items in the collection is true.
     *
     * Call the given predicate on each item in the collection.
     * As soon as a call returns true, this returns true.
     * Otherwise this returns false.
     * For an empty collection this returns false.
     * @expose
     * @param {function(T):boolean} pred This function must not have any side-effects.
     * @return {boolean} True if any predicate call is true; false otherwise.
     * @since 1.4
     */
    any(pred: ((a: T) => boolean)): boolean;
    /**
     * This is true if all invocations of the given predicate on items in the collection are true.
     *
     * Call the given predicate on each item in the collection.
     * As soon as a call returns false, this returns false.
     * Otherwise this returns true.
     * For an empty collection this returns true.
     * @expose
     * @param {function(T):boolean} pred This function must not have any side-effects.
     * @return {boolean} True if all predicate calls are true; false otherwise.
     * @since 1.4
     */
    all(pred: ((a: T) => boolean)): boolean;
    /**
     * Call the given function on each item in the collection.
     * @expose
     * @param {function(T)} func This function must not modify the collection.
     * @return {Set.<T>} This Set itself.
     * @since 1.4
     */
    each(func: ((a: T) => void)): Set<T>;
    /**
     * Call the given function on each item in the collection and collect the results in a new Set.
     *
     * Unlike Iterator#map, this returns a Set, not an Iterator.
     * @expose
     * @param {function(T):*} func This function must not modify the collection.
     * @return {Set.<S>}
     */
    map<S>(func: ((a: T) => S)): Set<S>;
    /**
     * Call the given predicate on each item in the collection and for each item that it returns true, collect the item in a new Set.
     *
     * Unlike Iterator#filter, this returns a Set, not an Iterator.
     * @expose
     * @param {function(T):boolean} pred This function must not have any side-effects.
     * @return {Set.<T>}
     */
    filter(pred: ((a: T) => boolean)): Set<T>;
    /**
     * Removes a value (if found) from the Set.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {T} val The value to insert.
     * @return {boolean} true if the value was found and removed, false otherwise.
     */
    remove(val: T): boolean;
    /**
     * Removes a value (if found) from the Set.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {T} val The value to insert.
     * @return {boolean} true if the value was found and removed, false otherwise.
     */
    delete(val: T): boolean;
    /**
     * Removes all of the values of a collection from this Set.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {Iterable.<T>|Array.<T>} coll the collection of items to remove.
     * @return {Set.<T>} This modified Set.
     */
    removeAll(coll: Iterable<T> | Array<T>): Set<T>;
    /**
     * Removes from this Set all items that are not in the given collection.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {Iterable.<T>} coll the collection of items that should be kept in this Set.
     * @return {Set.<T>} This modified Set.
     */
    retainAll(coll: Iterable<T>): Set<T>;
    /**
     * Clears the Set.
     * This sets the #count to zero.
     *
     * Be careful not to call this method while iterating over the collection.
     */
    clear(): void;
    /**
     * Makes a shallow copy of this Set.
     * The values are not copied,
     * so if they are objects they may continue to be shared with the original Set.
     * @expose
     * @return {Set.<T>} The new Set with the same elements.
     */
    copy(): Set<T>;
    /**
     * Produces a JavaScript Array from the contents of this Set.
     * @return {Array.<T>} A copy of the Set in Array form.
     */
    toArray(): Array<T>;
    /**
     * Converts the Set to a List.
     * Because there is no ordering within a Set,
     * the values in the List may be in any order.
     * @return {List.<T>} A copy of the contents of this Set in List form.
     */
    toList(): List<T>;
    /**
     * This read-only property is the number of elements in the Set.
     */
    readonly count: number;
    /**
     * This read-only property is the number of elements in the Set.
     */
    readonly size: number;
    /**
     * Gets an object that you can use for iterating over the Set.
     * The value will be a member of the Set.
     * Typical usage:
     * ```js
     *   var it = aSet.iterator;
     *   while (it.next()) {
     *     . . . " value: " + it.value . . .
     *   }
     * ```
     */
    readonly iterator: Iterator<T>;
    /**
     * Undocumented.
     * Set the unique hash ID for an object.
     * This should be called at the beginning of each constructor that does not inherit from another class.
     */
    static uniqueHash(obj: ObjectData): void;
    /**
     * Undocumented.
     * Get the unique hash ID for an object, making it if necessary.
     */
    static hashIdUnique(obj: ObjectData): number;
    /**
     * Undocumented.
     * Get the unique hash ID for an object; may return undefined.
     */
    static hashId(obj: ObjectData): number | undefined;
}
/**
 * Undocumented
 * @template K,V
 * @unrestricted
 */
export class KeyValuePair<K, V> {
    /**
     * @param {K} k
     * @param {V} v
     */
    constructor(k: K, v: V);
    /**
     * @return {string}
     */
    toString(): string;
    /**
     * Gets a key for a value in a Map.
     * @return {K} the key uniquely identifying a #value in a Map.
     */
    key: K;
    /**
     * Gets a value in a Map.
     * @return {V} a value in a Map corresponding to a #key.
     */
    value: V;
}
/**
 * Undocumented.
 * Iterating over Maps is very similar to an Iterator<KeyValuePair<K, V>>,
 * but not exactly, because the type of the value property is V, not KeyValuePair<K, V>.
 */
export interface IMapIterator<K, T> {
    /** @hidden */
    iterator: IMapIterator<K, T>;
    /** @hidden */
    next(): boolean;
    /** @hidden */
    first(): KeyValuePair<K, T> | null;
    /** @hidden */
    reset(): void;
    /** @hidden */
    any(pred: (x: KeyValuePair<K, T>) => boolean): boolean;
    /** @hidden */
    all(pred: (x: KeyValuePair<K, T>) => boolean): boolean;
    /** @hidden */
    each(func: (x: KeyValuePair<K, T>) => void): IMapIterator<K, T>;
    /** @hidden */
    map<S>(func: (x: KeyValuePair<K, T>) => S): Iterator<S>;
    /** @hidden */
    filter(pred: (x: KeyValuePair<K, T>) => boolean): Iterator<KeyValuePair<K, T>>;
    /** @hidden */
    readonly key: K;
    /** @hidden */
    readonly value: T;
    /** @hidden */
    readonly count: number;
}
/**
 * <span style="color: red; font-weight: bold;">NOTE: For 2.0 the #constructor arguments have changed.
 * Map now optionally accepts a collection, and only checks types in TypeScript.</span>
 *
 * An unordered iterable collection of key/value pairs that cannot contain two instances of the same key.
 * In TypeScript it is a generic class that enforces at compile-time the type of the key and the type of the associated value.
 *
 * To create a Map:
 * ```js
 *   var map = new go.Map(); // In TypeScript: new go.Map<string, number>();
 *   map.add("one", 1);
 *   map.add("two", 2);
 *   map.add("three", 3);
 *   // now map.count === 3
 *   // and map.getValue("two") === 2
 *   // and map.contains("zero") === false
 * ```
 *
 * You can iterate over the key/value pairs in a Map:
 * ```js
 *   var it = aMap.iterator;
 *   while (it.next()) {
 *     console.log(it.key + ": " + it.value);
 *   }
 * ```
 * Or:
 * ```js
 *   aMap.each(function(kvp) {
 *     console.log(kvp.key + ": " + kvp.value);
 *   });
 * ```
 * But note that there is no guaranteed ordering amongst the key/value pairs.
 *
 * Call #toKeySet to get a read-only Set that holds all of the keys of a Map.
 * Iterating over that Set will produce values that are the keys in the Map.
 *
 * Although not precisely implementing the features and semantics of the EcmaScript 6 **Map** class,
 * this **GoJS** Map class has synonyms for the following methods and property:
 *   - **get(key)**: #getValue, but returns null instead of undefined when key is not present
 *   - **set(key,val)**: #add
 *   - **has(key)**: #contains
 *   - **delete(key)**: #remove
 *   - **clear()**: #clear
 *   - **size**: #count
 *
 * The constructor now takes an optional Iterable or Array argument that provides the initial entries for the new Map.
 *
 * Note that GoJS iteration is quite different than ES6 iteration, so that functionality has not been made somewhat compatible.
 * These collection classes were defined in GoJS before the ES6 collection classes were proposed.
 * @template K, V
 * @unrestricted
 * @category Collection
 */
export class Map<K, V> {
    /**
     * There are two possible constructors:
     *
     * `new go.Map()`, for JavaScript
     *
     * `new go.Map<K, V>()` for TypeScript
     *
     * In TypeScript, the two optional generic arguments describe the types of keys
     * and the types of values that this Map may hold.
     *
     * For example, the expression:
     * ```js
     * // TypeScript:
     * new go.Map<string, go.Point>()
     * ```
     * produces a Map that has keys that must be strings and whose associated values
     * must be Points.
     * @param {Iterable.<KeyValuePair.<K,V>>|Array.<KeyValuePair.<K,V>>=} coll an optional collection of keys/values to add, or an Array of { key: ..., value: ... } objects.
     * Note that the key/value pairs are objects with "key" and "value" properties, not Arrays of length 2.
     */
    constructor(coll?: Iterator<KeyValuePair<K, V>> | Array<KeyValuePair<K, V>>);
    /**
     * @return {string}
     */
    toString(): string;
    /**
     * Adds a key-value association to the Map, or replaces the value associated with the key
     * if the key was already present in the map.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {K} key The key or index for storing the value in the Map.
     * @param {V} val The value to add to the Map, associated with the key.
     * @return {Map.<K,V>} This modified Map.
     */
    add(key: K, val: V): Map<K, V>;
    /**
     * Adds a key-value association to the Map, or replaces the value associated with the key
     * if the key was already present in the map.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {K} key The key or index for storing the value in the Map.
     * @param {V} val The value to add to the Map, associated with the key.
     * @return {Map.<K,V>} This modified Map.
     */
    set(key: K, val: V): Map<K, V>;
    /**
     * Adds all of the key-value pairs of another Map to this Map.
     * If a key is already present in this Map,
     * its value is replaced with the corresponding value from the given map.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {Iterable.<KeyValuePair.<K,V>>|Array.<KeyValuePair.<K,V>>} coll the collection of keys/values to add, or an Array of { key: ..., value: ... } objects.
     * @return {Map.<K,V>} This modified Map.
     */
    addAll(coll: Iterator<KeyValuePair<K, V>> | Array<KeyValuePair<K, V>>): Map<K, V>;
    /**
     * Returns the first key/value pair in the collection, or null if there is none.
     * @return {KeyValuePair.<K,V>} This returns null if there are no items in the collection.
     * @since 1.4
     */
    first(): KeyValuePair<K, V>;
    /**
     * This is true if any invocation of the given predicate on items in the collection is true.
     *
     * Call the given predicate on each key/value pair in the collection.
     * As soon as a call returns true, this returns true.
     * Otherwise this returns false.
     * For an empty collection this returns false.
     * @expose
     * @param {function(KeyValuePair.<K,V>):boolean} pred The argument to the predicate will be an object with both "key" and "value" properties.
     * This function must not have any side-effects.
     * @return {boolean} True if any predicate call is true; false otherwise.
     * @since 1.4
     */
    any(pred: ((a: KeyValuePair<K, V>) => boolean)): boolean;
    /**
     * This is true if all invocations of the given predicate on items in the collection are true.
     *
     * Call the given predicate on each key/value pair in the collection.
     * As soon as a call returns false, this returns false.
     * Otherwise this returns true.
     * For an empty collection this returns true.
     * @expose
     * @param {function(KeyValuePair.<K,V>):boolean} pred The argument to the predicate will be an object with both "key" and "value" properties.
     * This function must not have any side-effects.
     * @return {boolean} True if all predicate calls are true; false otherwise.
     * @since 1.4
     */
    all(pred: ((a: KeyValuePair<K, V>) => boolean)): boolean;
    /**
     * Call the given function on each key/value pair in the collection.
     * @expose
     * @param {function(KeyValuePair.<K,V>)} func The argument to the function will be an object with both "key" and "value" properties.
     * This function must not modify the collection.
     * @return {Map.<K,V>} This Map itself
     * @since 1.4
     */
    each(func: ((a: KeyValuePair<K, V>) => void)): Map<K, V>;
    /**
     * Call the given function on each key-value pair in the collection and associate the key with the result of the function in a new Map.
     *
     * Unlike Iterator#map, this returns a Map, not an Iterator.
     * @expose
     * @param {function(KeyValuePair.<K,V>):*} func The argument to the function will be an object with both "key" and "value" properties.
     * This function must not modify the collection.
     * @return {Map.<K,S>} a new Map with the same keys but values produced by the function
     */
    map<S>(func: ((a: KeyValuePair<K, V>) => S)): Map<K, S>;
    /**
     * Call the given predicate on each key-value pair in the collection and for each pair that it returns true, add the key-value association in a new Map.
     *
     * Unlike Iterator#filter, this returns a Map, not an Iterator.
     * @expose
     * @param {function(KeyValuePair.<K,V>):boolean} pred This function must not have any side-effects.
     * @return {Map.<K,V>}
     */
    filter(pred: ((a: KeyValuePair<K, V>) => boolean)): Map<K, V>;
    /**
     * Returns whether the given key is in this Map.
     * @param {K} key The key to look up in the Map.
     * @return {boolean} Whether or not the key is contained within the Map.
     */
    contains(key: K): boolean;
    /**
     * Returns whether the given key is in this Map.
     * @param {K} key The key to look up in the Map.
     * @return {boolean} Whether or not the key is contained within the Map.
     */
    has(key: K): boolean;
    /**
     * Returns the value associated with a key.
     * @param {K} key The key to look up in the Map.
     * @return {V|null} The value associated with the given key, or null if not present in the Map.
     */
    getValue(key: K): V | null;
    /**
     * Returns the value associated with a key.
     * @param {K} key The key to look up in the Map.
     * @return {V|null} The value associated with the given key, or null if not present in the Map.
     */
    get(key: K): V | null;
    /**
     * Removes a key (if found) from the Map.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {K} key The key to insert.
     * @return {boolean} true if the key was found and removed, false otherwise.
     */
    remove(key: K): boolean;
    /**
     * Removes a key (if found) from the Map.
     *
     * Be careful not to call this method while iterating over the collection.
     * @param {K} key The key to insert.
     * @return {boolean} true if the key was found and removed, false otherwise.
     */
    delete(key: K): boolean;
    /**
     * Clears the Map, removing all key-value associations.
     * This sets the #count to zero.
     *
     * Be careful not to call this method while iterating over the collection.
     */
    clear(): void;
    /**
     * Makes a shallow copy of this Map.
     * The keys and their values are not copied,
     * so if they are objects they may continue to be shared with the original Map.
     * @expose
     * @return {Map.<K,V>} The new Map with copies of the same entries.
     */
    copy(): Map<K, V>;
    /**
     * Produces a JavaScript Array of key/value pair objects from the contents of this Map.
     * @return {Array.<KeyValuePair.<K,V>>} A copy of the Map in Array form,
     * each element being an Object with 'key' and 'value' properties.
     */
    toArray(): Array<KeyValuePair<K, V>>;
    /**
     * Produces a Set that provides a read-only view onto the keys of this Map.
     * The collection of keys is not copied.
     * @return {Set.<K>}
     */
    toKeySet(): Set<K>;
    /**
     * This read-only property is the number of associations in the Map.
     */
    readonly count: number;
    /**
     * This read-only property is the number of associations in the Map.
     */
    readonly size: number;
    /**
     * Gets an object that you can use for iterating over the key-value pairs of a Map.
     * Typical usage:
     * ```js
     *   var it = aMap.iterator;
     *   while (it.next()) {
     *     console.log("the key: " + it.key + " has value: " + it.value);
     *   }
     * ```
     */
    readonly iterator: IMapIterator<K, V>;
    /**
     * Gets an object that you can use for iterating over the keys of a Map.
     * Typical usage:
     * ```js
     *   var it = aMap.iteratorKeys;
     *   while (it.next()) {
     *     console.log("key: " + it.value);
     *   }
     * ```
     * @since 1.4
     */
    readonly iteratorKeys: Iterator<K>;
    /**
     * Gets an object that you can use for iterating over the values of a Map.
     * Typical usage:
     * ```js
     *   var it = aMap.iteratorValues;
     *   while (it.next()) {
     *     console.log("value: " + it.value);
     *   }
     * ```
     * @since 1.4
     */
    readonly iteratorValues: Iterator<V>;
}
/**
 * A Point represents an x- and y-coordinate pair in two-dimensional space.
 *
 * Use the static functions Point.parse and Point.stringify to convert to and from
 * a standard string representation that is independent of the current locale.
 *
 * When an instance of this class is the value of a property of a GraphObject class or Diagram
 * or CommandHandler or a Tool class, you should treat the object
 * as if it were frozen or read-only -- you cannot modify its properties.
 * This allows the property to return a value without allocating a new instance.
 * If you need to do your own calculations with the value, call #copy to make
 * a new instance with the same values that you can modify.
 *
 * Many methods modify the object's properties and then return a reference to "this" object.
 * The only instance method to allocate a new object is the #copy method.
 * The static Point.parse method also allocates a new object.
 *
 * The "Debug" implementation of this class is significantly slower than the "Release" implementation,
 * mostly due to additional error checking.
 *
 * You cannot inherit from this class.
 * @category Geometry
 */
export class Point {
    /**
     * The default constructor produces the Point(0,0).
     * This constructor may take either zero arguments or two arguments.
     * @param {number=} x The x value.
     * @param {number=} y The y value.
     */
    constructor(x?: number, y?: number);
    /**
     * Modify this Point with new X and Y values.
     * @param {number} x
     * @param {number} y
     * @return {Point} this.
     */
    setTo(x: number, y: number): Point;
    /**
     * Modify this Point so that its X and Y values are the same as the given Point.
     * @param {Point} p the given Point.
     * @return {Point} this.
     */
    set(p: Point): Point;
    /**
     * Create a copy of this Point, with the same values.
     * @expose
     * @return {Point}
     */
    copy(): Point;
    /**
     * This static function can be used to read in a Point from a string that was produced by Point.stringify.
     *
     * `go.Point.parse("1 2")` produces the Point `new go.Point(1, 2)`.
     * @param {string} str
     * @return {Point}
     */
    static parse(str: string): Point;
    /**
     * This static function can be used to write out a Point as a string that can be read by Point.parse.
     *
     * `go.Point.stringify(new go.Point(1, 2))` produces the string "1 2".
     * @param {Point} val
     * @return {string}
     */
    static stringify(val: Point): string;
    /**
     * Indicates whether the given Point is equal to this Point.
     * @param {Point} p The Point to compare to the current Point.
     * @return {boolean} True if the two Points have identical X and Y values,
     * false otherwise.
     * @see #equalTo
     */
    equals(p: Point): boolean;
    /**
     * Indicates whether the given point (x, y) is equal to this Point.
     * @param {number} x
     * @param {number} y
     * @return {boolean} True if the two Points have identical X and Y values,
     * false otherwise.
     * @see #equals
     */
    equalTo(x: number, y: number): boolean;
    /**
     * Undocumented
     * @param {Point} p The Point to compare to the current Point.
     * @return {boolean} True if the two Points have X and Y values that
     * are equal with a tolerance of 0.5, false otherwise.
     */
    equalsApprox(p: Point): boolean;
    /**
     * Modify this point so that is the sum of the current Point and the
     * x and y co-ordinates of the given Point.
     * @param {Point} p The Point to add to this Point.
     * @return {Point} this.
     * @see #offset
     * @see #subtract
     */
    add(p: Point): Point;
    /**
     * Modify this point so that is the difference of this Point and the
     * x and y co-ordinates of the given Point.
     * @param {Point} p The Point to subtract from the current Point.
     * @return {Point} this.
     * @see #offset
     * @see #add
     */
    subtract(p: Point): Point;
    /**
     * Modify this point by shifting its values with the given DX and DY offsets.
     * @param {number} dx
     * @param {number} dy
     * @return {Point} this.
     * @see #add
     * @see #subtract
     */
    offset(dx: number, dy: number): Point;
    /**
     * Modify this Point so that has been rotated about the origin by the given angle.
     * @param {number} angle an angle in degrees.
     * @return {Point} this.
     */
    rotate(angle: number): Point;
    /**
     * Modify this Point so that its X and Y values have been scaled by given factors along the X and Y axes.
     * @param {number} sx
     * @param {number} sy
     * @return {Point} this.
     */
    scale(sx: number, sy: number): Point;
    /**
     * Returns the square of the distance from this Point to a given Point.
     * @param {Point} p the other Point to measure to.
     * @return {number} the square of the euclidean distance.
     * @see #distanceSquared
     */
    distanceSquaredPoint(p: Point): number;
    /**
     * Returns the square of the distance from this point to a given point (px, py).
     * @param {number} px
     * @param {number} py
     * @return {number} the square of the euclidean distance.
     * @see #distanceSquaredPoint
     */
    distanceSquared(px: number, py: number): number;
    /**
     * Modify this Point so that its X and Y values have been normalized to a unit length.
     * However, if this Point is the origin (zero, zero), its length remains zero.
     * @return {Point} this.
     */
    normalize(): Point;
    /**
     * Compute the angle from this Point to a given Point.
     * However, if the given Point is the same as this Point, the direction is zero.
     * @param {Point} p the other Point to which to measure the relative angle.
     * @return {number} the angle, in degrees, of the line from this Point to the given point.
     * @see #direction
     */
    directionPoint(p: Point): number;
    /**
     * Compute the angle from this Point to a given (px,py) point.
     * However, if the point is the same as this Point, the direction is zero.
     * @param {number} px
     * @param {number} py
     * @return {number} the angle, in degrees, of the line from this Point to the given point.
     * @see #directionPoint
     */
    direction(px: number, py: number): number;
    /**
     * Modify this point to be the closest point to this point that is on a finite line segment.
     * @param {number} px one end of the finite line segment
     * @param {number} py one end of the finite line segment
     * @param {number} qx the other end of the finite line segment
     * @param {number} qy the other end of the finite line segment
     * @return {Point} this modified Point
     * @since 1.3
     */
    projectOntoLineSegment(px: number, py: number, qx: number, qy: number): Point;
    /**
     * Modify this point to be the closest point to this point that is on a finite line segment.
     * @param {Point} p one end of the finite line segment
     * @param {Point} q the other end of the finite line segment
     * @return {Point} this modified Point
     * @since 1.3
     */
    projectOntoLineSegmentPoint(p: Point, q: Point): Point;
    /**
     * Modify this point to be at the nearest point on an infinite grid,
     * given the grid's origin and size of each grid cell.
     * @param {number} originx the top-left point of one of the grid cells
     * @param {number} originy the top-left point of one of the grid cells
     * @param {number} cellwidth the size of each grid cell -- must be a real number larger than zero
     * @param {number} cellheight the size of each grid cell -- must be a real number larger than zero
     * @return {Point} this modified Point
     * @since 1.3
     */
    snapToGrid(originx: number, originy: number, cellwidth: number, cellheight: number): Point;
    /**
     * Modify this point to be at the nearest point on an infinite grid,
     * given the grid's origin and size of each grid cell.
     * @param {Point} origin the top-left point of one of the grid cells
     * @param {Size} cellsize the size of each grid cell
     * @return {Point} this modified Point
     * @since 1.3
     */
    snapToGridPoint(origin: Point, cellsize: Size): Point;
    /**
     * Modify this Point so that its X and Y values correspond to a particular Spot
     * in a given Rect.
     *
     * The result is meaningless if Spot#isNoSpot is true for the given Spot.
     * @param {Rect} r the Rect for which we are finding the point.
     * @param {Spot} spot the Spot; Spot#isSpot must be true for this Spot.
     * @return {Point} this.
     * @see #setSpot
     */
    setRectSpot(r: Rect, spot: Spot): Point;
    /**
     * Modify this Point so that its X and Y values correspond to a particular Spot
     * in a given rectangle.
     *
     * The result is meaningless if Spot#isNoSpot is true for the given Spot.
     * @param {number} x The X coordinate of the Rect for which we are finding the point.
     * @param {number} y The Y coordinate of the Rect for which we are finding the point.
     * @param {number} w The Width of the Rect for which we are finding the point.
     * @param {number} h The Height of the Rect for which we are finding the point.
     * @param {Spot} spot the Spot; Spot#isSpot must be true for this Spot.
     * @return {Point} this the point of the spot in the rectangle.
     * @see #setRectSpot
     */
    setSpot(x: number, y: number, w: number, h: number, spot: Spot): Point;
    /**
     * This static function returns the square of the distance from the point P
     * to the finite line segment from point A to point B.
     * @param {number} px
     * @param {number} py
     * @param {number} ax
     * @param {number} ay
     * @param {number} bx
     * @param {number} by
     * @return {number} the euclidean distance.
     */
    static distanceLineSegmentSquared(px: number, py: number, ax: number, ay: number, bx: number, by: number): number;
    /**
     * This static function returns the square of the distance from the point P to the point Q.
     * @param {number} px
     * @param {number} py
     * @param {number} qx
     * @param {number} qy
     * @return {number}
     */
    static distanceSquared(px: number, py: number, qx: number, qy: number): number;
    /**
     * This static function returns the angle in degrees of the line from point P to point Q.
     * @param {number} px
     * @param {number} py
     * @param {number} qx
     * @param {number} qy
     * @return {number}
     */
    static direction(px: number, py: number, qx: number, qy: number): number;
    /**
     * Gets or sets the x value of the Point.
     */
    x: number;
    /**
     * Gets or sets the y value of the Point.
     */
    y: number;
    /**
     * True if this Point has X and Y values that are real numbers and not infinity.
     * @return {boolean}
     */
    isReal(): boolean;
    /**
     * @expose
     * @return {Point}
     */
    static alloc(): Point;
    /**
     * @expose
     * @param {number} x
     * @param {number} y
     * @return {Point}
     */
    static allocAt(x: number, y: number): Point;
    /**
     * @expose
     * @param {Point} temp
     */
    static free(temp: Point): void;
}
/**
 * A Size describes a width and a height in two-dimensional coordinates.
 * The width and height must both be non-negative.
 *
 * Use the static functions Size.parse and Size.stringify to convert to and from
 * a standard string representation that is independent of the current locale.
 *
 * When an instance of this class is the value of a property of a GraphObject class or Diagram
 * or CommandHandler or a Tool class, you should treat the object
 * as if it were frozen or read-only -- you cannot modify its properties.
 * This allows the property to return a value without allocating a new instance.
 * If you need to do your own calculations with the value, call #copy to make
 * a new instance with the same values that you can modify.
 *
 * Many methods modify the object's properties and then return a reference to "this" object.
 * The only instance method to allocate a new object is the #copy method.
 * The static Size.parse method also allocates a new object.
 *
 * The "Debug" implementation of this class is significantly slower than the "Release" implementation,
 * mostly due to additional error checking.
 *
 * You cannot inherit from this class.
 * @category Geometry
 */
export class Size {
    /**
     * The default constructor produces the Size(0,0).
     * This constructor may take either zero arguments or two arguments.
     * @param {number=} w The initial width (must be non-negative).
     * @param {number=} h The initial height (must be non-negative).
     */
    constructor(w?: number, h?: number);
    /**
     * Modify this Size with new Width and Height values.
     * @param {number} w the width.
     * @param {number} h the height.
     * @return {Size} this.
     */
    setTo(w: number, h: number): Size;
    /**
     * Modify this Size so that its Width and Height values are the same as the given Size.
     * @param {Size} s the given Size.
     * @return {Size} this.
     */
    set(s: Size): Size;
    /**
     * Create a copy of this Size, with the same values.
     * @expose
     * @return {Size}
     */
    copy(): Size;
    /**
     * This static function can be used to read in a Size from a string that was produced by Size.stringify.
     *
     * `go.Size.parse("1 2")` produces the Size `new go.Size(1, 2)`.
     * @param {string} str
     * @return {Size}
     */
    static parse(str: string): Size;
    /**
     * This static function can be used to write out a Size as a string that can be read by Size.parse.
     *
     * `go.Size.stringify(new go.Size(1, 2))` produces the string "1 2".
     * @param {Size} val
     * @return {string}
     */
    static stringify(val: Size): string;
    /**
     * Indicates whether the given Size is equal to the current Size.
     * @param {Size} s The Size to compare to the current Size.
     * @return {boolean} True if the Sizes have the same width and height,
     * false otherwise.
     * @see #equalTo
     */
    equals(s: Size): boolean;
    /**
     * Indicates whether the given size is equal to this Size.
     * @param {number} w the width.
     * @param {number} h the height.
     * @return {boolean} True if the Sizes have the same width and height,
     * false otherwise.
     * @see #equals
     */
    equalTo(w: number, h: number): boolean;
    /**
     * Undocumented
     * @param {Size} s The Size to compare to the current Size.
     * @return {boolean} True if the two Sizes have Width and Height values that
     * are equal with a tolerance of 0.5, false otherwise.
     */
    equalsApprox(s: Size): boolean;
    /**
     * Gets or sets the width value of the Size.
     * The value must not be negative.
     */
    width: number;
    /**
     * Gets or sets the height value of the Size.
     * The value must not be negative.
     */
    height: number;
    /**
     * True if this Size has Width and Height values that are real numbers and not infinity.
     * @return {boolean}
     */
    isReal(): boolean;
    /**
     * @return {Size}
     */
    static alloc(): Size;
    /**
     * @param {Size} temp
     */
    static free(temp: Size): void;
}
/**
 * A Rect describes a rectangular two-dimensional area as a top-left point (x and y values)
 * and a size (width and height values).
 *
 * Use the static functions Rect.parse and Rect.stringify to convert to and from
 * a standard string representation that is independent of the current locale.
 *
 * When an instance of this class is the value of a property of a GraphObject class or Diagram
 * or CommandHandler or a Tool class, you should treat the object
 * as if it were frozen or read-only -- you cannot modify its properties.
 * This allows the property to return a value without allocating a new instance.
 * If you need to do your own calculations with the value, call #copy to make
 * a new instance with the same values that you can modify.
 *
 * Many methods modify the object's properties and then return a reference to "this" object.
 * The only instance method to allocate a new object is the #copy method.
 * The static Rect.parse method also allocates a new object.
 * The #center, #position, and #size properties all allocate and return a new object.
 *
 * The "Debug" implementation of this class is significantly slower than the "Release" implementation,
 * mostly due to additional error checking.
 *
 * You cannot inherit from this class.
 * @category Geometry
 */
export class Rect {
    /**
     * There are four constructors: (), (Point, Point), (Point, Size),
     * and four numbers for (x, y, width, height).
     * The default constructor (no argument) results in a Rect(0,0,0,0).
     * @param {Point|number=} x Top-left Point, or x value.
     * @param {Point|Size|number=} y Bottom-right Point or Size or y value.
     * @param {number=} w Width to be used if x,y are specified;
     * must be non-negative.
     * @param {number=} h Height to be used if x,y are specified;
     * must be non-negative.
     */
    constructor(x?: Point | number, y?: Point | Size | number, w?: number, h?: number);
    /**
     * Modify this Rect with new X, Y, Width, and Height values.
     * @param {number} x
     * @param {number} y
     * @param {number} w the width.
     * @param {number} h the height.
     * @return {Rect} this.
     */
    setTo(x: number, y: number, w: number, h: number): Rect;
    /**
     * Modify this Rect so that its X, Y, Width, and Height values are the same as the given Rect.
     * @param {Rect} r the given Rect.
     * @return {Rect} this.
     */
    set(r: Rect): Rect;
    /**
     * Modify this Rect so that its X and Y values are the same as the given Point.
     * @param {Point} p the given Point.
     * @return {Rect} this.
     */
    setPoint(p: Point): Rect;
    /**
     * Modify this Rect so that its Width and Height values are the same as the given Size.
     * @param {Size} s the given Size.
     * @return {Rect} this.
     */
    setSize(s: Size): Rect;
    /**
     * Create a copy of this Rect, with the same values.
     * @expose
     * @return {Rect}
     */
    copy(): Rect;
    /**
     * This static function can be used to read in a Rect from a string that was produced by Rect.stringify.
     *
     * `go.Rect.parse("1 2 3 4")` produces the Rect `new go.Rect(1, 2, 3, 4)`.
     * @param {string} str
     * @return {Rect}
     */
    static parse(str: string): Rect;
    /**
     * This static function can be used to write out a Rect as a string that can be read by Rect.parse.
     *
     * `go.Rect.stringify(new go.Rect(1, 2, 3, 4))` produces the string "1 2 3 4".
     * @param {Rect} val
     * @return {string}
     */
    static stringify(val: Rect): string;
    /**
     * Indicates whether the given Rect is equal to the current Rect.
     * @param {Rect} r The rectangle to compare to the current rectangle.
     * @return {boolean} True if the Rects are equivalent in x, y, width,
     * and height.
     * @see #equalTo
     */
    equals(r: Rect): boolean;
    /**
     * Indicates whether the given Rect is equal to the current Rect.
     * @param {number} x
     * @param {number} y
     * @param {number} w the width.
     * @param {number} h the height.
     * @return {boolean} True if the Rects are equivalent in x, y, width, and height.
     * @see #equals
     */
    equalTo(x: number, y: number, w: number, h: number): boolean;
    /**
     * Undocumented
     * @param {Rect} r The Rect to compare to the current Rect.
     * @return {boolean} True if the two Rects have respective X, Y, Width,
     * and Height values within 0.5, false otherwise.
     */
    equalsApprox(r: Rect): boolean;
    /**
     * Indicates whether this Rect contains the given Point.
     * @param {Point} p The Point to check.
     * @return {boolean} True if the Point is contained within this Rect,
     * false otherwise.
     * @see #containsRect
     * @see #contains
     */
    containsPoint(p: Point): boolean;
    /**
     * Indicates whether this Rect contains the given Rect.
     * @param {Rect} r The Rect to check.
     * @return {boolean} True if the Rect is contained within this Rect,
     * false otherwise.
     * @see #containsPoint
     * @see #contains
     */
    containsRect(r: Rect): boolean;
    /**
     * Indicates whether this Rect contains the given Point/Rect.
     * @param {number} x The X coordinate of the Point or Rect to include in the new bounds.
     * @param {number} y The Y coordinate of the Point or Rect to include in the new bounds.
     * @param {number=} w The Width of the Rect to include in the new bounds, defaults to zero.
     * @param {number=} h The Height of the Rect to include in the new bounds, defaults to zero.
     * @return {boolean} True if the Point/Rect is contained within this Rect,
     * false otherwise.
     * @see #containsRect
     * @see #containsPoint
     */
    contains(x: number, y: number, w?: number, h?: number): boolean;
    /**
     * Modify this Rect by shifting its values with the given DX and DY offsets.
     * @param {number} dx
     * @param {number} dy
     * @return {Rect} this.
     */
    offset(dx: number, dy: number): Rect;
    /**
     * Modify this Rect so that its width and height are changed on all four sides,
     * equally on the left and right sides, and equally on the top and bottom sides.
     * When the arguments are negative, this operation deflates this Rect, but not beyond zero.
     * @param {number} w The additional width on each side, left and right; may be negative.
     * @param {number} h The additional height on each side, top and bottom; may be negative.
     * @return {Rect} this.
     * @see #grow
     * @see #addMargin
     * @see #subtractMargin
     */
    inflate(w: number, h: number): Rect;
    /**
     * Modify this Rect by adding the given Margin to each side of the Rect.
     * @param {Margin} m The Margin to add to the Rect.
     * @return {Rect} this bigger Rect.
     * @see #subtractMargin
     * @see #grow
     * @see #inflate
     */
    addMargin(m: Margin): Rect;
    /**
     * Modify this Rect by subtracting the given Margin from each side of the Rect.
     * @param {Margin} m The Margin to subtract from the Rect.
     * @return {Rect} this smaller Rect.
     * @see #addMargin
     * @see #grow
     * @see #inflate
     */
    subtractMargin(m: Margin): Rect;
    /**
     * Modifies this Rect by adding some distance to each side of the Rect.
     * @param {number} t the amount to move the top side upwards; may be negative.
     * @param {number} r the amount to move the right side rightwards; may be negative.
     * @param {number} b the amount to move the bottom side downwards; may be negative.
     * @param {number} l the amount to move the left side leftwards; may be negative.
     * @return {Rect} this modified Rect.
     * @see #inflate
     * @see #addMargin
     * @see #subtractMargin
     */
    grow(t: number, r: number, b: number, l: number): Rect;
    /**
     * Modify this Rect so that it is the intersection of this Rect and the given Rect.
     * @param {Rect} r Rect to intersect with.
     * @return {Rect} this.
     * @see #intersect
     * @see #intersects
     */
    intersectRect(r: Rect): Rect;
    /**
     * Modify this Rect so that it is the intersection of this Rect and the rectangle
     * defined by x, y, w, h.
     * @param {number} x
     * @param {number} y
     * @param {number} w
     * @param {number} h
     * @return {Rect} this.
     * @see #intersectRect
     * @see #intersects
     */
    intersect(x: number, y: number, w: number, h: number): Rect;
    /**
     * Determine if a given Rect is partly or wholly inside of this Rect.
     * @param {Rect} r Rect to test intersection with.
     * @return {boolean} true if there is an intersection.
     * @see #intersects
     * @see #intersectRect
     */
    intersectsRect(r: Rect): boolean;
    /**
     * Determine if this Rect partly or wholly overlaps the rectangle
     * defined by x, y, w, h.
     * @param {number} x
     * @param {number} y
     * @param {number} w
     * @param {number} h
     * @return {boolean} true if there is any overlap.
     * @see #intersectsRect
     * @see #intersect
     */
    intersects(x: number, y: number, w: number, h: number): boolean;
    /**
     * Modify this Rect to be exactly big enough to contain both the original Rect and the given Point.
     * @param {Point} p The Point to include in the new bounds.
     * @return {Rect} this.
     * @see #unionRect
     * @see #union
     */
    unionPoint(p: Point): Rect;
    /**
     * Modify this Rect to be exactly big enough to contain this Rect and the given Rect.
     * @param {Rect} r The Rect to include in the new bounds.
     * @return {Rect} this.
     * @see #unionPoint
     * @see #union
     */
    unionRect(r: Rect): Rect;
    /**
     * Modify this Rect to be exactly big enough to contain both the original Rect and the given rectangular area.
     * @param {number} x The X coordinate of the Point or Rect to include in the new bounds.
     * @param {number} y The Y coordinate of the Point or Rect to include in the new bounds.
     * @param {number=} w The Width of the Rect to include in the new bounds, defaults to zero.
     * @param {number=} h The Height of the Rect to include in the new bounds, defaults to zero.
     * @return {Rect} this.
     * @see #unionRect
     * @see #unionPoint
     */
    union(x: number, y: number, w?: number, h?: number): Rect;
    /**
     * Modify this Rect so that a given Spot is at a given (x,y) point using this Rect's size.
     * Return this rectangle for which the spot is at that point, without modifying the size.
     *
     * The result is meaningless if Spot#isNoSpot is true.
     * @param {number} x the point where the spot should be.
     * @param {number} y the point where the spot should be.
     * @param {Spot} spot a Spot; Spot#isSpot must be true.
     * @return {Rect} this.
     */
    setSpot(x: number, y: number, spot: Spot): Rect;
    /**
     * This static function indicates whether a Rect contains the given Point/Rect.
     * @param {number} rx The X coordinate of a Rect.
     * @param {number} ry The Y coordinate of a Rect.
     * @param {number} rw The Width of a Rect.
     * @param {number} rh The Height of a Rect.
     * @param {number} x The X coordinate of the Point or Rect that might be in the bounds (RX, RY, RW, RH).
     * @param {number} y The Y coordinate of the Point or Rect that might be in the bounds (RX, RY, RW, RH).
     * @param {number=} w The Width of the Rect to include in the new bounds, defaults to zero.
     * @param {number=} h The Height of the Rect to include in the new bounds, defaults to zero.
     * @return {boolean} True if the Point/Rect is contained within this Rect,
     * false otherwise.
     */
    static contains(rx: number, ry: number, rw: number, rh: number, x: number, y: number, w?: number, h?: number): boolean;
    /**
     * This static function indicates whether a Rect partly or wholly overlaps the given Rect.
     * @param {number} rx The X coordinate of a Rect.
     * @param {number} ry The Y coordinate of a Rect.
     * @param {number} rw The Width of a Rect.
     * @param {number} rh The Height of a Rect.
     * @param {number} x The X coordinate of the Point or Rect that might overlap the bounds (RX, RY, RW, RH).
     * @param {number} y The Y coordinate of the Point or Rect that might overlap the bounds (RX, RY, RW, RH).
     * @param {number} w
     * @param {number} h
     * @return {boolean} true if there is any overlap.
     */
    static intersects(rx: number, ry: number, rw: number, rh: number, x: number, y: number, w: number, h: number): boolean;
    /**
     * Gets or sets the top-left x coordinate of the Rect.
     */
    x: number;
    /**
     * Gets or sets the top-left y coordinate of the Rect.
     */
    y: number;
    /**
     * Gets or sets the width of the Rect.
     * The value must not be negative.
     */
    width: number;
    /**
     * Gets or sets the height of the Rect.
     * The value must not be negative.
     */
    height: number;
    /**
     * Gets or sets the leftmost value of the Rect.
     * This is the same as the X value.
     */
    left: number;
    /**
     * Gets or sets the topmost value of the Rect.
     * This is the same as the Y value.
     */
    top: number;
    /**
     * Gets or sets the x-axis value of the right of the Rect.
     * This is equal to the sum of the x value and the width.
     */
    right: number;
    /**
     * Gets or sets the y-axis value of the bottom of the Rect.
     * This is equal to the sum of the y value and the height.
     */
    bottom: number;
    /**
     * Gets or sets the x- and y-axis position of the Rect as a Point.
     */
    position: Point;
    /**
     * Gets or sets the width and height of the Rect as a Size.
     */
    size: Size;
    /**
     * Gets or sets the Point at the center of this Rect.
     * Setting this property just shifts the X and Y values.
     */
    center: Point;
    /**
     * Gets or sets the horizontal center X coordinate of the Rect.
     */
    centerX: number;
    /**
     * Gets or sets the vertical center Y coordinate of the Rect.
     */
    centerY: number;
    /**
     * True if this Rect has X, Y, Width, and Height values that are real numbers and not infinity.
     * @return {boolean}
     */
    isReal(): boolean;
    /**
     * True if this Rect has a Width and Height of zero.
     * @return {boolean}
     */
    isEmpty(): boolean;
    /**
     * @return {Rect}
     */
    static alloc(): Rect;
    /**
     * @param {number} x
     * @param {number} y
     * @param {number} w
     * @param {number} h
     * @return {Rect}
     */
    static allocAt(x: number, y: number, w: number, h: number): Rect;
    /**
     * @param {Rect} temp
     */
    static free(temp: Rect): void;
}
/**
 * The MarginLike type is the same as `Margin | number`.
 * A number in place of a Margin object is treated as a uniform Margin with that thickness.
 * @category Type
 */
export type MarginLike = Margin | number;
/**
 * A Margin represents a band of space outside or inside a rectangular area,
 * with possibly different values on each of the four sides.
 *
 * Example uses include GraphObject#margin, Panel#padding, and Diagram#padding.
 *
 * Use the static functions Margin.parse and Margin.stringify to convert to and from
 * a standard string representation that is independent of the current locale.
 *
 * When an instance of this class is the value of a property of a GraphObject class or Diagram
 * or CommandHandler or a Tool class, you should treat the object
 * as if it were frozen or read-only -- you cannot modify its properties.
 * This allows the property to return a value without allocating a new instance.
 * If you need to do your own calculations with the value, call #copy to make
 * a new instance with the same values that you can modify.
 *
 * Many methods modify the object's properties and then return a reference to "this" object.
 * The only instance method to allocate a new object is the #copy method.
 * The static Margin.parse method also allocates a new object.
 *
 * The "Debug" implementation of this class is significantly slower than the "Release" implementation,
 * mostly due to additional error checking.
 *
 * You cannot inherit from this class.
 * @category Geometry
 */
export class Margin {
    /**
     * If zero arguments are supplied, zero is used for all four sides.
     * If one argument is supplied, that number is used for all four sides.
     * If two arguments are supplied, the top and bottom sides get the first value,
     * and the left and right sides get the second value.
     * Otherwise there must be four arguments, in the order: top, right, bottom, left.
     * @param {number=} t the margin for the top side;
     *                    if not supplied, all sides are zero.
     * @param {number=} r the margin for the right side;
     *                    if not supplied, all sides have the value of the first argument.
     * @param {number=} b the margin for all bottom side;
     *                    if not supplied, the top and bottom get the value of the first argument,
     *                    and the right and left sides get the value of the second argument.
     * @param {number=} l the margin for the left side;
     *                    must be supplied if the third argument was supplied.
     */
    constructor(t?: number, r?: number, b?: number, l?: number);
    /**
     * Modify this Margin with new Top, Right, Bottom, and Left values.
     * @param {number} t top.
     * @param {number} r right.
     * @param {number} b bottom.
     * @param {number} l left.
     * @return {Margin} this.
     */
    setTo(t: number, r: number, b: number, l: number): Margin;
    /**
     * Modify this Margin so that its Top, Right, Bottom, and Left values
     * are the same as the given Margin.
     * @param {Margin} m the given Margin.
     * @return {Margin} this.
     */
    set(m: Margin): Margin;
    /**
     * Create a copy of this Margin, with the same values.
     * @expose
     * @return {Margin}
     */
    copy(): Margin;
    /**
     * This static function can be used to read in a Margin from a string that was produced by Margin.stringify.
     *
     * `go.Margin.parse("1 2 3 4")` produces the Margin `new go.Margin(1, 2, 3, 4)`.
     * @param {string} str
     * @return {Margin}
     */
    static parse(str: string): Margin;
    /**
     * This static function can be used to write out a Margin as a string that can be read by Margin.parse.
     *
     * `go.Margin.stringify(new go.Margin(1, 2, 3, 4))` produces the string "1 2 3 4".
     * @param {Margin} val
     * @return {string}
     */
    static stringify(val: Margin): string;
    /**
     * Indicates whether the given Margin is equal to this Margin.
     * @param {Margin} m The Margin to compare to this Margin.
     * @return {boolean} True if the two Margins have identical Top and Right and Bottom and Left values,
     * false otherwise.
     * @see #equalTo
     */
    equals(m: Margin): boolean;
    /**
     * Indicates whether the given margin is equal to this Margin.
     * @param {number} t top.
     * @param {number} r right.
     * @param {number} b bottom.
     * @param {number} l left.
     * @return {boolean} True if the two Margins have identical Top and Right and Bottom and Left values,
     * false otherwise.
     * @see #equals
     */
    equalTo(t: number, r: number, b: number, l: number): boolean;
    /**
     * Undocumented
     * @param {Margin} m The Margin to compare to the current Margin.
     * @return {boolean} True if the two Margins have Top, Right,
     * Bottom and Left values within 0.5, false otherwise.
     */
    equalsApprox(m: Margin): boolean;
    /**
     * Gets or sets the top value of this margin. Default is 0.
     */
    top: number;
    /**
     * Gets or sets the right value of this margin. Default is 0.
     */
    right: number;
    /**
     * Gets or sets the bottom value of this margin. Default is 0.
     */
    bottom: number;
    /**
     * Gets or sets the left value of this margin. Default is 0.
     */
    left: number;
    /**
     * True if this Margin has values that are real numbers and not infinity.
     * @return {boolean}
     */
    isReal(): boolean;
    /**
     * @return {Margin}
     */
    static alloc(): Margin;
    /**
     * @param {Margin} temp
     */
    static free(temp: Margin): void;
}
/**
 * A Spot represents a relative point from (0,0) to (1,1) within the bounds of
 * a rectangular area plus an absolute offset.
 *
 * Use the static functions Spot.parse and Spot.stringify to convert to and from
 * a standard string representation that is independent of the current locale.
 *
 * When an instance of this class is the value of a property of a GraphObject class or Diagram
 * or CommandHandler or a Tool class, you should treat the object
 * as if it were frozen or read-only -- you cannot modify its properties.
 * This allows the property to return a value without allocating a new instance.
 * If you need to do your own calculations with the value, call #copy to make
 * a new instance with the same values that you can modify.
 *
 * Many methods modify the object's properties and then return a reference to "this" object.
 * The only instance method to allocate a new object is the #copy method.
 * The static Spot.parse method also allocates a new object.
 *
 * The "Debug" implementation of this class is significantly slower than the "Release" implementation,
 * mostly due to additional error checking.
 *
 * You cannot inherit from this class.
 * @category Geometry
 */
export class Spot {
    /**
     * The default constructor produces the Spot(0, 0, 0, 0), at the top-left corner.
     * @param {number=} x The x value of the Spot. Default is zero.
     * @param {number=} y The y value of the Spot. Default is zero.
     * @param {number=} offx (Optional) The absolute x offset. Default is zero.
     * @param {number=} offy (Optional) The absolute y offset. Default is zero.
     */
    constructor(x?: number, y?: number, offx?: number, offy?: number);
    /**
     * Modify this Spot with new X, Y, OffsetX, and OffsetY values.
     * @param {number} x
     * @param {number} y
     * @param {number} offx
     * @param {number} offy
     * @return {Spot} this.
     */
    setTo(x: number, y: number, offx: number, offy: number): Spot;
    /**
     * Modify this Spot so that its X, Y, OffsetX, and OffsetY values are the same as the given Spot.
     * @param {Spot} s the given Spot.
     * @return {Spot} this.
     */
    set(s: Spot): Spot;
    /**
     * Create a copy of this Spot, with the same values.
     * @expose
     * @return {Spot}
     */
    copy(): Spot;
    /**
     * This static function can be used to read in a Spot from a string that was produced by Spot.stringify.
     *
     * `go.Spot.parse("0 1 2 3")` produces the Spot `new go.Spot(0, 1, 2, 3)`.
     * @param {string} str
     * @return {Spot}
     */
    static parse(str: string): Spot;
    /**
     * This static function can be used to write out a Spot as a string that can be read by Spot.parse.
     *
     * `go.Spot.stringify(new go.Spot(0, 1, 2, 3))` produces the string "0 1 2 3".
     * @param {Spot} val
     * @return {string}
     */
    static stringify(val: Spot): string;
    /**
     * Two spots are equal if all four property values are the same.
     * @param {Spot} spot The Spot to compare to the current Spot.
     * @return {boolean} True if the two spots are equal, false otherwise.
     */
    equals(spot: Spot): boolean;
    /**
     * Return a new spot that is opposite this spot.
     * The X and Y values will be an equal distance
     * away from the center on the other side of the center.
     * The OffsetX and OffsetY values are also negated.
     *
     * The result is meaningless if #isNoSpot is true.
     * @return {Spot}
     */
    opposite(): Spot;
    /**
     * This predicate is true if this Spot is a side that
     * includes the side(s) given by the argument Spot.
     * @param {Spot} side
     * @return {boolean} false if either this Spot or the argument Spot is not a "Side";
     * true if the side(s) that this Spot represents are a superset or the same set of
     * sides represented by the argument Spot.
     */
    includesSide(side: Spot): boolean;
    /**
     * Gets or sets the x value of the Spot, a fractional value between zero and one.
     *
     * The Spot(0,0) of a rectangle is at the top-left corner.
     * This value is commonly available as Spot.TopLeft.
     * The Spot(0.5,1) would be at the middle of the bottom side.
     * That value is commonly available as Spot.MiddleBottom.
     */
    x: number;
    /**
     * Gets or sets the y value of the Spot, a fractional value between zero and one.
     *
     * The Spot(0,1) of a rectangle is at the bottom-left corner.
     * This value is commonly available as Spot.BottomLeft.
     * The Spot(1,0.5) would be at the middle of the right side.
     * That value is commonly available as Spot.MiddleRight.
     */
    y: number;
    /**
     * Gets or sets the offsetX value of the Spot.
     * The value may be negative.
     * If this represents a side value instead of a specific spot, this value is meaningless.
     *
     * A Spot(0,0,5,5) of a large rectangle is near the top-left corner, inside the rectangle.
     * A Spot(1,1,5,5) would be near the bottom-right corner, outside the rectangle.
     * A Spot(1,1,-5,-5) would also be near the bottom-right corner, but inside the rectangle.
     */
    offsetX: number;
    /**
     * Gets or sets the offsetY value of the Spot.
     * The value may be negative.
     * If this represents a side value instead of a specific spot, this value is meaningless.
     *
     * A Spot(0,0,5,5) of a large rectangle is near the top-left corner, inside the rectangle.
     * A Spot(1,1,5,5) would be near the bottom-right corner, outside the rectangle.
     * A Spot(1,1,-5,-5) would also be near the bottom-right corner, but inside the rectangle.
     */
    offsetY: number;
    /**
     * True if this is a specific spot, not a side nor Spot.None.
     * @return {boolean}
     */
    isSpot(): boolean;
    /**
     * True if this is an unspecific special spot, such as Spot.None
     * or one of the sides.
     * @return {boolean}
     */
    isNoSpot(): boolean;
    /**
     * True if this is a special spot referring to one (or more) of the sides.
     * This is false if the spot is Spot.None.
     * @return {boolean}
     */
    isSide(): boolean;
    /**
     * True if this is a special spot referring to no particular spot or side.
     * @return {boolean}
     */
    isNone(): boolean;
    /**
     * True if this is a special spot indicating that the real spot value will come from another source.
     * @return {boolean}
     */
    isDefault(): boolean;
    /**
     * Use this Spot value to indicate no particular spot --
     * code looking for a particular point on an element will need to do their
     * own calculations to determine the desired point depending on the
     * circumstances.
     * @constant
     */
    static None: Spot;
    /**
     * Use this value to indicate that the real spot value is inherited from elsewhere.
     * @constant
     */
    static Default: Spot;
    /**
     * The specific point at the top-left corner of the bounding rectangle.
     * @constant
     */
    static TopLeft: Spot;
    /**
     * The specific point at the center of the top side of the bounding rectangle.
     * @constant
     */
    static TopCenter: Spot;
    /**
     * The specific point at the top-right corner of the bounding rectangle.
     * @constant
     */
    static TopRight: Spot;
    /**
     * The specific point at the middle of the left side of bounding rectangle.
     * @constant
     */
    static LeftCenter: Spot;
    /**
     * The specific point at the very center of the bounding rectangle.
     * @constant
     */
    static Center: Spot;
    /**
     * The specific point at the middle of the right side of bounding rectangle.
     * @constant
     */
    static RightCenter: Spot;
    /**
     * The specific point at the bottom-left corner of the bounding rectangle.
     * @constant
     */
    static BottomLeft: Spot;
    /**
     * The specific point at the middle of the bottom side of bounding rectangle.
     * @constant
     */
    static BottomCenter: Spot;
    /**
     * The specific point at the bottom-right corner of the bounding rectangle.
     * @constant
     */
    static BottomRight: Spot;
    /**
     * A synonym for Spot.TopCenter.
     * @constant
     */
    static MiddleTop: Spot;
    /**
     * A synonym for Spot.LeftCenter.
     * @constant
     */
    static MiddleLeft: Spot;
    /**
     * A synonym for Spot.RightCenter.
     * @constant
     */
    static MiddleRight: Spot;
    /**
     * A synonym for Spot.BottomCenter.
     * @constant
     */
    static MiddleBottom: Spot;
    /**
     * A synonym for Spot.TopCenter.
     * @constant
     */
    static Top: Spot;
    /**
     * A synonym for Spot.LeftCenter.
     * @constant
     */
    static Left: Spot;
    /**
     * A synonym for Spot.RightCenter.
     * @constant
     */
    static Right: Spot;
    /**
     * A synonym for Spot.BottomCenter.
     * @constant
     */
    static Bottom: Spot;
    /**
     * The set of points at the top side of the bounding rectangle.
     * @constant
     */
    static TopSide: Spot;
    /**
     * The set of points at the left side of the bounding rectangle.
     * @constant
     */
    static LeftSide: Spot;
    /**
     * The set of points at the right side of the bounding rectangle.
     * @constant
     */
    static RightSide: Spot;
    /**
     * The set of points at the bottom side of the bounding rectangle.
     * @constant
     */
    static BottomSide: Spot;
    /**
     * The set of points at the top or bottom sides of the bounding rectangle.
     * @constant
     */
    static TopBottomSides: Spot;
    /**
     * The set of points at the left or right sides of the bounding rectangle.
     * @constant
     */
    static LeftRightSides: Spot;
    /**
     * The set of points at the top or left sides of the bounding rectangle.
     * @constant
     */
    static TopLeftSides: Spot;
    /**
     * The set of points at the top or right sides of the bounding rectangle.
     * @constant
     */
    static TopRightSides: Spot;
    /**
     * The set of points at the left or bottom sides of the bounding rectangle.
     * @constant
     */
    static BottomLeftSides: Spot;
    /**
     * The set of points at the right or bottom sides of the bounding rectangle.
     * @constant
     */
    static BottomRightSides: Spot;
    /**
     * The set of points on all sides of the bounding rectangle except top side.
     * @constant
     */
    static NotTopSide: Spot;
    /**
     * The set of points on all sides of the bounding rectangle except left side.
     * @constant
     */
    static NotLeftSide: Spot;
    /**
     * The set of points on all sides of the bounding rectangle except right side.
     * @constant
     */
    static NotRightSide: Spot;
    /**
     * The set of points on all sides of bounding rectangle except bottom side.
     * @constant
     */
    static NotBottomSide: Spot;
    /**
     * The set of points on all sides of the bounding rectangle.
     * @constant
     */
    static AllSides: Spot;
    /**
     * @return {Spot}
     */
    static alloc(): Spot;
    /**
     * @param {Spot} temp
     */
    static free(temp: Spot): void;
}
/**
 * The Geometry class is used to define the "shape" of a Shape.
 * A Geometry can be simple straight lines, rectangles, or ellipses.
 * A Geometry can also be an arbitrarily complex path, consisting of a list of PathFigures.
 *
 * A Geometry must not be modified once it has been used by a Shape.
 * However, a Geometry may be shared by multiple Shapes.
 *
 * It is commonplace to create Geometries using geometry path string syntax:
 * <a href="../../intro/geometry.html">Geometry Path Strings</a>.
 * However it is much more efficient to create Geometries programmatically.
 * One way to do that is illustrated by several of the samples that evaluate JavaScript such as:
 * ```js
 * new go.Geometry()
 *     .add(new go.PathFigure(p.x, p.y)
 *          .add(new go.PathSegment(go.PathSegment.Arc, -sweep/2, sweep, 0, 0, radius+layerThickness, radius+layerThickness))
 *          .add(new go.PathSegment(go.PathSegment.Line, q.x, q.y))
 *          .add(new go.PathSegment(go.PathSegment.Arc, sweep/2, -sweep, 0, 0, radius, radius).close()));
 * ```
 * See samples that make use of Geometries in the <a href="../../samples/index.html#geometries">samples index</a>.
 * @unrestricted
 * @category Geometry
 */
export class Geometry {
    /**
     * @param {?=} type
     * Construct an empty Geometry.
     * The geometry type must be one of the following values:
     * Geometry.Line, Geometry.Ellipse, Geometry.Rectangle, Geometry.Path.
     */
    constructor(type?: EnumValue);
    /**
     * Create a copy of this Geometry, with the same values and figures.
     * @expose
     * @return {Geometry}
     */
    copy(): Geometry;
    /**
     * For drawing a simple straight line;
     * a value for Geometry#type.
     * The line goes from the point
     * (#startX, #startY) to the point (#endX, #endY).
     * @constant
     */
    static Line: EnumValue;
    /**
     * For drawing a rectangle;
     * a value for Geometry#type.
     * The rectangle goes from the point
     * (#startX, #startY) to the point (#endX, #endY).
     * @constant
     */
    static Rectangle: EnumValue;
    /**
     * For drawing an ellipse fitting within a rectangle;
     * a value for Geometry#type.
     * The ellipse fits within the rectangle that goes from the point
     * (#startX, #startY) to the point (#endX, #endY).
     * @constant
     */
    static Ellipse: EnumValue;
    /**
     * For drawing a complex path made of a list of PathFigures;
     * a value for Geometry#type.
     * @constant
     */
    static Path: EnumValue;
    /**
     * Undocumented
     * @param {Geometry} g
     * @return {boolean}
     */
    equalsApprox(g: Geometry): boolean;
    /**
     * This static function can be used to write out a Geometry as a string
     * that can be read by Geometry.parse.
     *
     * The string produced by this method is a superset of the SVG path
     * string rules that contains some additional GoJS-specific tokens.
     * See the <a href="../../intro/pictures.html">Introduction page on Geometry Parsing</a> for more details.
     * @param {Geometry} val
     * @return {string}
     * @since 1.1
     */
    static stringify(val: Geometry): string;
    /**
     * Given a SVG or GoJS path string, returns a congruent path string with each pathfigure filled.
     * For instance, "M0 0 L22 22 L33 0" would become "F M0 0 L22 22 L33 0".
     * @param {string} str
     * @return {string}
     * @since 1.1
     */
    static fillPath(str: string): string;
    /**
     * Produce a Geometry from a string that uses an SVG-like compact path geometry syntax.
     * The syntax accepts all SVG Path Syntax (<a href="https://www.w3.org/TR/SVG/paths.html" target="_blank">SVG Path Syntax (w3.org)</a>),
     * as well as three GoJS-specific tokens.
     * Specifically, the following tokens are allowed:
     *   - `M (x,y)+ ` - Move commands
     *   - `L (x,y)+, H (x)+, V (y)+` Lines commands, including horizontal and vertical lines
     *   - `C (x1 y1 x2 y2 x y)+, S (x2 y2 x y)+` Cubic bezier curves
     *   - `Q (x1 y1 x y)+, T (x y)+` Quadratic bezier curves
     *   - `A (rx ry x-axis-rotation large-arc-flag clockwise-flag x y)+` Arcs (following SVG arc conventions)
     *   - `Z` Denotes that the previous subpath is closed
     *
     * Additionally there are some tokens specific to GoJS:
     *   - `B (startAngle, sweepAngle, centerX, centerY, radius)+` Arcs following GoJS canvas arc conventions
     *   - `X` Used before M-commands to denote separate PathFigures instead of a subpath
     *   - `F` Denotes whether the current PathFigure is filled (true if F is present)
     *   - `U` Denotes that the PathFigure is not shadowed
     *
     * See the <a href="../../intro/geometry.html">Introduction page on Geometry Parsing</a> for more details.
     * @param {string} str
     * @param {boolean=} filled whether figures should be filled.
     * If true, all PathFigures in the string will be filled regardless of the presence
     * of an "F" command or not.
     * If false, all PathFigures will determine their own filled state by the presence of an "F" command or not.
     * Default is false.
     * @return {Geometry}
     */
    static parse(str: string, filled?: boolean): Geometry;
    /**
     * Computes the Geometry's bounds without adding an origin point, and returns those bounds as a rect.
     * This method does not modify the Geometry or its bounds.
     * @return {Rect}
     * @since 1.1
     */
    computeBoundsWithoutOrigin(): Rect;
    /**
     * Normalizes the Geometry points in place by ensuring the top-left bounds of the geometry lines up with (0, 0),
     * returning the Point (x, y) amount it was shifted.
     * After this method is called there will be no negative X or Y value for the Geometry's bounds,
     * and no empty space at the top-left of the Geometry.
     * @return {Point}
     * @since 1.1
     */
    normalize(): Point;
    /**
     * Offsets the Geometry in place by a given (x, y) amount
     * @param {number} x The x-axis offset factor.
     * @param {number} y The y-axis offset factor.
     * @return {Geometry} this
     * @since 1.1
     */
    offset(x: number, y: number): Geometry;
    /**
     * Scales the Geometry in place by a given (x, y) scale factor.
     *
     * If you want to flip a Geometry horizontally, call `geo.scale(-1, 1)`.
     * If you want to flip a Geometry vertically, call `geo.scale(1, -1)`.
     * @param {number} x The x-axis scale factor.
     * @param {number} y The y-axis scale factor.
     * @return {Geometry} this
     * @since 1.1
     */
    scale(x: number, y: number): Geometry;
    /**
     * Rotates the Geometry in place by a given angle, with optional x and y values to rotate the geometry about.
     * If no x and y value are given, (0, 0) is used as the rotation point.
     * @param {number} angle The angle to rotate by.
     * @param {number=} x The optional X point to rotate the geometry about. If no point is given, this value is 0.
     * @param {number=} y The optional Y point to rotate the geometry about. If no point is given, this value is 0.
     * @return {Geometry} this
     * @since 1.1
     */
    rotate(angle: number, x?: number, y?: number): Geometry;
    /**
     * Undocumented
     * @param {number} fraction A fractional amount between 0 and 1, inclusive.
     * @param {Point=} result an optional Point that is modified and returned.
     * @return {Point}
     */
    getPointAlongPath(fraction: number, result?: Point): Point;
    /**
     * Undocumented
     * @param {Point} pt A Point near the this Geometry.
     * @return {number}
     */
    getFractionForPoint(pt: Point): number;
    /**
     * Gets or sets the type of the Geometry.
     * The default type is Geometry.Path.
     * Other permissible values are Geometry.Line, Geometry.Ellipse,
     * or Geometry.Rectangle.
     */
    type: EnumValue;
    /**
     * Gets or sets the starting X coordinate of the Geometry if it is of type
     * Geometry.Line|Line, Geometry.Rectangle|Rectangle, or Geometry.Ellipse|Ellipse.
     * The initial value is zero.
     */
    startX: number;
    /**
     * Gets or sets the starting Y coordinate of the Geometry if it is of type
     * Geometry.Line|Line, Geometry.Rectangle|Rectangle, or Geometry.Ellipse|Ellipse.
     * The initial value is zero.
     */
    startY: number;
    /**
     * Gets or sets the ending X coordinate of the Geometry if it is of type
     * Geometry.Line|Line, Geometry.Rectangle|Rectangle, or Geometry.Ellipse|Ellipse.
     * The initial value is zero.
     */
    endX: number;
    /**
     * Gets or sets the ending Y coordinate of the Geometry if it is of type
     * Geometry.Line|Line, Geometry.Rectangle|Rectangle, or Geometry.Ellipse|Ellipse.
     * The initial value is zero.
     */
    endY: number;
    /**
     * Gets or sets the List of PathFigures
     * that describes the content of the
     * path for Geometries of type Geometry.Path|Path.
     */
    figures: List<PathFigure>;
    /**
     * Add a PathFigure to the #figures list.
     * @param {PathFigure} figure a newly allocated unshared PathFigure that will become owned by this Geometry
     * @return {Geometry} this
     * @since 1.5
     */
    add(figure: PathFigure): Geometry;
    /**
     * Undocumented
     * @param {number} f1x spot1.x
     * @param {number} f1y spot1.y
     * @param {number} f2x spot2.x
     * @param {number} f2y spot2.y
     * @param {number=} o1x offset1.x, default is zero
     * @param {number=} o1y offset1.y, default is zero
     * @param {number=} o2x offset2.x, default is zero
     * @param {number=} o2y offset2.y, default is zero
     * @return {Geometry} this
     */
    setSpots(f1x: number, f1y: number, f2x: number, f2y: number, o1x?: number, o1y?: number, o2x?: number, o2y?: number): Geometry;
    /**
     * Gets or sets the spot an "Auto" Panel will use for the top-left corner of any panel content
     * when the Shape#spot1 value is Spot.Default.
     * The default value is Spot.TopLeft, at the top-left point of the bounds of the Shape.
     */
    spot1: Spot;
    /**
     * Gets or sets the spot an "Auto" Panel will use for the bottom-right corner of any panel content
     * when the Shape#spot2 value is Spot.Default.
     * The default value is Spot.BottomRight, at the bottom-right point of the bounds of the Shape.
     */
    spot2: Spot;
    /**
     * Gets or sets the Shape#geometryStretch value the Shape should use by default
     * when the Shape#geometryStretch value is GraphObject.None.
     * The default value is GraphObject.Fill.
     * Some figure generators return a Geometry with this property set to GraphObject.Uniform,
     * in order to preserve its aspect ratio when used by a Shape that may have different sizes.
     * @since 1.5
     */
    defaultStretch: EnumValue;
    /**
     * This read-only property returns a rectangle that contains all points within the Geometry.
     * The result will always contain the origin (0, 0).
     */
    readonly bounds: Rect;
}
/**
 * A PathFigure represents a section of a Geometry.
 * It is a single connected series of
 * two-dimensional geometric PathSegments.
 *
 * The Geometry#figures property is a list of PathFigures.
 *
 * Filled PathFigures are painted with the Shape#fill.
 *
 * A PathFigure must not be modified once its containing Geometry
 * has been assigned to a Shape.
 * @unrestricted
 * @category Geometry
 */
export class PathFigure {
    /**
     * Constructs an empty figure.
     * The optional arguments specify the starting point of the figure.
     * You'll want to add a new instance of a PathFigure to the
     * Geometry#figures list of a Geometry.
     * @param {number=} sx optional: the X coordinate of the start point (default is zero).
     * @param {number=} sy optional: the Y coordinate of the start point (default is zero).
     * @param {boolean=} filled optional: whether the figure is filled (default is true).
     * @param {boolean=} shadowed optional: whether the figure may be drawn with a shadow (default is true).
     */
    constructor(sx?: number, sy?: number, filled?: boolean, shadowed?: boolean);
    /**
     * Create a copy of this PathFigure, with the same values and segments.
     * @expose
     * @return {PathFigure}
     */
    copy(): PathFigure;
    /**
     * Undocumented
     * @param {PathFigure} f
     * @return {boolean}
     */
    equalsApprox(f: PathFigure): boolean;
    /**
     * Gets or sets whether this PathFigure is drawn filled.
     * The default value is true.
     */
    isFilled: boolean;
    /**
     * Gets or sets whether this PathFigure will render a shadow if one is defined.
     * This is used for turning off shadows on specific subpaths.
     * The default value is true.
     */
    isShadowed: boolean;
    /**
     * Gets or sets the starting point X coordinate of the PathFigure.
     * The default value is zero.
     */
    startX: number;
    /**
     * Gets or sets the starting point Y coordinate of the PathFigure.
     * The default value is zero.
     */
    startY: number;
    /**
     * Gets or sets the List of PathSegments that define this PathFigure.
     */
    segments: List<PathSegment>;
    /**
     * Add a PathSegment to the #segments list.
     * @param {PathSegment} segment a newly allocated unshared PathSegment that will become owned by this PathFigure
     * @return {PathFigure}
     * @since 1.5
     */
    add(segment: PathSegment): PathFigure;
}
/**
 * A PathSegment represents a straight line or curved segment of a path between
 * two or more points that are part of a PathFigure.
 *
 * A PathSegment must not be modified once its containing PathFigure's
 * Geometry has been assigned to a Shape.
 * @unrestricted
 * @category Geometry
 */
export class PathSegment {
    /**
     * Constructs a segment that goes nowhere unless you specify some Points.
     *
     * The segment type must be one of the following values:
     * PathSegment.Line, PathSegment.Bezier,
     * PathSegment.QuadraticBezier, PathSegment.Arc, PathSegment.SvgArc.
     * You will want to add a new instance of a PathSegment to the
     * PathFigure#segments list of a PathFigure.
     * @param {EnumValue=} type if not supplied, the default PathSegment type is PathSegment.Line.
     *   But if the type is supplied, one must also provide the endpoint X and Y values,
     *   either as arguments in this constructor or by setting the #endX and #endY properties.
     *   If the type is QuadraticBezier, the X1 and Y1 control point values must both be supplied.
     *   If the type is Bezier, X1, Y1, X2, and Y2 control point values must all be supplied.
     *
     * If the type is PathSegment.Line|Line it needs the following arguments:
     *
     * `(go.PathSegment.Line, ex, ey)`
     *   - **ex, ey** describe the end point
     *
     * If the type is PathSegment.QuadraticBezier|QuadraticBezier it needs the following arguments:
     *
     * `(go.PathSegment.QuadraticBezier, ex, ey, x1, y1)`
     *   - **ex, ey** describe the end point
     *   - **x1, y1** describe the only control point
     *
     * If the type is PathSegment.Bezier|Bezier it needs the following arguments:
     *
     * `(go.PathSegment.Bezier, ex, ey, x1, y1, x2, y2)`
     *   - **ex, ey** describe the end point
     *   - **x1, y1** describe the first control point
     *   - **x2, y2** describe the second control point
     *
     * If the type is PathSegment.Arc|Arc it needs the following arguments:
     *
     * `(go.PathSegment.Arc, startAngle, sweepAngle, centerX, centerY, radiusX, radiusY)`
     *   - **startAngle** describes the start angle, in degrees
     *   - **sweepAngle** describes the sweep angle, in degrees
     *   - **centerX, centerY** describe the center point
     *   - **radiusX, radiusY** describe the radiusX and radiusY
     *
     * If the type is PathSegment.SvgArc|SvgArc it needs the following arguments:
     *
     * `(go.PathSegment.SvgArc, ex, ey, radiusX, radiusY, xAxisRotation, largeArcFlag, clockwiseFlag)`
     *
     * They are in the same order as arcs in SVG path strings, except the endpoint x and y values come first, not last.
     *   - **ex, ey** describe the endpoint
     *   - **radiusX, radiusY** describe the radius
     *   - **xAxisRotation** describes the #xAxisRotation (number in degrees)
     *   - **largeArcFlag** describes the #isLargeArc (true or false)
     *   - **clockwiseFlag** describes the #isClockwiseArc (true or false).
     * @param {number=} ex optional: the X coordinate of the end point, or the startAngle of an Arc.
     * @param {number=} ey optional: the Y coordinate of the end point, or the sweepAngle of an Arc.
     * @param {number=} x1 optional: the X coordinate of the first bezier control point, or the centerX of an Arc, or the radiusX of an SvgArc.
     * @param {number=} y1 optional: the Y coordinate of the first bezier control point, or the centerY of an Arc, or the radiusY of an SvgArc.
     * @param {number=} x2 optional: the X coordinate of the second cubic bezier control point, or the radiusX of an Arc, or the xAxisRotation of an SvgArc.
     * @param {number|boolean=} y2 optional: the Y coordinate of the second cubic bezier control point, or the radiusY of an Arc, or whether this is the larger arc of an SvgArc.
     * @param {boolean=} clockwise optional: whether an SvgArc goes clockwise or counterclockwise.
     */
    constructor(type?: EnumValue, ex?: number, ey?: number, x1?: number, y1?: number, x2?: number, y2?: number | boolean, clockwise?: boolean);
    /**
     * Create a copy of this PathSegment, of the same type and with the same point values.
     * @expose
     * @return {PathSegment}
     */
    copy(): PathSegment;
    /**
     * Undocumented
     * @param {PathSegment} s
     * @return {boolean}
     */
    equalsApprox(s: PathSegment): boolean;
    /**
     * For beginning a new subpath, a value for PathSegment#type.
     * @constant
     */
    static Move: EnumValue;
    /**
     * For drawing a straight line segment, a value for PathSegment#type.
     * @constant
     */
    static Line: EnumValue;
    /**
     * For drawing a cubic bezier segment, a value for PathSegment#type.
     * @constant
     */
    static Bezier: EnumValue;
    /**
     * For drawing a quadratic bezier segment, a value for PathSegment#type.
     * @constant
     */
    static QuadraticBezier: EnumValue;
    /**
     * For drawing an arc segment, a value for PathSegment#type.
     * @constant
     */
    static Arc: EnumValue;
    /**
     * For drawing an SVG arc segment, a value for PathSegment#type.
     *
     * See: <a href="https://www.w3.org/TR/SVG/paths.html#PathDataEllipticalArcCommands">SVG Arc specification (w3.org)</a>
     * @constant
     * @since 1.1
     */
    static SvgArc: EnumValue;
    /**
     * Closes the path after this PathSegment
     * @return {PathSegment} returns this PathSegment.
     */
    close(): PathSegment;
    /**
     * Gets or sets whether the path is closed after this PathSegment.
     * Default value is false.
     */
    isClosed: boolean;
    /**
     * Gets or sets the type of the PathSegment.
     * The value must be one of the following:
     * PathSegment.Line, PathSegment.Bezier,
     * PathSegment.QuadraticBezier, PathSegment.Arc,
     * PathSegment.Move, PathSegment.Arc, and PathSegment.SvgArc.
     */
    type: EnumValue;
    /**
     * Gets or sets the X coordinate of the end point for all types of PathSegment except PathSegment.Arc|Arc.
     * The default value is zero.
     */
    endX: number;
    /**
     * Gets or sets the Y coordinate of the end point for all types of PathSegment except PathSegment.Arc|Arc.
     * The default value is zero.
     */
    endY: number;
    /**
     * Gets or sets the X value of the first control point for a PathSegment
     * of type PathSegment.Bezier|Bezier or PathSegment.QuadraticBezier|QuadraticBezier.
     */
    point1X: number;
    /**
     * Gets or sets the Y value of the first control point for a PathSegment
     * of type PathSegment.Bezier|Bezier or PathSegment.QuadraticBezier|QuadraticBezier.
     */
    point1Y: number;
    /**
     * Gets or sets the X value of the second control point for a PathSegment
     * of type cubic PathSegment.Bezier|Bezier.
     */
    point2X: number;
    /**
     * Gets or sets the Y value of the second control point for a PathSegment
     * of type cubic PathSegment.Bezier|Bezier.
     */
    point2Y: number;
    /**
     * Gets or sets the center X value of the Arc for a PathSegment of type PathSegment.Arc|Arc.
     */
    centerX: number;
    /**
     * Gets or sets the center Y value of the Arc for a PathSegment of type PathSegment.Arc|Arc.
     */
    centerY: number;
    /**
     * Gets or sets the X value of the radius for a PathSegment of type PathSegment.Arc|Arc or PathSegment.SvgArc|SvgArc.
     * Value must be a positive number.
     */
    radiusX: number;
    /**
     * Gets or sets the Y value of the radius for a PathSegment of type PathSegment.Arc|Arc or PathSegment.SvgArc|SvgArc.
     * Value must be a positive number.
     */
    radiusY: number;
    /**
     * Gets or sets the starting angle for a PathSegment of type PathSegment.Arc|Arc.
     * Value must within the range: (0 <= value < 360).
     */
    startAngle: number;
    /**
     * Gets or sets the length of angle in degrees, or amount of arc to "sweep" for a PathSegment of type PathSegment.Arc|Arc.
     * Must be between -360 and 360, inclusive.
     */
    sweepAngle: number;
    /**
     * Gets or sets the clockwise-flag for a PathSegment of type PathSegment.SvgArc|SvgArc.
     * SVG Arcs specify a radius and an endpoint, and are always a portion of an ellipse.
     * The parameters allow for two potential ellipses and four potential arcs.
     * A clockwise-flag set to true will use one of the two possible positive-angle arcs,
     * and false will use one of the two negative-angle arcs. Which arc is chosen (small or large)
     * depends on the value of #isLargeArc.
     * For more information see the visual examples in the
     * <a href="https://www.w3.org/TR/SVG/paths.html#PathDataEllipticalArcCommands">SVG Arc specification (w3.org)</a>
     */
    isClockwiseArc: boolean;
    /**
     * Gets or sets the large-arc-flag for a PathSegment of type PathSegment.SvgArc|SvgArc.
     * SVG Arcs specify a radius and an endpoint, and are always a portion of an ellipse.
     * The parameters allow for two potential ellipses and four potential arcs.
     * A large-arc-flag set to true will choose the larger of the two arc sweeps.
     * Which way the arc sweeps (positive angle or negative angle) depends on the value of #isClockwiseArc
     * For more information see the visual examples in the
     * <a href="https://www.w3.org/TR/SVG/paths.html#PathDataEllipticalArcCommands">SVG Arc specification (w3.org)</a>
     */
    isLargeArc: boolean;
    /**
     * Gets or sets the X-axis rotation for a PathSegment of type PathSegment.SvgArc|SvgArc.
     * X-axis rotation is used to rotate the ellipse that the arc is created from,
     * and must be between 0 and 360 degrees.
     * Default is 0.
     */
    xAxisRotation: number;
}
/**
 * An InputEvent represents a mouse or keyboard or touch event.
 * The principal properties hold information about a particular input event.
 * These properties include the #documentPoint at which a mouse event
 * occurred in document coordinates,
 * the corresponding point in view/element coordinates, #viewPoint,
 * the #key for keyboard events,
 * and the #modifiers and #button at the time.
 * Additional descriptive properties include #clickCount, #delta,
 * #timestamp, and the source event #event (if available).
 *
 * Many of its properties are provided for convenient access to the state of the input event,
 * such as asking whether the #control key was held down at the time,
 * or the #targetObject (a GraphObject) that the mouse was over .
 *
 * When real events fire on the Diagram, InputEvents are created automatically set update the value of Diagram#lastInput.
 * These events set the value of #event with the backing browser-defined Event,
 * which may be a MouseEvent, KeyboardEvent, PointerEvent, TouchEvent, and so on.
 *
 * InputEvents backed by MouseEvents set both #button, the button that caused the action,
 * and #buttons, the set of buttons currently pressed after the action has happened.
 * By default a user-created InputEvent sets #button and #buttons as if the event was a left-click.
 *
 * You can create InputEvents and set the value of Diagram#lastInput in order to simulate user actions in tools.
 * This can be useful for testing. See the <a href="../../extensions/Robot.html">Robot extension sample</a> for
 * an example of creating InputEvents to simulate user input.
 * @unrestricted
 */
export class InputEvent {
    /**
     * The InputEvent class constructor produces an empty InputEvent.
     * You should never need to call this constructor.
     * But if you do, you must first set #diagram to the Diagram before any other use.
     */
    constructor();
    /**
     * Make a copy of this InputEvent.
     * @expose
     * @return {InputEvent}
     */
    copy(): InputEvent;
    /**
     * Gets the source diagram associated with the event.
     */
    diagram: Diagram;
    /**
     * Gets or sets the point at which this input event occurred.
     * The Point is in view coordinates within the viewport, not in document coordinates.
     * This should be valid for mouse events.
     * For keyboard events, this is the last available mouse point.
     * @see #documentPoint
     * @see Diagram#transformViewToDoc
     * @see Diagram#transformDocToView
     */
    viewPoint: Point;
    /**
     * Gets or sets the point at which this input event occurred, in document coordinates.
     * The Point is in document coordinates, the same as the GraphObject#position
     * for Parts in the diagram.
     * This should be valid for mouse events.
     * For keyboard events, this is the last available mouse point.
     * @see #viewPoint
     * @see Diagram#transformViewToDoc
     * @see Diagram#transformDocToView
     */
    documentPoint: Point;
    /**
     * Gets or sets the modifier keys that were used with the mouse or keyboard event.
     * The number will be a combination of flags representing Control, Shift, Alt or Meta.
     * @see #control
     * @see #shift
     * @see #alt
     * @see #meta
     */
    modifiers: number;
    /**
     * Gets or sets the numerical value representing the mouse button that caused this event.
     *
     * Only one mouse button can cause an event at a time,
     * but the set of all buttons currently pressed is kept in #buttons.
     *
     * This property describes nothing during a `mousemove` event,
     * since no button press causes the event. Instead, use the convenience properties
     * #left, #middle, or #right, or use the value
     * of #buttons to determine which mouse buttons are
     * held during `mousemove` events.
     *
     * Common values for this property:
     *   - 0: left mouse button
     *   - 1: middle mouse button
     *   - 2: right mouse button
     *
     * Other values are possible if the mouse has additional buttons.
     *
     * If there is no associated #event, setting this also
     * sets the #buttons flags to only this button.
     *
     * This property is valid if this is a mouse event.
     * @see #buttons
     * @see #left
     * @see #middle
     * @see #right
     */
    button: number;
    /**
     * Gets or sets the set of buttons that are currently being held down.
     * If this is a `mouseup` event, this set of buttons does not include
     * the button that was just released, which will be the value of #button.
     *
     * Common values for this property:
     *   - 0: not holding down any button
     *   - 1: left mouse button
     *   - 2: right mouse button
     *   - 3: both left and right mouse buttons
     *   - 4: middle mouse button
     *   - 5: middle and left mouse buttons
     *   - 6: middle and right mouse buttons
     *   - 7: all three common mouse buttons
     *
     * Other values are possible if the mouse has additional buttons.
     *
     * This property is valid if this is a mouse event.
     * @see #button
     * @see #left
     * @see #middle
     * @see #right
     */
    buttons: number;
    /**
     * Gets or sets the key pressed or released as this event.
     * This property is valid if this is a keyboard event.
     */
    key: string;
    /**
     * Gets or sets whether the InputEvent represents a mouse-down or a key-down event.
     * The default value is false.
     * @since 1.1
     */
    down: boolean;
    /**
     * Gets or sets whether the InputEvent represents a mouse-up or a key-up event.
     * The default value is false.
     * @since 1.1
     */
    up: boolean;
    /**
     * Gets or sets whether this event represents a click or a double-click.
     * It is zero if not a click; one if a single-click; two if a double-click.
     * This property is valid if this is a mouse event.
     */
    clickCount: number;
    /**
     * Gets or sets the amount of change associated with a mouse-wheel rotation.
     * It is an abstract number, either positive or negative.
     * This property is valid if this is a mouse-wheel event.
     */
    delta: number;
    /**
     * This property is true when the InputEvent is caused by a touch event that registered more than one touch.
     * @see #isTouchEvent
     * @since 1.5
     */
    isMultiTouch: boolean;
    /**
     * Gets or sets whether an InputEvent that applies to a GraphObject and bubbles
     * up the chain of containing Panels is stopped from continuing up the chain.
     *
     * Some event handlers may set this to true to avoid getting any behavior from the containing Panels.
     * The default value is false.
     */
    handled: boolean;
    /**
     * Undocumented
     * @see #handled
     */
    bubbles: boolean;
    /**
     * Gets or sets the platform's user-agent-supplied event for this event.
     * It may be null if no underlying event exists.
     * @see #timestamp
     */
    event: Event | null;
    /**
     * This read-only property is true when the InputEvent is caused by a touch event.
     * @see #isMultiTouch
     * @since 1.5
     */
    readonly isTouchEvent: boolean;
    /**
     * Gets or sets the time at which the event occurred, in milliseconds.
     */
    timestamp: number;
    /**
     * Gets or sets the diagram associated with the canvas that the event is currently targeting.
     */
    targetDiagram: Diagram | null;
    /**
     * Gets or sets the GraphObject that is at the current mouse point, if any.
     *
     * For those events that are bubbled up the chain of parent Panels,
     * this property provides access to the original GraphObject where the input event occurred.
     * @see #handled
     */
    targetObject: GraphObject | null;
    /**
     * Gets or sets whether the control key is being held down.
     * This is true if the key was pressed at the time of the event.
     * @see #modifiers
     * @see #shift
     * @see #alt
     * @see #meta
     */
    control: boolean;
    /**
     * Gets or sets whether the shift key is being held down.
     * This is true if the key was pressed at the time of the event.
     * @see #modifiers
     * @see #control
     * @see #alt
     * @see #meta
     */
    shift: boolean;
    /**
     * Gets or sets whether the alt key is being held down.
     * This is true if the key was pressed at the time of the event.
     * @see #modifiers
     * @see #control
     * @see #shift
     * @see #meta
     */
    alt: boolean;
    /**
     * Gets or sets whether the meta key is being held down.
     * This is true if the key was pressed at the time of the event.
     * @see #modifiers
     * @see #control
     * @see #shift
     * @see #alt
     */
    meta: boolean;
    /**
     * Gets or sets whether the logical left mouse button is being held down.
     * This is true if the button was pressed at the time of the event.
     *
     * If this InputEvent has a #event of type `MouseEvent` with `e.type` of "mouseup" or "mousedown",
     * this property uses the value of #button. Otherwise, it uses the value of #buttons.
     *
     * When setting, this sets the value of #buttons.
     * @see #button
     * @see #middle
     * @see #right
     */
    left: boolean;
    /**
     * Gets or sets whether the logical right mouse button is being held down.
     * This is true if the button was pressed at the time of the event.
     * @see #button
     * @see #left
     * @see #middle
     */
    right: boolean;
    /**
     * Gets or sets whether the logical middle mouse button is being held down.
     * This is true if the button was pressed at the time of the event.
     * @see #button
     * @see #left
     * @see #right
     */
    middle: boolean;
}
/**
 * A DiagramEvent represents a more abstract event than an InputEvent.
 * They are raised on the Diagram class.
 * One can receive such events by registering a DiagramEvent listener on a Diagram
 * by calling Diagram#addDiagramListener.
 * The listener function, when called, will be passed an instance of a DiagramEvent.
 * Use the #name property to decide what kind of diagram event it is.
 * The #diagram property refers to the Diagram, and you can get additional information
 * from that, such as the Diagram#lastInput, which in turn provides information
 * such as InputEvent#documentPoint that may be relevant for that kind of DiagramEvent.
 *
 * The #subject and #parameter properties optionally provide additional information
 * about the diagram event.  The subject could be a collection of Parts or it could be
 * an individual object such as a Link or a GraphObject within a Node.
 * Everything depends on the kind of diagram event that it is.
 *
 * Some DiagramEvents such as "ObjectSingleClicked" or "BackgroundDoubleClicked" are normally
 * associated with InputEvents.
 * Some DiagramEvents such as "SelectionMoved" or "PartRotated" are associated with the
 * results of Tool-handled gestures or CommandHandler actions.
 * Some DiagramEvents are not necessarily associated with any input events at all,
 * such as "ViewportBoundsChanged", which can happen due to programmatic
 * changes to the Diagram#position and Diagram#scale properties.
 *
 * DiagramEvents that occur during a transaction may be called before the state of the whole diagram has settled down.
 * This often means that such events occur before a layout, so nodes may not have their final positions,
 * links may not have their final routes, and the Diagram#documentBounds and Diagram#viewportBounds
 * may not yet have been updated.
 * Such events may make additional changes to the diagram, which may in turn result in additional side-effects.
 *
 * DiagramEvents that occur outside of a transaction require you to start and commit a transaction around any side-effects that you want to do.
 * However, some DiagramEvents do not allow you to make any changes to the Diagram or Model.
 *
 * Currently defined diagram event names include:
 *   - **"AnimationStarting"**, an animation is about to start;<br/>
 *     do not modify the Diagram or its Model in the event listener.
 *   - **"AnimationFinished"**, an animation just completed;<br/>
 *     do not modify the Diagram or its Model in the event listener.
 *   - **"BackgroundSingleClicked"**, a click that was not on any Part;<br/>
 *     if you make any changes, start and commit your own transaction.
 *   - **"BackgroundDoubleClicked"**, a double-click that was not on any Part;<br/>
 *     if you make any changes, start and commit your own transaction.
 *   - **"BackgroundContextClicked"**, a context-click that was not on any Part;<br/>
 *     if you make any changes, start and commit your own transaction.
 *   - **"ChangingSelection"**, an operation is about to change the Diagram#selection collection,<br/>
 *     which is also the value of the DiagramEvent#subject;<br/>
 *     do not make any changes to the selection or the diagram in the event listener;
 *     note that just setting Part#isSelected will not raise this event, but tools and commands will.
 *   - **"ChangedSelection"**, an operation has just changed the Diagram#selection collection,<br/>
 *     which is also the value of the DiagramEvent#subject;<br/>
 *     do not make any changes to the selection or the diagram in the event listener;
 *     note that just setting Part#isSelected will not raise this event, but tools and commands will.
 *   - **"ClipboardChanged"**, Parts have been copied to the clipboard by CommandHandler#copySelection;<br/>
 *     the DiagramEvent#subject is the collection of Parts;<br/>
 *     if you make any changes, start and commit your own transaction.
 *   - **"ClipboardPasted"**, Parts have been copied from the clipboard into the Diagram by CommandHandler#pasteSelection;<br/>
 *     the DiagramEvent#subject is the Diagram#selection,<br/>
 *     and this is called within a transaction.
 *   - **"DocumentBoundsChanged"**, the area of the diagram's Parts, Diagram#documentBounds, has changed;<br/>
 *     the DiagramEvent#parameter is the old Rect
 *   - **"ExternalObjectsDropped"**, Parts have been copied into the Diagram by drag-and-drop from outside of the Diagram;<br/>
 *     the DiagramEvent#subject is the set of Parts that were dropped (which is also the Diagram#selection),
 *     the DiagramEvent#parameter is the source Diagram,
 *     and this is called within a transaction.
 *   - **"GainedFocus"**, the diagram has gained keyboard focus, such as after a call to Diagram#focus.
 *   - **"InitialLayoutCompleted"**, the whole diagram layout has updated for the first time since a major change to the Diagram, such as replacing the Model;<br/>
 *     if you make any changes, you do not need to perform a transaction.
 *   - **"LayoutCompleted"**, the whole diagram layout has just been updated;<br/>
 *     if you make any changes, you do not need to perform a transaction.
 *   - **"LinkDrawn"**, the user has just created a new Link using LinkingTool;<br/>
 *     the DiagramEvent#subject is the new Link,<br/>
 *     and this is called within a transaction.
 *   - **"LinkRelinked"**, the user has just reconnected an existing Link using RelinkingTool or DraggingTool;<br/>
 *     the DiagramEvent#subject is the modified Link,<br/>
 *     the DiagramEvent#parameter is the GraphObject port that the link was disconnected from,<br/>
 *     and this is called within a transaction.
 *   - **"LinkReshaped"**, the user has just rerouted an existing Link using LinkReshapingTool;<br/>
 *     the DiagramEvent#subject is the modified Link,<br/>
 *     the DiagramEvent#parameter is the List of Points of the link's original route,
 *     and this is called within a transaction.
 *   - **"LostFocus"**, the diagram has lost keyboard focus, a.k.a. "blur".
 *   - **"Modified"**, the Diagram#isModified property has been set to a new value --
 *     useful for marking a window as having been modified since the last save;<br/>
 *     do not modify the Diagram or its Model in the event listener.
 *   - **"ObjectSingleClicked"**, a click that occurred on a GraphObject;<br/>
 *     the DiagramEvent#subject is the GraphObject;<br/>
 *     if you make any changes, start and commit your own transaction.
 *   - **"ObjectDoubleClicked"**, a double-click that occurred on a GraphObject;<br/>
 *     the DiagramEvent#subject is the GraphObject;<br/>
 *     if you make any changes, start and commit your own transaction.
 *   - **"ObjectContextClicked"**, a context-click that occurred on a GraphObject;<br/>
 *     the DiagramEvent#subject is the GraphObject;<br/>
 *     if you make any changes, start and commit your own transaction.
 *   - **"PartCreated"**, the user inserted a new Part by ClickCreatingTool;<br/>
 *     the DiagramEvent#subject is the new Part,<br/>
 *     and this is called within a transaction.
 *   - **"PartResized"**, the user has changed the size of a GraphObject by ResizingTool;<br/>
 *     the DiagramEvent#subject is the GraphObject,<br/>
 *     the DiagramEvent#parameter is the original Size,<br/>
 *     and this is called within a transaction.
 *   - **"PartRotated"**, the user has changed the angle of a GraphObject by RotatingTool;<br/>
 *     the DiagramEvent#subject is the GraphObject,<br/>
 *     the DiagramEvent#parameter is the original angle in degrees,<br/>
 *     and this is called within a transaction.
 *   - **"SelectionMoved"**, the user has moved selected Parts by DraggingTool;<br/>
 *     the DiagramEvent#subject is a Set of the moved Parts,<br/>
 *     and this is called within a transaction.
 *   - **"SelectionCopied"**, the user has copied selected Parts by DraggingTool;<br/>
 *     the DiagramEvent#subject is Set of the newly copied Parts,<br/>
 *     and this is called within a transaction.
 *   - **"SelectionDeleted"**, the user has deleted selected Parts by CommandHandler#deleteSelection;<br/>
 *     the DiagramEvent#subject is the collection of Parts that were deleted,<br/>
 *     and this is called within a transaction.
 *   - **"SelectionDeleting"**, the user is about to delete selected Parts by CommandHandler#deleteSelection;<br/>
 *     the DiagramEvent#subject is the Diagram#selection collection of Parts to be deleted,<br/>
 *     and this is called within a transaction.
 *   - **"SelectionGrouped"**, the user has made a new Group out of the selected Parts by CommandHandler#groupSelection;<br/>
 *     the DiagramEvent#subject is the new Group,<br/>
 *     and this is called within a transaction.
 *   - **"SelectionUngrouped"**, the user has removed a selected Group but kept its members by CommandHandler#ungroupSelection;<br/>
 *     the DiagramEvent#subject is the collection of Groups that were ungrouped,<br/>
 *     the DiagramEvent#parameter is the collection of former member Parts that were ungrouped,<br/>
 *     and this is called within a transaction.
 *   - **"SubGraphCollapsed"**, the user has collapsed selected Groups by CommandHandler#collapseSubGraph;<br/>
 *     the DiagramEvent#subject is the collection of Groups that were collapsed,<br/>
 *     and this is called within a transaction.
 *   - **"SubGraphExpanded"**, the user has expanded selected Groups by CommandHandler#expandSubGraph;<br/>
 *     the DiagramEvent#subject is the collection of Groups that were expanded,<br/>
 *     and this is called within a transaction.
 *   - **"TextEdited"**, the user has changed the string value of a TextBlock by TextEditingTool;<br/>
 *     the DiagramEvent#subject is the edited TextBlock,<br/>
 *     the DiagramEvent#parameter is the original string,<br/>
 *     and this is called within a transaction.
 *   - **"TreeCollapsed"**, the user has collapsed selected Nodes with subtrees by CommandHandler#collapseTree;<br/>
 *     the DiagramEvent#subject is the collection of Nodes that were collapsed,<br/>
 *     and this is called within a transaction.
 *   - **"TreeExpanded"**, the user has expanded selected Nodes with subtrees by CommandHandler#expandTree;<br/>
 *     the DiagramEvent#subject is the collection of Nodes that were expanded,<br/>
 *     and this is called within a transaction.
 *   - **"ViewportBoundsChanged"**, the visible area of the Diagram, Diagram#viewportBounds, has changed;<br/>
 *     the DiagramEvent#subject is an object whose "scale" property is the old Diagram#scale value,
 *     whose "position" property is the old Diagram#position value,
 *     and whose "bounds" property is the old Diagram#viewportBounds value;
 *     the DiagramEvent#parameter is also the old viewportBounds Rect.
 * @unrestricted
 */
export class DiagramEvent {
    /**
     * The DiagramEvent class constructor produces an empty DiagramEvent.
     * You should never need to call this constructor.
     * But if you do, you must first set #diagram to the Diagram before any other use.
     */
    constructor();
    /**
     * Gets the diagram associated with the event.
     */
    diagram: Diagram;
    /**
     * Gets or sets the name of the kind of diagram event that this represents.
     * This property should always be set to one of the recognized list of names,
     * as listed in the documentation for DiagramEvent.
     */
    name: string;
    /**
     * Gets or sets an optional object that is the subject of the diagram event.
     * This property defaults to null.
     */
    subject: any;
    /**
     * Gets or sets an optional object that describes the change to the subject of the diagram event.
     * This property defaults to null.
     */
    parameter: any;
}
/**
 * The signature for a function acting as a handler for DiagramEvents.
 */
export type DiagramEventHandler = (e: DiagramEvent) => void;
/**
 * (Undocumented, internal interface)
 */
export type DiagramEventName = 'AnimationStarting' | 'AnimationFinished' | 'BackgroundSingleClicked' | 'BackgroundDoubleClicked' | 'BackgroundContextClicked' | 'ChangingSelection' | 'ChangedSelection' | 'ClipboardChanged' | 'ClipboardPasted' | 'DocumentBoundsChanged' | 'ExternalObjectsDropped' | 'GainedFocus' | 'InitialLayoutCompleted' | 'LayoutCompleted' | 'LinkDrawn' | 'LinkRelinked' | 'LinkReshaped' | 'LostFocus' | 'Modified' | 'ObjectSingleClicked' | 'ObjectDoubleClicked' | 'ObjectContextClicked' | 'PartCreated' | 'PartResized' | 'PartRotated' | 'SelectionMoved' | 'SelectionCopied' | 'SelectionDeleted' | 'SelectionDeleting' | 'SelectionGrouped' | 'SelectionUngrouped' | 'SubGraphCollapsed' | 'SubGraphExpanded' | 'TextEdited' | 'TreeCollapsed' | 'TreeExpanded' | 'ViewportBoundsChanged' | 'InvalidateDraw';
/**
 * (Undocumented, internal interface)
 */
export interface DiagramEventsInterface {
    AnimationStarting?: DiagramEventHandler;
    AnimationFinished?: DiagramEventHandler;
    BackgroundSingleClicked?: DiagramEventHandler;
    BackgroundDoubleClicked?: DiagramEventHandler;
    BackgroundContextClicked?: DiagramEventHandler;
    ChangingSelection?: DiagramEventHandler;
    ChangedSelection?: DiagramEventHandler;
    ClipboardChanged?: DiagramEventHandler;
    ClipboardPasted?: DiagramEventHandler;
    DocumentBoundsChanged?: DiagramEventHandler;
    ExternalObjectsDropped?: DiagramEventHandler;
    GainedFocus?: DiagramEventHandler;
    InitialLayoutCompleted?: DiagramEventHandler;
    LayoutCompleted?: DiagramEventHandler;
    LinkDrawn?: DiagramEventHandler;
    LinkRelinked?: DiagramEventHandler;
    LinkReshaped?: DiagramEventHandler;
    LostFocus?: DiagramEventHandler;
    Modified?: DiagramEventHandler;
    ObjectSingleClicked?: DiagramEventHandler;
    ObjectDoubleClicked?: DiagramEventHandler;
    ObjectContextClicked?: DiagramEventHandler;
    PartCreated?: DiagramEventHandler;
    PartResized?: DiagramEventHandler;
    PartRotated?: DiagramEventHandler;
    SelectionMoved?: DiagramEventHandler;
    SelectionCopied?: DiagramEventHandler;
    SelectionDeleted?: DiagramEventHandler;
    SelectionDeleting?: DiagramEventHandler;
    SelectionGrouped?: DiagramEventHandler;
    SelectionUngrouped?: DiagramEventHandler;
    SubGraphCollapsed?: DiagramEventHandler;
    SubGraphExpanded?: DiagramEventHandler;
    TextEdited?: DiagramEventHandler;
    TreeCollapsed?: DiagramEventHandler;
    TreeExpanded?: DiagramEventHandler;
    ViewportBoundsChanged?: DiagramEventHandler;
}
/**
 * A ChangedEvent represents a change to an object, typically a GraphObject,
 * but also for model data, a Model, or a Diagram.
 * The most common case is for remembering the name of a property
 * and the before-and-after values for that property.
 *
 * You can listen for changed events on the model using Model#addChangedListener
 * or Diagram#addModelChangedListener, and on the Diagram using Diagram#addChangedListener.
 *
 * There are four kinds of changes, represented by enumerated values:
 * ChangedEvent.Property (the most common), ChangedEvent.Insert and ChangedEvent.Remove
 * (to represent inserting or removing objects from collections),
 * and ChangedEvent.Transaction (to notify about beginning or ending transactions or undo or redo).
 *
 * The most common kind of ChangedEvent is a Property change.
 * The name of the property is given by #propertyName.
 * The modified object is given by #object.
 * Use the #oldValue and #newValue properties for the before and after property values.
 *
 * For an Insert ChangedEvent, the modified collection (often an Array) is a property value on the #object.
 * The #propertyName helps distinguish between different collections on the object.
 * Use the #newValue property to indicate the value that was inserted.
 * Use the #newParam property to indicate where or how, such as an array index or dictionary key.
 *
 * For a Remove ChangedEvent, the modified collection is a property value on the #object.
 * The #propertyName helps distinguish between different collections on the object.
 * Use the #oldValue property to indicate the value that was removed.
 * Use the #oldParam property to indicate where or how, such as an array index or dictionary key.
 *
 * Transaction ChangedEvents are generated by the UndoManager.
 * The #propertyName names the nature of the ChangedEvent.
 * For the very first transaction, the property name is "StartingFirstTransaction".
 * This ChangedEvent precedes a ChangedEvent whose property name is "StartedTransaction",
 * which occurs for every top-level transaction.
 *
 * When ending a transaction, there is first a ChangedEvent whose name is "ComittingTransaction".
 * This is followed by one with either "CommittedTransaction" or "RolledBackTransaction",
 * depending on how the transaction is ending.
 * The #oldValue provides the transaction name and the #object is the Transaction being finished.
 * (Note that the Transaction value may be null if no Transaction is available at that time,
 * perhaps because there were no changes made during the transaction.)
 * That Transaction can be scanned to look for ChangedEvents that you may wish to record in a database,
 * all within a single database transaction.
 *
 * There are also Transaction ChangedEvents corresponding to "StartingUndo", "FinishedUndo",
 * "StartingRedo", and "FinishedRedo".
 * The #object property provides the Transaction that is about-to-be or just-was undone or redone.
 *
 * Non-Transaction ChangedEvents are remembered by the UndoManager, if UndoManager#isEnabled,
 * and held in the UndoManager#history as Transactions which hold lists of ChangedEvents.
 * That is why ChangedEvent implements #undo and #redo of the change that it remembers.
 *
 * When the ChangedEvent represents a change to a Model, the value of #model is non-null
 * and the value of #diagram is meaningless.
 * If the change is a structural change to the model,
 * the value of #modelChange indicates the kind of change.
 * Currently defined model changed event names include:
 *   - **"nodeDataArray"**, after the model's Model#nodeDataArray is replaced, inserted into or removed from
 *     (setting Model#nodeDataArray or calling Model#addNodeData or Model#removeNodeData)
 *   - **"nodeKey"**, after changing a node data's unique key (Model#setKeyForNodeData)
 *   - **"nodeCategory"**, after changing a node data's category (Model#setCategoryForNodeData)
 *   - **"linkDataArray"**, after the model's GraphLinksModel#linkDataArray is replaced, inserted into or removed from
 *     (setting GraphLinksModel#linkDataArray or calling GraphLinksModel#addLinkData
 *     or GraphLinksModel#removeLinkData)
 *   - **"linkKey"**, after changing a link data's unique key (GraphLinksModel#setKeyForLinkData)
 *   - **"linkCategory"**, after changing a link data's category (GraphLinksModel#setCategoryForLinkData)
 *   - **"linkFromKey"**, after changing a link data's "from" node key (GraphLinksModel#setFromKeyForLinkData)
 *   - **"linkToKey"**, after changing a link data's "to" node key (GraphLinksModel#setToKeyForLinkData)
 *   - **"linkFromPortId"**, after changing a link data's "from" port identifier string (GraphLinksModel#setFromPortIdForLinkData)
 *   - **"linkToPortId"**, after changing a link data's "to" port identifier string (GraphLinksModel#setToPortIdForLinkData)
 *   - **"linkLabelKeys"**, after replacing, inserting into, or removing from a link data's array of keys to label nodes
 *     (calling GraphLinksModel#setLabelKeysForLinkData, GraphLinksModel#addLabelKeyForLinkData,
 *     or GraphLinksModel#removeLabelKeyForLinkData)
 *   - **"nodeGroupKey"**, after changing a node data's key for a containing group data (GraphLinksModel#setGroupKeyForNodeData)
 *   - **"nodeParentKey"**, after changing a node data's "parent" node key (TreeModel#setParentKeyForNodeData)
 *   - **"parentLinkCategory"**, after changing a node data's "parent" link's category(TreeModel#setParentLinkCategoryForNodeData)
 *   - other names are for internal implementation use only, only on Transaction ChangedEvents
 *
 * The value of ChangedEvent#propertyName indicates the actual name of the property that was modified.
 * ChangedEvent#modelChange is a non-empty string only when there is a known structural change to the model,
 * not just the setting of some property on some object.
 *
 * When the ChangedEvent represents a change to a Diagram or a GraphObject within a diagram,
 * the value of #diagram is non-null and
 * the values of #model and #modelChange are meaningless.
 *
 * Please note that ChangedEvents can be raised for many different causes.
 * You may not be interested in changes to temporary objects -- in that case ignore the ChangedEvent when
 * Model#skipsUndoManager or Diagram#skipsUndoManager is true.
 * @unrestricted
 * @category Model
 */
export class ChangedEvent {
    /**
     * The ChangedEvent class constructor produces an empty ChangedEvent object.
     */
    constructor();
    /**
     * For informational events, such as transactions and undo/redo operations,
     * and used as the value for ChangedEvent#change.
     * The ChangedEvent#object refers to the Transaction affected, if any.
     * The ChangedEvent#propertyName distinguishes the different transaction or undo or redo stages.
     * The ChangedEvent#oldValue may provide the transaction name, if available, as given to UndoManager#commitTransaction.
     * @constant
     */
    static Transaction: EnumValue;
    /**
     * For simple property changes,
     * and used as the value for ChangedEvent#change.
     * The name of the property is given by ChangedEvent#propertyName.
     * The modified object is given by ChangedEvent#object.
     * Use the ChangedEvent#oldValue and ChangedEvent#newValue properties for the previous and next property values.
     *
     * For model changes, the ChangedEvent#modelChange may be non-empty, indicating a structural change to the model.
     * @constant
     */
    static Property: EnumValue;
    /**
     * For inserting into collections,
     * and used as the value for ChangedEvent#change.
     * The modified object is given by ChangedEvent#object.
     * Use the optional ChangedEvent#propertyName to distinguish between different collections on the object.
     * Use the ChangedEvent#newValue property to indicate the value that was inserted.
     * Use the optional ChangedEvent#newParam property to indicate where or how, such as an array index or dictionary key.
     * @constant
     */
    static Insert: EnumValue;
    /**
     * For removing from collections,
     * and used as the value for ChangedEvent#change.
     * The modified object is given by ChangedEvent#object.
     * Use the optional ChangedEvent#propertyName to distinguish between different collections on the object.
     * Use the ChangedEvent#oldValue property to indicate the value that was removed.
     * Use the optional ChangedEvent#oldParam property to indicate where or how, such as an array index or dictionary key.
     * @constant
     */
    static Remove: EnumValue;
    /**
     * Forget any object references that this ChangedEvent may have.
     */
    clear(): void;
    /**
     * Make a copy of this ChangedEvent.
     * ChangedEvents are copied when the UndoManager adds to a Transaction.
     * @return {ChangedEvent}
     */
    copy(): ChangedEvent;
    /**
     * This is a convenient method to get the right value, depending on the value of undo,
     * when implementing a state change as part of an undo or a redo.
     * @param {boolean} undo If true, returns the oldValue, otherwise returns the newValue.
     * @return {*} Either the oldValue or the newValue.
     */
    getValue(undo: boolean): any;
    /**
     * This is a convenient method to get the right parameter value, depending on the value of undo,
     * when implementing a state change as part of an undo or a redo.
     * @param {boolean} undo If true, returns the oldParam, otherwise returns the newParam.
     * @return {*} Either the oldParam or the newParam.
     */
    getParam(undo: boolean): any;
    /**
     * This predicate returns true if you can call undo().
     * @return {boolean} True if ready for undo() to be called.
     */
    canUndo(): boolean;
    /**
     * Reverse the effects of this object change.
     * canUndo() must be true for this method to have any effect.
     */
    undo(): void;
    /**
     * This predicate returns true if you can call redo().
     * @return {boolean} True if ready for redo() to be called.
     */
    canRedo(): boolean;
    /**
     * Re-perform this object change after an undo().
     * canRedo() must be true for this method to have any effect.
     */
    redo(): void;
    /**
     * Gets or sets the Model or TreeModel or GraphLinksModel that was modified.
     * When this property is non-null, the #diagram property will be null.
     * However this property and the #diagram property may both be null simultaneously,
     * when no particular model or diagram applies.
     */
    model: Model | null;
    /**
     * Gets or sets the Diagram that was modified.
     * When this property is non-null, the #model property will be null.
     * However this property and the #model property may both be null simultaneously,
     * when no particular model or diagram applies.
     */
    diagram: Diagram | null;
    /**
     * Gets or sets the nature of change that occurred.
     * The default is ChangedEvent.Property.
     * Other values are ChangedEvent.Insert, ChangedEvent.Remove,
     * and ChangedEvent.Transaction.
     */
    change: EnumValue;
    /**
     * Gets or sets the name of the model change, reflecting a change to
     * model data in addition to a change to the model itself.
     *
     * The default is an empty string, which indicates that this is just
     * a regular change to some object's state, probably its property.
     * For a list of possible model change names, see the documentation for ChangedEvent.
     * The names are compared in a case-sensitive manner.
     */
    modelChange: string;
    /**
     * Gets or sets the name of the property change.
     * The default is an empty string, which is not a valid property name.
     * This property can be useful even when the type of change is
     * not ChangedEvent.Property, because it can help identify
     * the collection in the #object that was modified
     * (for ChangedEvent.Insert or ChangedEvent.Remove)
     * or the stage of the current transaction (for ChangedEvent.Transaction).
     */
    propertyName: string | ((a: ObjectData, b: any) => any);
    /**
     * This read-only property is true when this ChangedEvent is of type ChangedEvent.Transaction and represents the end of a transactional change.
     * It is implemented as:
     * ```js
     * return (this.change === ChangedEvent.Transaction &&
     *         (this.propertyName === "CommittedTransaction" ||
     *          this.propertyName === "FinishedUndo" ||
     *          this.propertyName === "FinishedRedo"));
     * ```
     */
    readonly isTransactionFinished: boolean;
    /**
     * Gets or sets the Object that was modified.
     * The default is null.
     *
     * For ChangedEvent.Transaction changes, this may be the Transaction.
     */
    object: ObjectData | null;
    /**
     * Gets or sets the previous or old value that the property had.
     * The default is null.
     */
    oldValue: any;
    /**
     * Gets or sets an optional value associated with the old value.
     * Most properties do not require any parameter to describe the change.
     * This is typically a value that helps distinguish the old value, such as an index into an array.
     * It is null if it is not used.
     * The default is null.
     */
    oldParam: any;
    /**
     * Gets or sets the next or current value that the property has.
     * The default is null.
     */
    newValue: any;
    /**
     * Gets or sets an optional value associated with the new value.
     * Most properties do not require any parameter to describe the change.
     * This is typically a value that helps distinguish the new value, such as an index into an array.
     * It is null if it is not used.
     * The default is null.
     */
    newParam: any;
}
/**
 * The signature for a function acting as a handler for ChangedEvents.
 */
export type ChangedEventHandler = (e: ChangedEvent) => void;
/**
 * A Transaction holds a list of ChangedEvents collected during a transaction,
 * as the value of the read-only #changes property.
 *
 * Start a transaction by calling UndoManager#startTransaction
 * (or Model#startTransaction or Diagram#startTransaction, which call that method).
 * Be sure to finish a transaction with a matching call to UndoManager#commitTransaction
 * (or Model#commitTransaction or Diagram#commitTransaction),
 * or a call to UndoManager#rollbackTransaction
 * (or the same named methods on Model or Diagram).
 *
 * If you are performing multiple or repeated changes to a model or diagram,
 * surround all of the code with calls to start and commit the transaction --
 * do not perform repeated calls to start-commit-start-commit.
 * Typically each change caused by the user, such as a button click or a change of focus or a mouse drag,
 * should perform one transaction in which all changes are made.
 * All of the predefined commands and tools perform transactions.
 *
 * Undoing or redoing a transaction is done by calling UndoManager#undo or UndoManager#redo.
 * Those methods call the #undo or #redo methods here.
 *
 * The UndoManager holds a list of Transactions in its UndoManager#history.
 * @unrestricted
 * @category Model
 */
export class Transaction {
    /**
     * Construct an object holding an empty list of ChangedEvents
     * and no #name.
     */
    constructor();
    /**
     * Clear all of the saved changes.
     */
    clear(): void;
    /**
     * This predicate returns true if you can call #undo,
     * namely when #isComplete is true.
     * @return {boolean} true if ready for #undo to be called.
     */
    canUndo(): boolean;
    /**
     * Undo all of the changes, in reverse order.
     * This is normally called only by the UndoManager.
     * #canUndo must be true for this method to have any effect.
     */
    undo(): void;
    /**
     * This predicate returns true if you can call #redo,
     * namely when #isComplete is true.
     * @return {boolean} true if ready for #redo to be called.
     */
    canRedo(): boolean;
    /**
     * Re-perform these changes after an #undo.
     * This is normally called only by the UndoManager.
     * #canRedo must be true for this method to have any effect.
     */
    redo(): void;
    /**
     * This read-only property returns the list of ChangedEvents.
     * The changes are stored in order of occurrence.
     *
     * You should not modify this list.
     */
    readonly changes: List<ChangedEvent>;
    /**
     * Gets or sets the transaction name for this collection of changes.
     * This is set by a top-level call to UndoManager#commitTransaction.
     */
    name: string;
    /**
     * Gets or sets whether we can add more ChangedEvents to this list of changes.
     * This is initially false.
     * It is set to true by UndoManager#commitTransaction and UndoManager#rollbackTransaction.
     */
    isComplete: boolean;
}
/**
 * An UndoManager observes and records model and diagram changes in transactions and
 * supports undo/redo operations.
 * You will need to set the #isEnabled property to true in order for
 * the UndoManager to record changes and for users to perform an undo or a redo.
 *
 * Typically an operation will call #startTransaction,
 * make some changes to the Model and/or Diagram,
 * and then call #commitTransaction.
 * Any ChangedEvents that occur will be recorded in a
 * Transaction object.
 * If for some reason you do not wish to complete the transaction
 * successfully, you can call #rollbackTransaction instead
 * of #commitTransaction.
 *
 * For convenience the Diagram#commit and Model#commit methods
 * execute a function within a transaction and then perform a commit,
 * or else a rollback upon an error.
 *
 * The #history property is a list of Transactions.
 * #commitTransaction will add the #currentTransaction
 * to the #history list.
 * #rollbackTransaction will undo the changes remembered in the
 * #currentTransaction and then discard it, without changing the #history.
 * You can limit how many transactions are remembered in the history
 * by setting #maxHistoryLength.
 *
 * Transactions may be nested.
 * Be sure to call either #commitTransaction or #rollbackTransaction
 * for each call to #startTransaction.
 * Avoid repeated start-commit-start-commit calls as a result of a user's actions.
 * Instead, start, make all changes, and then commit.
 *
 * If you want to restore the diagram to the state before the latest complete
 * transaction, call #undo.
 * Call #redo to change the diagram to a later state.
 * If after some number of undo's you start a transaction,
 * all of the history after the current state is discarded,
 * and a new transaction may be recorded.
 * You cannot undo or redo during a transaction.
 *
 * Initially each Model has its own UndoManager.
 * UndoManagers may be shared by multiple Models by replacing
 * the standard Model#undoManager created by the model constructor.
 *
 * There are several informational properties:
 *   - #isInTransaction is true when a top-level transaction has been started that has not yet been committed or rolled-back.
 *   - #currentTransaction holds the flattened list of all ChangedEvents
 *     that have happened within the current transaction.
 *   - #transactionLevel indicates the current depth of nesting.
 *   - #nestedTransactionNames holds the stack of transaction names supplied to
 *     #startTransaction calls.
 *   - #history holds only complete top-level transactions.
 *   - #isUndoingRedoing is true during a call to #undo or #redo.
 *   - #historyIndex indicates which Transaction in the #history is the next to be "undone";
 *     this is decremented by each undo and incremented by each redo.
 *   - #transactionToUndo and #transactionToRedo indicate which Transaction may be undone or redone next, if any.
 *   - #models returns an iterator over all of the Models that this UndoManager is handling.
 *
 * A transaction may not be ongoing when replacing a Diagram#model,
 * because it would not make sense to be replacing the UndoManager (the Model#undoManager)
 * while changes are being recorded.
 *
 * Replacing a Diagram#model copies certain properties from the old UndoManager to the new one,
 * including #isEnabled and #maxHistoryLength.
 * @unrestricted
 * @category Model
 */
export class UndoManager {
    /**
     * The constructor produces an empty UndoManager
     * with no transaction history.
     */
    constructor();
    /**
     * Clear all of the Transactions and clear all other state,
     * including any ongoing transaction without rolling back.
     * However, this maintains its references to its Models.
     *
     * You should not call this method during a transaction.
     */
    clear(): void;
    /**
     * @expose
     * Undocumented.
     * Copy persistent properties from an old UndoManager to this new one.
     * This is called by the Diagram#model property setter.
     * @param old
     */
    copyProperties(old: UndoManager): void;
    /**
     * Make sure this UndoManager knows about a Model for which
     * it may receive ChangedEvents when the given Model is changed.
     * The model will also receive notifications about transactions and undo or redo operations.
     *
     * You should not call this method during a transaction.
     * @param {Model} model A Model that this UndoManager is managing.
     * @see #models
     * @see #removeModel
     */
    addModel(model: Model): void;
    /**
     * Inform this UndoManager that it will no longer be receiving ChangedEvents
     * when the given Model is changed.
     * The model will no longer receive notifications about transactions and undo or redo operations.
     *
     * You should not call this method during a transaction.
     * If you call this method between transactions when there is a transaction history,
     * you should be careful that there are no ChangedEvents referring to that model in any Transactions.
     * @param {Model} model A Model that this UndoManager should no longer manage.
     * @see #models
     * @see #addModel
     */
    removeModel(model: Model): void;
    /**
     * Begin a transaction, where the changes are held by a Transaction object
     * as the value of #currentTransaction.
     * You must call either #commitTransaction or #rollbackTransaction afterwards.
     *
     * For convenience, this method is called by Model#startTransaction and Diagram#startTransaction.
     *
     * Transactions can be nested.
     * Starting or ending a nested transaction will return false.
     * Nested transactions will share the same Transaction list of ChangedEvents.
     *
     * Starting a transaction will not necessarily cause #currentTransaction to be non-null.
     * A Transaction object is usually only created by #handleChanged when a ChangedEvent first occurs.
     * @param {string=} tname a short string describing the transaction, pushed onto the #nestedTransactionNames stack.
     * @return {boolean} true if starting a top-level transaction.
     * @see #commitTransaction
     * @see #rollbackTransaction
     */
    startTransaction(tname?: string): boolean;
    /**
     * Commit the current transaction started by a call to #startTransaction.
     *
     * For convenience, this method is called by Model#commitTransaction and Diagram#commitTransaction.
     *
     * If this call stops a top-level transaction,
     * we mark the #currentTransaction as complete (Transaction#isComplete),
     * we add the Transaction to the #history list,
     * and we return true.
     * Committing a transaction when there have been some undos without corresponding
     * redos will throw away the Transactions holding changes that happened
     * after the current state, before adding the new Transaction to the
     * #history list.
     * @param {string=} tname a short string describing the transaction;
     *        this is recorded as the Transaction#name and need not be the same as the string passed to #startTransaction.
     *        If the value is an empty string or not supplied, this will use the name given to #startTransaction.
     * @return {boolean} true if ending a top-level transaction.
     * @see #startTransaction
     * @see #rollbackTransaction
     */
    commitTransaction(tname?: string): boolean;
    /**
     * Rollback the current transaction started by a call to #startTransaction, undoing any changes.
     *
     * For convenience, this method is called by Model#rollbackTransaction and Diagram#rollbackTransaction.
     *
     * This undoes and then discards the changes in the #currentTransaction.
     * You must have started a transaction previously.
     * @return {boolean} true if ending a top-level transaction.
     * @see #startTransaction
     * @see #commitTransaction
     */
    rollbackTransaction(): boolean;
    /**
     * This predicate returns true if you can call #undo.
     * This will return false if #isEnabled is false (as it is by default),
     * if any transaction is ongoing, or
     * if there is no #transactionToUndo that can be undone.
     * @return {boolean} true if ready for #undo to be called.
     */
    canUndo(): boolean;
    /**
     * Reverse the effects of the #transactionToUndo.
     * #canUndo must be true for this method to have any effect.
     *
     * This is called by CommandHandler#undo.
     *
     * This will raise a "StartingUndo" ChangedEvent of type ChangedEvent.Transaction,
     * perform the Transaction#undo on the #transactionToUndo, and then
     * raise a "FinishedUndo" ChangedEvent of type ChangedEvent.Transaction.
     * The two ChangedEvents are to let model listeners know that an undo is about to take place
     * and that it just finished.
     * #isUndoingRedoing will temporarily be set to true during this operation.
     */
    undo(): void;
    /**
     * This predicate returns true if you can call #redo.
     * This will return false if #isEnabled is false (as it is by default),
     * if any transaction is ongoing, or
     * if there is no #transactionToRedo that can be redone.
     * @return {boolean} true if ready for #redo to be called.
     */
    canRedo(): boolean;
    /**
     * After an #undo, re-perform the changes in #transactionToRedo.
     * #canRedo must be true for this method to have any effect.
     *
     * This is called by CommandHandler#redo.
     *
     * This will raise a "StartingRedo" ChangedEvent of type ChangedEvent.Transaction,
     * perform the Transaction#redo on the #transactionToRedo, and then
     * raise a "FinishedRedo" ChangedEvent of type ChangedEvent.Transaction.
     * The two ChangedEvents are to let model listeners know that a redo is about to take place
     * and that it just finished.
     * #isUndoingRedoing will temporarily be set to true during this operation.
     */
    redo(): void;
    /**
     * Maybe record a ChangedEvent in the #currentTransaction.
     * This calls #skipsEvent to see if this should ignore the change.
     * If #skipsEvent returns false, this creates a copy of the ChangedEvent
     * and adds it to the #currentTransaction.
     * If there is no #currentTransaction, this first creates and remembers it.
     *
     * This method always ignores all changes while performing
     * an #undo or #redo.
     * This method is also a no-op if #isEnabled is false.
     * @param {ChangedEvent} e a ChangedEvent.
     */
    handleChanged(e: ChangedEvent): void;
    /**
     * This predicate is called by #handleChanged to decide if a ChangedEvent
     * is not interesting enough to be remembered.
     *
     * Transactional events (of change type ChangedEvent.Transaction) are always skipped.
     * Changed events for GraphObjects that are in Layer#isTemporary layers are also skipped.
     *
     * Sometimes changed events do not even get to #handleChanged because
     * Model#skipsUndoManager or Diagram#skipsUndoManager is true.
     * @expose
     * @param {ChangedEvent} e the ChangedEvent received by #handleChanged.
     * @return {boolean} true to not record the change.
     */
    skipsEvent(e: ChangedEvent): boolean;
    /**
     * This read-only property returns an iterator for all of the Models that this UndoManager is handling.
     * @see #addModel
     * @see #removeModel
     */
    readonly models: Iterator<Model>;
    /**
     * Gets or sets whether this UndoManager records any changes.
     * The default value is false -- you need to set this to true if
     * you want the user to be able to undo or redo.
     *
     * You can temporarily turn off recording by setting Diagram#skipsUndoManager
     * and Model#skipsUndoManager to true.
     */
    isEnabled: boolean;
    /**
     * This read-only property returns the Transaction in the #history to be undone next.
     * The value may be null if the UndoManager is not ready to perform an undo.
     * @see #transactionToRedo
     */
    readonly transactionToUndo: Transaction | null;
    /**
     * This read-only property returns the Transaction in the #history to be redone next.
     * The value may be null if the UndoManager is not ready to perform a redo.
     * @see #transactionToUndo
     */
    readonly transactionToRedo: Transaction | null;
    /**
     * This read-only property is true during a call to #undo or #redo.
     */
    readonly isUndoingRedoing: boolean;
    /**
     * This read-only property returns the whole history, a list of all of the Transactions,
     * each representing a transaction with some number of ChangedEvents.
     *
     * You should not modify this List.
     */
    readonly history: List<Transaction>;
    /**
     * Gets or sets the maximum number of transactions that this undo manager will remember.
     * When a transaction is committed and the number exceeds this value,
     * the UndoManager will discard the oldest transaction(s) in order to meet this limit.
     * The initial value is 999.
     * Any new value must be an integer.
     * A negative value is treated as if there were no limit.
     * A zero value will not remember any Transactions in the #history,
     * but will allow commits and rollbacks to occur normally,
     * including raising "Transaction" type ChangedEvents.
     *
     * This property is useful in helping limit the memory consumption of typical applications.
     * But this does not limit the number of ChangedEvents that are recorded,
     * because there may be an unlimited number of those within each Transaction.
     * Decreasing this value will not necessarily remove any existing Transactions
     * if there currently exist more in #history than the new value permits.
     */
    maxHistoryLength: number;
    /**
     * This read-only property returns the index into #history for the current undoable Transaction.
     * The value is -1 if there is no undoable Transaction to be undone.
     */
    readonly historyIndex: number;
    /**
     * This read-only property returns the current Transaction for recording additional model change events.
     * This is initialized and augmented by #handleChanged
     * before it is added to #history by a top-level call
     * to #commitTransaction.
     * The value will be null between transactions.
     */
    readonly currentTransaction: Transaction | null;
    /**
     * This read-only property returns the current transaction level.
     * The value is zero when there is no ongoing transaction.
     * The initial value is zero.
     * #startTransaction will increment this value;
     * #commitTransaction or #rollbackTransaction will decrement it.
     * When this value is greater than zero, #canUndo
     * and #canRedo will be false, because
     * additional logically related model change events may occur.
     */
    readonly transactionLevel: number;
    /**
     * This read-only property is true after the first call to #startTransaction
     * and before a corresponding call to #commitTransaction or #rollbackTransaction.
     *
     * During a transaction #canUndo and #canRedo will be false.
     * #currentTransaction may be non-null if any ChangedEvents were recorded.
     */
    readonly isInTransaction: boolean;
    /**
     * This read-only property returns a stack of ongoing transaction names.
     * The outermost transaction name will be the first item in the list.
     * The last one will be the name of the most recent (nested) call
     * to #startTransaction.
     *
     * You should not modify this List.
     */
    readonly nestedTransactionNames: List<string>;
}
/**
 * Tools handle mouse, keyboard, and touch events.
 * The currently running tool, Diagram#currentTool, receives all input events from the Diagram
 * via canonicalized InputEvents.
 * <p class="boxrun">
 * For more discussion, see <a href="../../intro/tools.html">Introduction to Tools</a>.
 * See samples that make use of tools in the <a href="../../samples/index.html#tools">samples index</a>.
 *
 * Most tools are "mode-less" tools that are managed by the ToolManager,
 * which chooses the current tool based on the kind and position of the mouse event and the parts in the diagram.
 * The ToolManager has properties holding instances of most of the pre-defined Tool classes.
 * These classes include:
 *   - In the ToolManager#mouseDownTools List:
 *     - ActionTool
 *     - RelinkingTool
 *     - LinkReshapingTool
 *     - RotatingTool
 *     - ResizingTool
 *   - In the ToolManager#mouseMoveTools List:
 *     - LinkingTool
 *     - DraggingTool
 *     - DragSelectingTool
 *     - PanningTool
 *   - In the ToolManager#mouseUpTools List:
 *     - ContextMenuTool
 *     - TextEditingTool
 *     - ClickCreatingTool
 *     - ClickSelectingTool
 *
 * The ToolManager chooses a tool to run as the diagram's current tool by finding in its lists of tools the first tool
 * whose #canStart method returns true.  The ToolManager then sets Diagram#currentTool to be that tool.
 *
 * A tool is in the "running" state when it is the value of Diagram#currentTool.
 * The Diagram#currentTool property setter will call #doStop on the old tool
 * and then call #doStart on the new tool.
 *
 * A tool can then go into the "active" state once it decides it can actually do something.
 * This happens with a call to #doActivate, normally called by the ToolManager.
 * Later it is deactivated (#doDeactivate) and then stopped.
 * #isActive should be true when the tool is "active".
 * Often tools should ignore certain common events, such as calls to #doMouseMove,
 * unless the tool #isActive.
 *
 * You can prevent a "mode-less" tool (i.e. one managed by the ToolManager)
 * from being started by the ToolManager by setting #isEnabled to false.
 *
 * You can also go into a particular "mode" by setting Diagram#currentTool explicitly,
 * thereby circumventing the normal operation of the ToolManager.
 * This ignores the #isEnabled property and does not call the #canStart predicate.
 * The behavior will depend on the tool -- not all of the predefined tools support operating as a "modal" tool.
 *
 * Tools cannot be shared amongst multiple Diagrams.
 *
 * If you define a Tool subclass, you may override any of the methods whose names start with "do"
 * and any other methods that are documented to be overridable, such as #canStart.
 * However you must seriously consider calling the base method in order to gets its default behavior.
 * There may be situations where not calling the base method may cause subtle bugs.
 * But that depends on the method and the tool.
 * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call the base method.
 * @unrestricted
 * @category Tool
 */
export abstract class Tool {
    /**
     * Don't construct this directly -- this is an abstract class.
     */
    constructor();
    /**
     * This read-only property returns the Diagram that owns this tool and
     * for which this tool is handling input events.
     */
    diagram: Diagram;
    /**
     * The diagram asks each tool to update any adornments the tool might
     * use for a given part.
     * If the tool uses its own tool handles,
     * this should display them or hide them as appropriate.
     * Typically this should only show them if the part is selected.
     *
     * By default this method does nothing.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Part} part
     */
    updateAdornments(part: Part): void;
    /**
     * This predicate is used by the ToolManager to decide if this tool can be started mode-lessly
     * by mouse and touch events.
     * Implementations of this method can look at Diagram#lastInput to get the mouse
     * event and input state.
     *
     * By default this method returns #isEnabled.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean} true if #isEnabled is true and
     * if the Diagram#toolManager can make this tool the Diagram#currentTool and
     * then call the #doStart method.
     */
    canStart(): boolean;
    /**
     * The Diagram calls this method when this tool becomes the current tool; you should not call this method.
     * Tool implementations should perform their per-use initialization here, such
     * as setting up internal data structures, or capturing the mouse.
     * Implementations of this method can look at Diagram#lastInput to get the mouse
     * event and input state.
     *
     * You should not call this method -- only the Diagram#currentTool property setter should call this method.
     *
     * By default this method does nothing.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * If you override this method, it is commonplace to also override #doStop to clean up whatever you set up in this method.
     * @expose
     */
    doStart(): void;
    /**
     * The Diagram calls this method after setting Diagram#currentTool,
     * to make the new tool active.
     * This should set #isActive to true.
     * Overrides of this method might call #startTransaction,
     * if this tool's activity involves modification of the model.
     * Implementations of this method can look at Diagram#lastInput to get the mouse
     * event and input state.
     *
     * You should call this method only after setting Diagram#currentTool to the Tool that you want to activate.
     *
     * By default this only sets #isActive to true.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * If you override this method, it is commonplace to also override #doDeactivate to clean up whatever you set up in this method.
     * @expose
     */
    doActivate(): void;
    /**
     * The Diagram calls this method on the old tool when
     * Diagram#currentTool is set to a new tool.
     * This needs to set #isActive to false.
     * Overrides of this method might call #stopTransaction,
     * if this tool's activity involves modification of the model.
     *
     * You should have no reason to call this method, because it is automatically called by the
     * Diagram#currentTool property setter on the old tool.
     *
     * By default this only sets #isActive to false.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * It is commonplace to override this method in order to clean up whatever you have set up in an override of #doActivate.
     * @expose
     */
    doDeactivate(): void;
    /**
     * The Diagram calls this method when this tool stops being the current tool; you should not call this method.
     * Tool implementations should perform their per-use cleanup here,
     * such as releasing mouse capture.
     *
     * You should not call this method -- only the Diagram#currentTool property setter should call this method.
     * If you want to stop a tool unexpectedly, you should call #doCancel.
     * If your implementation of a tool wants to stop itself, you should call #stopTool.
     *
     * By default this method does nothing.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * It is commonplace to override this method in order to clean up whatever you have set up in an override of #doStart.
     * @expose
     */
    doStop(): void;
    /**
     * The diagram will call this method when the user wishes to cancel the
     * current tool's operation.
     * Typically this is called when the user hits the ESCAPE key.
     * This should restore the original state of what was modified by this tool, and then it should call #stopTool.
     * This method is not responsible for cleaning up any side-effects that should be performed
     * by #doDeactivate and/or #doStop, which will always be called whether the tool stops normally or abnormally.
     *
     * By default this method just sets #transactionResult to null and calls #stopTool.
     * You will want to override this method even in tools that call #startTransaction and #stopTransaction,
     * because the UndoManager might not be enabled.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doCancel(): void;
    /**
     * If the Diagram#currentTool is this tool,
     * stop this tool and start the Diagram#defaultTool
     * by making it be the new current tool.
     * The implementation of various tool methods can call this method to stop the current tool.
     * This will call #doStop -- you should not call that method directly.
     *
     * If you want to stop the current tool and have it restore the original state, call #doCancel.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    stopTool(): void;
    /**
     * The diagram will call this method upon a mouse down event.
     * This is normally overridden for mouse-down tools;
     * it is not called for mouse-move or mouse-up tools.
     * However it may also be called when the tool is run in a modal fashion,
     * when code explicitly sets the diagram's Diagram#currentTool.
     * Implementations of this method can look at Diagram#lastInput to get the mouse
     * event and input state.
     *
     * By default this method checks #isActive; if that is false it calls #canStart.
     * If that in turn is true, this calls #doActivate.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doMouseDown(): void;
    /**
     * The diagram will call this method upon a mouse move event.
     * This is normally overridden for mouse-move tools;
     * it is not called for mouse-up tools.
     * However it may also be called when the tool is run in a modal fashion,
     * when code explicitly sets the diagram's Diagram#currentTool.
     * An override of this method usually does nothing when #isActive is false.
     * Implementations of this method can look at Diagram#lastInput to get the mouse
     * event and input state.
     *
     * By default this method does nothing.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doMouseMove(): void;
    /**
     * The diagram will call this method upon a mouse up event.
     * This is normally overridden for mouse-up tools.
     * An override of this method usually does nothing when #isActive is false,
     * except for calling #stopTool.
     * Tools normally stop upon a mouse up, by calling #stopTool.
     * If you want to handle multiple mouse down-up gestures in one tool activation,
     * you will need to override this method to only stop the tool when you want.
     * Implementations of this method can look at Diagram#lastInput to get the mouse
     * event and input state.
     *
     * By default this method just calls #stopTool.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doMouseUp(): void;
    /**
     * The diagram will call this method as the mouse wheel is rotated.
     * Implementations of this method can look at Diagram#lastInput to get the mouse
     * event and input state.
     *
     * By default this method does nothing.
     * (But the ToolManager#doMouseWheel override will call Tool#standardMouseWheel.)
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doMouseWheel(): void;
    /**
     * Called by ToolManager#doMouseDown and ToolManager#doMouseMove,
     * this method determines whether or not to allow pinch zooming from a multi-touch event.
     * By default this predicate just returns true.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     * @see #standardPinchZoomStart
     * @see #standardPinchZoomMove
     * @since 1.5
     */
    canStartMultiTouch(): boolean;
    /**
     * Initiates pinch-zooming on multi-touch devices.
     *
     * This is called by ToolManager#doMouseDown if the Diagram#lastInput has InputEvent#isMultiTouch
     * set to true and #canStartMultiTouch returns true.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canStartMultiTouch
     * @see #standardPinchZoomMove
     * @since 1.5
     */
    standardPinchZoomStart(): void;
    /**
     * Continues pinch-zooming (started by #standardPinchZoomStart on multi-touch devices.
     *
     * This is called by ToolManager#doMouseMove if the Diagram#lastInput has InputEvent#isMultiTouch
     * set to true and #canStartMultiTouch returns true.
     * By default this calls #doCancel in order to cancel the regular tool behavior caused by the multitouch events.
     * This then calculates the appropriate zoom level and calls CommandHandler#canResetZoom to decide
     * whether to call CommandHandler#resetZoom to actually set Diagram#scale.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canStartMultiTouch
     * @see #standardPinchZoomStart
     * @since 1.5
     */
    standardPinchZoomMove(): void;
    /**
     * The diagram will call this method upon a key down event.
     * By default this just calls #doCancel if the key is the ESCAPE key.
     * Implementations of this method can look at Diagram#lastInput to get the key.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doKeyDown(): void;
    /**
     * The diagram will call this method upon a key up event.
     * Implementations of this method can look at Diagram#lastInput to get the key.
     *
     * By default this method does nothing.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doKeyUp(): void;
    /**
     * Call Diagram#startTransaction with the given transaction name.
     * This always sets #transactionResult to null.
     *
     * This is normally called in an override of #doActivate, if the tool modifies the model,
     * along with a call to #stopTransaction in an override of #doDeactivate.
     * Alternatively, you can surround a block of code that sets the #transactionResult
     * with calls to startTransaction and stopTransaction.
     * @param {string=} tname a string describing the transaction.
     * @return {boolean} the value of the call to Diagram#startTransaction.
     */
    startTransaction(tname?: string): boolean;
    /**
     * If #transactionResult is null, call Diagram#rollbackTransaction,
     * otherwise call Diagram#commitTransaction.
     *
     * This is normally called in an override of #doDeactivate,
     * if #startTransaction was called in #doActivate.
     * Alternatively, you can surround a block of code that sets the #transactionResult
     * with calls to startTransaction and stopTransaction.
     * @return {boolean} the result of the call to rollback or commit the transaction.
     */
    stopTransaction(): boolean;
    /**
     * Implement the standard behavior for selecting parts with the mouse,
     * depending on the control and shift modifier keys.
     *
     * Control-clicking on a part will select it if it wasn't already,
     * and will deselect if it had been selected.
     * Shift-clicking on a part will add it to the selection (if it wasn't already).
     * Otherwise, clicking on a part will select it (if it wasn't already).
     *
     * Note that there are restrictions on selection.
     * For example, a part cannot be selected in this manner if Part#selectable is false,
     * or if Diagram#maxSelectionCount would be exceeded.
     *
     * A left click in the background of the diagram with no modifier keys clears the selection.
     *
     * This method does not implement any click event behavior -- that is implemented by #standardMouseClick.
     *
     * The ClickSelectingTool calls this method in its override of #doMouseUp in order to change the selection.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    standardMouseSelect(): void;
    /**
     * Implement the standard behavior for mouse clicks,
     * searching for and calling click handler functions on GraphObjects
     * or on Diagram,
     * and raising the corresponding DiagramEvent.
     *
     * A click on a GraphObject of the diagram will raise one of the following DiagramEvents:
     * "ObjectSingleClicked", "ObjectDoubleClicked", or "ObjectContextClicked".
     * This will also look at the corresponding click property:
     * GraphObject#click, GraphObject#doubleClick, or GraphObject#contextClick.
     * If the value is a function, this will call it, passing the current InputEvent
     * and the GraphObject.
     * If the value is null, it tries looking at the parent GraphObject#panel,
     * and so on, walking up the visual tree until it finds the appropriate function to call.
     * After calling the click function, if the value of InputEvent#handled is false,
     * this method will continue walking up the visual tree looking for more click functions
     * to call.
     * Once it has looked at the top-level object (a Part)
     * for a click function, this method stops.
     *
     * A click in the background of the diagram will raise one of the following DiagramEvents:
     * "BackgroundSingleClicked", "BackgroundDoubleClicked", or "BackgroundContextClicked".
     * This will also look at the corresponding click property:
     * Diagram#click, Diagram#doubleClick, or Diagram#contextClick.
     * If the value is a function, this will call it, passing the current InputEvent.
     *
     * This method is not responsible for selecting or deselecting any parts.
     * Call #standardMouseSelect for that functionality.
     *
     * Note that this calls GraphObject#isEnabledObject on the target object;
     * if it returns false, no click action will occur.
     *
     * The ClickSelectingTool calls this method in its override of #doMouseUp in order to raise "click" events.
     * Note that by default GraphObjects in Layers that are Layer#isTemporary will not be "clicked".
     * To change that behavior it is easiest to set GraphObject#isActionable to true on those objects for which you wish to handle "click" events.
     * Then the ActionTool's #doMouseUp override will raise the standard "click" events.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {function(GraphObject):GraphObject | null=} navig An optional custom navigation
     * function to find target objects.
     * @param {function(GraphObject):boolean | null=} pred An optional custom predicate
     * function to find target objects. No value means that only objects in layers holding permanent objects.
     * @return {boolean} true if InputEvent#handled had been set to true on the Diagram#lastInput.
     */
    standardMouseClick<T extends GraphObject>(navig?: (a: GraphObject) => (T | null), pred?: (a: T) => boolean): boolean;
    /**
     * Implement the standard behavior for mouse enter, over, and leave events,
     * where the mouse is moving but no button is pressed.
     * This should be called by mouse move event handlers when wanting to
     * detect and invoke mouse enter/over/leave event handlers.
     *
     * The GraphObject#mouseEnter property provides a function to call
     * when the mouse first enters an object or any of its contained objects
     * (if the object is actually a Panel).
     *
     * The GraphObject#mouseLeave property provides a function to call
     * when the mouse leaves an object and all of its contained objects
     * (if the object is actually a Panel).
     *
     * The GraphObject#mouseOver property and Diagram#mouseOver
     * properties provide functions to call when the mouse moves but stays
     * within the same GraphObject or when the mouse moves in the
     * background of the Diagram.
     *
     * This method is also responsible for updating the Diagram#currentCursor
     * according to the value of GraphObject#cursor and Diagram#defaultCursor.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    standardMouseOver(): void;
    /**
     * Implement the standard behavior for mouse wheel events.
     * ToolManager#doMouseWheel calls this method.
     *
     * Turning the mouse wheel if Diagram#allowVerticalScroll is true
     * causes the diagram to scroll up or down.
     * If InputEvent#shift and Diagram#allowHorizontalScroll are true,
     * the diagram scrolls left or right.
     *
     * If InputEvent#control and Diagram#allowZoom are true,
     * turning the mouse wheel changes the diagram's scale,
     * zooming in or out while trying to keep the point in the model
     * at the same point as the mouse.
     *
     * The value of ToolManager#mouseWheelBehavior affects what operations might occur
     * upon mouse wheel events.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    standardMouseWheel(): void;
    /**
     * This is called to start a new timer to call #doWaitAfter after a given delay.
     * It first cancels any previously running "WaitAfter" timer, by calling #cancelWaitAfter.
     *
     * This is normally used to implement mouse hover and mouse hold events.
     * If the mouse has moved, it must not have moved beyond the distance as determined by
     * Tool#isBeyondDragSize for it be considered "stationary".
     * So the regular ToolManager#doMouseMove implementation only calls
     * this method when the mouse has moved beyond the drag size.
     *
     * This method is rarely overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {number} delay The delay, in milliseconds.
     * @param {InputEvent=} event An optional event that caused this timer. Defaults to Diagram#lastInput.
     * This gets passed on to #doWaitAfter.
     */
    standardWaitAfter(delay: number, event?: InputEvent): void;
    /**
     * This is called to cancel any running "WaitAfter" timer.
     *
     * This is called when a tool is stopped.
     *
     * This method is rarely overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    cancelWaitAfter(): void;
    /**
     * This is called a certain delay after a call to #standardWaitAfter if there
     * has not been any call to #cancelWaitAfter.
     * The ToolManager overrides this method in order to implement support for mouse-hover behavior and tooltips.
     *
     * By default this does nothing.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {InputEvent} event The event that caused #standardWaitAfter.
     */
    doWaitAfter(event: InputEvent): void;
    /**
     * This convenience function finds the front-most GraphObject
     * that is at a given point and that is an element of an Adornment
     * that is of a given category.
     * The tool handle must be an immediate element of the Adornment,
     * not a GraphObject that is nested within Panels within the Adornment.
     *
     * This method is very infrequently overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point} p a Point in document coordinates.
     * @param {string} category the required Part#category of the Adornment.
     * @return {GraphObject}
     */
    findToolHandleAt(p: Point, category: string): GraphObject | null;
    /**
     * Return true when the last mouse point is far enough away from the first mouse down point
     * to constitute a drag operation instead of just a potential click.
     *
     * This uses the value of ToolManager#dragSize.
     * On touch devices the value is automatically increased to accommodate the unavoidable movement of fingers.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point=} first Point in view coordinates, defaults to Diagram#firstInput's InputEvent#viewPoint.
     * @param {Point=} last Point in view coordinates, defaults to Diagram#lastInput's InputEvent#viewPoint.
     * @return {boolean}
     */
    isBeyondDragSize(first?: Point, last?: Point): boolean;
    /**
     * Gets or sets the name of this tool.
     * The default name is an empty string,
     * but the constructor for each instance of a subclass of Tool will initialize it appropriately.
     * For example, the name of the DragSelectingTool is "DragSelecting".
     *
     * This name is sometimes used by tools that use Adornments as the Part#category for their Adornments.
     * It is also sometimes used by tools that conduct transactions as the transaction name.
     */
    name: string;
    /**
     * Gets or sets whether this tool can be started by a mouse event.
     *
     * Set this to false to prevent #canStart from returning true.
     * Setting this property to false should prevent this tool from being used in a mode-less fashion
     * by the ToolManager with a mouse down/move/up event.
     * However, even when this property is false, this tool can still be used in a modal fashion:
     * it can still be started by explicitly setting the
     * Diagram#currentTool property to this tool.
     *
     * The default value is true.
     */
    isEnabled: boolean;
    /**
     * Gets or sets whether this tool is started and is actively doing something.
     *
     * You can set this to true after your tool is started (i.e. when it is the
     * Diagram#currentTool and #doStart
     * had been called), but when it is not yet in a state
     * that it is actually "doing" something, because it is waiting for the right
     * circumstances.  This is typically only important when the tool is used in
     * a modal fashion.
     *
     * The default value is false.
     * This is normally set by #doActivate and #doDeactivate.
     */
    isActive: boolean;
    /**
     * Gets or sets the name of the transaction to be committed by #stopTransaction
     *
     * If null, the transaction will be rolled back.
     *
     * If this is non-null at the time of a call to #stopTransaction,
     * it calls Diagram#commitTransaction with this transaction name;
     * if this is null at that time, it calls Diagram#rollbackTransaction.
     *
     * The default value is null; #startTransaction will also set this to null.
     * Because a value of null when #stopTransaction is called will rollback the transaction,
     * it is important that your code sets this property to a non-null value when it thinks it has succeeded.
     *
     * This property exists so that no matter what execution path occurs to end the usage of a tool,
     * any ongoing transaction can be properly committed or rolled-back.
     * Many tools call #startTransaction and #stopTransaction; thus they set this property
     * for their transaction to be committed.
     * #doCancel also sets this property to null.
     */
    transactionResult: string | null;
}
/**
 * This special Tool is responsible for managing all of the Diagram's
 * mode-less tools, which you can access as the Diagram#toolManager.
 *
 * Mode-less tools are tools that are present in one of the following lists:
 * #mouseDownTools, #mouseMoveTools, or #mouseUpTools.
 * This ToolManager tool is normally the Diagram#defaultTool,
 * so it is also usually the Diagram#currentTool when the user is doing "nothing".
 *
 * When this tool is running as the current tool, it handles mouse-down,
 * mouse-move, and mouse-up events and the corresponding touch events.
 * For each event it iterates over each of the tools in the corresponding list,
 * calling the tool's Tool#canStart predicate.
 * If that predicate returns true, it starts that tool by making it the
 * diagram's current tool.
 * It then activates the tool and passes on the event to the tool by calling
 * the corresponding method (either Tool#doMouseDown,
 * Tool#doMouseMove, or Tool#doMouseUp).
 *
 * Because this tool is typically the one running as the diagram's current tool
 * when the user isn't "doing" anything, this tool can also handle other events,
 * such as mouse wheel events and keyboard commands.
 *
 * Keyboard events are just passed on to the Diagram#commandHandler's
 * CommandHandler#doKeyDown or CommandHandler#doKeyUp method.
 *
 * This tool also is responsible for showing tooltips.
 * Tooltip Adornments may be declared as any GraphObject#toolTip,
 * or as the Diagram#toolTip if the mouse or finger remains motionless in the background of the diagram.
 * You can set #toolTipDuration to control how long the tooltip remains visible after being motionless.
 *
 * This tool does not utilize any tool handles.
 * This tool does not conduct any transactions.
 * But of course some of the tools that the ToolManager starts can show tool handles and/or
 * conduct their own transactions.
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class ToolManager extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the Diagram#toolManager, which you can modify.
     *
     * The Tool#name of this tool is "ToolManager".
     *
     * The constructor produces a ToolManager that manages no tools.
     * Call #initializeStandardTools to create various tools,
     * initialize the tool properties such as #draggingTool,
     * and initialize the three mouse tool lists with those newly created tools.
     */
    constructor();
    /**
     * This default value for #mouseWheelBehavior indicates that mouse wheel events scroll the diagram.
     * @constant
     */
    static WheelScroll: EnumValue;
    /**
     * This value for #mouseWheelBehavior indicates that the mouse wheel events change the scale of the diagram.
     * @constant
     */
    static WheelZoom: EnumValue;
    /**
     * This value for #mouseWheelBehavior indicates that the mouse wheel events are ignored,
     * although scrolling or zooming by other means may still be allowed.
     * @constant
     * @since 1.2
     */
    static WheelNone: EnumValue;
    /**
     * This value for #gestureBehavior indicates that the pointer/touch pinch gestures
     * on the canvas intend to zoom the Diagram.
     * @constant
     * @since 1.5
     */
    static GestureZoom: EnumValue;
    /**
     * This value for #gestureBehavior indicates that the pointer/touch pinch gestures
     * on the canvas intend to have no effect on the Diagram, but also no effect on the page.
     * @constant
     * @since 1.5
     */
    static GestureCancel: EnumValue;
    /**
     * This value for #gestureBehavior indicates that the pointer/touch pinch gestures
     * on the canvas intend to have no effect on the Diagram, but will not be prevented,
     * and may bubble up the page to have other effects (such as zooming the page).
     * @constant
     * @since 1.5
     */
    static GestureNone: EnumValue;
    /**
     * Gets or sets the ToolManager's mouse wheel behavior. Allowed values are ToolManager.WheelScroll and
     * ToolManager.WheelZoom and ToolManager.WheelNone.
     *
     * The default value is ToolManager.WheelScroll.
     */
    mouseWheelBehavior: EnumValue;
    /**
     * Gets or sets the ToolManager's default gestureBehavior behavior. Allowed values are ToolManager.GestureZoom and
     * ToolManager.GestureCancel and ToolManager.GestureNone.
     *
     * The default value is ToolManager.GestureZoom.
     * @since 1.5
     */
    gestureBehavior: EnumValue;
    /**
     * Initialize the three mouse tool lists with instances of the standard tools.
     * This adds new instances of tools to the following three mouse tool lists:
     * #mouseDownTools, #mouseMoveTools, or #mouseUpTools.
     * This also sets the various tool properties of this ToolManager
     * to those newly created tools.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    initializeStandardTools(): void;
    /**
     * Iterate over the #mouseDownTools list and start the first tool
     * for which its Tool#canStart predicate returns true.
     *
     * Starting a tool replaces the Diagram#currentTool with the new tool.
     * Successfully doing so also activates the new tool by calling Tool#doActivate
     * and passes on the mouse-down event to it by calling Tool#doMouseDown.
     *
     * Not finding any startable tools causes this tool manager to activate,
     * thereby enabling the mouse-move and mouse-up behaviors and starts
     * detection of a mouse-hold event after #holdDelay milliseconds.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doMouseDown(): void;
    /**
     * Iterate over the #mouseMoveTools list and start the first tool
     * for which its Tool#canStart predicate returns true.
     *
     * Starting a tool replaces the Diagram#currentTool with the new tool.
     * Successfully doing so also activates the new tool by calling Tool#doActivate
     * and passes on the mouse-move event to it by calling Tool#doMouseMove.
     *
     * If no tool is found and activated, this:
     *   - Calls Tool#standardMouseOver to invoke any mouse-enter, mouse-over, or mouse-leave functions.
     *   - if the mouse has moved sufficiently, such that Tool#isBeyondDragSize is true,
     *     this also calls Tool#standardWaitAfter for a possible call to
     *     #doWaitAfter after #hoverDelay milliseconds.
     *   - Sets the Diagram#lastInput to bubble.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doMouseMove(): void;
    /**
     * Implement the standard behavior for when the mouse has not moved for a period of time.
     * This is due to an expired timer started by calling Tool#standardWaitAfter.
     *
     * This calls #doMouseHover and, if not mouse-down, #doToolTip.
     * Afterwards, if the event that called Tool#standardWaitAfter was a touch event,
     * this simulates a right click, enabling context menu functionality and other actions on touch devices.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {InputEvent} event The event that caused #standardWaitAfter. In this override,
     * if it is a touch event and the Diagram#lastInput event is not handled (InputEvent#handled set to false),
     * then this method will simulate a right click.
     */
    doWaitAfter(event: InputEvent): void;
    /**
     * Implement the standard behavior for mouse hover and mouse hold events,
     * called by #doWaitAfter when the mouse has not moved for a period of time.
     *
     * If there has been no mouse down, the timer is set for #hoverDelay milliseconds.
     * If it executes, it calls any GraphObject#mouseHover function on the object
     * at the mouse or on any of its containing panels,
     * or it calls any Diagram#mouseHover function for a background mouse-hover event.
     *
     * If there had been a mouse down, the timer is set for #holdDelay milliseconds.
     * If it executes, it calls any GraphObject#mouseHold function on the object
     * at the mouse or on any of its containing panels,
     * or it calls any Diagram#mouseHold function for a background mouse-held-down event.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doMouseHover(): void;
    /**
     * Implement the standard behavior for tooltips,
     * called by #doWaitAfter when the mouse has not moved for a period of time.
     *
     * This looks for a GraphObject at the latest mouse point.
     * If it finds an object, it checks for a GraphObject#toolTip.
     * If it has none, this method searches up the visual tree for a containing
     * Panel that does have a tooltip.
     *
     * If it didn't find any object, this looks for a Diagram#toolTip.
     *
     * If it eventually finds a tooltip, this calls #showToolTip.
     * Otherwise this calls #hideToolTip.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doToolTip(): void;
    /**
     * Show a tooltip Adornment or HTMLInfo.
     * This is called by #doToolTip once that method has found a tooltip to display.
     *
     * This calls #positionToolTip to make it easier to customize how the tooltip
     * is positioned relative to the object with the tooltip.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Adornment|HTMLInfo} tooltip
     * @param {GraphObject} obj The GraphObject getting the tooltip; this is null if the tooltip is being shown for the diagram background.
     */
    showToolTip(tooltip: Adornment | HTMLInfo, obj: GraphObject): void;
    /**
     * This is called by #showToolTip to position the part within the viewport.
     * It normally goes just below the cursor.
     * But if the mouse is too close to the right edge or the bottom edge of the viewport,
     * it is positioned left and/or above the cursor.
     *
     * This method only operates if the tooltip, an Adornment, does not have a Adornment#placeholder.
     * When there is a Placeholder in the tooltip, that Adornment is automatically positioned
     * so that the Placeholder is positioned at the adorned object, the second argument to this method.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * If you override this method to position the tooltip,
     * the tooltip has already been measured but not arranged,
     * so you can use its GraphObject#measuredBounds width and height
     * but not its GraphObject#actualBounds.
     * @expose
     * @param {Adornment} tooltip
     * @param {GraphObject} obj The GraphObject getting the tooltip,
     * or null if the tooltip is for the diagram background.
     */
    positionToolTip(tooltip: Adornment, obj: GraphObject): void;
    /**
     * Hide any tooltip.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    hideToolTip(): void;
    /**
     * This read-only property returns the currently showing tooltip, or null if there is none.
     */
    currentToolTip: Adornment | HTMLInfo | null;
    /**
     * Iterate over the #mouseUpTools list and start the first tool
     * for which its Tool#canStart predicate returns true.
     *
     * Starting a tool replaces the Diagram#currentTool with the new tool.
     * Successfully doing so also activates the new tool by calling Tool#doActivate
     * and passes on the mouse-up event to it by calling Tool#doMouseUp.
     *
     * If no startable tool is found it deactivates this tool manager,
     * to get ready for a mouse-down and ignore mouse-move and mouse-up events.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doMouseUp(): void;
    /**
     * The diagram will call this method as the mouse wheel is rotated.
     *
     * By default this just calls Tool#standardMouseWheel.
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doMouseWheel(): void;
    /**
     * This just calls CommandHandler#doKeyDown on the diagram's Diagram#commandHandler.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doKeyDown(): void;
    /**
     * This just calls CommandHandler#doKeyUp on the diagram's Diagram#commandHandler.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doKeyUp(): void;
    /**
     * Find a mouse tool of a given name.
     * This searches the #mouseDownTools, #mouseMoveTools, and #mouseUpTools lists.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {string} name the type of tool, such as "Dragging" or "ClickSelecting".
     * @return {Tool} a Tool whose Tool#name exactly matches the given name,
     * or null if no such tool is found in any of the three lists.
     */
    findTool(name: string): Tool | null;
    /**
     * Replace a mouse tool of a given name with a new tool, or remove an existing tool (if the *newtool* is null).
     * This searches the #mouseDownTools, #mouseMoveTools,
     * and #mouseUpTools lists.
     * The new tool is inserted into the same list in which the same-named tool is found,
     * at the same position as the old tool.
     * However, if no existing tool with the given name is present, this does **not** add the new tool to any list, since it cannot know where it should be added.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {string} name the type of tool, such as "Dragging" or "ClickSelecting".
     * @param {Tool} newtool If null, any tool that the search finds will just be removed
     * from the list in which it was found.
     * @return {Tool} the old tool that was replaced by the new one; this is null if none was found and the new tool was not added to any mouse tool list
     */
    replaceTool(name: string, newtool: Tool): Tool;
    /**
     * This read-only property returns the list of Tools that might be started upon a mouse or finger press event.
     * When the ToolManager handles a mouse-down or touch-down event in #doMouseDown,
     * it searches this list in order, starting the first tool for which
     * Tool#canStart returns true.
     *
     * This list may be modified, but it must not be modified while any tool
     * is handling events.
     *
     * #initializeStandardTools installs the following tools, in order:
     *   - #actionTool, an ActionTool
     *   - #relinkingTool, a RelinkingTool
     *   - #linkReshapingTool, a LinkReshapingTool
     *   - #rotatingTool, a RotatingTool
     *   - #resizingTool, a ResizingTool
     */
    readonly mouseDownTools: List<Tool>;
    /**
     * This read-only property returns the list of Tools that might be started upon a mouse or finger move event.
     * When the ToolManager handles a mouse-move or touch-move event in #doMouseMove,
     * it searches this list in order, starting the first tool for which
     * Tool#canStart returns true.
     *
     * This list may be modified, but it must not be modified while any tool
     * is handling events.
     *
     * #initializeStandardTools installs the following tools, in order:
     *   - #linkingTool, a LinkingTool
     *   - #draggingTool, a DraggingTool
     *   - #dragSelectingTool, a DragSelectingTool
     *   - #panningTool, a PanningTool
     */
    readonly mouseMoveTools: List<Tool>;
    /**
     * This read-only property returns the list of Tools that might be started upon a mouse or finger up event.
     * When the ToolManager handles a mouse-up or touch-up event in #doMouseUp,
     * it searches this list in order, starting the first tool for which
     * Tool#canStart returns true.
     *
     * This list may be modified, but it must not be modified while any tool
     * is handling events.
     *
     * #initializeStandardTools installs the following tools, in order:
     *   - #contextMenuTool, a ContextMenuTool
     *   - #textEditingTool, a TextEditingTool
     *   - #clickCreatingTool, a ClickCreatingTool
     *   - #clickSelectingTool, a ClickSelectingTool
     */
    readonly mouseUpTools: List<Tool>;
    /**
     * Gets or sets the time between when the mouse stops moving and a hover event,
     * in milliseconds. This value affects the delay before GraphObject#toolTips are shown.
     * The default value is 850 milliseconds.
     */
    hoverDelay: number;
    /**
     * Gets or sets the time between when the mouse stops moving and a hold event,
     * in milliseconds.
     * The default value is 850 milliseconds.
     */
    holdDelay: number;
    /**
     * Gets or sets the distance in view coordinates within which a mouse down-and-up is considered a click and beyond which a mouse movement is considered a drag.
     * The default value is 2 pixels horizontally and vertically for mouse events, and increases by 6 pixels for touch events.
     * This value is used by Tool#isBeyondDragSize.
     * @since 1.2
     */
    dragSize: Size;
    /**
     * Gets or sets how long a tool tip is visible.
     * The default value is 5000 milliseconds.
     *
     * This is used by #showToolTip to determine how long to wait before calling #hideToolTip.
     */
    toolTipDuration: number;
    /**
     * Gets or sets the mode-less ActionTool, normally one of the #mouseDownTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    actionTool: ActionTool;
    /**
     * Gets or sets the mode-less RelinkingTool, normally one of the #mouseDownTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    relinkingTool: RelinkingTool;
    /**
     * Gets or sets the mode-less LinkReshapingTool, normally one of the #mouseDownTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    linkReshapingTool: LinkReshapingTool;
    /**
     * Gets or sets the mode-less ResizingTool, normally one of the #mouseDownTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    resizingTool: ResizingTool;
    /**
     * Gets or sets the mode-less RotatingTool, normally one of the #mouseDownTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    rotatingTool: Tool;
    /**
     * Gets or sets the mode-less LinkingTool, normally one of the #mouseMoveTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    linkingTool: LinkingTool;
    /**
     * Gets or sets the mode-less DraggingTool, normally one of the #mouseMoveTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    draggingTool: DraggingTool;
    /**
     * Gets or sets the mode-less DragSelectingTool, normally one of the #mouseMoveTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    dragSelectingTool: DragSelectingTool;
    /**
     * Gets or sets the mode-less PanningTool, normally one of the #mouseMoveTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    panningTool: PanningTool;
    /**
     * Gets or sets the mode-less ContextMenuTool, normally one of the #mouseUpTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    contextMenuTool: ContextMenuTool;
    /**
     * Gets or sets the mode-less TextEditingTool, normally one of the #mouseUpTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    textEditingTool: TextEditingTool;
    /**
     * Gets or sets the mode-less ClickCreatingTool, normally one of the #mouseUpTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    clickCreatingTool: ClickCreatingTool;
    /**
     * Gets or sets the mode-less ClickSelectingTool, normally one of the #mouseUpTools.
     *
     * You can disable this tool by setting its Tool#isEnabled property to false.
     */
    clickSelectingTool: ClickSelectingTool;
}
/**
 * The DraggingTool is used to move or copy selected parts with the mouse.
 * This sets the Part#location property; you may want to save the location to the model
 * by using a TwoWay Binding on the "location" property in your Parts/Nodes/Groups templates.
 *
 * Dragging the selection moves parts for which Part#canMove is true.
 * If the user holds down the Control key (Option key on Mac), this tool will make a copy of the parts being dragged,
 * for those parts for which Part#canCopy is true.
 *
 * When the drag starts it calls #computeEffectiveCollection to find the actual collection
 * of Parts to be dragged.
 * Normally this collection includes not only the Diagram#selection, but also parts that belong
 * to those selected parts, such as members of groups.
 * If #dragsTree is true, the effective collection also includes all of the nodes and links that
 * constitute the subtree starting from selected nodes.
 * The result of #computeEffectiveCollection is not a Set but a Map
 * which remembers the original Part#location for all of the dragged parts.
 * This map is saved as the value of #draggedParts.
 *
 * During the drag if the user holds down the Control/Option key this tool makes a copy of the #draggedParts
 * and proceeds to drag it around.
 * (It only copies the Diagram#selection, not the whole effective collection,
 * if #copiesEffectiveCollection is false.)
 * The collection of copied parts is held by #copiedParts.
 * It too is a Map remembering the original locations of the parts.
 * #copiedParts will be null when this tool is moving (not copying) at the moment.
 *
 * Each Part's movement is limited by the Diagram#computeMove method.
 * By default it limits the Part#location to be within the bounds given by Part#minLocation and Part#maxLocation.
 * (Those default to minus Infinity to plus Infinity.)
 * As a further convenience, the value of NaN in minLocation and maxLocation cause Diagram#computeMove to use the
 * part's current location.
 * So, for example, an easy way to declare that the user may only drag a node horizontally is to just set:
 * ```js
 * $(go.Node,
 *   . . .
 *   { minLocation: new go.Point(-Infinity, NaN), maxLocation: new go.Point(Infinity, NaN) },
 *   . . .
 * )
 * ```
 *
 * If you set #isGridSnapEnabled to true, dragged or copied parts will be snapped to points on a grid.
 * The snapping occurs continuously during a drag unless you set #isGridSnapRealtime to false.
 * Normally the grid points come from the Diagram#grid, even if that grid is not GraphObject#visible.
 * However you can override those grid's properties for the snapping grid cell size and offset
 * by setting the properties here: #gridSnapCellSize and #gridSnapOrigin.
 * This computes the point to snap to for each dragged part.
 * The resulting point is used as the new Part#location.
 *
 * For the most general control over where a part may be dragged, either set the Part#dragComputation property
 * or override Diagram#computeMove.
 * For the common case of wanting to keep member nodes within the Group that they are members of,
 * you can do something like:
 * ```js
 *  // this is a Part.dragComputation function for limiting where a Node may be dragged
 *   function stayInGroup(part, pt, gridpt) {
 *     // don't constrain top-level nodes
 *     var grp = part.containingGroup;
 *     if (grp === null) return pt;
 *     // try to stay within the background Shape of the Group
 *     var back = grp.resizeObject;
 *     if (back === null) return pt;
 *     // allow dragging a Node out of a Group if the Shift key is down
 *     if (part.diagram.lastInput.shift) return pt;
 *     var p1 = back.getDocumentPoint(go.Spot.TopLeft);
 *     var p2 = back.getDocumentPoint(go.Spot.BottomRight);
 *     var b = part.actualBounds;
 *     var loc = part.location;
 *     // find the padding inside the group's placeholder that is around the member parts
 *     var m = grp.placeholder.padding;
 *     // now limit the location appropriately
 *     var x = Math.max(p1.x + m.left, Math.min(pt.x, p2.x - m.right - b.width - 1)) + (loc.x-b.x);
 *     var y = Math.max(p1.y + m.top, Math.min(pt.y, p2.y - m.bottom - b.height - 1)) + (loc.y-b.y);
 *     return new go.Point(x, y);
 *   }
 * ```
 * Note that this expects there to be a "SHAPE" object within the Group's visual tree that delimits
 * where the part may be dragged within the group.
 * This also expects that Group#computesBoundsIncludingLinks is false.
 * Then in your node template(s), just set:
 * ```js
 * $(go.Node,
 *   . . .,
 *   { dragComputation: stayInGroup },
 *   . . .
 * )
 * ```
 *
 * This tool does not utilize any Adornments or tool handles.
 * If the drag is successful, it raises the "SelectionMoved" or "SelectionCopied" DiagramEvent
 * and produces a "Move" or a "Copy" transaction.
 *
 * If you want to programmatically start a new user's dragging of a particular existing node,
 * you can make sure that node is selected, set the #currentPart property, and then start and activate the tool.
 * ```js
 *   var node = ...;
 *   myDiagram.select(node);        // in this case the only selected node
 *   var tool = myDiagram.toolManager.draggingTool;
 *   tool.currentPart = node;       // the DraggingTool will not call standardMouseSelect
 *   myDiagram.currentTool = tool;  // starts the DraggingTool
 *   tool.doActivate();             // activates the DraggingTool
 * ```
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class DraggingTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#draggingTool, which you can modify.
     *
     * The Tool#name of this tool is "Dragging".
     */
    constructor();
    /**
     * Gets or sets whether for any internal copying operation is permitted by control-drag-and-drop.
     * This property affects the behavior of #mayCopy,
     * but does not affect whether copied objects may be dropped into this diagram from a different diagram.
     *
     * The default value is true.
     * Setting this property does not raise any events.
     * @since 1.4
     */
    isCopyEnabled: boolean;
    /**
     * Gets or sets whether for a copying operation the extended selection
     * is copied or only the selected parts.
     * The default value is true.
     * Setting this property does not raise any events.
     *
     * The CommandHandler#copiesConnectedLinks property serves a similar role for the CommandHandler#copySelection command,
     * when the user types control-C to copy the currently selected parts.
     */
    copiesEffectiveCollection: boolean;
    /**
     * Gets or sets the DraggingTool's DraggingOptions instance, which controls several dragging properties.
     *
     * Several DraggingTool properties are just convenience properties:
     *   - isGridSnapEnabled
     *   - isGridSnapRealtime
     *   - gridSnapCellSize
     *   - gridSnapCellSpot
     *   - gridSnapOrigin
     *   - dragsLink
     *   - dragsTree
     *
     * Setting any of these properties really sets the corresponding dragOptions property.
     *
     * Setting this property does not raise any events.
     * @since 2.0
     */
    dragOptions: DraggingOptions;
    /**
     * Gets or sets whether the DraggingTool snaps objects to grid points.
     * Whether the snapping movement of the dragged parts occurs during the drag or only upon a drop
     * is determined by the value of #isGridSnapRealtime.
     *
     * This property does not affect dragging disconnected links,
     * but those links to respect the Part#dragComputation, which can be used to snap them.
     *
     * By default this property is false.
     * Setting this property does not raise any events.
     *
     * This property is a convenience getter/setter, and sets a value on dragOptions.
     */
    isGridSnapEnabled: boolean;
    /**
     * Gets or sets whether link routing takes some short-cuts during dragging.
     * When false Links whose routing is AvoidsNodes are not routed to avoid Nodes, in order to improve dragging performance.
     * The default value is true.
     * @since 1.4
     */
    isComplexRoutingRealtime: boolean;
    /**
     * Gets or sets whether the DraggingTool snaps objects to grid points during the drag.
     * This property is ignored unless #isGridSnapEnabled is true.
     * By default this property is true; when false parts are only snapped to grid locations upon the drop (i.e. mouse-up).
     * Setting this property does not raise any events.
     *
     * This property is a convenience getter/setter, and sets a value on dragOptions.
     * @since 1.1
     */
    isGridSnapRealtime: boolean;
    /**
     * Gets or sets the size of the grid cell used when snapping during a drag
     * if the value of #isGridSnapEnabled is true.
     * By default this property is the Size(NaN, NaN), which causes this tool to use the Panel#gridCellSize value of the Diagram#grid.
     * Setting this property does not raise any events.
     *
     * This property is a convenience getter/setter, and sets a value on dragOptions.
     */
    gridSnapCellSize: Size;
    /**
     * Gets or sets the Spot that specifies what point in the grid cell dragged parts snap to,
     * if the value of #isGridSnapEnabled is true.
     * By default this property is Spot.TopLeft:
     * node locations will snap exactly to the grid point.
     * Setting this property does not raise any events.
     *
     * This property is a convenience getter/setter, and sets a value on dragOptions.
     */
    gridSnapCellSpot: Spot;
    /**
     * Gets or sets the snapping grid's origin point, in document coordinates,
     * if the value of #isGridSnapEnabled is true.
     * By default this property is the Point(NaN, NaN),
     * which causes this tool to use the Panel#gridOrigin value from the Diagram#grid.
     * Setting this property does not raise any events.
     *
     * This property is a convenience getter/setter, and sets a value on dragOptions.
     */
    gridSnapOrigin: Point;
    /**
     * Gets or sets whether the user can drag a single Link,
     * disconnecting it from its connected nodes and possibly connecting it
     * to valid ports when the link is dropped.
     * The default value is false.
     * Setting this property does not raise any events.
     *
     * In order to avoid too many cases of having both ends of a dragged Link connect to the same node (if allowed),
     * it is commonplace to decrease the LinkingBaseTool#portGravity to a smaller value such as 10 or 20.
     *
     * This property is a convenience getter/setter, and sets a value on dragOptions.
     * @since 1.3
     */
    dragsLink: boolean;
    /**
     * Gets or sets whether moving or copying a node also includes all of the
     * node's tree children and their descendants, along with the links to those additional nodes.
     * The default value is false.
     * Setting this property does not raise any events.
     *
     * The CommandHandler#copiesTree property serves a similar role for the CommandHandler#copySelection command,
     * when the user types control-C to copy the currently selected parts.
     *
     * This property is a convenience getter/setter, and sets a value on dragOptions.
     */
    dragsTree: boolean;
    /**
     * Gets the Part found at the mouse point.
     * This is normally set by a call to #standardMouseSelect.
     */
    currentPart: Part | null;
    /**
     * Gets the collection of Parts that this tool has copied.
     * The value is a Map mapping Parts to DraggingInfo Objects
     * that have a "point" property remembering the original location of that Part.
     * The value is null when moving instead of copying.
     *
     * #draggedParts provides the map of Parts that are being moved and from which this collection was copied.
     */
    copiedParts: Map<Part, DraggingInfo> | null;
    /**
     * Gets the collection of Parts being moved.
     * The value is a Map mapping Parts to DraggingInfo Objects
     * that have a "point" property remembering the original location of that Part.
     *
     * #copiedParts provides the map of Parts that have been copied during a copying operation, if any.
     */
    draggedParts: Map<Part, DraggingInfo> | null;
    /**
     * Gets or sets the mouse point from which parts start to move.
     * The value is a Point in document coordinates.
     * This property is normally set to the diagram's mouse-down point in #doActivate,
     * but may be set to a different point if parts are being copied from a different control.
     * Setting this property does not raise any events.
     */
    startPoint: Point;
    /**
     * On touch gestures only, this property gets or sets the time in milliseconds
     * for which the mouse must be stationary before this tool can be started.
     * The default value is 100 milliseconds.
     * Setting this property does not raise any events.
     */
    delay: number;
    /**
     * This tool can run if the diagram allows selection and moves/copies/dragging-out,
     * if the mouse has moved far enough away to be a drag and not a click,
     * and if #findDraggablePart has found a selectable part at the mouse-down point.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Return the selectable and movable/copyable Part at the mouse-down point.
     * This is called by #canStart to decide if this tool is ready to run.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {Part}
     */
    findDraggablePart(): Part | null;
    /**
     * This override prevents the Control modifier unselecting an already selected part.
     * This also remembers the selectable #currentPart at the current mouse point.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    standardMouseSelect(): void;
    /**
     * Start the dragging operation.
     * This calls #computeEffectiveCollection and saves the result as #draggedParts.
     *
     * This starts a "Drag" transaction.
     * Depending on what happens, the transaction may be finished as a "Move" or a "Copy"
     * transaction, or it may be rolled-back if the tool is cancelled.
     *
     * Normally when this method is called the value of #currentPart will be null,
     * in which case this will call Tool#standardMouseSelect which will set #currentPart.
     * But if when this method is called the value of #currentPart has already been set
     * because the programmer wants the user to start dragging that Part,
     * then this method will not need to call Tool#standardMouseSelect because
     * the Part(s) to be selected and dragged have already been determined by the caller.
     */
    doActivate(): void;
    /**
     * This just calls CommandHandler#computeEffectiveCollection. The implementation of this method was moved to CommandHandler for 2.0.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Iterable.<Part>} parts A Set or List of Parts.
     * @param {DraggingOptions} options Potential options for the collection computation
     * @return {Map.<Part,DraggingInfo>}  a Map mapping Parts to DraggingInfo Objects
     *         that have a "point" property remembering the original location of that Part.
     */
    computeEffectiveCollection(parts: Iterable<Part>, options: DraggingOptions): Map<Part, DraggingInfo>;
    /**
     * Stop the dragging operation by stopping the transaction and cleaning up any temporary state.
     */
    doDeactivate(): void;
    /**
     * Abort any dragging operation.
     */
    doCancel(): void;
    /**
     * Handle switching between copying and moving modes as the Control/Option key is pressed or released.
     */
    doKeyDown(): void;
    /**
     * Handle switching between copying and moving modes as the Control/Option key is pressed or released.
     */
    doKeyUp(): void;
    /**
     * Move a collection Map of Parts by a given offset.
     *
     * If *check* is true this respects the Part#canMove predicate for Nodes or simple Parts
     * when this is the Diagram#currentTool.
     * It also respects #isGridSnapEnabled in order to try to automatically snap part locations to a grid.
     * And it also uses the Part#dragComputation function, if any, to determine the new location for each part.
     *
     * The first argument is a Map as produced by #computeEffectiveCollection,
     * not a List or Set or Iterator of Parts.
     * Call Diagram#moveParts if you want to move a simple collection of Parts without having to create the argument Map.
     * @expose
     * @param {Map.<Part,DraggingInfo>} parts  a Map mapping Parts to DraggingInfo Objects
     *        that have a "point" property remembering the original location of that Part.
     * @param {Point} offset The offset, before snapping, to move parts.
     * This offset reflects the total amount moved during tool operation,
     * based on original Part locations remembered when the DraggingTool activated.
     * @param {boolean} check  Whether to check Part#canMove on each part.
     * @since 1.1
     */
    moveParts(parts: Map<Part, DraggingInfo>, offset: Point, check: boolean): void;
    /**
     * Perform any additional side-effects during a drag, whether an internal move or copy or an external drag,
     * that may affect the existing non-moved object(s).
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point} pt a Point in document coordinates.
     * @param {GraphObject} obj the GraphObject at the point,
     * excluding what is being dragged or temporary objects;
     * the argument may be null if the drag is occurring in the background of the diagram.
     * Use GraphObject#part to get the Node or Part at the root of
     * the visual tree of the stationary object.
     * @since 1.1
     */
    doDragOver(pt: Point, obj: GraphObject | null): void;
    /**
     * Perform any additional side-effects after a drop, whether an internal move or copy or an external drop,
     * that may affect the existing non-moved object(s).
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point} pt a Point in document coordinates.
     * @param {GraphObject} obj the GraphObject where the drop occurred,
     * excluding what was dropped or temporary objects;
     * the argument may be null if the drop occurred in the background of the diagram.
     * Use GraphObject#part to get the Node or Part at the root of
     * the visual tree of the stationary object.
     * @since 1.1
     */
    doDropOnto(pt: Point, obj: GraphObject | null): void;
    /**
     * Move the #draggedParts (or if copying, the #copiedParts) to follow the current mouse point.
     *
     * If this creates any temporary parts, by default it adds them to the Tool layer.
     *
     * This calls #doDragOver for any side-effects on stationary parts.
     */
    doMouseMove(): void;
    /**
     * On a mouse-up finish moving or copying the effective selection.
     *
     * This calls #doDropOnto for any side-effects on stationary parts.
     *
     * This also updates the diagram's bounds,
     * raises a "SelectionCopied" or "SelectionMoved" DiagramEvent, and stops this tool.
     */
    doMouseUp(): void;
    /**
     * Undocumented
     * @expose
     * @param {Event} e
     * @param {Point} modelpt
     * @param {Diagram} overdiag the diagram attached to the target of the mouse move event, if any.
     * @return {boolean}
     */
    simulatedMouseMove(e: Event | Touch | null, modelpt: Point, overdiag: Diagram | null): boolean;
    /**
     * Undocumented
     * @expose
     * @param {Event} e
     * @param {Diagram} other
     * @param {Point} modelpt
     * @param {Diagram} curdiag
     * @return {boolean}
     */
    simulatedMouseUp(e: Event | Touch | null, other: Diagram, modelpt: Point, curdiag: Diagram | null): boolean;
    /**
     * This predicate is true when the diagram allows objects to be copied and inserted,
     * and some object in the selection is copyable,
     * and the user is holding down the Control key (Option key on Mac).
     *
     * This method may be overridden, although in most cases it is easiest to set Part#copyable.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    mayCopy(): boolean;
    /**
     * This predicate is true when the diagram allows objects to be moved,
     * and some object in the selection is movable.
     *
     * This method may be overridden, although in most cases it is easiest to set Part#movable.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    mayMove(): boolean;
    /**
     * This method computes the new location for a Node or simple Part,
     * given a new desired location and an optional Map of dragged parts,
     * taking any grid-snapping into consideration,
     * any Part#dragComputation function,
     * and any Part#minLocation and Part#maxLocation.
     *
     * As of 2.0, this just calls Diagram#computeMove and remains for compatibility.
     *
     * This method may be overridden, but should usually be overridden on Diagram.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Part} n  the Node or simple Part that is being moved
     * @param {Point} newloc the proposed new location
     * @param {Map.<Part,DraggingInfo>=} draggedparts  an optional Map mapping Parts to DraggingInfo Objects
     *        that have a "point" property remembering the original location of that Part.
     * @param {Point=} result  an optional Point that is modified and returned
     * @return {Point}  the possibly grid-snapped computed Point that is within the minimum and maximum permitted locations
     * @since 1.1
     */
    computeMove(n: Part, newloc: Point, draggedparts?: Map<Part, DraggingInfo> | null, result?: Point): Point;
}
/**
 * This abstract class is the base class for the LinkingTool and RelinkingTool classes.
 *
 * This class includes properties for defining and accessing any temporary nodes and temporary link
 * that are used during any linking operation, as well as access to the existing diagram's nodes and link
 * (if any) that are involved with the linking operation.
 * <p class="boxread">
 * For a general discussion of link routing, see:
 * <a href="../../intro/links.html">Introduction to Links</a>,
 * <a href="../../intro/linkLabels.html">Introduction to Link Labels</a>, and
 * <a href="../../intro/connectionPoints.html">Introduction to Link Connection Points</a>.
 * For customizing the linking tools, see <a href="../../intro/tools.html#LinkingAndRelinkingTools">Introduction to the Linking Tools</a>.
 * For customizing the reshaping of Links, see <a href="../../intro/tools.html#LinkReshapingTool">Introduction to the LinkReshapingTool</a>.
 * For a general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export abstract class LinkingBaseTool extends Tool {
    /**
     * Don't construct this directly -- this is an abstract class.
     */
    constructor();
    /**
     * Gets or sets the distance at which link snapping occurs.
     * The default value is 100.0.
     * The value must be non-negative.
     * Setting this property does not raise any events.
     */
    portGravity: number;
    /**
     * Gets or sets whether it is valid to have partly or completely unconnected links.
     *
     * The default value is false -- the user cannot draw or reconnect a link to "nowhere".
     * Setting this property does not raise any events.
     * @since 1.3
     */
    isUnconnectedLinkValid: boolean;
    /**
     * Gets or sets the temporary Link that is shown while the user is drawing or reconnecting a link.
     * Setting this property does not raise any events.
     */
    temporaryLink: Link;
    /**
     * Gets or sets the temporary Node at the "from" end of the #temporaryLink
     * while the user is drawing or reconnecting a link.
     * Setting this property does not raise any events.
     */
    temporaryFromNode: Node;
    /**
     * Gets or sets the GraphObject that is the port at the "from" end of the #temporaryLink
     * while the user is drawing or reconnecting a link.
     * Setting this property does not raise any events.
     */
    temporaryFromPort: GraphObject;
    /**
     * Gets or sets the temporary Node at the "to" end of the #temporaryLink
     * while the user is drawing or reconnecting a link.
     * Setting this property does not raise any events.
     */
    temporaryToNode: Node;
    /**
     * Gets or sets the GraphObject that is the port at the "to" end of the #temporaryLink
     * while the user is drawing or reconnecting a link.
     * Setting this property does not raise any events.
     */
    temporaryToPort: GraphObject;
    /**
     * Gets or sets the original Link being reconnected by the RelinkingTool.
     * Setting this property does not raise any events.
     */
    originalLink: Link | null;
    /**
     * Gets or sets the original Node from which the new link is being drawn
     * or from which the #originalLink was connected when being relinked.
     * Setting this property does not raise any events.
     */
    originalFromNode: Node | null;
    /**
     * Gets or sets the GraphObject that is the port in the #originalFromNode.
     * Setting this property does not raise any events.
     */
    originalFromPort: GraphObject | null;
    /**
     * Gets or sets the original Node to which the new link is being drawn
     * or to which the #originalLink was connected when being relinked.
     * Setting this property does not raise any events.
     */
    originalToNode: Node | null;
    /**
     * Gets or sets the GraphObject that is the port in the #originalToNode.
     * Setting this property does not raise any events.
     */
    originalToPort: GraphObject | null;
    /**
     * Gets whether the linking operation is in the forwards direction,
     * connecting from the "From" port to the "To" port.
     */
    isForwards: boolean;
    /**
     * Gets or sets a proposed GraphObject port for connecting a link.
     * Whether this is a "to" port or a "from" port depends on the direction
     * (#isForwards) in which the link is being drawn or reconnected.
     *
     * This is set when the mouse is being dragged and when a mouse-up event occurs
     * with the result of a call to #findTargetPort.
     * Setting this property does not raise any events.
     */
    targetPort: GraphObject | null;
    /**
     * Make a temporary port look and act like a real one.
     *
     * This is called by #doMouseMove as the tool finds new valid target ports.
     * If #findTargetPort returns null, it calls #setNoTargetPortProperties instead.
     *
     * If the #portTargeted property is a function, that function is called.
     * It is easier to customize the behavior by setting that functional property than it is to override
     * this method and the #setNoTargetPortProperties method.
     * But you may want to call this method to get the standard behavior for dynamically adapting
     * the temporary node/port to "act like" the target port.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} realnode
     * @param {GraphObject} realport
     * @param {Node} tempnode
     * @param {GraphObject} tempport
     * @param {boolean} toend
     * @since 1.2
     */
    protected copyPortProperties(realnode: Node | null, realport: GraphObject | null, tempnode: Node, tempport: GraphObject, toend: boolean): void;
    /**
     * Reset a temporary port's properties to neutral values when there is no target port.
     *
     * This is called by #doMouseMove when the tool finds no new valid target port.
     * If #findTargetPort returns non-null, it calls #copyPortProperties instead.
     *
     * If the #portTargeted property is a function, that function is called with null values as the first two arguments.
     * It is easier to customize the behavior by setting that functional property than it is to override
     * this method and the #copyPortProperties method.
     * But you may want to call this method to get the standard behavior for dynamically adapting
     * the temporary node/port to "act like" it is not connecting with any target port.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} tempnode
     * @param {GraphObject} tempport
     * @param {boolean} toend
     * @since 1.2
     */
    protected setNoTargetPortProperties(tempnode: Node, tempport: GraphObject, toend: boolean): void;
    /**
     * Mouse movement results in a temporary node moving to where a valid target port is located,
     * or to where the mouse is if there is no valid target port nearby.
     *
     * This calls #findTargetPort to update the #targetPort
     * given the new mouse point.
     * If a valid target port is found this moves the temporary node/port and
     * makes them appear like the target node/port by calling #copyPortProperties.
     * If no valid target port is found, this moves the temporary node to
     * where the mouse currently is and removes any node/port appearance,
     * by calling #setNoTargetPortProperties.
     *
     * This method may be overridden, but we recommend that you call this base method.
     */
    doMouseMove(): void;
    /**
     * Find a port with which the user could complete a valid link.
     *
     * This finds objects near to the current mouse point for which a valid link connection is possible.
     *
     * For each port element found, this calls #isValidLink to find out if a link between
     * the original node/port and the found node/port would be valid.
     * The result is saved in a cache for faster decisions later during operation of this tool.
     * The closest valid port is returned.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {boolean} toend true if looking for a "to" port.
     * @return {GraphObject} a valid port, or null if no such port is near the current mouse point
     * within the #portGravity distance.
     */
    findTargetPort(toend: boolean): GraphObject | null;
    /**
     * This predicate is true if it is permissible to connect a link from a given node/port.
     * This is called by #isValidLink.
     * <p class="boxread">
     * For a more general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} fromnode
     * @param {GraphObject} fromport
     * @return {boolean}
     * False if the node is in a Layer that does not Layer#allowLink.
     * False if the port's GraphObject#fromLinkable is either false or null.
     * False if the number of links connected to the port would exceed the port's GraphObject#fromMaxLinks.
     * Otherwise true.
     */
    isValidFrom(fromnode: Node | null, fromport: GraphObject | null): boolean;
    /**
     * This predicate is true if it is permissible to connect a link to a given node/port.
     * This is called by #isValidLink.
     * <p class="boxread">
     * For a more general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} tonode
     * @param {GraphObject} toport
     * @return {boolean}
     * False if the node is in a Layer that does not Layer#allowLink.
     * False if the port's GraphObject#toLinkable is either false or null.
     * False if the number of links connected from the port would exceed the port's GraphObject#toMaxLinks.
     * Otherwise true.
     */
    isValidTo(tonode: Node | null, toport: GraphObject | null): boolean;
    /**
     * This predicate is true if both argument ports are in the same Node.
     * This is called by #isValidLink.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * It is rare that you will want to override this method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {GraphObject} fromport
     * @param {GraphObject} toport
     * @return {boolean}
     */
    isInSameNode(fromport: GraphObject | null, toport: GraphObject | null): boolean;
    /**
     * This predicate is true if there is a link in the diagram going from the given port to the given port.
     * This is called by #isValidLink.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * It is rare that you will want to override this method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {GraphObject} fromport
     * @param {GraphObject} toport
     * @return {boolean}
     */
    isLinked(fromport: GraphObject | null, toport: GraphObject | null): boolean;
    /**
     * This predicate should be true when it is logically valid to connect a new link from
     * one node/port to another node/port.
     * When this is called by the RelinkingTool, that tool sets #originalLink
     * to be the Link being reconnected.
     * <p class="boxread">
     * For a more general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
     *
     * This method may be overridden, although it is usually much easier to just set #linkValidation
     * or Node#linkValidation in order to add some application-specific link validation.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} fromnode the "from" Node.
     * @param {GraphObject} fromport the "from" GraphObject port.
     * @param {Node} tonode the "to" Node.
     * @param {GraphObject} toport the "to" GraphObject port.
     * @return {boolean}
     * False if #isValidFrom is false for the "from" node/port.
     * False if #isValidTo is false for the "to" node/port.
     * False if #isInSameNode is true unless GraphObject#fromLinkableSelfNode
     * and GraphObject#toLinkableSelfNode are true for the two ports.
     * False if #isLinked is true unless GraphObject#fromLinkableDuplicates
     * and GraphObject#toLinkableDuplicates are true for the two ports.
     * False if trying to link to the link's own label node(s).
     * If Node#linkValidation is a predicate on either or both of the "from" node and the "to" node,
     * the predicate is called and if it returns false, this predicate returns false.
     * If #linkValidation is a predicate and if it returns false, this predicate returns false.
     * Otherwise this predicate is true.
     */
    isValidLink(fromnode: Node | null, fromport: GraphObject | null, tonode: Node | null, toport: GraphObject | null): boolean;
    /**
     * Checks whether a proposed link would be valid according to Diagram#validCycle.
     * This does not distinguish between different ports on a node, so this method does not need to take port arguments.
     * This is called by #isValidLink.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * It is rare that you will want to override this method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} from
     * @param {Node} to
     * @param {Link} ignore may be null; this is useful during relinking to ignore the #originalLink
     * @return {boolean}
     * @since 1.5
     */
    isValidCycle(from: Node | null, to: Node | null, ignore?: Link | null): boolean;
    /**
     * Gets or sets a predicate that determines whether or not a new link between two ports would be valid.
     * This predicate is called in addition to the normal link checking performed by #isValidLink
     * and any Node#linkValidation predicates on the "from" and "to" nodes.
     * When relinking, the Link being considered for reconnection is passed as the fifth argument.
     * The default predicate is null, which is equivalent to simply returning true.
     * <p class="boxread">
     * For a more general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
     *
     * The function, if supplied, must not have any side-effects.
     */
    linkValidation: ((fromNode: Node, fromPort: GraphObject, toNode: Node, toPort: GraphObject, link: Link) => boolean) | null;
    /**
     * Gets or sets a function that is called as the tool targets the nearest valid port.
     * The first two arguments specify the port by providing the Node that it is in
     * and the GraphObject that is the actual port object.
     * The next two arguments are the temporary port that has been moved and styled to be like the valid port.
     * These values will be either the #temporaryToNode and #temporaryToPort
     * or the #temporaryFromNode and #temporaryFromPort, depending on
     * which end of the temporary link is being updated.
     * The fifth argument is true if the target port represents a potential "to" end of a link;
     * it is false if it is for the "from" end of a link.
     *
     * When there is no valid port within the #portGravity, the first two arguments are null.
     *
     * The function, if supplied, must not add or remove any links or nodes or port objects,
     * nor may it change the validity of any potential link connection.
     * @since 1.2
     */
    portTargeted: ((node: Node, port: GraphObject, tempNode: Node, tempPort: GraphObject, toEnd: boolean) => void) | null;
}
/**
 * The LinkingTool lets a user draw a new Link between two ports,
 * using a mouse-drag operation.
 *
 * By default an instance of this tool is installed as a mouse-move tool in the
 * Diagram#toolManager as the ToolManager#linkingTool.
 * However this tool may be used modally, as described below.
 *
 * #canStart calls #findLinkablePort to find a valid "port" element
 * from which (or to which) the user may interactively draw a new link.
 * #doActivate sets up a temporary link and two temporary nodes,
 * one at the start port and one following the mouse.
 * <p class="boxread">
 * For a general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
 *
 * This tool does not utilize any Adornments or tool handles.
 *
 * This tool conducts a transaction while the tool is active.
 * A successful linking will result in a "LinkDrawn" DiagramEvent and a "Linking" transaction.
 *
 * If you want to programmatically start a new user mouse-gesture to draw a new link
 * from a given GraphObject that either is a "port" or may be within
 * the visual tree of a "port", set the #startObject property
 * to let #findLinkablePort find the real "port" element.
 * Then start and activate this tool:
 * ```js
 *   var tool = myDiagram.toolManager.linkingTool;
 *   tool.startObject = ...;
 *   myDiagram.currentTool = tool;
 *   tool.doActivate();
 * ```
 * @extends LinkingBaseTool
 * @unrestricted
 * @category Tool
 */
export class LinkingTool extends LinkingBaseTool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#linkingTool, which you can modify.
     *
     * The Tool#name of this tool is "Linking".
     */
    constructor();
    /**
     * This value for LinkingTool#direction indicates that users may draw new links in either direction.
     * @constant
     */
    static Either: EnumValue;
    /**
     * This value for LinkingTool#direction indicates that users may draw new links forwards only
     * (i.e. from "from" node to "to" node).
     * @constant
     */
    static ForwardsOnly: EnumValue;
    /**
     * This value for LinkingTool#direction indicates that users may draw new links backwards only
     * (i.e. from "to" node to "from" node).
     * @constant
     */
    static BackwardsOnly: EnumValue;
    /**
     * Gets or sets a data object that is copied by #insertLink
     * and added to the GraphLinksModel when creating a new Link.
     * The default value is an empty Object, which will be copied.
     * The value must be an Object for the linking operation to succeed.
     * A null value will cause #insertLink to fail.
     *
     * The copied link data's GraphLinksModel#linkFromKeyProperty
     * and GraphLinksModel#linkToKeyProperty properties are set to
     * the corresponding node's data's key values.
     * If the ports have GraphObject#portId values that are not null,
     * the link data's GraphLinksModel#linkFromPortIdProperty
     * and GraphLinksModel#linkToPortIdProperty properties are also set.
     *
     * Setting this property does not raise any events.
     * This property is ignored if the Diagram#model is not a GraphLinksModel.
     */
    archetypeLinkData: ObjectData | null;
    /**
     * Gets or sets an optional node data object representing a link label, that is copied by #insertLink
     * and added to the GraphLinksModel when creating a new Link.
     *
     * The default value is null, which causes no such label node data to be added to the model
     * along with the new link data.
     *
     * #insertLink calls GraphLinksModel#addLabelKeyForLinkData
     * to associate the label node data with the link data.
     *
     * Setting this property does not raise any events.
     * This property is ignored if the Diagram#model is not a GraphLinksModel.
     */
    archetypeLabelNodeData: ObjectData | null;
    /**
     * Gets or sets the direction in which new links may be drawn.
     * Possible values are LinkingTool.ForwardsOnly, LinkingTool.BackwardsOnly, or LinkingTool.Either.
     * This defaults to LinkingTool.Either.
     * Setting this property does not raise any events.
     */
    direction: EnumValue;
    /**
     * Gets or sets the GraphObject at which #findLinkablePort should start its search.
     * The default value is null.
     * Setting this property does not raise any events.
     *
     * If you want to explicitly start a new user mouse-gesture to draw a new link
     * from a given GraphObject that may be a "port" object or may be within
     * the visual tree of a "port" object, set this property to that object
     * to let #findLinkablePort find the real "port" object.
     * Then start and activate this tool:
     * ```js
     *   var tool = myDiagram.toolManager.linkingTool;
     *   tool.startObject = ...;
     *   myDiagram.currentTool = tool;
     *   tool.doActivate();
     * ```
     */
    startObject: GraphObject | null;
    /**
     * This tool can run when the diagram allows linking, the model is modifiable,
     * the left-button mouse drag has moved far enough away to not be a click, and
     * when #findLinkablePort has returned a valid port
     * or when #startObject is a valid port.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Return the GraphObject at the mouse-down point,
     * if it is part of a node and if it is valid to link with it.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {GraphObject}
     * If the #direction is LinkingTool.Either or LinkingTool.ForwardsOnly,
     * this checks the element and its parent Node by calling LinkingBaseTool#isValidFrom.
     * If the #direction is LinkingTool.Either or LinkingTool.BackwardsOnly,
     * this checks the element and its parent Node by calling LinkingBaseTool#isValidTo.
     * In either case finding a matching port will return that port and set LinkingBaseTool#isForwards appropriately.
     * Otherwise this will return null.
     */
    findLinkablePort(): GraphObject | null;
    /**
     * Start the linking operation.
     *
     * If the #startObject is already set, it uses that object to find the starting port.
     * If it is not set, this calls #findLinkablePort and remembers it as the starting port.
     *
     * It then start a transaction, captures the mouse, and changes the cursor.
     * Next it initializes and adds the LinkingBaseTool#temporaryFromNode,
     * LinkingBaseTool#temporaryToNode, and LinkingBaseTool#temporaryLink to the diagram.
     * The temporary nodes that are positioned and sized to be like the real LinkingBaseTool#originalFromPort
     * and LinkingBaseTool#originalToPort ports.
     * The temporary link connects the two temporary ports, of course.
     *
     * This method may be overridden, but we recommend that you call this base method.
     */
    doActivate(): void;
    /**
     * Finishing the linking operation stops the transaction, releases the mouse, and resets the cursor.
     *
     * This method may be overridden, but we recommend that you call this base method.
     */
    doDeactivate(): void;
    /**
     * A mouse-up ends the linking operation; if there is a valid #targetPort nearby,
     * this adds a new Link by calling #insertLink.
     *
     * If there is a new link, it is selected and the "LinkDrawn" DiagramEvent is raised
     * with the new link as the DiagramEvent#subject.
     * If there is no new link, this calls #doNoLink.
     * In any case this stops the tool.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * It is usually easier to override #insertLink or just set #archetypeLinkData.
     * It is also common to implement a "LinkDrawn" DiagramEvent listener on the Diagram.
     */
    doMouseUp(): void;
    /**
     * Make a copy of the #archetypeLinkData, set its node and port properties, and add it to the GraphLinksModel.
     * If the model is a TreeModel, set the parent key of the child's node data object.
     *
     * For GraphLinksModels, if #archetypeLabelNodeData is non-null, this method also adds that node data as an initial label node
     * for the new link data.
     *
     * For TreeModels, if Diagram#isTreePathToChildren is false, the roles of the *fromnode* and *tonode* arguments are exchanged.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * Usually though it is easiest to just set #archetypeLinkData to control what kind of link is created.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} fromnode
     * @param {GraphObject} fromport the from-port, or null to use the node itself
     * @param {Node} tonode
     * @param {GraphObject} toport the to-port, or null to use the node itself
     * @return {Link} the newly created Link, or null if it failed.
     * @see #doNoLink
     */
    insertLink(fromnode: Node | null, fromport: GraphObject | null, tonode: Node | null, toport: GraphObject | null): Link | null;
    /**
     * This method is called upon a mouse up when no new link is drawn,
     * either because no valid LinkingBaseTool#targetPort was found or because
     * #insertLink returned null.
     *
     * This method may be overridden.  By default this method does nothing.
     * If you want to successfully perform any side-effects, you will need to set Tool#transactionResult to a string;
     * otherwise this tool's transaction will be rolled-back.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} fromnode
     * @param {GraphObject} fromport the from-port, or null to use the node itself
     * @param {Node} tonode
     * @param {GraphObject} toport the to-port, or null to use the node itself
     * @since 1.7
     */
    doNoLink(fromnode: Node | null, fromport: GraphObject | null, tonode: Node | null, toport: GraphObject | null): void;
}
/**
 * The RelinkingTool allows the user to reconnect an existing Link
 * if the Link#relinkableTo and/or Link#relinkableFrom properties are true.
 * <p class="boxread">
 * For a general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
 *
 * By default an instance of this tool is installed as a mouse-down tool in the
 * Diagram#toolManager as the ToolManager#relinkingTool.
 *
 * This tool makes use of two Adornments,
 * each including a relink handle (potentially one for each end of the link),
 * shown when a link is selected.
 *
 * This tool conducts a transaction while the tool is active.
 * A successful relinking will result in a "LinkRelinked" DiagramEvent and a "Relinking" transaction.
 *
 * If you want to programmatically start a new user's relinking of a Link,
 * you can set the #handle property to the specific "RelinkFrom" or "RelinkTo" handle and
 * then start and activate the tool.
 * ```js
 *   var tool = myDiagram.toolManager.relinkingTool;
 *   tool.originalLink = ...;   // specify which Link to have the user reconnect
 *   tool.isForwards = true;    // specify which end of the Link to reconnect
 *   myDiagram.currentTool = tool;  // starts the RelinkingTool
 *   tool.doActivate();             // activates the RelinkingTool
 * ```
 * @extends LinkingBaseTool
 * @unrestricted
 * @category Tool
 */
export class RelinkingTool extends LinkingBaseTool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#relinkingTool, which you can modify.
     *
     * The Tool#name of this tool is "Relinking".
     */
    constructor();
    /**
     * Show an Adornment for each end of the Link that the user may reconnect.
     * The Adornment may have a copy of #fromHandleArchetype and/or #toHandleArchetype.
     *
     * This method may be overridden.
     * @param {Part} part
     */
    updateAdornments(part: Part): void;
    /**
     * Undocumented
     * @expose
     * @param {GraphObject} selelt the GraphObject of the Link being relinked.
     * @param {boolean} toend
     * @return {Adornment}
     */
    makeAdornment(selelt: GraphObject, toend: boolean): Adornment;
    /**
     * Gets or sets a small GraphObject that is copied as a relinking handle for the selected link path
     * at the "from" end of the link.
     * By default this is a Shape that is a small blue diamond.
     * Setting this property does not raise any events.
     *
     * Here is an example of changing the default handle to be larger green triangles:
     * ```js
     * myDiagram.toolManager.relinkingTool.toHandleArchetype =
     *   $(go.Shape, "Triangle",
     *     { width: 10, height: 10, fill: "limegreen", segmentIndex: 0 });
     * ```
     * @see #toHandleArchetype
     */
    fromHandleArchetype: GraphObject | null;
    /**
     * Gets or sets a small GraphObject that is copied as a relinking handle for the selected link path
     * at the "to" end of the link.
     * By default this is a Shape that is a small blue diamond.
     * Setting this property does not raise any events.
     *
     * Here is an example of changing the default handle to be larger orange triangles:
     * ```js
     *   myDiagram.toolManager.relinkingTool.toHandleArchetype =
     *   $(go.Shape, "Triangle",
     *     { width: 10, height: 10, fill: "orange", segmentIndex: -1 });
     * ```
     * @see #fromHandleArchetype
     */
    toHandleArchetype: GraphObject | null;
    /**
     * Returns the GraphObject that is the tool handle being dragged by the user.
     * This will be contained by an Adornment whose category is "RelinkFrom" or "RelinkTo".
     * Its Adornment#adornedPart is the same as the #originalLink.
     *
     * This property is also settable, but should only be set either within an override of #doActivate
     * or prior to calling #doActivate.
     */
    handle: GraphObject | null;
    /**
     * This tool can run when the diagram allows relinking, the model is modifiable,
     * and there is a relink handle at the mouse-down point.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Start the relinking operation.
     *
     * Find the relink handle by calling Tool#findToolHandleAt
     * looking for either the "RelinkFrom" adornment or the "RelinkTo" adornment,
     * saving the result in #handle.
     *
     * Normally when this method is called the value of LinkingBaseTool#originalLink and #handle will be null,
     * resulting in a call to Tool#findToolHandleAt to find a "RelinkFrom" or "RelinkTo" tool handle,
     * which is then remembered as the value of #handle.
     * If when this method is called the value of #handle is already set,
     * then there is no need to call Tool#findToolHandleAt,
     * because the programmer has already set up which relinking handle they want the user to be relinking.
     * Finding a handle is not necessary if LinkingBaseTool#originalLink and
     * #isForwards have been set before calling this method.
     *
     * This starts a transaction, captures the mouse, and sets the cursor.
     *
     * If LinkingBaseTool#originalLink or #handle was not set beforehand,
     * the value of #isForwards is set depending on the category of the relink handle found.
     * The LinkingBaseTool#originalLink property and various
     * "Original..." port and node properties are set too.
     * The temporary nodes and temporary link are also initialized.
     *
     * This method may be overridden, but we recommend that you call this base method.
     */
    doActivate(): void;
    /**
     * Make a temporary link look and act like the real Link being relinked.
     * By default this method copies many of the routing-oriented properties
     * from the LinkingBaseTool#originalLink to the LinkingBaseTool#temporaryLink.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Link} reallink
     * @param {Link} templink
     * @since 1.3
     */
    protected copyLinkProperties(reallink: Link | null, templink: Link): void;
    /**
     * Finishing the linking operation stops the transaction, releases the mouse, and resets the cursor.
     *
     * This method may be overridden, but we recommend that you call this base method.
     */
    doDeactivate(): void;
    /**
     * A mouse-up ends the relinking operation; if there is a valid #targetPort nearby,
     * this modifies the old link to connect with the target port.
     *
     * A successful relinking calls #reconnectLink to actually change the link.
     * The "LinkRelinked" DiagramEvent is raised with the link as the DiagramEvent#subject
     * and with the now-disconnected original port as the DiagramEvent#parameter.
     * If the link was not reconnected, this calls #doNoRelink.
     * In any case this stops the tool.
     *
     * A failure to find a valid target port results in no changes and no DiagramEvent.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * You might find it easier to override #reconnectLink.
     * It is actually most common to implement a "LinkRelinked" DiagramEvent listener on the Diagram.
     */
    doMouseUp(): void;
    /**
     * Modify an existing Link to connect to a new node and port.
     *
     * This method may be overridden, but we recommend that you call this base method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Link} existinglink
     * @param {Node} newnode the Node to connect to or from.
     * @param {GraphObject} newport the GraphObject port to connect to or from.
     * @param {boolean} toend If true, this modifies the link's "to" node and port; otherwise it modifies the "from" node and port.
     * @return {boolean} true if successful.
     * @see #doNoRelink
     */
    reconnectLink(existinglink: Link, newnode: Node | null, newport: GraphObject | null, toend: boolean): boolean;
    /**
     * This method is called upon a mouse up when #reconnectLink is not called.
     *
     * This method may be overridden.  By default this method does nothing.
     * If you want to successfully perform any side-effects, you will need to set Tool#transactionResult to a string;
     * otherwise this tool's transaction will be rolled-back.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Link} existinglink
     * @param {boolean} toend If true, the user was trying to modify the link's "to" node and port.
     * @since 1.7
     */
    doNoRelink(existinglink: Link, toend: boolean): void;
}
/**
 * The LinkReshapingTool is used to interactively change the route of a Link
 * by setting its Link#points list.
 * You may want to save the route to the model by using a TwoWay Binding on the "points" property of the Link.
 *
 * This tool makes use of an Adornment, shown when the adorned Link is selected,
 * that includes some number of reshape handles.
 * This tool conducts a transaction while the tool is active.
 * A successful reshaping will result in a "LinkReshaped" DiagramEvent and a "LinkReshaping" transaction.
 *
 * <p class="boxread">
 * For a general discussion of link routing, see:
 * <a href="../../intro/links.html">Introduction to Links</a>,
 * <a href="../../intro/linkLabels.html">Introduction to Link Labels</a>, and
 * <a href="../../intro/connectionPoints.html">Introduction to Link Connection Points</a>.
 * For customizing the linking tools, see <a href="../../intro/tools.html#LinkingAndRelinkingTools">Introduction to the Linking Tools</a>.
 * For a general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class LinkReshapingTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#linkReshapingTool, which you can modify.
     *
     * The Tool#name of this tool is "LinkReshaping".
     */
    constructor();
    /**
     * Disallow dragging.
     * @constant
     */
    static None: EnumValue;
    /**
     * Allow only horizontal (left-and-right) dragging.
     * @constant
     */
    static Horizontal: EnumValue;
    /**
     * Allow only vertical (up-and-down) dragging.
     * @constant
     */
    static Vertical: EnumValue;
    /**
     * Allow dragging in any direction.
     * @constant
     */
    static All: EnumValue;
    /**
     * Get the permitted reshaping behavior for a particular reshape handle.
     * @param {GraphObject} obj a reshape handle in the "LinkReshaping" Adornment.
     * @return {EnumValue} one of LinkReshapingTool.All, .Vertical, .Horizontal, or .None
     * @since 1.3
     */
    protected getReshapingBehavior(obj: GraphObject): EnumValue;
    /**
     * Set the permitted reshaping behavior for a particular reshape handle.
     * @param {GraphObject} obj a reshape handle in the "LinkReshaping" Adornment.
     * @param {EnumValue} behavior one of LinkReshapingTool.All, .Vertical, .Horizontal, or .None
     * @since 1.3
     */
    protected setReshapingBehavior(obj: GraphObject, behavior: EnumValue): void;
    /**
     * Show an Adornment with reshape handles at each of the interesting points of the link's route,
     * if the link is selected and visible and if Part#canReshape is true.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Part} part
     */
    updateAdornments(part: Part): void;
    /**
     * Undocumented
     * @expose
     * @param {GraphObject} pathshape
     * @return {Adornment}
     */
    makeAdornment(pathshape: GraphObject): Adornment;
    /**
     * Undocumented
     * @expose
     * @param {GraphObject} pathshape the GraphObject path of the link being reshaped.
     * @param {number} idx the index of the route point to be moved.
     * @return {GraphObject}
     */
    makeHandle(pathshape: GraphObject, idx: number): GraphObject;
    /**
     * Gets or sets a small GraphObject that is copied as a reshape handle at each movable point in the selected link's route.
     * By default this is a Shape that is a small blue rectangle.
     * Setting this property does not raise any events.
     *
     * Here is an example of changing the default handles to be larger yellow circles:
     * ```js
     *   myDiagram.toolManager.linkReshapingTool.handleArchetype =
     *     $(go.Shape, "Circle",
     *       { width: 10, height: 10, fill: "yellow" });
     * ```
     * @see #midHandleArchetype
     */
    handleArchetype: GraphObject | null;
    /**
     * Undocumented
     * @expose
     * @param {GraphObject} pathshape the GraphObject path of the link being reshaped.
     * @param {number} idx the index of the route point to be moved.
     * @return {GraphObject}
     */
    makeResegmentHandle(pathshape: GraphObject, idx: number): GraphObject;
    /**
     * Gets or sets a small GraphObject that is copied as a resegment handle at each mid-point in the selected Link's route.
     * By default this is a Shape that is a small blue diamond.
     * Setting this property does not raise any events.
     *
     * Here is an example of changing the default handle to be larger cyan triangles:
     * ```js
     *   myDiagram.toolManager.linkReshapingTool.midHandleArchetype =
     *     $(go.Shape, "Triangle", { width: 10, height: 10, fill: "cyan" });
     * ```
     * @see #handleArchetype
     * @since 1.2
     */
    midHandleArchetype: GraphObject | null;
    /**
     * Returns the GraphObject that is the tool handle being dragged by the user.
     * This will be contained by an Adornment whose category is "LinkReshaping".
     * Its Adornment#adornedPart is the same as the #adornedLink.
     * This is normally set by #doActivate, remembering the result of the call to Tool#findToolHandleAt.
     */
    handle: GraphObject | null;
    /**
     * This read-only property returns the Link that is being routed manually.
     */
    readonly adornedLink: Link | null;
    /**
     * This tool may run when there is a mouse-down event on a reshape handle.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Start reshaping, if #findToolHandleAt finds a reshape handle at the mouse down point.
     *
     * If successful this sets #handle to be the reshape handle that it finds
     * and #adornedLink to be the Link being routed.
     * It also remembers the original link route (a list of Points) in case this tool is cancelled.
     * And it starts a transaction.
     */
    doActivate(): void;
    /**
     * This stops the current reshaping operation with the link route shaped the way it is.
     */
    doDeactivate(): void;
    /**
     * Restore the link route to be the original points and stop this tool.
     */
    doCancel(): void;
    /**
     * Undocumented
     * @expose
     * @return {Point} the point where resegmenting begins
     */
    protected getResegmentingPoint(): Point;
    /**
     * Call #reshape with a new point determined by the mouse
     * to change the route of the #adornedLink.
     */
    doMouseMove(): void;
    /**
     * Reshape the route with a point based on the most recent mouse point by calling #reshape,
     * and then raise a "LinkReshaped" DiagramEvent before stopping this tool.
     */
    doMouseUp(): void;
    /**
     * Change the route of the #adornedLink by moving the point corresponding to the current
     * #handle to be at the given Point.
     * This is called by #doMouseMove and #doMouseUp with the result of calling
     * #computeReshape to constrain the input point.
     *
     * This method respects the orthogonality of the link, if necessary limiting movement along only one axis.
     * To maintain orthogonality it may need to modify more than one point in the route.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point} newPoint the value of the call to #computeReshape.
     */
    reshape(newPoint: Point): void;
    /**
     * This is called by #doMouseMove and #doMouseUp to limit the input point
     * before calling #reshape.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point} p the point where the handle is being dragged.
     * @return {Point} Either the same Point p or one constrained by the reshape behavior to be moved
     * only vertically or only horizontally.
     */
    computeReshape(p: Point): Point;
    /**
     * This read-only property returns the Point that was the original location of the handle that is being dragged to reshape the Link.
     * @since 1.1
     */
    readonly originalPoint: Point;
    /**
     * This read-only property returns the List of Points that was the original route of the Link that is being reshaped.
     * This List should not be modified; its value is indeterminate until a reshaping has been activated.
     * @since 1.1
     */
    readonly originalPoints: List<Point>;
}
/**
 * The ResizingTool is used to interactively change the size of a GraphObject
 * in the selected Part or Node by setting its GraphObject#desiredSize property.
 * You may want to save the size to the model by using a TwoWay Binding on the "desiredSize" property
 * of the GraphObject that is named by Part#resizeObjectName.
 * This tool does not operate on Links.
 *
 * You can limit the permitted minimum and maximum dimensions by setting
 * #minSize and #maxSize.
 * The resizing will also respect the GraphObject#minSize and
 * GraphObject#maxSize properties.
 * Width or height values that are NaN do not constrain the resizing.
 * Override #computeMinSize and/or #computeMaxSize to change this behavior.
 *
 * You can also limit the width and/or height to be multiples of a particular size by
 * setting Part#resizeCellSize.
 * If either or both of these values are NaN, as they are by default,
 * it will get the values from this tool's #cellSize.
 * Finally it will consider the Diagram#grid's Panel#gridCellSize
 * if #isGridSnapEnabled is true.
 * Override #computeCellSize to change this behavior.
 *
 * Pressing the Shift key or resizing a Shape with a Shape#geometryStretch of
 * GraphObject.Uniform or GraphObject.UniformToFill will maintain the aspect ratio during the resize.
 * Override #computeReshape to change this behavior.
 *
 * This tool makes use of an Adornment, shown when the Part or Node is selected,
 * that includes some number of resize handles.
 * The resize handles are normally copies of ResizingTool#handleArchetype,
 * unless you specify a custom resize Adornment by setting Part#resizeAdornmentTemplate.
 * The resize Adornment is normally a "Spot" Panel with eight resize handles,
 * each with GraphObject#alignment set to one of the eight standard Spot values --
 * the four corners and the four side middles.
 * The GraphObject#alignment is what identifies and distinguishes each of the handles and
 * the behavior when the user drags the handle.
 *
 * This tool conducts a transaction while the tool is active.
 * A successful resizing will result in a "PartResized" DiagramEvent and a "Resizing" transaction.
 * <p class="boxread">
 * For a general discussion of the sizing of objects, see: <a href="../../intro/sizing.html">Introduction to the sizing of GraphObjects</a>.
 * For customizing the ResizingTool, see <a href="../../intro/tools.html#ResizingTool">Introduction to the ResizingTool</a>.
 *
 * If you want to programmatically start a user's resizing of the Part#resizeObject of an existing selected node,
 * you can set the #handle property to the specific resize handle and then start and activate the tool.
 * ```js
 *   var node = ...;
 *   myDiagram.select(node);
 *   var adorn = node.findAdornment("Resizing");
 *   var tool = myDiagram.toolManager.resizingTool;
 *   // specify which resize handle of the "Resizing" Adornment of the selected node
 *   tool.handle = adorn.elt(...);
 *   myDiagram.currentTool = tool;  // starts the ResizingTool
 *   tool.doActivate();             // activates the ResizingTool
 * ```
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class ResizingTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#resizingTool, which you can modify.
     *
     * The Tool#name of this tool is "Resizing".
     */
    constructor();
    /**
     * Show an Adornment with the resize handles at points along the edge of the bounds of the
     * selected Part's Part#resizeObject.
     *
     * First this finds the object in the visual tree of the Part that should
     * get the resize adornment and that the user will be able to resize interactively.
     * It finds the object that has the Part#resizeObjectName property of the Part.
     * If the Part#resizeObjectName property is an empty string, as it is by default,
     * it uses the whole part.
     *
     * It then builds the adornment, associating it with the chosen resize object.
     * If Part#resizeAdornmentTemplate is non-null, it is copied.
     * Otherwise it constructs a new Adornment with a Placeholder and eight copies of #handleArchetype,
     * four at the corners and four at the middle of each side.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Part} part
     */
    updateAdornments(part: Part): void;
    /**
     * Undocumented
     * @expose
     * @param {GraphObject} resizeObj the GraphObject being resized.
     * @return {Adornment}
     */
    makeAdornment(resizeObj: GraphObject): Adornment;
    /**
     * Undocumented
     * @expose
     * @param {GraphObject} elt
     * @param {number} angle
     */
    updateResizeHandles(elt: GraphObject, angle: number): void;
    /**
     * Gets or sets a small GraphObject that is copied as a resizing handle for the selected part.
     * By default this is a Shape that is a small blue rectangle.
     * Setting this property does not raise any events.
     *
     * Here is an example of changing the default handle to be larger yellow circles:
     * ```js
     *   myDiagram.toolManager.resizingTool.handleArchetype =
     *     $(go.Shape, "Circle",
     *       { width: 10, height: 10, fill: "yellow" });
     * ```
     *
     * This property is ignored when a custom resizing Adornment is specified as the Part#resizeAdornmentTemplate.
     * That property is normally null, in which case this tool will automatically construct Adornments
     * holding eight copies of this handle archetype, each with a GraphObject#alignment being one of the
     * standard eight Spots.
     */
    handleArchetype: GraphObject | null;
    /**
     * Returns the GraphObject that is the tool handle being dragged by the user.
     * This will be contained by an Adornment whose category is "ResizingTool".
     * Its Adornment#adornedObject is the same as the #adornedObject.
     * This is normally set by #doActivate, remembering the result of the call to Tool#findToolHandleAt.
     *
     * This property is also settable, but should only be set either within an override of #doActivate
     * or prior to calling #doActivate.
     */
    handle: GraphObject | null;
    /**
     * Gets the GraphObject that is being resized.
     * This may be the same object as the selected Part or it may be contained within that Part.
     *
     * This property is also settable, but should only be set when overriding functions
     * in ResizingTool, and not during normal operation.
     */
    adornedObject: GraphObject | null;
    /**
     * This tool may run when there is a mouse-down event on a resize handle,
     * the diagram is not read-only and it allows resizing,
     * the left mouse button is being used,
     * and this tool's adornment's resize handle is at the current mouse point.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Find the #handle, remember the object's original bounds,
     * save the results of calling #computeMinSize, #computeMaxSize, and #computeCellSize,
     * capture the mouse, and start a transaction.
     *
     * Normally when this method is called the value of #handle will be null,
     * resulting in a call to Tool#findToolHandleAt to find a "Resizing" tool handle,
     * which is then remembered as the value of #handle.
     * If when this method is called the value of #handle is already set,
     * then there is no need to call Tool#findToolHandleAt,
     * because the programmer has already set up which resize handle they want the user to be resizing.
     */
    doActivate(): void;
    /**
     * Stop the current transaction, forget the #handle and #adornedObject, and release the mouse.
     */
    doDeactivate(): void;
    /**
     * Restore the original size of the GraphObject.
     */
    doCancel(): void;
    /**
     * Call #resize with a new size determined by the current mouse point.
     * This determines the new bounds by calling #computeResize.
     *
     * When this calls #computeResize it passes as the reshape argument the result of calling #computeReshape.
     * The min, max, and cell arguments will be the saved results of #doActivate's calls to #computeMinSize, #computeMaxSize, and #computeCellSize.
     */
    doMouseMove(): void;
    /**
     * Call #resize with the final bounds based on the most recent mouse point,
     * commit the transaction, and raise the "PartResized" DiagramEvent.
     * This determines the new bounds by calling #computeResize.
     *
     * When this calls #computeResize it passes as the reshape argument the result of calling #computeReshape.
     */
    doMouseUp(): void;
    /**
     * Change the size of the selected part's Part#resizeObject to have the given bounds.
     * This modifies its GraphObject#desiredSize and maybe its Part#location.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Rect} newr a Rectangle in the #adornedObject's local coordinates, not in document coordinates
     */
    resize(newr: Rect): void;
    /**
     * Given a Spot in the original bounds of the object being resized and a new Point,
     * compute the new Rect.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point} newPoint a Point in local coordinates.
     * @param {Spot} spot the alignment spot of the handle being dragged.
     * @param {Size} min the result of the call to #computeMinSize.
     * @param {Size} max the result of the call to #computeMaxSize.
     * @param {Size} cell the result of the call to #computeCellSize.
     * @param {boolean} reshape true if the new size may change the aspect ratio from that of the natural bounds of the #adornedObject.
     * @return {Rect} a Rectangle in the #adornedObject's local coordinates, not in document coordinates
     */
    computeResize(newPoint: Point, spot: Spot, min: Size, max: Size, cell: Size, reshape: boolean): Rect;
    /**
     * Decide whether to allow arbitrary reshaping or whether to keep the same aspect ratio of the object being resized.
     * If the #adornedObject is a Shape,
     * then if the Shape#geometryStretch is GraphObject.Uniform,
     * this method will return false to restrict reshaping to maintain the object's current ratio of height to width.
     * Also, if the user is holding down the Shift key, this method will return false.
     *
     * This is called on each mouse-move and on mouse-up; the result is passed to the call to #resize.
     * This permits the user to change the behavior dynamically during resizing.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * For example, to always keep the object's original aspect ratio, override this method to return false.
     * When calling GraphObject.make to initialize a Diagram:
     * ```js
     * $(go.Diagram, . . .,
     *   {
     *     "resizingTool.computeReshape": function() { return false; },
     *     . . . })
     * ```
     * Or when overriding the method dynamically:
     * ```js
     *   myDiagram.toolManager.resizingTool.computeReshape = function() { return false; }
     * ```
     * Your override might want to look at the `this.adornedObject.part.data` properties to decide whether to allow reshaping.
     * @expose
     * @return {boolean} true to allow any aspect ratio; false to preserve the #adornedObject's height/width ratio
     * @since 1.7
     */
    computeReshape(): boolean;
    /**
     * The effective minimum resizing size is the maximum of #minSize and the
     * #adornedObject's GraphObject#minSize.
     *
     * This is called once when the tool is activated.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {Size}
     */
    computeMinSize(): Size;
    /**
     * The effective maximum resizing size is the minimum of the #maxSize and the
     * #adornedObject's GraphObject#maxSize.
     *
     * This is called once when the tool is activated.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {Size}
     */
    computeMaxSize(): Size;
    /**
     * The size should be a multiple of the value returned by this method.
     *
     * This is called once when the tool is activated.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {Size}
     */
    computeCellSize(): Size;
    /**
     * Gets or sets the minimum size to which the user can resize.
     * The effective minimum size is the maximum of this value and the GraphObject#minSize,
     * independently in each direction.
     *
     * The default value is `go.Size(1, 1)`.
     * Any new value must be of type Size; `NaN` width or height values are treated as zero.
     * Setting this property does not raise any events.
     */
    minSize: Size;
    /**
     * Gets or sets the maximum size to which the user can resize.
     * The effective maximum size is the minimum of this value and the GraphObject#maxSize,
     * independently in each direction.
     *
     * The default value is `go.Size(9999, 9999)`.
     * Any new value must be of type Size; `NaN` width or height values are treated as Infinity.
     * Setting this property does not raise any events.
     */
    maxSize: Size;
    /**
     * Gets or sets the width and height multiples with which the user must resize.
     * The effective cell size is computed by first looking at the Adornment#adornedPart's Part#resizeCellSize.
     * If either or both of its width and height are `NaN`, it will use this property, `cellSize`.
     * If either or both of this property's width and height are `NaN`, it will
     * consider the Diagram#grid's Panel#gridCellSize.
     *
     * The default value is `go.Size(NaN, NaN)`.
     * Setting this property does not raise any events.
     */
    cellSize: Size;
    /**
     * Gets or sets whether the ResizingTool snaps object sizes to the diagram's background grid during the resize.
     * By default this property is false.
     * Setting this property does not raise any events.
     */
    isGridSnapEnabled: boolean;
    /**
     * This read-only property returns the Size that was the original value of the GraphObject#desiredSize
     * of the element that is being resized.
     * @since 1.1
     */
    readonly originalDesiredSize: Size;
    /**
     * This read-only property returns the Point that was the original value of the Part#location
     * of the Part that is being resized.
     * @since 1.1
     */
    readonly originalLocation: Point;
}
/**
 * The RotatingTool is used to interactively change the GraphObject#angle of a GraphObject
 * by setting its GraphObject#angle property.
 * You may want to save the angle to the model by using a TwoWay Binding on the "angle" property
 * of the GraphObject that is named by Part#rotateObjectName.
 *
 * This tool allows the user to rotate the Part#rotateObject of the selected Part.
 * The Part must be Part#rotatable, which is false by default.
 * Normally this works with Parts or Nodes; it does not make sense for whole Links
 * or Link#paths, so if you want to rotate a label on a Link, make sure to name that label
 * and refer to it as the Part#rotateObjectName.
 *
 * You can control the point about which the object is rotated by setting Part#rotationSpot.
 * The rotation spot can be computed dynamically by overriding #computeRotationPoint.
 * The value of that method call is saved as #rotationPoint.
 *
 * You can limit the permitted angles by setting #snapAngleMultiple and #snapAngleEpsilon.
 * For example, if you want to permit only angles that are multiples of 90 degrees,
 * set #snapAngleMultiple to 90 and #snapAngleEpsilon to 45.
 * Pressing the Shift key during rotation ignores these two properties.
 *
 * This tool makes use of an Adornment that includes a rotation handle.
 * It is shown when the selected Part is Part#rotatable.
 * You can control its direction relative to the #rotationPoint
 * by setting #handleAngle to a multiple of 90 degrees,
 * and its distance from the rotation point by setting #handleDistance.
 * The rotate handle is normally a copy of RotatingTool#handleArchetype.
 * unless you specify a custom rotate Adornment by setting Part#rotateAdornmentTemplate.
 *
 * This tool conducts a transaction while the tool is active.
 * A successful rotation will result in a "PartRotated" DiagramEvent and a "Rotating" transaction.
 * <p class="boxread">
 * For customizing the RotatingTool, see <a href="../../intro/tools.html#RotatingTool">Introduction to the RotatingTool</a>.
 *
 * If you want to programmatically start a user's rotating of the Part#rotateObject of an existing selected node,
 * you can set the #handle property to the rotate handle and then start and activate the tool.
 * ```js
 *   var node = ...;
 *   myDiagram.select(node);
 *   var adorn = node.findAdornment("Rotating");
 *   var tool = myDiagram.toolManager.rotatingTool;
 *   // specify the rotation handle of the "Rotating" Adornment of the selected node
 *   tool.handle = adorn.elt(0);
 *   myDiagram.currentTool = tool;  // starts the RotatingTool
 *   tool.doActivate();             // activates the RotatingTool
 * ```
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class RotatingTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#rotatingTool, which you can modify.
     *
     * The Tool#name of this tool is "Rotating".
     */
    constructor();
    /**
     * Show an Adornment with a rotate handle at a point to the side of the adorned object
     * if the part is selected and visible and if Part#canRotate() is true.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Part} part the part.
     */
    updateAdornments(part: Part): void;
    /**
     * Undocumented
     * @expose
     * @param {GraphObject} rotateObj the GraphObject being rotated.
     * @return {Adornment}
     */
    makeAdornment(rotateObj: GraphObject): Adornment;
    /**
     * Gets or sets a small GraphObject that is copied as a rotation handle for the selected part.
     * By default this is a Shape that is a small yellow circle.
     * Setting this property does not raise any events.
     *
     * Here is an example of changing the default handle to be green "X":
     * ```js
     *   myDiagram.toolManager.rotatingTool.handleArchetype =
     *     $(go.Shape, "XLine",
     *       { width: 8, height: 8, stroke: "green", fill: "transparent" });
     * ```
     *
     * This property is ignored when a custom rotating Adornment is specified as the Part#rotateAdornmentTemplate.
     */
    handleArchetype: GraphObject | null;
    /**
     * This read-only property returns the GraphObject that is the tool handle being dragged by the user.
     * This will be contained by an Adornment whose category is "RotatingTool".
     * Its Adornment#adornedObject is the same as the #adornedObject.
     *
     * This property is also settable, but should only be set either within an override of #doActivate
     * or prior to calling #doActivate.
     */
    handle: GraphObject | null;
    /**
     * Gets the GraphObject that is being rotated.
     * This may be the same object as the selected Part or it may be contained within that Part.
     *
     * This property is also settable, but should only be set when overriding functions
     * in RotatingTool, and not during normal operation.
     */
    adornedObject: GraphObject | null;
    /**
     * This tool may run when there is a mouse-down event on a rotate handle,
     * the diagram is not read-only and it allows rotation,
     * the left mouse button is being used,
     * and this tool's adornment's rotate handle is at the current mouse point.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Capture the mouse, remember the original GraphObject#angle,
     * and start a transaction.
     *
     * Normally when this method is called the value of #handle will be null,
     * resulting in a call to Tool#findToolHandleAt to find a "Rotating" tool handle,
     * which is then remembered as the value of #handle.
     * If when this method is called the value of #handle is already set,
     * then there is no need to call Tool#findToolHandleAt,
     * because the programmer has already set up which rotate handle they want the user to be rotating.
     */
    doActivate(): void;
    /**
     * Compute the Point about which things should rotate.
     * This is called when this tool is activated and sometimes when updating the Adornment.
     * The value of the call is remembered as #rotationPoint.
     * @expose
     * @param {GraphObject} obj  the Part#rotateObject
     * @return {Point}
     */
    computeRotationPoint(obj: GraphObject): Point;
    /**
     * Stop the current transaction, forget the #handle and #adornedObject, and release the mouse.
     */
    doDeactivate(): void;
    /**
     * Restore the original GraphObject#angle of the adorned object.
     */
    doCancel(): void;
    /**
     * Call #rotate with a new angle determined by the current mouse point.
     * This determines the new angle by calling #computeRotate.
     */
    doMouseMove(): void;
    /**
     * Call #rotate with an angle based on the most recent mouse point,
     * commit the transaction, and raise the "PartRotated" DiagramEvent.
     */
    doMouseUp(): void;
    /**
     * Change the angle of the selected part's Part#rotateObject.
     * This modifies its GraphObject#angle.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {number} newangle in degrees.
     */
    rotate(newangle: number): void;
    /**
     * Compute the new angle given a point.
     * If the Shift key is pressed, this method ignores the #snapAngleMultiple and #snapAngleEpsilon.
     *
     * If the angle is close (by #snapAngleEpsilon degrees)
     * to a multiple of #snapAngleMultiple degrees,
     * make it exactly that multiple.
     *
     * This method is called by both #doMouseMove and #doMouseUp.
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point} newPoint in document coordinates.
     * @return {number} the new angle, in degrees.
     */
    computeRotate(newPoint: Point): number;
    /**
     * Gets or sets the preferred angles for the selected object.
     *
     * The default value is 45 degrees, meaning that angles that are multiples
     * of 45 degrees are automatically preferred, if the actual angle is
     * close to that multiple.
     * The closeness is determined by the #snapAngleEpsilon property.
     * A value of zero for #snapAngleMultiple results in no snapping at all.
     * Setting this property does not raise any events.
     */
    snapAngleMultiple: number;
    /**
     * Gets or sets the the closeness to a desired angle at which the angle is "snapped to".
     *
     * The default value is 2 degrees, meaning that any angle within 2 degrees
     * of a multiple of the #snapAngleMultiple automatically
     * snaps to that multiple.
     * Values are limited to half of the #snapAngleMultiple;
     * such values restrict user selected angles only to exact multiples of
     * #snapAngleMultiple -- no other angles between them.
     * Setting this property does not raise any events.
     */
    snapAngleEpsilon: number;
    /**
     * This read-only property returns the angle that was the original value of the GraphObject#angle
     * of the GraphObject that is being rotated.
     * The value is invalid when this tool is not active.
     * @since 1.1
     */
    readonly originalAngle: number;
    /**
     * This read-only property returns the value returned by the call to #computeRotationPoint.
     * The value is invalid when this tool is not active.
     * @since 2.0
     */
    readonly rotationPoint: Point;
    /**
     * Gets or sets the spot to locate the Adornment for the rotation handle when it does not have a Placeholder.
     * This assumes the Adornment's location will be at the center of the rotation handle.
     *
     * By default this is zero degrees, which locates the Adornment
     * so that it is placed to the right of an un-rotated object.
     * You can set this to a multiple of 90 degrees to position the rotation handle differently
     * with respect to the #rotationPoint.
     * @since 2.0
     */
    handleAngle: number;
    /**
     * Gets or sets the spot to locate the Adornment at a particular distance from the #adornedObject.
     * This assumes the Adornment's location will be at the center of the rotation handle.
     *
     * By default this is 50, which locates the Adornment
     * so that it is placed to the right of an un-rotated object.
     * @since 2.0
     */
    handleDistance: number;
}
/**
 * The ClickSelectingTool selects and deselects Parts when there is a click.
 * It does this by calling Tool#standardMouseSelect.
 * It is also responsible for handling and dispatching click events on GraphObjects
 * by calling Tool#standardMouseClick.
 *
 * Note that this tool avoids raising click events on objects that are in temporary layers.
 * This is to prevent parts such as selection adornments from interfering with clicking on selected nodes or links.
 * (Adornments are in the "Adornment" Layer, which Layer#isTemporary.)
 * However this means that if you add a GraphObject#click event handler on a GraphObject in an Adornment,
 * it will not be called.
 * You can get it to be called by setting GraphObject#isActionable to true on that object in the adornment.
 *
 * This tool is a standard mouse-up tool, the ToolManager#clickSelectingTool.
 *
 * This tool does not utilize any Adornments or tool handles.
 * This tool does not modify the model or conduct any transaction.
 *
 * An example customization of this tool is shown in the <a href="../../extensions/TreeMap.html">Tree Map</a> sample,
 * where the Tool#standardMouseSelect method is overridden to permit the user to cycle through
 * the chain of containing groups, changing the selection on each click to the next containing group.
 *
 * If you want to programmatically select a Part, you can set Part#isSelected or call Diagram#select.
 * If you want to know which Part is at a particular point, you can call Diagram#findPartAt.
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class ClickSelectingTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#clickSelectingTool.
     *
     * The Tool#name of this tool is "ClickSelecting".
     */
    constructor();
    /**
     * This tool can run whenever a click occurs.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Upon a click, this calls Tool#standardMouseSelect to change the Diagram#selection collection,
     * then calls Tool#standardMouseClick to perform the normal click behaviors,
     * and then stops this tool.
     */
    doMouseUp(): void;
}
/**
 * The ActionTool is responsible for handling and dispatching mouse events on GraphObjects
 * that have GraphObject#isActionable set to true.
 * This is how one implements "controls", such as buttons or sliders or knobs, as GraphObjects
 * that can be inside Parts without interfering with the standard tool behaviors.
 *
 * This tool allows individual GraphObjects (usually Panels) to handle mouse-down-move-up events without
 * having to define new Tools.
 * It does this by calling GraphObject#actionDown, GraphObject#actionMove, GraphObject#actionUp,
 * and GraphObject#actionCancel on objects that have GraphObject#isActionable set to true.
 *
 * This tool is a standard mouse-down tool, the ToolManager#actionTool.
 *
 * This tool does not utilize any Adornments or tool handles.
 * This tool does not modify the model or conduct any transaction, although the actions that this invokes may do so.
 *
 * It would be very unusual to want to customize this tool.
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class ActionTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#actionTool.
     *
     * The Tool#name of this tool is "Action".
     */
    constructor();
    /**
     * This tool can run when there is a mouse-down on an object with GraphObject#isActionable true
     * or if the object is within a Panel that "isActionable".
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * If there is a GraphObject found with GraphObject#isActionable set to true,
     * call that object's GraphObject#actionDown event, if it exists.
     */
    doMouseDown(): void;
    /**
     * If this tool is active call GraphObject#actionMove, if it exists, on the active object.
     */
    doMouseMove(): void;
    /**
     * Calls the GraphObject#actionUp event if defined, then effectively calls
     * Tool#standardMouseClick to perform the normal click behaviors,
     * and then stops this tool.
     */
    doMouseUp(): void;
    /**
     * Call the GraphObject#actionCancel event if defined on the current object.
     */
    doCancel(): void;
}
/**
 * The ClickCreatingTool lets the user create a node by clicking where they want the new node to be.
 * By default a double-click is required to start this tool;
 * set #isDoubleClick to false if you want a single-click to create a node.
 *
 * This tool is a standard mouse-up tool, the ToolManager#clickCreatingTool.
 * However, it will not be able to start running unless you have set the
 * #archetypeNodeData property to an object that can be copied and added to the diagram's model.
 *
 * If you set #isGridSnapEnabled to true, newly created parts will be snapped to points on a grid.
 *
 * This tool does not utilize any Adornments or tool handles.
 * This tool does conduct a transaction when inserting the new node.
 *
 * If you want to programmatically create a Part, you can call #insertPart.
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class ClickCreatingTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#clickCreatingTool, which you can modify.
     *
     * The Tool#name of this tool is "ClickCreating".
     */
    constructor();
    /**
     * This tool can run when the diagram is not read-only and supports creating new nodes,
     * and when there has been a click (or double-click if #isDoubleClick is true)
     * in the background of the diagram (not on a Part),
     * and #archetypeNodeData is an object that can be copied and added to the model.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Upon a click, call #insertPart and stop this tool.
     */
    doMouseUp(): void;
    /**
     * Create a node by adding a copy of the #archetypeNodeData object
     * to the diagram's model, assign its Part#location
     * to be the given point, and select the new part.
     *
     * This starts and commits a transaction and also raises the "PartCreated" DiagramEvent.
     * The actual part that is added to the diagram may be a Part, a Node,
     * or even a Group, depending on the properties of the #archetypeNodeData
     * and the type of the template that is copied to create the part.
     *
     * This method may be overridden, although it is usually much easier to just set #archetypeNodeData.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point} loc a Point in document coordinates.
     * @return {Part} the newly created Part, or null if it failed.
     */
    insertPart(loc: Point): Part;
    /**
     * Gets or sets a data object that will be copied and added to the diagram's model each time this tool executes.
     * The default value is null.
     * The value must be non-null for this tool to be able to run.
     * Setting this property does not raise any events.
     */
    archetypeNodeData: ObjectData | null;
    /**
     * Gets or sets whether a double click rather than a single-click is required
     * to insert a new Part at the mouse-up point.
     * The default value is true -- only a double-click will cause a node to be created.
     * Setting this property does not raise any events.
     */
    isDoubleClick: boolean;
    /**
     * Gets or sets whether the ClickCreatingTool snaps object sizes to the diagram's background grid during creation.
     * By default this property is false.
     * Setting this property does not raise any events.
     *
     * @since 2.0
     */
    isGridSnapEnabled: boolean;
}
/**
 * The DragSelectingTool lets the user select multiple parts within a rectangular area drawn by the user.
 * There is a temporary part, the #box,
 * that shows the current area encompassed between the mouse-down
 * point and the current mouse point.
 * The default drag selection box is a magenta rectangle.
 * You can change the #box to customize its appearance -- see its documentation for an example.
 *
 * This tool is a standard mouse-move tool, the ToolManager#dragSelectingTool.
 * However this cannot start running unless there has been a motionless delay
 * after the mouse-down event of at least #delay milliseconds.
 *
 * This tool does not utilize any Adornments or tool handles,
 * but it does temporarily add the #box part to the diagram.
 * This tool does not modify the model or conduct any transaction.
 *
 * Selection occurs on a mouse-up when it calls #selectInRect
 * with the value of #computeBoxBounds.
 * Selectable parts are selected when their bounds fall entirely within the rectangle,
 * unless #isPartialInclusion is set to true.
 * <p class="boxread">
 * For customizing the DragSelectingTool, see <a href="../../intro/tools.html#DragSelectingTool">Introduction to the DragSelectingTool</a>.
 *
 * If you implement your own drag-in-the-background-to-do-something tool, you may need to disable
 * this tool or insert your new tool in the ToolManager#mouseMoveTools list before this tool,
 * in order for your tool to run.  There are examples of such tools defined in the extensions directory:
 * <a href="../../extensions/RealtimeDragSelecting.html">Realtime Drag Selecting Tool</a>,
 * <a href="../../extensions/DragCreating.html">Drag Creating Tool</a>, and
 * <a href="../../extensions/DragZooming.html">Drag Zooming Tool</a>.
 *
 * If you want to programmatically select some Parts in a rectangular area,
 * you can call #selectInRect.
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class DragSelectingTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#dragSelectingTool, which you can modify.
     *
     * The Tool#name of this tool is "DragSelecting".
     */
    constructor();
    /**
     * This tool can run when the diagram allows selection,
     * there has been delay of at least #delay milliseconds
     * after the mouse-down before a mouse-move,
     * there has been a mouse-drag far enough away not to be a click,
     * and there is no selectable part at the mouse-down point.
     *
     * The delay required to start this tool enables both this tool and
     * the ToolManager#panningTool to co-exist as mode-less mouse-move tools.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Capture the mouse and show the #box.
     */
    doActivate(): void;
    /**
     * Release the mouse and remove any #box.
     */
    doDeactivate(): void;
    /**
     * Update the #box's position and size according to the value
     * of #computeBoxBounds.
     */
    doMouseMove(): void;
    /**
     * Call #selectInRect with the value of a call to #computeBoxBounds.
     *
     * This method changes the cursor to "wait" and
     * raises the "ChangingSelection" DiagramEvent before calling #selectInRect,
     * and raises the "ChangedSelection" DiagramEvent afterward.
     */
    doMouseUp(): void;
    /**
     * This just returns a Rect stretching from the mouse-down point to the current mouse point.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {Rect} a Rect in document coordinates.
     */
    computeBoxBounds(): Rect;
    /**
     * This method is called to select some parts within the area of a given rectangle.
     *
     * The normal behavior is to set the diagram's selection collection to only those parts
     * in the given rectangle according to the #isPartialInclusion policy.
     * However, if the Shift key modifier is used, no parts are deselected --
     * this adds to the selection the parts in the rectangle not already selected.
     * If the Control key (Command on Mac) modifier is used, this toggles the selectedness of the parts in the rectangle.
     * If the Control key (Command on Mac) and Shift key modifiers are both used, this deselects the parts in the rectangle.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Rect} r a rectangular bounds in document coordinates.
     */
    selectInRect(r: Rect): void;
    /**
     * Gets or sets the time in milliseconds for which the mouse must be stationary
     * before this tool can be started.
     * The default value is 175 milliseconds.
     * Setting this property does not raise any events.
     */
    delay: number;
    /**
     * Gets or sets whether a selectable Part may be only partly
     * or must be completely enclosed by the rectangle given to #selectInRect.
     * The default value is false: parts must be completely inside the rectangle.
     * Setting this property does not raise any events.
     */
    isPartialInclusion: boolean;
    /**
     * Gets or sets the Part used as the "rubber-band selection box"
     * that is stretched to follow the mouse, as feedback for what area will
     * be passed to #selectInRect upon a mouse-up.
     *
     * Initially this is a Part containing only a simple magenta rectangular Shape.
     * The object to be resized during dragging should be named "SHAPE".
     * Setting this property does not raise any events.
     *
     * Here is an example of changing the selection box to be a thicker bright green rectangle:
     * ```js
     *   myDiagram.toolManager.dragSelectingTool.box =
     *     $(go.Part,
     *       { layerName: "Tool", selectable: false },
     *       $(go.Shape,
     *         { name: "SHAPE", fill: null, stroke: "chartreuse", strokeWidth: 3 }));
     * ```
     * Note that the Part should be put into a Layer that Layer#isTemporary.
     *
     * Modifying this property while this tool Tool#isActive might have no effect.
     */
    box: Part | null;
}
/**
 * The PanningTool supports manual panning, where the user can shift the
 * Diagram#position by dragging the mouse.
 *
 * This tool is a standard mouse-move tool, the ToolManager#panningTool.
 * Although the ToolManager#dragSelectingTool takes precedence over this tool,
 * the DragSelectingTool only starts if there is a delay after a mouse-down event.
 * If it does not start, then this PanningTool gets a chance to start.
 *
 * This tool does not utilize any Adornments or tool handles.
 * This tool does not modify the model or conduct any transaction.
 *
 * If you want to programmatically "pan" the diagram, you can just set Diagram#position
 * or call methods such as Diagram#scroll, Diagram#scrollToRect, or Diagram#centerRect.
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class PanningTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#panningTool.
     *
     * The Tool#name of this tool is "Panning".
     */
    constructor();
    /**
     * This tool can run when the diagram allows scrolling and
     * the mouse has been dragged with the left button far enough away
     * from the mouse-down point to avoid being a click.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Capture the mouse, change the diagram cursor to "move", and remember the Diagram#position.
     */
    doActivate(): void;
    /**
     * Release the mouse and restore the default diagram cursor.
     */
    doDeactivate(): void;
    /**
     * Restore the Diagram#position to what it was when this tool activated.
     */
    doCancel(): void;
    /**
     * Modify the Diagram#position according to how much the mouse has moved.
     */
    doMouseMove(): void;
    /**
     * Modify the Diagram#position according to how much the mouse has moved.
     */
    doMouseUp(): void;
    /**
     * Gets or sets whether panning actions will allow
     * events to bubble **instead** of panning in the diagram.
     *
     * Set this to true to allow mobile devices to scroll the page with panning gestures on the diagram.
     * Otherwise, the panning events will be captured and used to pan inside of the diagram.
     *
     * The default value is false.
     */
    bubbles: boolean;
    /**
     * This read-only property returns the Point that was the original value of Diagram#position when the panning operation started.
     * @since 1.1
     */
    readonly originalPosition: Point;
}
/**
 * HTMLInfo is used to show and hide custom HTML page elements, such as a context menu, tooltip, or text editor made of HTML.
 *
 * Properties that can be set to an HTMLInfo include:
 *   - TextEditingTool#defaultTextEditor
 *   - TextBlock#textEditor
 *   - GraphObject#contextMenu
 *   - Diagram#contextMenu
 *   - GraphObject#toolTip
 *   - Diagram#toolTip
 *
 * When a context menu is set to an instance of HTMLInfo,
 * ContextMenuTool#showContextMenu and ContextMenuTool#hideContextMenu
 * call #show and #hide respectively. You may define #mainElement
 * instead of #hide in order to automatically use a default hide method.
 *
 * When a tooltip is set to an instance of HTMLInfo,
 * ToolManager#showToolTip and ToolManager#hideToolTip
 * call #show and #hide respectively.
 *
 * When a text editor is set to an instance of HTMLInfo,
 * TextEditingTool#doActivate calls #show and TextEditingTool#doDeactivate calls #hide.
 *
 * For HTMLInfo to work, you must define #show
 * and either #hide or #mainElement.
 * Typical usage will also stop the ContextMenuTool once the desired context action occurs,
 * typically by calling `diagram.currentTool.stopTool();`.
 *
 * Example usage of HTMLInfo can be found in the
 * <a href="../../samples/customContextMenu.html">Custom Context Menu</a> and
 * <a href="../../samples/htmlLightBoxContextMenu.html">HTML LightBox Context Menu</a> samples, the
 * <a href="../../samples/customTextEditingTool.html">Custom TextEditingTool sample</a>, and the
 * <a href="../../extensions/textEditor.html">Text Editor implementation extension</a>.
 *
 * Here is the outline for typical usage of HTMLInfo as a context menu:
 * ```js
 * // Assign an HTMLInfo to the Diagram:
 * myDiagram.contextMenu = $(go.HTMLInfo, {
 *   show: showContextMenu,
 *   hide: hideContextMenu
 * });
 *
 * function showContextMenu(obj, diagram, tool) {
 *   // Show the context menu HTML element:
 *   SomeDOMElement.style.display = "block";
 *
 *   // Also show relevant buttons given the current state
 *   // and the GraphObject obj; if null, the context menu is for the whole Diagram
 * }
 *
 * function hideContextMenu() {
 *   SomeDOMElement.style.display = "none";
 * }
 *
 * function buttonClick() {
 *   // do some action when a context menu button is clicked
 *
 *   // then:
 *   myDiagram.currentTool.stopTool();
 * }
 * ```
 *
 * By default, TextEditingTool#defaultTextEditor is an instance of HTMLInfo.
 * You can see its default implementation details <a href="../../extensions/textEditor.html">here</a>.
 * @unrestricted
 * @since 1.7
 * @category Tool
 */
export class HTMLInfo {
    /**
     * A newly constructed HTMLInfo can be assigned as a value of TextEditingTool#defaultTextEditor, TextBlock#textEditor
     * GraphObject#contextMenu, Diagram#contextMenu, GraphObject#toolTip, or Diagram#toolTip.
     */
    constructor();
    /**
     * Gets or sets the primary HTML Element that represents this HTMLInfo.
     * In a context menu, this would be the outermost HTML element, the one which typically
     * shows and hides. If this is set and #hide is not, HTMLInfo will automatically execute:
     * ```js
     *   tool.mainElement.style.display = "none";
     * ```
     *
     * when #hide would typically be called.
     *
     * This is set only for convenience; the default value for this property is null.
     * @see #hide
     */
    mainElement: HTMLElement | null;
    /**
     * Gets or sets the function to call when an HTMLInfo is to be shown,
     * such as when used as the GraphObject#contextMenu or Diagram#toolTip or TextBlock#textEditor.
     *
     * If this is called by:
     *   - ContextMenuTool#showContextMenu, the first argument is the GraphObject for which the context menu is being shown, or null for the diagram background.
     *   - ToolManager#showToolTip, the first argument is the GraphObject for which the tooltip is being shown, or null for the diagram background.
     *   - TextEditingTool#doActivate, the first argument is the TextBlock for which the text editor is being shown.
     *
     * If you need access to any bound data for the first argument, if it is non-null, you can get it via:
     * `obj.part.data`
     * and then you can look at any of the properties you have put on that data.
     *
     * When used as a context menu, typically shown elements, such as buttons, should call
     * `diagram.currentTool.stopTool();` when their action is completed.
     */
    show: ((a: GraphObject, b: Diagram, c: Tool) => void) | null;
    /**
     * Gets or sets the function to call when an HTMLInfo is to be hidden.
     * The function should "hide" the HTMLInfo, either by removing any traces of it or otherwise
     * returning the page state to one where the HTMLInfo content is no longer considered active.
     *
     * Setting this is not strictly necessary, if no action is to be taken when hiding the HTMLInfo.
     *
     * If this is called by:
     *   - ContextMenuTool#hideContextMenu, the second argument is the ContextMenuTool.
     *   - ToolManager#hideToolTip, the second argument is the ToolManager.
     *   - TextEditingTool#doDeactivate, the second argument is the TextEditingTool.
     *
     * @see #mainElement
     */
    hide: ((a: Diagram, b: Tool) => void) | null;
    /**
     * Gets or sets a function that returns the primary value associated with this HTMLInfo,
     * such as the string value of a text editor, which would be solicited by the TextEditingTool.
     *
     * This typically returns a string.
     */
    valueFunction: (() => any) | null;
}
/**
 * The ContextMenuTool is used to create and show a context menu.
 * It automatically disables any browser context menu.
 *
 * Define context menus on individual GraphObjects by setting GraphObject#contextMenu.
 * Define a context menu for the diagram background by setting Diagram#contextMenu.
 *
 * This tool is a standard mouse-down tool, the ToolManager#contextMenuTool.
 *
 * This tool does not utilize any tool handles.
 * This tool does not modify the model or conduct any transaction,
 * although any code invoked by context menu commands might do so.
 *
 * There are examples of customizing this tool in the
 * <a href="../../samples/customContextMenu.html">Custom Context Menu</a> and
 * <a href="../../samples/htmlLightBoxContextMenu.html">HTML LightBox Context Menu</a> samples.
 *
 * If you want to programmatically show a context menu for a particular GraphObject or for the
 * whole diagram, call CommandHandler#showContextMenu.
 * That command method is also invoked by the Menu key on the keyboard.
 *
 * Normally this shows a context menu (if available) on a right-mouse-up event.
 * If you want it to happen on a right-mouse-down event, you'll need to move this tool
 * from the ToolManager#mouseUpTools list to the ToolManager#mouseDownTools list:
 * ```js
 * myDiagram.toolManager.mouseDownTools.add(myDiagram.toolManager.replaceTool("ContextMenu", null));
 * ```
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class ContextMenuTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#contextMenuTool, which you can modify.
     *
     * The Tool#name of this tool is "ContextMenu".
     */
    constructor();
    /**
     * Return true if it's a mouse right click that hasn't moved Tool#isBeyondDragSize
     * and that is on a GraphObject with a GraphObject#contextMenu.
     * This is also true if the mouse right click is in the diagram background
     * and the diagram's Diagram#contextMenu is non-null.
     *
     * On touch devices, a special default context menu will appear even if no object with a context menu is found.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * Find a GraphObject at the current mouse point with a GraphObject#contextMenu,
     * or return the Diagram if there is a Diagram#contextMenu.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {GraphObject|Diagram=} obj Optional GraphObject with which to start searching for a context menu.
     * If null, the Diagram will be used. If no argument is specified, this method will look for an object at the current mouse point.
     * @return {GraphObject|Diagram|null} something with a `contextMenu`,
     * or null if nothing can be found with a context menu at the current mouse point.
     */
    findObjectWithContextMenu(obj?: GraphObject | Diagram): GraphObject | Diagram | null;
    /**
     * Do nothing, activation is special and relies on doMouseUp
     */
    doActivate(): void;
    /**
     * Activate this but also execute the normal behavior that would happen on a mouse-up
     * if this tool is in the ToolManager#mouseDownTools list.
     * Perform ActionTool#doMouseDown if a ContextMenuButton was hit.
     */
    doMouseDown(): void;
    /**
     * If there is something found by #findObjectWithContextMenu,
     * call #showContextMenu with that object's GraphObject#contextMenu
     * or the Diagram#contextMenu.
     *
     * Once a context menu is being shown,
     * if a click occurs on a part of the context menu, call Tool#standardMouseClick.
     * Otherwise if the click occurs elsewhere, just stop this tool.
     * Unlike most tools, the first mouse-up should not stop this tool.
     */
    doMouseUp(): void;
    /**
     * Handle mouse-enter, mouse-over, and mouse-leave events, as well as tooltips.
     */
    doMouseMove(): void;
    /**
     * Show an Adornment or HTMLInfo as a context menu.
     *
     * This method is called by the context click (Tool#doMouseDown) and CommandHandler#showContextMenu.
     * If you want to programmatically show a context menu for a particular GraphObject or for the
     * whole diagram, do not call this method, which only does one small piece of the process of
     * bringing up a context menu.
     * Instead call CommandHandler#showContextMenu, which will start this tool
     * and eventually call this method and handle additional input events.
     *
     * For Adornment context menus: If the object's containing Part is data-bound,
     * this sets the contextmenu's `Part.data` to the same value.
     * The Adornment#adornedObject property is set to the GraphObject for which the menu is being shown.
     *
     * This method sets the #currentContextMenu.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Adornment|HTMLInfo} contextmenu
     * @param {GraphObject} obj the GraphObject for which the context menu is being shown;
     *   this is null if the contextmenu is being shown for the diagram background.
     */
    showContextMenu(contextmenu: Adornment | HTMLInfo, obj: GraphObject | null): void;
    /**
     * This is called by #showContextMenu to position the context menu within the viewport.
     * It normally goes just below the cursor.
     * But if the mouse is too close to the right edge or the bottom edge of the viewport,
     * it is positioned left and/or above the cursor.
     *
     * This method only operates if the context menu, an Adornment, does not have a Adornment#placeholder.
     * When there is a Placeholder in the context menu, that Adornment is automatically positioned
     * so that the Placeholder is positioned at the adorned object, the second argument to this method.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * If you override this method to position the context menu,
     * the context menu has already been measured but not arranged,
     * so you can use its GraphObject#measuredBounds width and height
     * but not its GraphObject#actualBounds.
     * @expose
     * @param {Adornment} contextmenu
     * @param {GraphObject} obj The GraphObject getting the context menu,
     * or null if the context menu is for the diagram background.
     */
    positionContextMenu(contextmenu: Adornment, obj: GraphObject | null): void;
    /**
     * Hide any context menu.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    hideContextMenu(): void;
    /**
     * This is the HTMLInfo#show method for the #defaultTouchContextMenu.
     *
     * If the object's containing Part is data-bound,
     * set the contextmenu's `Part.data` to the same value.
     * The Adornment#adornedObject property is set to the GraphObject
     * for which the menu is being shown.
     *
     * The menu carries some default CSS styling and uses the following CSS classes:
     *
     * * `goCXforeground` for the DIV containing the buttons
     * * `goCXbackground` for the darker DIV behind the context menu
     * * `goCXul` for the HTML `ul` items
     * * `goCXli` for the HTML `li` tag items
     * * `goCXa` for the HTML `a` tag items
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a>
     * for how to override methods and how to call this base method.
     * @expose
     */
    showDefaultContextMenu(): void;
    /**
     * This is the HTMLInfo#hide method for the #defaultTouchContextMenu.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a>
     * for how to override methods and how to call this base method.
     * @expose
     */
    hideDefaultContextMenu(): void;
    /**
     * Gets or sets the currently showing context menu, or null if there is none.
     * This is typically only set in #showContextMenu and not by the user.
     * It is also typically set to null in #hideContextMenu.
     */
    currentContextMenu: Adornment | HTMLInfo | null;
    /**
     * Gets or sets the HTMLInfo or Adornment that acts as the default touch context menu.
     * On touch devices, this context menu will appear even there is no context menu defined.
     *
     * You can disable this functionality by setting this property to null.
     *
     * By default shows a set of HTML elements acting as a context menu.
     * @since 1.7
     */
    defaultTouchContextMenu: Adornment | HTMLInfo | null;
    /**
     * Gets or sets the GraphObject found at the mouse point that has a context menu.
     * This property remembers the result returned by #findObjectWithContextMenu if it is a GraphObject.
     * This value is passed to #showContextMenu as the second argument.
     * The value will be null if the context menu is for the diagram rather than for a particular GraphObject.
     * @since 1.4
     */
    currentObject: GraphObject | null;
    /**
     * This read-only property returns the original mouse-down point in document coordinates.
     */
    readonly mouseDownPoint: Point;
}
/**
 * The TextEditingTool is used to let the user interactively edit text in place.
 * This sets the TextBlock#text property; you may want to save the changed text to the model
 * by using a TwoWay Binding on the "text" property of editable TextBlocks.
 *
 * Typically this is used by setting the TextBlock#editable property to true
 * on a particular TextBlock in a part.
 * When the part is selected and the user clicks on the TextBlock
 * or invokes the CommandHandler#editTextBlock command,
 * this tool is started and it uses an HTMLTextArea to perform in-place text editing.
 * (For more details see the description for TextEditingTool#doActivate.)
 *
 * The TextBlock is accessible as the TextEditingTool#textBlock property.
 * The text editor is accessible as the TextEditingTool#currentTextEditor property.
 * From the text editor control one can access the TextBlock being edited via the
 * 'textEditingTool' property to get to this tool, from which one can use
 * the TextEditingTool#textBlock property.
 *
 * You can disable mouse clicking from starting this text editing tool
 * by setting Tool#isEnabled to false.
 * You can disable the F2 key from starting this text editing tool
 * by making sure Part#canEdit returns false,
 * by either setting Diagram#allowTextEdit to false
 * or by setting Part#textEditable to false.
 *
 * If you want to programmatically start the user editing a particular TextBlock,
 * call CommandHandler#editTextBlock.
 * That command method is also invoked by the F2 key on the keyboard.
 * <p class="boxread">
 * For a general discussion of text editing validation, see: <a href="../../intro/validation.html#TextEditingValidation">Introduction to Text Validation</a>.
 * For customizing the TextEditingTool, read about HTMLInfo and see <a href="../../intro/HTMLInteraction.html#TextEditors">Introduction to Text Editors</a>.
 * @extends Tool
 * @unrestricted
 * @category Tool
 */
export class TextEditingTool extends Tool {
    /**
     * You do not normally need to create an instance of this tool
     * because one already exists as the ToolManager#textEditingTool, which you can modify.
     *
     * The Tool#name of this tool is "TextEditing".
     */
    constructor();
    /**
     * A possible value for TextEditingTool#acceptText, the text editing control has lost focus.
     * @constant
     */
    static LostFocus: EnumValue;
    /**
     * A possible value for TextEditingTool#acceptText, the user has clicked somewhere else in the diagram.
     * @constant
     */
    static MouseDown: EnumValue;
    /**
     * A possible value for TextEditingTool#acceptText, the user has typed TAB.
     * @constant
     */
    static Tab: EnumValue;
    /**
     * A possible value for TextEditingTool#acceptText, the user has typed ENTER.
     * @constant
     */
    static Enter: EnumValue;
    /**
     * A possible value for TextEditingTool#starting,
     * A single click on a TextBlock with TextBlock#editable property set to true
     * will start in-place editing.
     * @constant
     */
    static SingleClick: EnumValue;
    /**
     * A possible value for TextEditingTool#starting,
     * A single click on a TextBlock with TextBlock#editable property set to true
     * will start in-place editing, but only if the Part that the TextBlock is in is already selected.
     * @constant
     */
    static SingleClickSelected: EnumValue;
    /**
     * A possible value for TextEditingTool#starting,
     * A double click on a TextBlock with TextBlock#editable property set to true
     * will start in-place editing.
     * @constant
     */
    static DoubleClick: EnumValue;
    /**
     * A possible value for #state,
     * this is the starting value before the tool is activated.
     * @constant
     */
    static StateNone: EnumValue;
    /**
     * A possible value for #state,
     * TextEditingTool#doActivate sets the value of #state to this.
     * @constant
     */
    static StateActive: EnumValue;
    /**
     * A possible value for #state,
     * The default text editor receiving focus sets the value of #state to this.
     * @constant
     */
    static StateEditing: EnumValue;
    /**
     * A possible value for #state,
     * TextEditingTool.acceptText sets the value of #state to this.
     * @constant
     */
    static StateValidating: EnumValue;
    /**
     * A possible value for #state,
     * If validation fails, TextEditingTool.acceptText sets the value of #state to this.
     * @constant
     */
    static StateInvalid: EnumValue;
    /**
     * A possible value for #state,
     * Once accepted and the tool begins the "TextEditing" transaction, the value of #state is set to this.
     * @constant
     */
    static StateValidated: EnumValue;
    /**
     * Gets or sets the TextBlock that is being edited.
     * This property is initially null and is set in TextEditingTool#doActivate
     * as the TextBlock at the mouse click point.
     * However, if you set this property beforehand, TextEditingTool#doActivate
     * will not set it, and this tool will edit the given TextBlock.
     */
    textBlock: TextBlock | null;
    /**
     * Gets or sets the HTMLInfo that is editing the text.
     */
    currentTextEditor: HTMLInfo | null;
    /**
     * Gets or sets the default HTMLInfo that edits the text.
     *
     * When TextEditingTool#doActivate is called, the #currentTextEditor is set to this value by default.
     * If a TextBlock#textEditor is specified on the TextBlock, that editor is used instead.
     *
     * This tool will call HTMLInfo#show during #doActivate,
     * and HTMLInfo#hide during #doDeactivate.
     *
     * By default the value is an HTMLInfo, and the HTMLInfo#mainElement is an `HTMLTextArea`.
     * You can see the default implementation details <a href="../../extensions/textEditor.html">here</a>.
     *
     * For typical operation, HTMLInfo implementations should have a way of calling TextEditingTool#acceptText.
     */
    defaultTextEditor: HTMLInfo;
    /**
     * Gets or sets how user gestures can start in-place editing of text.
     *
     * Possible values are TextEditingTool.SingleClickSelected, TextEditingTool.SingleClick, and TextEditingTool.DoubleClick.
     *
     * The default is TextEditingTool.SingleClickSelected
     */
    starting: EnumValue;
    /**
     * This may run when there is a mouse-click on a TextBlock for which the
     * TextBlock#editable property is true in a Part
     * that Part#isSelected.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     */
    canStart(): boolean;
    /**
     * This calls TextEditingTool#doActivate if there is a
     * #textBlock set. #doActivate attempts to set #textBlock if it is null.
     */
    doStart(): void;
    /**
     * Start editing the text for a #textBlock.
     *
     * If TextEditingTool#textBlock is not already specified, this looks for one at the current
     * mouse point. If none is found, this method does nothing.
     *
     * This method sets #currentTextEditor.
     * If TextBlock#textEditor is defined on the TextBlock it will use that as the value.
     * By default, it uses the value of #defaultTextEditor, which is an HTMLInfo
     * showing an HTML textarea, with the CSS class `goTXarea`.
     *
     * If the #currentTextEditor is an HTMLInfo, this method calls HTMLInfo#show on that instance.
     *
     * This sets Tool#isActive to true.
     * Custom text editors should call TextEditingTool#acceptText to finish the edit
     * by modifying the TextBlock and committing the edit transaction.
     * Or call TextEditingTool#doCancel to abort the edit.
     */
    doActivate(): void;
    /**
     * Abort any text editing operation.
     */
    doCancel(): void;
    /**
     * A click (mouse up) calls TextEditingTool#doActivate if this tool is not already active
     * and if TextEditingTool#canStart returns true.
     */
    doMouseUp(): void;
    /**
     * This calls #acceptText with the reason TextEditingTool.MouseDown,
     * if this tool Tool#isActive.
     */
    doMouseDown(): void;
    /**
     * Finish editing by trying to accept the new text.
     *
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {EnumValue} reason The reason must be either TextEditingTool.LostFocus,
     * TextEditingTool.MouseDown, TextEditingTool.Tab, or TextEditingTool.Enter.
     */
    acceptText(reason: EnumValue): void;
    /**
     * Release the mouse.
     *
     * If the #currentTextEditor is an HTMLInfo, this calls HTMLInfo#hide.
     */
    doDeactivate(): void;
    /**
     * This predicate checks any TextBlock#textValidation predicate and
     * this tool's #textValidation predicate to make sure the TextBlock#text
     * property may be set to the new string.
     *
     * This method may be overridden, although usually it is sufficient to set #textValidation.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {TextBlock} textblock the TextBlock that is being edited.
     * @param {string} oldstr the previous string value.
     * @param {string} newstr the proposed new string value.
     * @return {boolean} true if the new string is valid for the given TextBlock.
     */
    isValidText(textblock: TextBlock, oldstr: string, newstr: string): boolean;
    /**
     * Gets or sets the predicate that determines whether or not a string of text is valid.
     * If this is non-null, this predicate is called in addition to any TextBlock#textValidation predicate.
     * See #isValidText for more details.
     * The default predicate is null, which is equivalent to simply returning true.
     *
     * The function, if supplied, must not have any side-effects.
     */
    textValidation: ((aTextBlock: TextBlock, oldString: string, newString: string) => boolean) | null;
    /**
     * Gets or sets whether to select (highlight) the editable text when the TextEditingTool is activated.
     * The default is true.
     * @since 1.5
     */
    selectsTextOnActivate: boolean;
    /**
     * Gets or sets the state of the TextEditingTool.
     * The only accepted values are listed as constant properties of TextEditingTool, including:
     *   - TextEditingTool.StateNone
     *   - TextEditingTool.StateActive
     *   - TextEditingTool.StateEditing
     *   - TextEditingTool.StateValidating
     *   - TextEditingTool.StateValidated
     *
     * The starting value value is TextEditingTool.StateNone,
     * TextEditingTool#doActivate sets the value to TextEditingTool.StateActive.
     * The default text editor receiving focus sets the value to TextEditingTool.StateEditing.
     * TextEditingTool#acceptText sets the value to TextEditingTool.StateValidating.
     * Once accepted and the tool begins the "TextEditing" transaction, the value is set to TextEditingTool.StateValidated.
     * @since 1.7
     */
    state: EnumValue;
    /**
     * This method returns a temporary TextBlock used for measuring text during editing.
     * The TextBlock.text is set to the parameter's value,
     * and the TextBlock is measured with the last available width of the #textBlock.
     *
     * Text editors can use the GraphObject#measuredBounds and TextBlock#lineCount
     * to determine a reasonable size for their text areas.
     * @param {string} text the text to measure
     * @return {TextBlock}
     * @since 1.7
     */
    measureTemporaryTextBlock(text: string): TextBlock;
}
/**
 * AnimationManager handles animations in a Diagram. Each Diagram has one, Diagram#animationManager.
 * Layouts, Group expansion and Tree expansion automatically start animations.
 *
 * Animation is enabled by default, setting the #isEnabled property to false will turn off animations for a Diagram.
 *
 * When an animations begins it raises the "AnimationStarting" diagram event,
 * upon completion it raises the "AnimationFinished" diagram event.
 *
 * Animation will stop if a new transaction is started, if an undo or redo is called,
 * if a layout is invalidated, or if a model is replaced.
 * When an animation is stopped, the Diagram immediately finishes the animation and draws the final state.
 * Animations can be stopped programatically with the method AnimationManager#stopAnimation.
 * @since 1.4
 * @unrestricted
 */
export class AnimationManager {
    /**
     * You do not normally need to create an instance of this class because one already exists as the Diagram#animationManager, which you can modify.
     */
    constructor();
    /**
     * Undocumented
     */
    prepareAutomaticAnimation(reason: string, options?: {
        onChange?: () => void;
        onComplete?: () => void;
        easing?: (a: number, b: number, c: number, d: number) => number;
        duration?: number;
    }): void;
    /**
     * Undocumented.
     */
    prepareAnimation(reason: string, options?: {
        onChange?: () => void;
        onComplete?: () => void;
        easing?: (a: number, b: number, c: number, d: number) => number;
        duration?: number;
    }): void;
    /**
     * Undocumented.
     */
    addToAnimation(obj: GraphObject, propname: string, start: any, end: any, cosmetic?: boolean, remainsVisible?: boolean): void;
    /**
     * Stops any running animation and updates the Diagram to its final state.
     *
     * If an animation was about to begin, it is cancelled.
     */
    stopAnimation(): void;
    /**
     * Gets or sets whether this AnimationManager operates.
     *
     * The default value is true.
     * Setting this to false does not stop an animation, it only stops future animations.
     * To stop any ongoing animation, use #stopAnimation.
     * Setting this property does not raise any events.
     */
    isEnabled: boolean;
    /**
     * Gets or sets the duration for animations, in milliseconds.
     *
     * The default value is 600 milliseconds.
     * The value must be a number greater than or equal to 1.
     * Setting this property does not raise any events.
     */
    duration: number;
    /**
     * This read-only property is true when the animation manger is currently animating.
     *
     * This value cannot be set, but animation can be stopped by calling #stopAnimation,
     * and it can be prevented by setting #isEnabled.
     */
    readonly isAnimating: boolean;
    /**
     * This read-only property is true when the animation manger is in the middle of an animation tick.
     * Animation only operates on GraphObjects during ticks, but code outside of AnimationManager's control may execute between ticks.
     *
     * isTicking can only be true when #isAnimating is also true.
     */
    readonly isTicking: boolean;
    /**
     * Gets or sets whether an animation is performed on an initial layout.
     *
     * The default value is true.
     * Changing the value does not affect any ongoing animation.
     * Setting this property does not raise any events.
     * @since 1.6
     */
    isInitial: boolean;
}
/**
 * Layers are how named collections of Parts are drawn in front or behind other collections of Parts in a Diagram.
 * Layers can only contain Parts, such as Nodes and Links. They cannot hold GraphObjects directly.
 *
 * You put a Part into a Layer by assigning Part#layerName with the name of the Layer.
 * You can use data binding to initialize and remember a Part's layer's name.
 * You can change a Part's layer by modifying its Part#layerName, which changes its Part#layer.
 *
 * Each Diagram starts off with the following list of Layers:
 * "Grid", "Background", "" (the default layer), "Foreground", "Adornment", "Tool".
 * Parts are normally put in the default layer.
 * The "Grid", "Adornment", and "Tool" layers are considered #isTemporary.
 * Changes to objects in temporary layers are not recorded by the UndoManager.
 * Parts in temporary layers are not selected and are not considered to be part of the document.
 * Objects in temporary layers do not receive click events unless you set their GraphObject#isActionable to true.
 * The "Grid" layer is the furthest back; it also contains "temporary" parts that cannot be selected.
 * Furthermore the "Grid" layer has #pickable set to false so that mouse or touch events
 * and calls to the "find..." methods do not even consider any parts in that layer.
 *
 * Layers have many properties that control what actions users are permitted to perform involving the parts in the layer.
 * These properties are very much like the similarly named properties on Diagram.
 * <h3>Z-ordering</h3>
 *
 * Layers are drawn and presented in order.
 * You can add your own layers by calling Diagram#addLayerBefore or Diagram#addLayerAfter
 * to insert a new layer at a particular place in the Z-order, or to re-order existing layers.
 * Use Diagram#findLayer to get the Layer with a particular name.
 * Parts can be individually z-ordered within a layer by setting Part#zOrder.
 * @unrestricted
 */
export class Layer {
    /**
     * This constructs an empty Layer; you should set the #name before adding the Layer to a Diagram.
     */
    constructor();
    /**
     * Find the front-most GraphObject in this layer
     * at the given point in document coordinates.
     *
     * If #visible is false, this method will not find any objects in this layer.
     * However, #opacity does not affect this method.
     * @param {Point} p A Point in document coordinates.
     * @param {function(GraphObject):GraphObject | null=} navig A function taking a GraphObject and
     * returning a GraphObject, defaulting to the identity.
     * @param {function(GraphObject):boolean | null=} pred A function taking the GraphObject
     * returned by navig and returning true if that object should be returned,
     * defaulting to a predicate that always returns true.
     * @return {GraphObject} The first GraphObject in the Z-order, or else null.
     */
    findObjectAt<T extends GraphObject>(p: Point, navig?: ((a: GraphObject) => (T | null)) | null, pred?: ((a: T) => boolean) | null): (T | null);
    /**
     * Return a collection of the GraphObjects of this layer
     * at the given point in document coordinates.
     *
     * If #visible is false, this method will not find any objects in this layer.
     * However, #opacity does not affect this method.
     * @param {Point} p A Point in document coordinates.
     * @param {function(GraphObject):GraphObject | null=} navig A function taking a GraphObject and
     * returning a GraphObject, defaulting to the identity.
     * If this function returns null, the given GraphObject will not be included in the results.
     * @param {function(GraphObject):boolean | null=} pred A function taking the GraphObject
     * returned by navig and returning true if that object should be returned,
     * defaulting to a predicate that always returns true.
     * @param {List.<GraphObject>|Set.<GraphObject>=} coll An optional collection (List or Set) to add the results to.
     * @return {Iterable.<GraphObject>} a collection of GraphObjects that will contain all GraphObjects
     * located at Point p, or else an empty collection.
     * If a List or Set was passed in, it is returned.
     */
    findObjectsAt<T extends GraphObject, S extends List<T> | Set<T> = Set<T>>(p: Point, navig?: ((a: GraphObject) => (T | null)) | null, pred?: ((a: T) => boolean) | null, coll?: S): S;
    /**
     * Returns a collection of all GraphObjects that are inside or that intersect
     * a given Rect in document coordinates.
     *
     * If #visible is false, this method will not find any objects in this layer.
     * However, #opacity does not affect this method.
     * @param {Rect} r A Rect in document coordinates.
     * @param {function(GraphObject):GraphObject | null=} navig A function taking a GraphObject and
     * returning a GraphObject, defaulting to the identity.
     * If this function returns null, the given GraphObject will not be included in the results.
     * @param {function(GraphObject):boolean | null=} pred A function taking the GraphObject
     * returned by navig and returning true if that object should be returned,
     * defaulting to a predicate that always returns true.
     * @param {boolean=} partialInclusion Whether an object can match if it merely intersects the rectangular area (true) or
     * if it must be entirely inside the rectangular area (false).  The default value is false.
     * @param {List.<GraphObject>|Set.<GraphObject>=} coll An optional collection (List or Set) to add the results to.
     * @return {Iterable.<GraphObject>} a collection of GraphObjects that will contain all GraphObjects
     * located in or near Rect r, or else an empty collection.
     * If a List or Set was passed in, it is returned.
     */
    findObjectsIn<T extends GraphObject, S extends List<T> | Set<T> = Set<T>>(r: Rect, navig?: ((a: GraphObject) => (T | null)) | null, pred?: ((a: T) => boolean) | null, partialInclusion?: boolean, coll?: S): S;
    /**
     * Returns a collection of all GraphObjects that are within a certain distance
     * of a given point in document coordinates.
     *
     * If #visible is false, this method will not find any objects in this layer.
     * However, #opacity does not affect this method.
     * @param {Point} p A Point in document coordinates.
     * @param {number} dist The distance from the point.
     * @param {function(GraphObject):GraphObject | null=} navig A function taking a GraphObject and
     * returning a GraphObject, defaulting to the identity.
     * If this function returns null, the given GraphObject will not be included in the results.
     * @param {function(GraphObject):boolean | null=} pred A function taking the GraphObject
     * returned by navig and returning true if that object should be returned,
     * defaulting to a predicate that always returns true.
     * @param {*=} partialInclusion Whether an object can match if it merely intersects the circular area (true) or
     * if it must be entirely inside the circular area (false).  The default value is true.
     * @param {List.<GraphObject>|Set.<GraphObject>=} coll An optional collection (List or Set) to add the results to.
     * @return {Iterable.<GraphObject>} a collection of GraphObjects that will contain all GraphObjects
     * located at Point p, or else an empty collection.
     * If a List or Set was passed in, it is returned.
     */
    findObjectsNear<T extends GraphObject, S extends List<T> | Set<T> = Set<T>>(p: Point, dist: number, navig?: ((a: GraphObject) => (T | null)) | null, pred?: ((a: T) => boolean) | null, partialInclusion?: boolean | S, coll?: S): S;
    /**
     * This read-only property returns an iterator for this Layer's Parts.
     * The Parts can be Nodes, Links, Groups, Adornments, or simple Parts.
     */
    readonly parts: Iterator<Part>;
    /**
     * This read-only property returns a backwards iterator for this Layer's Parts,
     * for iterating over the parts in reverse order.
     * The Parts can be Nodes, Links, Groups, Adornments, or simple Parts.
     */
    readonly partsBackwards: Iterator<Part>;
    /**
     * This read-only property returns the Diagram that is using this Layer.
     */
    readonly diagram: Diagram | null;
    /**
     * Gets or sets the name for this layer.
     * The initial value is an empty string, which is also the name of the default layer.
     * The name should be unique among the diagram's Diagram#layers.
     */
    name: string;
    /**
     * Gets or sets the opacity for all parts in this layer.
     * The value must be between 0.0 (fully transparent) and 1.0 (no additional transparency).
     * This value is multiplicative with any existing transparency,
     * for instance from a Brush or image transparency.
     * The default value is 1.
     *
     * This property, unlike #visible, does not change whether any objects are found by the "find..." methods.
     * @since 1.1
     * @see GraphObject#opacity
     */
    opacity: number;
    /**
     * Gets or sets whether the objects in this layer are considered temporary.
     *
     * Parts in temporary layers are not selectable,
     * and changes to Parts in temporary layers are not recorded in the UndoManager.
     * Objects in temporary layers do not receive click events unless you set their GraphObject#isActionable to true.
     *
     * Temporary layers are excluded from bounds calculations, with the exception of the "Tool" layer,
     * so that temporary objects created while dragging are included in the bounds.
     *
     * Default value is false.
     */
    isTemporary: boolean;
    /**
     * Gets or sets whether the user may view any of the objects in this layer.
     *
     * The default value is true -- all visible Parts are drawn.
     * When this property is false, all of the "find..." methods will fail to find parts that are in this layer.
     * @see GraphObject#visible
     */
    visible: boolean;
    /**
     * Gets or sets whether methods such as #findObjectAt find any of the objects in this layer.
     *
     * The default value is true.
     * When this property is false, all of the "find..." methods will fail to find parts that are in this layer.
     *
     * Note that setting pickable to false does not prevent users from selecting nodes.
     * It does prevent them from selecting nodes by clicking on them,
     * but does not prevent selection through other mechanisms such as the DragSelectingTool or
     * CommandHandler#selectAll or calls to Diagram#select.
     *
     * You can control whether individual GraphObjects are "hittable" by setting GraphObject#pickable.
     * @since 1.2
     * @see GraphObject#pickable
     */
    pickable: boolean;
    /**
     * Gets or sets whether the user may copy objects in this layer.
     * The initial value is true.
     */
    allowCopy: boolean;
    /**
     * Gets or sets whether the user may delete objects in this layer.
     * The initial value is true.
     */
    allowDelete: boolean;
    /**
     * Gets or sets whether the user may do in-place text editing in this layer.
     * The initial value is true.
     */
    allowTextEdit: boolean;
    /**
     * Gets or sets whether the user may group parts together in this layer.
     * The initial value is true.
     */
    allowGroup: boolean;
    /**
     * Gets or sets whether the user may ungroup existing groups in this layer.
     * The initial value is true.
     */
    allowUngroup: boolean;
    /**
     * Gets or sets whether the user may draw new links in this layer.
     * The initial value is true.
     */
    allowLink: boolean;
    /**
     * Gets or sets whether the user may reconnect existing links in this layer.
     * The initial value is true.
     */
    allowRelink: boolean;
    /**
     * Gets or sets whether the user may move objects in this layer.
     * The initial value is true.
     */
    allowMove: boolean;
    /**
     * Gets or sets whether the user may reshape parts in this layer.
     * The initial value is true.
     */
    allowReshape: boolean;
    /**
     * Gets or sets whether the user may resize parts in this layer.
     * The initial value is true.
     */
    allowResize: boolean;
    /**
     * Gets or sets whether the user may rotate parts in this layer.
     * The initial value is true.
     */
    allowRotate: boolean;
    /**
     * Gets or sets whether the user may select objects in this layer.
     * The initial value is true.
     */
    allowSelect: boolean;
}
/**
 * A Diagram is associated with an HTML DIV element. Constructing a Diagram creates
 * an HTML Canvas element which it places inside of the given DIV element, in addition to several helper DIVs.
 * **GoJS** will manage the contents of this DIV -- you should not modify the contents of the DIV,
 * although you may style the given DIV (background, border, etc) and position and size it as needed.
 *
 * Minimal Diagram construction looks like this. HTML:
 * ```html
 * <div id="myDiagramDiv" style="border: solid 1px black; width:400px; height:400px"></div>
 * ```
 *
 * JavaScript:
 * ```js
 * var $ = go.GraphObject.make;  // for conciseness
 *
 * myDiagram = $(go.Diagram, "myDiagramDiv",  // create a Diagram for the DIV HTML element
 *               {
 *                 "undoManager.isEnabled": true  // enable undo & redo
 *               });
 * ```
 *
 * The diagram will draw onto an HTML Canvas element, created inside the Diagram DIV.
 *
 * Each Diagram holds a set of Layers each of which holds some number of Parts
 * such as Nodes and Links.
 * Each Part consists of GraphObjects such as TextBlocks and Shapes
 * and Panels holding yet more GraphObjects.
 *
 * A Diagram and its Parts provide the visual representation of a Model that holds JavaScript
 * data objects for the nodes and the links.
 * The model provides the way to recognize the relationships between the data.
 * <p class="boxrun">
 * Two Diagrams can display and manipulate the same Model. (<a href="../../samples/updateDemo.html">Example</a>)
 *
 * A diagram will automatically create Nodes and Links corresponding to the model data.
 * The diagram has a number of named templates it uses to create the actual parts:
 * #nodeTemplateMap, #groupTemplateMap, and #linkTemplateMap.
 * Each template may have some data Bindings that set the part's GraphObjects' properties
 * based on the value of properties of the data.
 *
 * A simple Node template and Model data (both nodes and links) may look like this:
 *
 * ```js
 * var $ = go.GraphObject.make;  // for conciseness
 *
 * // define a simple Node template
 * myDiagram.nodeTemplate =
 *   $(go.Node, "Auto",  // the Shape will go around the TextBlock
 *     $(go.Shape, "RoundedRectangle",
 *       // Shape.fill is bound to Node.data.color
 *       new go.Binding("fill", "color")),
 *     $(go.TextBlock,
 *       { margin: 3 },  // some room around the text
 *       // TextBlock.text is bound to Node.data.key
 *       new go.Binding("text", "key"))
 *   );
 *
 * // create the model data that will be represented by Nodes and Links
 * myDiagram.model = new go.GraphLinksModel(
 * [
 *   { key: "Alpha", color: "lightblue" },
 *   { key: "Beta", color: "orange" },
 *   { key: "Gamma", color: "lightgreen" },
 *   { key: "Delta", color: "pink" }
 * ],
 * [
 *   { from: "Alpha", to: "Beta" },
 *   { from: "Alpha", to: "Gamma" },
 *   { from: "Beta", to: "Beta" },
 *   { from: "Gamma", to: "Delta" },
 *   { from: "Delta", to: "Alpha" }
 * ]);
 * ```
 * <p class="boxrun">
 * The above code is used to make the <a href="../../samples/minimal.html">Minimal sample</a>, a simple example of
 * creating a Diagram and setting its model.
 * <p class="boxread">
 * Read about models on the <a href="../../intro/usingModels.html">Using Models</a> page in the introduction.
 * A diagram is responsible for scrolling (#position) and zooming (#scale) all of the parts that it shows.
 * Each Part occupies some area given by its GraphObject#actualBounds.
 *
 * The union of all of the parts' bounds constitutes the #documentBounds.
 * The document bounds determines the area that the diagram can be scrolled to.
 * There are several properties that you can set, such as #initialContentAlignment,
 * that control the initial size and position of the diagram contents.
 *
 * At any later time you can also explicitly set the #position and/or #scale to
 * get the appearance that you want.  But you may find it easier to call methods to get the desired effect.
 * For example, if you want to make a particular Node be centered in the viewport,
 * call either #centerRect or #scrollToRect with the Node's GraphObject#actualBounds,
 * depending on whether or not you want the view to be scrolled if the node is already in view.
 * <p class="boxread">
 * Read in the Introduction about <a href="../../intro/viewport.html">Viewports</a>
 * and the <a href="../../intro/initialView.html">Initial Viewport</a>.
 * You can have the diagram perform automatic layouts of its nodes and links by setting
 * #layout to an instance of the Layout subclass of your choice.
 * The default #layout is an instance of the Layout base class that ignores links and
 * only positions Nodes that do not have a location.
 * This default layout will allow you to programmatically position nodes (including by loading
 * from a database) and will also allow the user to manually position nodes using the DraggingTool.
 *
 * If you do supply a particular layout as the #layout, you can control which Parts it operates
 * on by setting Part#isLayoutPositioned.
 * Normally, of course, it works on all top-level nodes and links.
 * The layout is performed both after the model is first loaded as well as after any part is added or removed
 * or changes visibility or size.
 * You can disable the initial layout by setting Layout#isInitial to false.
 * You can disable later automatic layouts by setting Layout#isOngoing to false.
 * <p class="boxread">
 * See the <a href="../../intro/layouts.html">Layouts</a> page in the Introduction for a summary of layout behavior.
 *
 * A diagram maintains a collection of selected parts, the Diagram#selection.
 * To select a Part you set its Part#isSelected property to true.
 *
 * There are many properties, named "allow...", that control what operations the user
 * may perform on the parts in the diagram.  These correspond to the same named
 * properties on Layer that govern the behavior for those parts in a particular layer.
 * Furthermore for some of these properties there are corresponding properties on
 * Part, named "...able", that govern the behavior for that individual part.
 * For example, the #allowCopy property corresponds to Layer#allowCopy and
 * to the property Part#copyable.
 * The Part#canCopy predicate is false if any of these properties is false.
 * <p class="boxread">
 * See the <a href="../../intro/permissions.html">Permissions</a> page for a more thorough discussion.
 *
 * The #commandHandler implements various standard commands,
 * such as the CommandHandler#deleteSelection method and the
 * CommandHandler#canDeleteSelection predicate.
 * <p class="boxread">
 * See the <a href="../../intro/commands.html">Commands</a> page for a listing of keyboard commands and
 * the use of commands in general.
 *
 * The diagram supports modular behavior for mouse events by implementing "tools".
 * All mouse and keyboard events are represented by InputEvents and redirected
 * to the #currentTool.
 * The default tool is an instance of ToolManager which keeps three lists of mode-less tools:
 * ToolManager#mouseDownTools, ToolManager#mouseMoveTools, and ToolManager#mouseUpTools.
 * The ToolManager searches these lists when a mouse event happens to find the first tool that can run.
 * It then makes that tool the new #currentTool, where it can continue to process input events.
 * When the tool is done, it stops itself, causing the #defaultTool to be the new #currentTool.
 *
 * Mouse-down tools include:
 *   - ToolManager#actionTool, to support objects like "buttons"
 *   - ToolManager#relinkingTool, to reconnect an existing link
 *   - ToolManager#linkReshapingTool, to modify the route of an existing link
 *   - ToolManager#rotatingTool, to change the angle of an object
 *   - ToolManager#resizingTool, to change the size of an object
 *
 * Mouse-move tools include:
 *   - ToolManager#linkingTool, to draw a new link
 *   - ToolManager#draggingTool, to move or copy the selection
 *   - ToolManager#dragSelectingTool, to select parts within a rectangular area
 *   - ToolManager#panningTool, to pan the diagram
 *
 * Mouse-up tools include:
 *   - ToolManager#contextMenuTool, to manage context menus
 *   - ToolManager#textEditingTool, to support in-place text editing
 *   - ToolManager#clickCreatingTool, to create new parts where the user clicks
 *   - ToolManager#clickSelectingTool, to select parts
 *
 * You can also run a tool in a modal fashion by explicitly setting #currentTool.
 * That tool will keep running until some code replaces the #currentTool.
 * This normally happens when the current tool calls Tool#stopTool, such as on a mouse-up event.
 * <p class="boxread">
 * See the <a href="../../intro/tools.html">Tools</a> page for a listing of predefined tools and how they operate.
 *
 * A diagram raises various DiagramEvents when interesting things happen that may have affected the whole diagram.
 * See the documentation for DiagramEvent for a complete listing.
 *
 * When you need to display multiple Models, but not at the same time, you can do so by using only one Diagram and setting the #model to a different one.
 * You can also have two Diagrams share a DIV by swapping the #div to `null` on one Diagram and setting it on the other.
 * When permanently removing a Diagram,t o clear any memory used,
 * set the #div to `null` and remove all references to the Diagram. These scenarios are discussed more on the
 * <a href="../../intro/replacingDeleting.html">Replacing Diagrams and Models</a> intro page.
 * @unrestricted
 * @category Diagram
 */
export class Diagram {
    /**
     * Construct an empty Diagram for a particular DIV HTML element.
     *
     * You will normally initialize properties of the Diagram that control its appearance and behavior.
     * These properties include:
     *   - templates (such as #nodeTemplate or #linkSelectionAdornmentTemplate),
     *   - an automatic layout (if desired, #layout),
     *   - positioning or scaling (such as #initialContentAlignment or #initialAutoScale),
     *   - permissions (such as #isReadOnly or #allowDelete), or
     *   - Tool or CommandHandler customization (such as CommandHandler#archetypeGroupData or
     *     Tool#isEnabled): `diagram.toolManager.dragSelectingTool.isEnabled = false;`.
     *
     * Then you will need to construct a Model (usually a GraphLinksModel) for the Diagram,
     * initialize its data by setting its Model#nodeDataArray and other properties,
     * and then set the diagram's #model.
     *
     * Finally, if you want to disassociate the Diagram from the HTML Div element, set Diagram#div to null.
     * If you remove a part of the HTML DOM containing a Div with a Diagram, you will need to
     * set #div to null in order for the page to recover the memory.
     *
     * It is commonplace to use the static function GraphObject.make to build a Diagram:
     * ```js
     * var $ = go.GraphObject.make;
     *
     * var diagram =
     *   $(go.Diagram, "myDiagramDiv",
     *     {
     *       allowZoom: false,
     *       "animationManager.isEnabled": false,  // turn off automatic animations
     *       "grid.visible": true,  // display a background grid for the whole diagram
     *       "grid.gridCellSize": new go.Size(20, 20),
     *       // allow double-click in background to create a new node
     *       "clickCreatingTool.archetypeNodeData": { text: "Node" },
     *       // allow Ctrl-G to call the groupSelection command
     *       "commandHandler.archetypeGroupData":
     *         { text: "Group", isGroup: true, color: "blue" },
     *       "commandHandler.copiesTree": true,  // for the copy command
     *       "commandHandler.deletesTree": true, // for the delete command
     *       "toolManager.hoverDelay": 100,  // how quickly tooltips are shown
     *       // mouse wheel zooms instead of scrolls
     *       "toolManager.mouseWheelBehavior": go.ToolManager.WheelZoom,
     *       "draggingTool.dragsTree": true,  // dragging for both move and copy
     *       "draggingTool.isGridSnapEnabled": true,
     *       layout: $(go.TreeLayout,
     *                 { angle: 90, sorting: go.TreeLayout.SortingAscending }),
     *       "undoManager.isEnabled": true,  // enable undo & redo
     *       // a Changed listener on the Diagram.model
     *       "ModelChanged": function(e) { if (e.isTransactionFinished) saveModel(); }
     *     });
     * ```
     * @param {Element|string=} div A reference to a DIV HTML element or its ID as a string.
     * If no DIV is supplied one will be created in memory. The Diagram's Diagram#div property
     * can then be set later on.
     */
    constructor(div?: Element | string);
    /**
     * Undocumented
     * This static function returns true if Diagrams are using a DOM, false otherwise.
     * This can be false if there is no root <code>document</code> in the JavaScript context,
     * or if the DOM was disabled through #useDOM.
     */
    static isUsingDOM(): boolean;
    /**
     * Undocumented
     * This static function sets whether or not GoJS should use a DOM, if one exists.
     * Can be set to false to simulate a DOM-less environment.
     * @param {boolean} value
     */
    static useDOM(value: boolean): void;
    /**
     * Removes all Parts from the Diagram, including unbound Parts except for the background grid,
     * and also clears out the Model and UndoManager and clipboard.
     * This operation is not undoable.
     *
     * Alternative actions are to replace the #model with a new Model
     * (probably a GraphLinksModel or a TreeModel),
     * or to set Model#nodeDataArray with an empty JavaScript Array
     * (and GraphLinksModel#linkDataArray).
     *
     * This does not remove any listeners from the diagram.
     */
    clear(): void;
    /**
     * Undocumented
     */
    reset(): void;
    /**
     * @expose
     * Used in Diagram constructor setup, this computes the pixel width of the scrollbars
     * @param {HTMLElement=} elem
     */
    setRTL(elem?: HTMLElement): void;
    /**
     * Undocumented
     * @expose
     * @param {HTMLElement=} elem
     */
    setScrollWidth(elem?: HTMLElement): void;
    /**
     * This static function gets the Diagram that is attached to an HTML DIV element.
     * @param {Element|string} div
     * @return {Diagram} null if there is no Diagram associated with the given DIV,
     * or if the argument is not a DIV element nor a string naming such a DIV element in the HTML document.
     */
    static fromDiv(div: Element | string): Diagram | null;
    /**
     * Gets or sets the Diagram's HTMLDivElement, via an HTML Element ID.
     * This is typically set automatically when a Div is supplied as an argument to Diagram's constructor.
     *
     * Setting this property to a new value will clobber any HTML and
     * inner DOM elements inside of both the new and the old divs.
     * It will then populate the Div with the elements
     * (inner Divs, Canvases) needed for the Diagram to function.
     *
     * If you want to disassociate the Diagram from the HTML Div element, set Diagram#div to null.
     * If you remove a part of the HTML DOM containing a Div with a Diagram, you will need to
     * set #div to null in order for the page to recover the memory.
     *
     * You should not attempt to manually modify the contents of this Div.
     * Changing this property value does not raise a Changed event.
     */
    div: HTMLDivElement | null;
    /**
     * Undocumented, but may be useful for change detection calls in Angular.
     * @expose
     */
    addEventListener(DOMElement: Element | Window | Document, name: string, listener: any, capture: boolean): void;
    /**
     * Undocumented, but may be useful for change detection calls in Angular.
     * @expose
     */
    removeEventListener(DOMElement: Element | Window | Document, name: string, listener: any, capture: boolean): void;
    /**
     * Undocumented
     * @expose
     * @return {number}
     */
    computePixelRatio(): number;
    /**
     * Undocumented
     * @expose
     */
    doFocus(): void;
    /**
     * Explicitly bring HTML focus to the Diagram's canvas.
     * This is called by tools that may create other HTML elements such as TextEditingTool.
     *
     * If #scrollsPageOnFocus is false, this tries to keep the page at the same scroll position
     * that it had before calling #focus.
     * This method is not overridable.
     */
    focus(): void;
    /**
     * This is called during a Diagram update to determine a new value for #documentBounds.
     * By default this computes the union of the bounds of all the visible
     * GraphObjects in this Diagram, unless Diagram#fixedBounds is set.
     *
     * To compute the bounds of a collection of Parts, call #computePartsBounds.
     * @expose
     * @return {Rect} a Rect in document coordinates.
     */
    protected computeBounds(): Rect;
    /**
     * Find the union of the GraphObject#actualBounds of all of the Parts in the given collection,
     * excluding Links unless the second argument is true.
     *
     * Unlike #computeBounds, this ignores the visibility of each Part and does not add any padding to the result.
     * @expose
     * @param {Iterable.<Part>|Array.<Part>} coll an iterable collection or Array of Parts.
     * @param {boolean=} includeLinks defaults to false
     * @return {Rect} This returns the bounding area of the given Parts;
     * if there are no Parts in the collection, this returns a Rect with zero width and height and an X and Y that are NaN.
     * @since 1.1
     */
    computePartsBounds(coll: Iterable<Part> | Array<Part>, includeLinks?: boolean): Rect;
    /**
     * Scales the Diagram to uniformly fit into the viewport. To have this done automatically,
     * set the Diagram's #autoScale to Diagram.Uniform.
     */
    zoomToFit(): void;
    /**
     * Modifies the #scale and #position of the Diagram
     * so that the viewport displays a given document-coordinates rectangle.
     * @param {Rect} r rectangular bounds in document coordinates.
     * @param {EnumValue=} scaling an optional value of either Diagram.Uniform (the default) or Diagram.UniformToFill.
     * @since 1.1
     */
    zoomToRect(r: Rect, scaling?: EnumValue): void;
    /**
     * Aligns the Diagram's #position based on a desired document Spot and viewport Spot.
     * @param {Spot} documentspot
     * @param {Spot} viewportspot
     */
    alignDocument(documentspot: Spot, viewportspot: Spot): void;
    /**
     * This convenience function finds the front-most Part
     * that is at a given point that might be selectable and that is not in a temporary layer.
     *
     * This just calls #findObjectAt with appropriate arguments,
     * but ignoring Layers that are Layer#isTemporary.
     * @param {Point} p a Point in document coordinates.
     * @param {boolean=} selectable Whether to only consider parts that are Part#selectable. The default is true.
     * @return {Part}
     * @see #findObjectAt
     * @see #findObjectsAt
     * @see #findObjectsIn
     * @see #findObjectsNear
     * @see #findPartsAt
     * @see #findPartsIn
     * @see #findPartsNear
     */
    findPartAt(p: Point, selectable?: boolean): Part | null;
    /**
     * Find the front-most GraphObject at the given point in document coordinates.
     *
     * If Layer#visible is false, this method will not find any objects in that layer.
     * However, Layer#opacity does not affect this method.
     *
     * Example usage:
     * ```js
     * // Returns the top-most object that is a selectable Part, or null if there isn't one
     * myDiagram.findObjectAt(
     *   myDiagram.lastInput.documentPoint,
     *   // Navigation function
     *   function(x) { return x.part; },
     *   // Because of the navigation function, x will always be a Part.
     *   function(x) { return x.canSelect(); }
     * );
     * ```
     * @param {Point} p A Point in document coordinates.
     * @param {function(GraphObject):GraphObject | null=} navig A function taking a GraphObject and
     * returning a GraphObject, defaulting to the identity.
     * @param {function(GraphObject):boolean | null=} pred A function taking the GraphObject
     * returned by navig and returning true if that object should be returned,
     * defaulting to a predicate that always returns true.
     * @return {GraphObject} The first GraphObject returned by the *navig* function
     * and satisfying the *pred* function that is at the point *p*, in Z-order from front to back,
     * or else null if nothing is found.
     * @see #findObjectsAt
     * @see #findObjectsIn
     * @see #findObjectsNear
     * @see #findPartAt
     */
    findObjectAt<T extends GraphObject>(p: Point, navig?: ((a: GraphObject) => (T | null)) | null, pred?: ((a: T) => boolean) | null): (T | null);
    /**
     * This convenience function finds all Parts that are at a point in document coordinates
     * and that are not in temporary layers.
     *
     * @param {Point} p A Point in document coordinates.
     * @param {boolean=} selectable Whether to only consider parts that are Part#selectable. The default is true.
     * @param {List.<Part>|Set.<Part>=} coll An optional collection (List or Set) to add the results to.
     * @return {Iterable.<Part>}
     * @see #findPartAt
     * @see #findPartsIn
     * @see #findPartsNear
     * @see #findObjectsAt
     * @since 2.0
     */
    findPartsAt<T extends Part, S extends List<T> | Set<T> = Set<T>>(p: Point, selectable?: boolean, coll?: S): S;
    /**
     * Return a collection of the GraphObjects at the given point in document coordinates.
     *
     * If Layer#visible is false, this method will not find any objects in that layer.
     * However, Layer#opacity does not affect this method.
     *
     * Example usage:
     * ```js
     * // Returns the Nodes that are at a given point, overlapping each other
     * myDiagram.findObjectsAt(somePoint,
     *   // Navigation function -- only return Nodes
     *   function(x) { var p = x.part; return (p instanceof go.Node) ? p : null; }
     * );
     * ```
     * @param {Point} p A Point in document coordinates.
     * @param {function(GraphObject):GraphObject | null=} navig A function taking a GraphObject and
     * returning a GraphObject, defaulting to the identity.
     * If this function returns null, the given GraphObject will not be included in the results.
     * @param {function(GraphObject):boolean | null=} pred A function taking the GraphObject
     * returned by navig and returning true if that object should be returned,
     * defaulting to a predicate that always returns true.
     * @param {List.<GraphObject>|Set.<GraphObject>=} coll An optional collection (List or Set) to add the results to.
     * @return {Iterable.<GraphObject>} a collection of GraphObjects returned by the *navig* function
     * and satisfying the *pred* that are located at the point *p*, or else an empty collection.
     * If a List or Set was passed in, it is returned.
     * @see #findObjectAt
     * @see #findObjectsIn
     * @see #findObjectsNear
     * @see #findPartsAt
     */
    findObjectsAt<T extends GraphObject, S extends List<T> | Set<T> = Set<T>>(p: Point, navig?: ((a: GraphObject) => (T | null)) | null, pred?: ((a: T) => boolean) | null, coll?: S): S;
    /**
     * This convenience function finds Parts that are inside or that intersect
     * a given Rect in document coordinates.
     *
     * This just calls #findObjectsIn with appropriate arguments,
     * but ignoring Layers that are Layer#isTemporary.
     * @param {Rect} r a Rect in document coordinates.
     * @param {boolean=} partialInclusion Whether a Part can match if it merely intersects the rectangular area (true) or
     * if it must be entirely inside the rectangular area (false). The default value is false.
     * @param {boolean=} selectable Whether to only consider parts that are Part#selectable. The default is true.
     * @param {List.<Part>|Set.<Part>=} coll An optional collection (List or Set) to add the results to.
     * @return {Iterable.<Part>}
     * @see #findPartAt
     * @see #findPartsAt
     * @see #findPartsNear
     * @see #findObjectsIn
     * @since 2.0
     */
    findPartsIn<T extends Part, S extends List<T> | Set<T> = Set<T>>(r: Rect, partialInclusion?: boolean, selectable?: boolean, coll?: S): S;
    /**
     * Returns a collection of all GraphObjects that are inside or that intersect
     * a given Rect in document coordinates.
     *
     * If Layer#visible is false, this method will not find any objects in that layer.
     * However, Layer#opacity does not affect this method.
     *
     * Example usage:
     * ```js
     * // Returns the Links that intersect a given rectangle and have a certain data property
     * myDiagram.findObjectsIn(someRect,
     *   // Navigation function -- only return Links
     *   function(x) { var p = x.part; return (p instanceof go.Link) ? p : null; },
     *   // Predicate that always receives a Link, due to above navigation function
     *   function(link) { return link.data.someProp > 17; },
     *   // the links may only partly overlap the given rectangle
     *   true
     * );
     * ```
     * @param {Rect} r A Rect in document coordinates.
     * @param {function(GraphObject):GraphObject | null=} navig A function taking a GraphObject and
     * returning a GraphObject, defaulting to the identity.
     * If this function returns null, the given GraphObject will not be included in the results.
     * @param {function(GraphObject):boolean | null=} pred A function taking the GraphObject
     * returned by navig and returning true if that object should be returned,
     * defaulting to a predicate that always returns true.
     * @param {boolean=} partialInclusion Whether an object can match if it merely intersects the rectangular area (true) or
     * if it must be entirely inside the rectangular area (false). The default value is false.
     * @param {List.<GraphObject>|Set.<GraphObject>=} coll An optional collection (List or Set) to add the results to.
     * @return {Iterable.<GraphObject>} a collection of GraphObjects returned by the *navig* function
     * and satisfying the *pred* function that are within the rectangle *r*, or else an empty collection.
     * If a List or Set was passed in, it is returned.
     * @see #findObjectAt
     * @see #findObjectsAt
     * @see #findObjectsNear
     * @see #findPartsIn
     */
    findObjectsIn<T extends GraphObject, S extends List<T> | Set<T> = Set<T>>(r: Rect, navig?: ((a: GraphObject) => (T | null)) | null, pred?: ((a: T) => boolean) | null, partialInclusion?: boolean, coll?: S): S;
    /**
     * This convenience function finds Parts that are within a certain distance
     * of a given point in document coordinates.
     *
     * This just calls #findObjectsNear with appropriate arguments,
     * but ignoring Layers that are Layer#isTemporary.
     * @param {Point} p A Point in document coordinates.
     * @param {number} dist The distance from the point.
     * @param {boolean=} partialInclusion Whether an object can match if it merely intersects the circular area (true) or
     * if it must be entirely inside the circular area (false). The default is true.
     * @param {boolean=} selectable Whether to only consider parts that are Part#selectable. The default is true.
     * @param {List.<Part>|Set.<Part>=} coll An optional collection (List or Set) to add the results to.
     * @return {Iterable.<Part>}
     * @see #findPartAt
     * @see #findPartsAt
     * @see #findPartsIn
     * @see #findObjectsNear
     * @since 2.0
     */
    findPartsNear<T extends Part, S extends List<T> | Set<T> = Set<T>>(p: Point, dist: number, partialInclusion?: boolean, selectable?: boolean, coll?: S): S;
    /**
     * Returns a collection of all GraphObjects that are within a certain distance
     * of a given point in document coordinates.
     *
     * If Layer#visible is false, this method will not find any objects in that layer.
     * However, Layer#opacity does not affect this method.
     *
     * Example usage:
     * ```js
     * // Returns the Nodes that intersect a given circular area and have a certain data property
     * myDiagram.findObjectsNear(somePoint,
     *   // The circular area is centered at somePoint and has radius 100
     *   100,
     *   // Navigation function -- only return Nodes
     *   function(x) { var p = x.part; return (p instanceof go.Node) ? p : null; },
     *   // Predicate that always receives a Node, due to above navigation function
     *   function(node) { return node.data.someProp > 17; },
     *   // the nodes may only partly overlap the given circular area
     *   true
     * );
     * ```
     * @param {Point} p A Point in document coordinates.
     * @param {number} dist The distance from the point.
     * @param {function(GraphObject):GraphObject | null=} navig A function taking a GraphObject and
     * returning a GraphObject, defaulting to the identity.
     * If this function returns null, the given GraphObject will not be included in the results.
     * @param {function(GraphObject):boolean | null=} pred A function taking the GraphObject
     * returned by navig and returning true if that object should be returned,
     * defaulting to a predicate that always returns true.
     * @param {*=} partialInclusion Whether an object can match if it merely intersects the circular area (true) or
     * if it must be entirely inside the circular area (false).  The default value is true.
     * The default is true.
     * @param {List.<GraphObject>|Set.<GraphObject>=} coll An optional collection (List or Set) to add the results to.
     * @return {Iterable.<GraphObject>} a collection of GraphObjects returned by the *navig* function
     * and satisfying the *pred* that are located near the point *p*, or else an empty collection.
     * If a List or Set was passed in, it is returned.
     * @see #findObjectAt
     * @see #findObjectsAt
     * @see #findObjectsIn
     * @see #findPartsNear
     */
    findObjectsNear<T extends GraphObject, S extends List<T> | Set<T> = Set<T>>(p: Point, dist: number, navig?: ((a: GraphObject) => (T | null)) | null, pred?: ((a: T) => boolean) | null, partialInclusion?: boolean | S, coll?: S): S;
    /**
     * Undocumented
     *
     * Invalidates all non-layout diagram state and forces an immediate redraw.
     * Because this can be very inefficent, to discourage its use, it remains an undocumented part of the API.
     */
    redraw(): void;
    /**
     * Updates the diagram immediately, then resets initialization flags so that
     * actions taken in the argument function will be considered part of Diagram initialization,
     * and will participate in initial layouts, #initialAutoScale, #initialContentAlignment, etc.
     *
     * This is useful in situations where you do not wish for the first content added to the diagram to be considered
     * the "initial" content, such as with a Node that represents a "Loading" bar.
     * @param {function()|null=} func an optional function of actions to perform as part of another diagram initialization.
     * @since 1.1
     */
    delayInitialization(func?: (() => void) | null): void;
    /**
     * Usage of this method is uncommon and may affect performance,
     * for efficiency do not call this method unless you have a well-defined need.
     * Normally, GoJS updates the diagram automatically, and completeing a transaction ensures an immediate update.
     *
     * The most common reason to call this method when the HTML Div has changed size but the window has not changed size,
     * and the Diagram needs to be notified of this DOM change. See an example of resizing diagrams
     * <a href="../../intro/resizing.html">here</a>.
     *
     * Requests that in the near-future the diagram makes sure all GraphObjects are arranged,
     * recomputes the document bounds, updates the scrollbars, and redraws the viewport.
     *
     * @param {boolean=} alwaysQueueUpdate If true the Diagram will queue another update,
     * even if an update is already occurring. The default value is false.
     * Side effects in an "InitialLayoutCompleted" DiagramEvent listener might necessitate setting this parameter.
     * @since 1.6
     */
    requestUpdate(alwaysQueueUpdate?: boolean): void;
    /**
     * Undocumented
     */
    maybeUpdate(): void;
    /**
     * Undocumented
     * @param {string} name
     * @return {*}
     */
    getRenderingHint(name: string): any;
    /**
     * Undocumented
     * @param {string} name
     * @param {*} val
     */
    setRenderingHint(name: string, val: any): void;
    /**
     * Undocumented
     * @param {string} name
     * @return {*}
     */
    getInputOption(name: string): any;
    /**
     * Undocumented
     * @param {string} name
     * @param {*} val
     */
    setInputOption(name: string, val: any): void;
    /**
     * This method sets a collection of properties according to the property/value pairs that have been set on the given Object,
     * in the same manner as GraphObject.make does when constructing a Diagram with an argument that is a simple JavaScript Object.
     *
     * You can set properties on an object that is the value of a property of the Diagram,
     * or on the Diagram#toolManager, by using a *subpropname.property* syntax for the property name.
     * At the current time only a single dot is permitted in the property "name".
     *
     * The property name may also be the name of a DiagramEvent, in which case this calls #addDiagramListener with that DiagramEvent name.
     * ```js
     * aDiagram.setProperties({
     *   allowDelete: false,
     *   // display a background grid for the whole diagram
     *   "grid.visible": true,
     *   "grid.gridCellSize": new go.Size(20, 20),
     *   "animationManager.isEnabled": false,  // turn off automatic animations
     *   // specify a group membership validation predicate
     *   "commandHandler.memberValidation": function(group, part) { return ...; },
     *   "commandHandler.copiesTree": true,  // for the copy command
     *   // mouse wheel zooms instead of scrolls
     *   "toolManager.mouseWheelBehavior": go.ToolManager.WheelZoom,
     *   "draggingTool.dragsTree": true,  // dragging for both move and copy
     *   "draggingTool.isGridSnapEnabled": true,
     *   layout: $(go.TreeLayout),
     *   // add a DiagramEvent listener
     *   "ExternalObjectsDropped": function(e) { e.subject.each(function(part) { ... }); }
     * });
     * ```
     * @param {Object} props a plain JavaScript object with various property values to be set on this Diagram or on a part of this Diagram.
     * @since 1.5
     */
    setProperties(props: ObjectData): void;
    /**
     * Adds a Part to the Layer that matches the Part's Part#layerName,
     * or else the default layer, which is named with the empty string.
     *
     * Normally parts added to a diagram are top-level parts.
     * If you want nodes to be members of a Group, in addition to calling this method
     * call Group#addMembers or set each Part#containingGroup.
     * @param {Part} part
     * @see #remove
     */
    add(part: Part): void;
    /**
     * Removes a Part from its Layer, provided the Layer is in this Diagram.
     * Removing a Node will also remove any Links that are connected with it.
     * Removing a Group will also remove all of its members.
     * Removing a Link will also remove all of its label Nodes, if it has any.
     * @param {Part} part
     * @see #add
     */
    remove(part: Part): void;
    /**
     * This method removes from this Diagram all of the Parts in a collection.
     * Removing a Node will also remove any Links that are connected with it.
     * Removing a Group will also remove all of its members.
     * Removing a Link will also remove all of its label Nodes, if it has any.
     *
     * This does not perform a transaction nor does it raise a DiagramEvent.
     * Call CommandHandler#deleteSelection, which calls this method,
     * if you want to delete all selected Parts.
     * The CommandHandler#deleteSelection command may delete other Parts as well, depending on CommandHandler#deletesTree.
     *
     * At this time there is no "addParts" method -- just call Diagram#add on each Part.
     * @param {Iterable.<Part>|Array.<Part>} coll A List or Set or Iterator or Array of Parts.
     * @param {boolean} check Whether to check Part#canDelete on each part.
     * @since 1.3
     */
    removeParts(coll: Iterable<Part> | Array<Part>, check: boolean): void;
    /**
     * Make a copy of a collection of Parts and return them in a Map mapping each original Part to its copy.
     * It may optionally add them to a given Diagram.
     * Copying a Group will also copy its member Nodes and Links.
     * Copying a Link will also copy any label Nodes that it owns.
     *
     * This does not perform a transaction nor does it raise a DiagramEvent.
     * Call CommandHandler#copySelection, which calls this method,
     * if you want to copy all selected Parts into the clipboard.
     * The CommandHandler#copySelection command may also copy additional Parts as well, depending on CommandHandler#copiesTree.
     * @param {Iterable.<Part>|Array.<Part>} coll A List or a Set or Iterator of Parts, or an Array of Parts.
     * @param {Diagram} diagram The destination diagram; if null, the copied parts are not added to this diagram.
     * @param {boolean} check Whether to check Part#canCopy on each part.
     * @return {Map.<Part,Part>}
     * @since 1.3
     */
    copyParts(coll: Iterable<Part> | Array<Part>, diagram: Diagram | null, check: boolean): Map<Part, Part>;
    /**
     * Move a collection of Parts in this Diagram by a given offset.
     * Moving a Group will also move its member Nodes and Links.
     * Moving with a zero X and a zero Y offset is potentially useful in order to snap Parts to the grid if DraggingTool#isGridSnapEnabled is true.
     *
     * This does not perform a transaction nor does it raise a DiagramEvent.
     *
     * @param {Iterable.<Part>|Array.<Part>} coll A List or a Set or Iterator of Parts, or an Array of Parts,
     * or null to move all of the Parts in this Diagram.
     * @param {Point} offset the amount to move each Part, in document coordinates.
     * @param {boolean} check Whether to check Part#canMove on each part.
     * @param {DraggingOptions=} dragOptions Optional dragging options. By default this uses the settings from the Diagram's DraggingTool.
     * @since 1.3
     */
    moveParts(coll: Iterable<Part> | Array<Part>, offset: Point, check: boolean, dragOptions?: DraggingOptions): void;
    /**
     * You do not need to call or orverride this method, but you may want to override DraggingTool#computeMove, which calls this method.
     *
     * This method computes the new location for a Node or simple Part,
     * given a new desired location and an optional Map of dragged parts,
     * taking any grid-snapping into consideration,
     * any Part#dragComputation function,
     * and any Part#minLocation and Part#maxLocation.
     *
     * @expose
     * @param {Part} n the Node or simple Part that is being moved
     * @param {Point} newloc the proposed new location
     * @param {DraggingOptions} dragOptions the proposed new location
     * @param {Point=} result an optional Point that is modified and returned
     * @return {Point} the possibly grid-snapped computed Point that is within the minimum and maximum permitted locations
     * @since 2.0
     */
    computeMove(n: Part, newloc: Point, dragOptions: DraggingOptions, result?: Point): Point;
    /**
     * Adds a new Layer to the list of layers.
     * If Layer#isTemporary is false, the layer is added after all existing non-temporary layers.
     * If Layer#isTemporary is true, the layer is added as the very last layer.
     * @param {Layer} layer The new Layer to add.  It is an error if the Layer already belongs to a Diagram.
     * @see #addLayerBefore
     * @see #addLayerAfter
     * @see #removeLayer
     * @see #findLayer
     */
    addLayer(layer: Layer): void;
    /**
     * Adds a layer to the list of layers before a specified layer.
     * This method can also re-order layers.
     * @param {Layer} layer the new Layer to add or existing Layer to move in Z-order.
     * @param {Layer} existingLayer the other Layer in this Diagram which should come just after the new or moved layer.
     * @see #addLayer
     * @see #addLayerAfter
     * @see #removeLayer
     */
    addLayerBefore(layer: Layer, existingLayer: Layer): void;
    /**
     * Adds a layer to the list of layers after a specified layer.
     * This method can also re-order layers.
     * @param {Layer} layer the new Layer to add or existing Layer to move in Z-order.
     * @param {Layer} existingLayer the other Layer in this Diagram which should come just before the new or moved layer.
     * @see #addLayer
     * @see #addLayerBefore
     * @see #removeLayer
     */
    addLayerAfter(layer: Layer, existingLayer: Layer): void;
    /**
     * Removes the given layer from the list of layers.
     *
     * Removing a layer does not remove the Parts in the layer. Instead, those Parts are placed into the default layer.
     * To remove all Parts in a layer you can call Diagram#removeParts with Layer#parts as the argument.
     *
     * You cannot remove the default layer, the one named with the empty string.
     * @param {Layer} layer
     * @see #addLayer
     * @see #addLayerBefore
     * @see #addLayerAfter
     * @see #findLayer
     */
    removeLayer(layer: Layer): void;
    /**
     * Finds a layer with a given name.
     * @param {string} name
     * @return {Layer} a Layer with the given name, or null if no such layer was found.
     * @see #addLayerBefore
     * @see #addLayerAfter
     * @see #removeLayer
     */
    findLayer(name: string): Layer | null;
    /**
     * Register an event handler on this Diagram's Diagram#model that is called when there is a ChangedEvent
     * on the Model, not in this diagram.
     * Be sure to call #removeModelChangedListener when you are done with the diagram.
     *
     * This is convenient when the Diagram#model may be replaced.
     * Using this method to register a Model Changed listener is more convenient than calling Model#addChangedListener directly
     * because when this diagram's Model is replaced, one does not need to call Model#removeChangedListener on the old Model
     * and then Model#addChangedListener again on the new Model.
     *
     * You can establish Model Changed listeners when you create a Diagram with GraphObject.make.  For example:
     * ```js
     * $(go.Diagram, . . .,
     *    {
     *        "ModelChanged": function(e) { if (e.isTransactionFinished) saveModel(); }
     *        . . .
     *    })
     * ```
     *
     * Do not add or remove Changed listeners during the execution of a Changed listener.
     * @param {function(ChangedEvent)} listener a function that takes a ChangedEvent as its argument.
     * @see #removeModelChangedListener
     * @since 1.6
     */
    addModelChangedListener(listener: ((e: ChangedEvent) => void)): void;
    /**
     * Unregister a ChangedEvent handler from this Diagram's Diagram#model.
     * @param {function(ChangedEvent)} listener a function that takes a ChangedEvent as its argument.
     * @see #addModelChangedListener
     * @since 1.6
     */
    removeModelChangedListener(listener: ((e: ChangedEvent) => void)): void;
    /**
     * Register an event handler that is called when there is a ChangedEvent because this Diagram
     * or one of its Parts has changed, but not because the Model or any model data has changed.
     *
     * It is unusual to listen for Diagram ChangedEvents --
     * it is far more common to listen for specific DiagramEvents by calling #addDiagramListener,
     * or to listen for Model ChangedEvents (i.e. changes to the model) by calling #addModelChangedListener.
     *
     * Do not add or remove Changed listeners during the execution of a Changed listener.
     * @param {function(ChangedEvent)} listener a function that takes a ChangedEvent as its argument.
     * @see #removeChangedListener
     */
    addChangedListener(listener: ((e: ChangedEvent) => void)): void;
    /**
     * Unregister a ChangedEvent handler.
     * @param {function(ChangedEvent)} listener a function that takes a ChangedEvent as its argument.
     * @see #addChangedListener
     */
    removeChangedListener(listener: ((e: ChangedEvent) => void)): void;
    /**
     * This read-only property returns the AnimationManager for this Diagram.
     * @since 1.4
     */
    readonly animationManager: AnimationManager;
    /**
     * This read-only property returns the UndoManager for this Diagram, which actually belongs to the #model.
     *
     * The default UndoManager has its UndoManager#isEnabled property set to false.
     * If you want users to undo and redo, you should set that property to true once you have initialized the Diagram or its Model.
     *
     * Note that the UndoManager might be shared with other Diagrams that are showing the same Model.
     * The UndoManager might also be shared with other Models too.
     */
    readonly undoManager: UndoManager;
    /**
     * Gets or sets whether ChangedEvents are not recorded by the UndoManager.
     * The initial and normal value is false.
     * WARNING: while this property is true do not perform any changes that cause any previous transactions
     * to become impossible to undo.
     *
     * While this property is true, changing the Diagram or any GraphObject does not call UndoManager#handleChanged.
     * Even when this property is true,
     * transactions (such as calls to #startTransaction) and
     * undo/redo (such as calls to CommandHandler#undo)
     * are still delegated to the #undoManager.
     *
     * You should set this to true only temporarily, and you should remember its previous value
     * before setting this to true.
     * When finishing the period for which you want the UndoManager to be disabled,
     * you should set this back to the remembered value it had before it was set to true.
     *
     * For more permanent disabling of the UndoManager, set UndoManager#isEnabled to false.
     *
     * Setting this property also sets Model#skipsUndoManager to the same value.
     * Setting this property does not notify about any changed event.
     */
    skipsUndoManager: boolean;
    /**
     * Undocumented
     */
    delaysLayout: boolean;
    /**
     * Begin a transaction, where the changes are held by a Transaction object
     * in the UndoManager.
     * This just calls UndoManager#startTransaction.
     * @param {string=} tname a descriptive name for the transaction.
     * @return {boolean} the value returned by UndoManager#startTransaction.
     * @see #commit
     */
    startTransaction(tname?: string): boolean;
    /**
     * Commit the changes of the current transaction.
     * This just calls UndoManager#commitTransaction.
     * @param {string=} tname a descriptive name for the transaction.
     * @return {boolean} the value returned by UndoManager#commitTransaction.
     */
    commitTransaction(tname?: string): boolean;
    /**
     * Rollback the current transaction, undoing any recorded changes.
     * This just calls UndoManager#rollbackTransaction.
     * @return {boolean} the value returned by UndoManager#rollbackTransaction.
     */
    rollbackTransaction(): boolean;
    /**
     * Starts a new transaction, calls the provided function, and commits the transaction.
     * Code is called within a try-finally statement.
     * If the function does not return normally, this rolls back the transaction rather than committing it.
     * Example usage:
     * ```js
     * myDiagram.commit(d => d.remove(somePart), "Remove Part");
     * ```
     * @param {Function} func the function to call as the transaction body
     * @param {(string|null)=} tname a descriptive name for the transaction, or null to temporarily set #skipsUndoManager to true;
     *        if no string transaction name is given, an empty string is used as the transaction name
     * @since 1.8
     */
    commit(func: (d: Diagram) => void, tname?: string | null): void;
    /**
     * Update all of the data-bound properties of Nodes and Links in this diagram,
     * without having to call Model#setDataProperty.
     * This copies/converts model data properties to set properties on Parts.
     * This method does not conduct a transaction, so you need to start and commit one yourself.
     *
     * It is better to call Model#setDataProperty to modify data properties,
     * because that will both record changes for undo/redo and will update all bindings
     * that make depend on that property.
     * Simply modifying the data and calling an "update..." method will not be able to record
     * the previous value(s) of properties in the model data to support undo.
     *
     * If you know which model data objects have been modified,
     * it will be more efficient to update only the Parts that need it
     * by calling Panel#updateTargetBindings.
     *
     * To update relationships between nodes, call #updateAllRelationshipsFromData.
     * @param {string=} srcprop An optional source data property name:
     *   when provided, only evaluates those Bindings that use that particular property;
     *   when not provided or when it is the empty string, all bindings are evaluated.
     * @see #updateAllRelationshipsFromData
     */
    updateAllTargetBindings(srcprop?: string): void;
    /**
     * Add or remove any nodes or links according to additional or missing data objects in the model
     * and update all of the references to nodes, in case they had been modified in the model without
     * properly notifying the model by calling Model#addNodeData or
     * GraphLinksModel#removeLinkData or
     * GraphLinksModel#setGroupKeyForNodeData or
     * GraphLinksModel#setToKeyForLinkData or other similar methods.
     * This method does not conduct a transaction, so you need to start and commit one yourself.
     *
     * It is better to call Model#addNodeData, Model#removeNodeData,
     * GraphLinksModel#addLinkData, GraphLinksModel#removeLinkData,
     * Model#setDataProperty, and other model methods to add/remove/modify data,
     * because those methods will both record changes for undo/redo and will update all bindings
     * that make depend on that property.
     * Simply modifying the data and calling an "update..." method will not be able to record
     * the previous value(s) of properties in the model data to support undo.
     *
     * This only adds, removes, or updates the relationships between nodes and links,
     * to have them reflect what is now declared in the model data.
     * If you know which model data objects have been modified,
     * it will be more efficient to update only the Parts that need it
     * by calling Part#updateRelationshipsFromData.
     *
     * To update GraphObject properties that are data bound, call #updateAllTargetBindings.
     * @see #updateAllTargetBindings
     * @since 1.5
     */
    updateAllRelationshipsFromData(): void;
    /**
     * Deselect all selected Parts.
     * This removes all parts from the #selection collection.
     * This method raises the "ChangingSelection" and "ChangedSelection" Diagram events.
     * @see #select
     * @see #selectCollection
     */
    clearSelection(): void;
    /**
     * Make the given object the only selected object.
     * This method raises the "ChangingSelection" and "ChangedSelection" Diagram events.
     * @param {Part} part a Part that is already in a layer of this Diagram.
     * If the value is null, this does nothing.
     * @see #selectCollection
     * @see #clearSelection
     */
    select(part: Part | null): void;
    /**
     * Select all of the Parts supplied in the given collection, and deselect all other Parts.
     * This method raises the "ChangingSelection" and "ChangedSelection" Diagram events.
     * @param {Iterable.<Part>|Array.<Part>} coll a List or Set or Iterator or Array, of Parts to be selected.
     * @see #select
     * @see #clearSelection
     */
    selectCollection(coll: Iterable<Part> | Array<Part>): void;
    /**
     * Remove highlights from all Parts.
     * This removes all parts from the #highlighteds collection.
     * @see #highlight
     * @see #highlightCollection
     * @see Part#isHighlighted
     * @since 1.4
     */
    clearHighlighteds(): void;
    /**
     * Make the given part the only highlighted part.
     * @param {Part} part a Part that is already in a layer of this Diagram.
     * If the value is null, this does nothing.
     * @see Part#isHighlighted
     * @see #highlightCollection
     * @see #clearHighlighteds
     * @since 1.4
     */
    highlight(part: Part | null): void;
    /**
     * Highlight all of the Parts supplied in the given collection, and clear all other highlighted Parts.
     * @param {Iterable.<Part>|Array.<Part>} coll a List or Set or Iterator or Array, of Parts to be highlighted.
     * @see Part#isHighlighted
     * @see #highlight
     * @see #clearHighlighteds
     * @since 1.4
     */
    highlightCollection(coll: Iterable<Part> | Array<Part>): void;
    /**
     * Scrolling function used by primarily by #commandHandler's CommandHandler#doKeyDown.
     * @param {string} unit A string representing the unit of the scroll operation. Can only be 'pixel', 'line', 'page', or 'document'.
     * @param {string} dir The direction of the scroll operation. Can only be 'up', 'down', 'left', or 'right'.
     * @param {number=} dist An optional distance multiplier, for multiple pixels, lines, or pages. The default value is 1.
     *                       This argument is ignored when the unit is 'document'.
     * @see #scrollToRect
     * @see #centerRect
     */
    scroll(unit: ('pixel' | 'line' | 'page' | 'document'), dir: ('up' | 'down' | 'left' | 'right'), dist?: number): void;
    /**
     * Modifies the #position to show a given Rect of the Diagram by centering the
     * viewport on that Rect. Does nothing if the Rect is already in view.
     *
     * See also #centerRect
     * @param {Rect} r
     * @see #centerRect
     * @see #scroll
     */
    scrollToRect(r: Rect): void;
    /**
     * Modifies the #position to show a given Rect of the Diagram by centering the
     * viewport on that Rect.
     * @param {Rect} r
     * @see #scrollToRect
     * @see #scroll
     */
    centerRect(r: Rect): void;
    /**
     * Given a Point in document coorindates, return a new Point in viewport coordinates.
     * @param {Point} p
     * @return {Point} The given Point converted into View coordinates.
     * @see #transformViewToDoc
     * @see GraphObject#getDocumentPoint
     */
    transformDocToView(p: Point): Point;
    /**
     * Given a point in viewport coordinates, return a new Point in document coordinates.
     * @param {Point} p
     * @return {Point} The given point converted into Document coordinates.
     * @see #transformDocToView
     * @see GraphObject#getDocumentPoint
     */
    transformViewToDoc(p: Point): Point;
    /**
     * The default autoScale type, used as the value of Diagram#autoScale:
     * The Diagram does not attempt to scale so that its documentBounds would fit the view.
     * @constant
     */
    static None: EnumValue;
    /**
     * Diagrams with this autoScale type, used as the value of Diagram#autoScale,
     * are scaled uniformly until the whole documentBounds fits in the view.
     * @constant
     */
    static Uniform: EnumValue;
    /**
     * Diagrams with this autoScale type, used as the value of Diagram#autoScale,
     * are scaled until the documentBounds fits in the view in one direction while
     * a scrollbar is still needed in the other direction.
     * @constant
     */
    static UniformToFill: EnumValue;
    /**
     * This value for Diagram#validCycle states that
     * there are no restrictions on making cycles of links.
     * @constant
     */
    static CycleAll: EnumValue;
    /**
     * This value for Diagram#validCycle states that
     * a valid link from a node will not produce a directed cycle in the graph.
     * @constant
     */
    static CycleNotDirected: EnumValue;
    /**
     * This value for Diagram#validCycle states that
     * a valid link from a node will not produce an undirected cycle in the graph.
     * @constant
     */
    static CycleNotUndirected: EnumValue;
    /**
     * This value for Diagram#validCycle states that
     * any number of destination links may go out of a node, but at most one
     * source link may come into a node, and there are no directed cycles.
     *
     * This value assumes that the graph does not already have any cycles in it,
     * or else the behavior may be indeterminate.
     * @constant
     */
    static CycleDestinationTree: EnumValue;
    /**
     * This value for Diagram#validCycle states that
     * any number of source links may come into a node, but at most one
     * destination link may go out of a node, and there are no directed cycles.
     *
     * This value assumes that the graph does not already have any cycles in it,
     * or else the behavior may be indeterminate.
     * @constant
     */
    static CycleSourceTree: EnumValue;
    /**
     * This value for Diagram#scrollMode states that
     * the viewport constrains scrolling to the Diagram document bounds.
     * @constant
     */
    static DocumentScroll: EnumValue;
    /**
     * This value for Diagram#scrollMode states that
     * the viewport does not constrain scrolling to the Diagram document bounds.
     * @constant
     */
    static InfiniteScroll: EnumValue;
    /**
     * Gets or sets what kinds of graphs this diagram allows the user to draw.
     * By default this property is Diagram.CycleAll -- all kinds of cycles are permitted.
     * Common values include Diagram.CycleDestinationTree and Diagram.CycleNotDirected.
     */
    validCycle: EnumValue;
    /**
     * This read-only property returns an iterator for this Diagram's Layers.
     * @see #addLayer
     * @see #removeLayer
     */
    readonly layers: Iterator<Layer>;
    /**
     * Gets or sets whether the Diagram's Diagram#model is Model#isReadOnly.
     * @see #isReadOnly
     * @see #isEnabled
     */
    isModelReadOnly: boolean;
    /**
     * Gets or sets whether the Diagram may be modified by the user,
     * while still allowing the user to scroll, zoom, and select.
     * The initial value is false.
     * @see #isModelReadOnly
     * @see #isEnabled
     */
    isReadOnly: boolean;
    /**
     * Gets or sets whether the user may interact with the Diagram.
     * @see #isReadOnly
     * @see #isModelReadOnly
     */
    isEnabled: boolean;
    /**
     * Gets or sets whether the user may copy to or paste parts from the internal clipboard.
     * This allows use of CommandHandler#cutSelection,
     * CommandHandler#copySelection and CommandHandler#pasteSelection.
     * The initial value is true.
     */
    allowClipboard: boolean;
    /**
     * Gets or sets whether the user may copy objects.
     * The initial value is true.
     */
    allowCopy: boolean;
    /**
     * Gets or sets whether the user may delete objects from the Diagram.
     * The initial value is true.
     */
    allowDelete: boolean;
    /**
     * Gets or sets whether the user may start a drag-and-drop in this Diagram,
     * possibly dropping in a different element.
     * The initial value is false.
     */
    allowDragOut: boolean;
    /**
     * Gets or sets whether the user may end a drag-and-drop operation in this Diagram.
     * This is typically set to true when a Diagram is used with a Palette.
     *
     * The initial value is true.
     */
    allowDrop: boolean;
    /**
     * Gets or sets whether the user may do in-place text editing.
     * The initial value is true.
     */
    allowTextEdit: boolean;
    /**
     * Gets or sets whether the user may group parts together.
     * The initial value is true.
     */
    allowGroup: boolean;
    /**
     * Gets or sets whether the user may ungroup existing groups.
     * The initial value is true.
     */
    allowUngroup: boolean;
    /**
     * Gets or sets whether the user may add parts to the Diagram.
     * The initial value is true.
     */
    allowInsert: boolean;
    /**
     * Gets or sets whether the user may draw new links.
     * The initial value is true.
     */
    allowLink: boolean;
    /**
     * Gets or sets whether the user may reconnect existing links.
     * The initial value is true.
     */
    allowRelink: boolean;
    /**
     * Gets or sets whether the user may move objects.
     * The initial value is true.
     */
    allowMove: boolean;
    /**
     * Gets or sets whether the user may reshape parts.
     * The initial value is true.
     */
    allowReshape: boolean;
    /**
     * Gets or sets whether the user may resize parts.
     * The initial value is true.
     */
    allowResize: boolean;
    /**
     * Gets or sets whether the user may rotate parts.
     * The initial value is true.
     */
    allowRotate: boolean;
    /**
     * Gets or sets whether the user may select objects.
     * The initial value is true.
     */
    allowSelect: boolean;
    /**
     * Gets or sets whether the user may undo or redo any changes.
     * The initial value is true.
     */
    allowUndo: boolean;
    /**
     * Gets or sets whether the user may zoom into or out of the Diagram.
     * The initial value is true.
     */
    allowZoom: boolean;
    /**
     * Gets or sets whether the Diagram has a vertical Scrollbar.
     *
     * To enable or disable scrolling itself, use #allowHorizontalScroll and #allowVerticalScroll.
     *
     * Adding or removing a scrollbar modifies the diagram's viewport.
     *
     * The initial value is true.
     * @see #allowVerticalScroll
     * @see #hasHorizontalScrollbar
     */
    hasVerticalScrollbar: boolean;
    /**
     * Gets or sets whether the Diagram has a horizontal Scrollbar.
     *
     * To enable or disable scrolling itself, use #allowHorizontalScroll and #allowVerticalScroll.
     *
     * Adding or removing a scrollbar modifies the diagram's viewport.
     *
     * The initial value is true.
     * @see #allowHorizontalScroll
     * @see #hasVerticalScrollbar
     */
    hasHorizontalScrollbar: boolean;
    /**
     * Gets or sets whether the user is allowed to use the horizontal scrollbar.
     * The initial value is true.
     * @see #hasHorizontalScrollbar
     * @see #allowVerticalScroll
     */
    allowHorizontalScroll: boolean;
    /**
     * Gets or sets whether the user is allowed to use the vertical scrollbar.
     * The initial value is true.
     * @see #hasVerticalScrollbar
     * @see #allowHorizontalScroll
     */
    allowVerticalScroll: boolean;
    /**
     * Gets or sets the distance in screen pixels that the horizontal scrollbar will scroll
     * when scrolling by a line.
     *
     * The default value is 16.
     * @see #scrollVerticalLineChange
     */
    scrollHorizontalLineChange: number;
    /**
     * Gets or sets the distance in screen pixels that the vertical scrollbar will scroll
     * when scrolling by a line.
     *
     * The default value is 16.
     * @see #scrollHorizontalLineChange
     */
    scrollVerticalLineChange: number;
    /**
     * Gets or sets the last InputEvent that occurred.
     *
     * This property is useful in tools and real-time operations for determining
     * where the mouse pointer was most recently located.
     *
     * Setting this property does not notify about any changed event.
     * @see #firstInput
     */
    lastInput: InputEvent;
    /**
     * Gets or sets the most recent mouse-down InputEvent that occurred.
     *
     * Setting this property does not notify about any changed event.
     * @see #lastInput
     */
    firstInput: InputEvent;
    /**
     * Gets or sets the current cursor for the Diagram, overriding the #defaultCursor.
     *
     * Valid CSS cursors are accepted,
     * such as "auto", "default", "none", "context-menu", "help", "pointer", "progress", "wait", etc.
     *
     * It is possible to use custom cursors with the syntax "url(path_to_image), default".
     * A fallback (like default here) is necessary for a custom cursor to work.
     *
     * To read more about cursor syntax, go to:
     * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/cursor">CSS cursors (mozilla.org)</a>.
     *
     * If the specified cursor is not accepted by the platform, GoJS will append
     * `-webkit-` or `-moz-` prefixes.
     *
     * Setting this property does not notify about any changed event.
     * Setting this value to the empty string ('') returns the Diagram's cursor to the #defaultCursor.
     * @see #defaultCursor
     * @see GraphObject#cursor
     */
    currentCursor: string;
    /**
     * Gets or sets the cursor to be used for the Diagram
     * when no GraphObject specifies a different cursor.
     *
     * Valid CSS cursors are accepted,
     * such as "auto", "default", "none", "context-menu", "help", "pointer", "progress", "wait", etc.
     *
     * It is possible to use custom cursors with the syntax "url(path_to_image), default".
     * A fallback (like default here) is necessary for a custom cursor to work.
     *
     * To read more about cursor syntax, go to:
     * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/cursor">CSS cursors (mozilla.org)</a>.
     * The default value is "auto".
     * @see #currentCursor
     * @see GraphObject#cursor
     */
    defaultCursor: string;
    /**
     * Gets or sets the function to execute when the user single-primary-clicks
     * on the background of the Diagram.
     * This typically involves a mouse-down followed by a prompt mouse-up
     * at approximately the same position using the left (primary) mouse button.
     * This property is used by the ClickSelectingTool
     * when the user clicks on no object.
     * The function is called in addition to the DiagramEvent
     * that is raised with the name "BackgroundSingleClicked".
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call #startTransaction and #commitTransaction.
     * @see #doubleClick
     * @see #contextClick
     * @see GraphObject#click
     */
    click: ((e: InputEvent) => void) | null;
    /**
     * Gets or sets the function to execute when the user double-primary-clicks
     * on the background of the Diagram.
     * This typically involves a mouse-down/up/down/up in rapid succession
     * at approximately the same position using the left (primary) mouse button.
     * This property is used by the ClickSelectingTool
     * when the user clicks on no object.
     * The function is called in addition to the DiagramEvent
     * that is raised with the name "BackgroundDoubleClicked".
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call #startTransaction and #commitTransaction.
     * @see #click
     * @see #contextClick
     * @see GraphObject#doubleClick
     */
    doubleClick: ((e: InputEvent) => void) | null;
    /**
     * Gets or sets the function to execute when the user single-secondary-clicks
     * on the background of the Diagram.
     * This typically involves a mouse-down followed by a prompt mouse-up
     * at approximately the same position using the right (secondary) mouse button.
     * This property is used by the ClickSelectingTool
     * when the user clicks on no object.
     * The function is called in addition to the DiagramEvent
     * that is raised with the name "BackgroundContextClicked".
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call #startTransaction and #commitTransaction.
     * @see #click
     * @see #doubleClick
     * @see GraphObject#contextClick
     */
    contextClick: ((e: InputEvent) => void) | null;
    /**
     * Gets or sets the function to execute when the user moves the mouse in
     * the background of the Diagram without holding down any buttons,
     * not over any GraphObjects.
     * This property is used by the ToolManager.
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * This function is called with Diagram#skipsUndoManager temporarily set to true,
     * so that any changes to GraphObjects are not recorded in the UndoManager.
     * You do not need to start and commit any transaction in this function.
     * After calling this function the diagram will be updated immediately.
     * @see #mouseHover
     * @see GraphObject#mouseOver
     */
    mouseOver: ((e: InputEvent) => void) | null;
    /**
     * Gets or sets the function to execute when the user holds the mouse stationary in
     * the background of the Diagram without holding down any buttons,
     * not over any GraphObjects.
     * This property is used by the ToolManager.
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call #startTransaction and #commitTransaction.
     * @see #mouseOver
     * @see GraphObject#mouseHover
     * @see ToolManager#doMouseHover
     */
    mouseHover: ((e: InputEvent) => void) | null;
    /**
     * Gets or sets the function to execute when the user holds the mouse stationary in
     * the background of the Diagram while holding down a button,
     * not over any GraphObjects.
     * This property is used by the ToolManager.
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call #startTransaction and #commitTransaction.
     * @see GraphObject#mouseHold
     * @seev ToolManager#doMouseHover
     */
    mouseHold: ((e: InputEvent) => void) | null;
    /**
     * Gets or sets the function to execute when the user is dragging the selection in
     * the background of the Diagram during a DraggingTool drag-and-drop,
     * not over any GraphObjects.
     *
     * If this property value is a function, it is called with an InputEvent.
     * It is called within the transaction performed by the DraggingTool.
     * By default this property is null.
     *
     * Note that for a drag-and-drop that originates in a different diagram, the target diagram's
     * selection collection will not be the parts that are being dragged.
     * Instead the temporary parts being dragged can be found as the source diagram's DraggingTool#copiedParts.
     *
     * This function is called with Diagram#skipsUndoManager temporarily set to true,
     * so that any changes to GraphObjects are not recorded in the UndoManager.
     * You do not need to start and commit any transaction in this function.
     * After calling this function the diagram will be updated immediately.
     *
     * For example, if you want to prevent the user from dropping Parts into the background of the diagram,
     * and want to provide feedback about that during a drag:
     * ```js
     *   myDiagram.mouseDragOver = function(e) {
     *     myDiagram.currentCursor = "no-drop";
     *   }
     * ```
     * @see #mouseDrop
     * @see GraphObject#mouseDragEnter
     * @see GraphObject#mouseDragLeave
     */
    mouseDragOver: ((e: InputEvent) => void) | null;
    /**
     * Gets or sets the function to execute when the user drops the selection in
     * the background of the Diagram at the end of a DraggingTool drag-and-drop,
     * not onto any GraphObjects.
     *
     * If this property value is a function, it is called with an InputEvent.
     * It is called within the transaction performed by the DraggingTool.
     * By default this property is null.
     *
     * For example, if you want to prevent the user from dropping Parts into the background of the diagram:
     * ```js
     *   myDiagram.mouseDrop = function(e) {
     *     myDiagram.currentTool.doCancel();
     *   }
     * ```
     * @see #mouseDragOver
     * @see GraphObject#mouseDrop
     */
    mouseDrop: ((e: InputEvent) => void) | null;
    /**
     * Gets or sets whether drag-and-drop events may be bubbled up to the diagram if not handled by a part.
     * The default value is false -- each Node or Link that in the diagram needs to define
     * its own GraphObject#mouseDragEnter, GraphObject#mouseDragLeave, and GraphObject#mouseDrop
     * event handlers if you want dragging/dropping on a part to act as if the user were acting on the diagram.
     *
     * If set to true, this will call Diagram#mouseDragOver during a drag, even while dragging over top-level parts,
     * and Diagram#mouseDrop will be called even when dropping onto parts.
     *
     * This property will have no impact while dropping on a Group.
     * The Group's mouseDrop and Group#handlesDragDropForMembers should be set if desired.
     *
     * GraphObjects do not have a mouseDragOver property, so if this is set to true, the Diagram's mouseDragOver will
     * always be called, even when dragging over a part.
     * @see #mouseDragOver
     * @see #mouseDrop
     * @see Group#handlesDragDropForMembers
     * @see GraphObject#mouseDrop
     * @since 2.0
     */
    handlesDragDropForTopLevelParts: boolean;
    /**
     * Gets or sets the function to execute when the mouse enters the Diagram.
     * (When the browser's <a href="https://developer.mozilla.org/en-US/docs/Web/Events/mouseenter">mouseEnter</a> event fires on the Diagram canvas.)
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call #startTransaction and #commitTransaction.
     * @see #mouseLeave
     * @see GraphObject#mouseEnter
     * @since 2.0
     */
    mouseEnter: ((e: InputEvent) => void) | null;
    /**
     * Gets or sets the function to execute when the mouse leaves the Diagram.
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call #startTransaction and #commitTransaction.
     * @see #mouseEnter
     * @see GraphObject#mouseLeave
     * @since 2.0
     */
    mouseLeave: ((e: InputEvent) => void) | null;
    /**
     * This Adornment or HTMLInfo is shown when the mouse stays motionless in the background.
     * The default value is null, which means no tooltip is shown.
     *
     * Here is a simple example:
     * ```js
     *  diagram.toolTip =
     *    $(go.Adornment, "Auto",
     *      $(go.Shape, { fill: "#CCFFCC" }),
     *      $(go.TextBlock, { margin: 4 },
     *        "This diagram lets you control the world.")
     *    );
     * ```
     * @see GraphObject#toolTip
     * @see ToolManager#doToolTip
     */
    toolTip: Adornment | HTMLInfo | null;
    /**
     * This Adornment or HTMLInfo is shown when the use context clicks in the background.
     * The default value is null, which means no context menu is shown.
     * On touch devices, a special default context menu will appear even there is no context menu defined.
     * See ContextMenuTool#defaultTouchContextMenu for details.
     *
     * ```js
     *  diagram.contextMenu =
     *    $("ContextMenu",
     *      $("ContextMenuButton",
     *        $(go.TextBlock, "Undo"),
     *        { click: function(e, obj) { e.diagram.commandHandler.undo(); } },
     *        new go.Binding("visible", "", function(o) {
     *            return o.diagram.commandHandler.canUndo();
     *          }).ofObject()),
     *      $("ContextMenuButton",
     *        $(go.TextBlock, "Redo"),
     *        { click: function(e, obj) { e.diagram.commandHandler.redo(); } },
     *        new go.Binding("visible", "", function(o) {
     *            return o.diagram.commandHandler.canRedo();
     *          }).ofObject())
     *    );
     * ```
     * @see GraphObject#contextMenu
     * @see ContextMenuTool
     */
    contextMenu: Adornment | HTMLInfo | null;
    /**
     * Gets or sets the CommandHandler for this Diagram.
     *
     * This is set to a new instance of CommandHandler on Diagram instantiation.
     *
     * Setting this property does not notify about any changed event.
     * The value cannot be null and must not be shared with other Diagrams.
     */
    commandHandler: CommandHandler;
    /**
     * Gets or sets the ToolManager for this Diagram.
     * This tool is used for mode-less operation.
     * It is responsible for choosing a particular tool to run as the #currentTool.
     *
     * This tool is normally also the #defaultTool.
     * If you don't want the ToolManager to run at all,
     * replace the #defaultTool with your own tool.
     *
     * Setting this property does not notify about any changed event.
     * The value cannot be null and must not be shared with other Diagrams.
     * If you set this property, you will probably also want to set #defaultTool.
     * @see #defaultTool
     */
    toolManager: ToolManager;
    /**
     * Gets or sets the default tool for this Diagram that becomes the current tool when the current tool stops.
     * Initially this value is the same tool as #toolManager, which is an instance of ToolManager.
     *
     * Setting this property also sets the #currentTool if the old default tool is the currently running tool.
     *
     * Setting this property does not notify about any changed event.
     * The value cannot be null and must not be shared with other Diagrams.
     * @see #currentTool
     * @see #toolManager
     */
    defaultTool: Tool;
    /**
     * Gets or sets the current tool for this Diagram that handles all input events.
     * This value is frequently replaced by the #toolManager as different tools run.
     *
     * Each Diagram has a number of tools that define its behavior when responding to mouse events.
     * These include ClickSelectingTool, DraggingTool, DragSelectingTool, LinkingTool, and ResizingTool, among others.
     *
     * Initially this is set to the value of #defaultTool.
     * Setting this to a null value is treated as if it were set to the #defaultTool,
     * because there should always be a currently running tool, except when the diagram is being initialized.
     *
     * A ToolManager is the default tool used by a Diagram - it chooses to run one of the other tools
     * depending on the circumstances.
     *
     * Setting this property to a new tool stops the previous current tool
     *
     * Setting this property does not notify about any changed event.
     * @see #defaultTool
     * @see #toolManager
     */
    currentTool: Tool;
    /**
     * This read-only property returns the read-only collection of selected objects.
     *
     * Do not modify this collection.
     * If you want to select or deselect a particular object in a Diagram,
     * set the Part#isSelected property.
     * If you want to deselect all objects, call #clearSelection.
     * If you want to deselect all objects and select a single object, call #select.
     *
     * You can limit how many objects the user can select by setting #maxSelectionCount.
     */
    readonly selection: Set<Part>;
    /**
     * Gets or sets the maximum number of selected objects.
     * The default value is a large positive integer.
     * Values must be non-negative.
     * Decreasing this value may cause objects to be removed from #selection
     * in order to meet the new lower limit.
     */
    maxSelectionCount: number;
    /**
     * Gets or sets the default selection Adornment template, used to adorn selected Parts other than Groups or Links.
     *
     * Each Node or simple Part can have its own Part#selectionAdornmentTemplate,
     * which if non-null will take precedence over this Diagram property.
     *
     * This Adornment must not be in the visual tree of any Diagram.
     */
    nodeSelectionAdornmentTemplate: Adornment;
    /**
     * Gets or sets the default selection Adornment template, used to adorn selected Groups.
     *
     * Each Group can have its own Part#selectionAdornmentTemplate, which if non-null will take precedence over this Diagram property.
     *
     * This Adornment must not be in the visual tree of any Diagram.
     */
    groupSelectionAdornmentTemplate: Adornment;
    /**
     * Gets or sets the default selection Adornment template, used to adorn selected Links.
     *
     * Each Link can have its own Part#selectionAdornmentTemplate, which if non-null will take precedence over this Diagram property.
     *
     * This Adornment must not be in the visual tree of any Diagram.
     */
    linkSelectionAdornmentTemplate: Adornment;
    /**
     * This read-only property returns the read-only collection of highlighted parts.
     *
     * Do not modify this collection.
     * If you want to highlight or remove the highlight for a particular Part in a Diagram,
     * set the Part#isHighlighted property.
     * If you want to remove all highlights, call #clearHighlighteds.
     * If you want to removal all highlights and highlight a single object, call #highlight.
     */
    readonly highlighteds: Set<Part>;
    /**
     * Gets or sets whether this Diagram's state has been modified.
     * Setting this property does not notify about any changed event,
     * but it does raise the "Modified" DiagramEvent,
     * although perhaps not immediately.
     *
     * Returns true if the Diagram has been changed,
     * if the #undoManager has recorded any changes, or
     * if an undo has been performed without a corresponding redo.
     *
     * Replacing the #model automatically sets this property to false after the initial layout has completed.
     * The "Modified" DiagramEvent is also raised when an undo or a redo has finished.
     * A "Modified" DiagramEvent listener must not modify this Diagram or its Model.
     */
    isModified: boolean;
    /**
     * Gets or sets the Model holding data corresponding to the
     * data-bound nodes and links of this Diagram.
     *
     * Replacing this value causes all of the bound Nodes and Links to be deleted and re-created
     * from the new model data.
     *
     * Models may be shared by multiple Diagrams. One common approach is to have two
     * Diagrams displaying the same Model but using different templates
     * (see #nodeTemplate, #nodeTemplateMap, and the associated link and group properties) and sometimes even different Layouts.
     *
     * Setting this property does not notify about any changed event; the new value must not be null.
     * Typically a new Model will have its own UndoManager, thereby replacing the Diagram's
     * current UndoManager.
     *
     * Replacing or re-setting the model will re-initialize the Diagram, taking in to account
     * #initialPosition, #initialScale, #initialAutoScale, and #initialContentAlignment.
     * It will also set #isModified to false.
     *
     * The default behavior when replacing the model is to copy a few UndoManager properties to the
     * new UndoManager, including UndoManager#isEnabled and UndoManager#maxHistoryLength.
     *
     * It is an error to replace the Diagram.model while a transaction is in progress.
     */
    model: Model;
    /**
     * Gets or sets the license key.
     * @since 2.0
     */
    static licenseKey: string;
    /**
     * Gets the current GoJS version.
     * @since 2.0
     */
    static readonly version: string;
    /**
     * Remove all of the Parts created from model data
     * and then create them again.
     * This must be called after modifying or replacing any of the template maps
     * such as #nodeTemplateMap.
     * This re-selects all of the new Parts that were created from data of the original selected Parts.
     *
     * If you modify a template Map, there is no notification that the map has changed.
     * You will need to call rebuildParts explicitly.
     * If you are replacing the #nodeTemplate or the #nodeTemplateMap or
     * the corresponding properties for Groups or Links,
     * the Diagram property setters will automatically call rebuildParts.
     *
     * It is extremely wasteful to call this method after making some model data changes that you want to
     * be reflected in the diagram.  Instead, it is better call Model#setDataProperty,
     * Model#addNodeData, Model#removeNodeData, or other model methods.
     * Not only do those methods update efficiently, they also preserve unbound state and support undo/redo.
     */
    rebuildParts(): void;
    /**
     * Look for a Part or Node or Group corresponding to a model's data object's unique key.
     * This will find a Link if the model is a GraphLinksModel that is maintaining a key on the link data objects.
     * @param {(string|number|undefined)} key a string or number.
     * @return {Part} null if a data with that key cannot be found in the model,
     * or if a corresponding Part cannot be found in the Diagram.
     * This will not return a Link unless the model is a GraphLinksModel and
     * GraphLinksModel#linkKeyProperty has been set.
     * If the same key is used for both a node data object and a link data object, this will return a Node.
     */
    findPartForKey(key: Key): Part | null;
    /**
     * Look for a Node or Group corresponding to a model's node data object's unique key.
     * @param {(string|number|undefined)} key a string or number.
     * @return {Node} null if a node data with that key cannot be found in the model,
     * or if a corresponding Node or Group cannot be found in the Diagram,
     * or if what is found is just a Part.
     */
    findNodeForKey(key: Key): Node | null;
    /**
     * Look for a Part, Node, Group, or Link corresponding to a Model's data object.
     * We recommend that you call #findNodeForData or #findLinkForData if you are looking for a Node or a Link.
     * @param {Object} data a JavaScript object matched by reference identity
     * @return {Part} an existing Part in this Diagram that was
     * created because its `Part.data` was the data in the Diagram's Model.
     */
    findPartForData(data: ObjectData): Part | null;
    /**
     * Look for a Node or Group  corresponding to a model's node data object.
     * @param {Object} nodedata a JavaScript object matched by reference identity;
     *   use #findNodesByExample if you want to find those Nodes whose data matches an example data object
     * @return {Node} an existing Node or Group in this Diagram that was
     * created because its `Part.data` was the node data in the Diagram's Model.
     * This will be null if there is no such part or if it's just a Part or Link.
     */
    findNodeForData(nodedata: ObjectData): Node | null;
    /**
     * Look for a Link corresponding to a GraphLinksModel's link data object.
     * @param {Object} linkdata a JavaScript object matched by reference identity;
     *   use #findLinksByExample if you want to find those Links whose data matches an example data object
     * @return {Link} an existing Link in this Diagram that was
     * created because its `Part.data` was the link data in the Diagram's Model.
     */
    findLinkForData(linkdata: ObjectData): Link | null;
    /**
     * Search for Nodes or Groups by matching the Node data with example data holding values, RegExps, or predicates.
     *
     * For example, calling this method with an argument object:
     * `{ sex: "M", name: /^Alex/i, age: function(n) { return n >= 18; } }`
     * will return an iterator of Nodes whose `Node.data` is a JavaScript object whose:
     *   - sex is "M" (a case-sensitive string comparison), and
     *   - name starts with the string "Alex" (using a case-insensitive match), and
     *   - age is greater than or equal to 18
     *
     * Here is how an example value can match the corresponding data value:
     *   - A string, number, or boolean is compared with the === operator.
     *   - A function is treated as a predicate and applied to the data value.
     *   - A regular expression (RexExp) is matched against the toString() of the data value.
     *     Common expressions include:
     *     - `/abc/` matches any string that includes exactly the substring "abc"
     *     - `/abc/i` matches any string that includes the substring "abc", but uses a case-insensitive comparison
     *     - `/^no/i` matches any string that starts with "no", ignoring case
     *     - `/ism$` matches any string that ends with "ism" exactly
     *     - `/(green|red) apple/` matches any string that includes either "green apple" or "red apple"
     *
     *     For more details read <a href="https://developer.mozilla.org/en-US/docs/Web/JavaScript/Guide/Regular_Expressions">Regular Expressions (mozilla.org)</a>.
     *   - An Array requires the data value to also be an Array of equal or greater length.
     *     Each example array item that is not undefined is matched with the corresponding data array item.
     *   - An Object is recursively matched with the data value, which must also be an Object.
     *
     * All properties given by the argument example data must be present on the node data,
     * unless the example property value is undefined.
     * All other data properties are ignored when matching.
     *
     * When multiple argument objects are given, if any of the objects match the node's data,
     * the node is included in the results.
     * @param {...Object} examples one or more JavaScript Objects whose properties are either predicates to be
     *     applied or RegExps to be tested or values to be compared to the corresponding data property value
     * @return {Iterator.<Node>}
     * @see #findLinksByExample
     * @since 1.5
     */
    findNodesByExample(...examples: Array<ObjectData>): Iterator<Node>;
    /**
     * Search for Links by matching the Link data with example data holding values, RegExps, or predicates.
     *
     * See the documentation of #findNodesByExample for how the example
     * data can match data of bound Parts.
     * @param {...Object} examples one or more JavaScript Objects whose properties are either predicates to be
     *     applied or RegExps to be tested or values to be compared to the corresponding data property value
     * @return {Iterator.<Link>}
     * @see #findNodesByExample
     * @since 1.5
     */
    findLinksByExample(...examples: Array<ObjectData>): Iterator<Link>;
    /**
     * Gets or sets the default Node template used as the archetype
     * for node data that is added to the #model.
     * Setting this property just modifies the #nodeTemplateMap
     * by replacing the entry named with the empty string.
     *
     * The value must not be null.
     * The template may be either a Node or a simple Part,
     * but not a Link or a Group.
     *
     * This Part must not be in the visual tree of any Diagram.
     */
    nodeTemplate: Part;
    /**
     * Gets or sets a Map mapping template names to Parts.
     * These nodes are copied for each node data that is added to the #model.
     *
     * The new value must not be null and must contain Nodes or simple Parts.
     * These Parts must not be in the visual tree of any Diagram.
     * Replacing this Map will automatically call #rebuildParts.
     *
     * If you modify this Map, by replacing a Node or by adding or
     * removing a map entry, you need to explicitly call #rebuildParts afterwards.
     * Any new map values must not be Links or Groups.
     *
     * If you want to create Groups, use #groupTemplateMap instead.
     */
    nodeTemplateMap: Map<string, Part>;
    /**
     * Gets or sets the default Group template used as the archetype
     * for group data that is added to the #model.
     *
     * Setting this property just modifies the #groupTemplateMap
     * by replacing the entry named with the empty string.
     * The value must not be null and must be a Group, not a Node or simple Part.
     * This Part must not be in the visual tree of any Diagram.
     */
    groupTemplate: Group;
    /**
     * Gets or sets a Map mapping template names to Groups.
     * These groups are copied for each group data that is added to the #model.
     *
     * The new value must not be null, nor may it contain a Node or Link or simple Part.
     * The Links must not be in the visual tree of any Diagram.
     * Replacing this Map will automatically call #rebuildParts.
     *
     * If you modify this Map, by replacing a Group in it or by adding or
     * removing a map entry, you need to explicitly call #rebuildParts afterwards.
     */
    groupTemplateMap: Map<string, Group>;
    /**
     * Gets or sets the default Link template used as the archetype
     * for link data that is added to the #model.
     *
     * Setting this property just modifies the #linkTemplateMap
     * by replacing the entry named with the empty string.
     * The value must not be null and must be a Link, not a Node or simple Part.
     * This Link must not be in the visual tree of any Diagram.
     */
    linkTemplate: Link;
    /**
     * Gets or sets a Map mapping template names to Links.
     * These links are copied for each link data that is added to the #model.
     *
     * The new value must not be null and must contain only Links, not Nodes or simple Parts.
     * The Links must not be in the visual tree of any Diagram.
     * Replacing this Map will automatically call #rebuildParts.
     *
     * If you modify this Map, by replacing a Link in it or by adding or
     * removing a map entry, you need to explicitly call #rebuildParts afterwards.
     */
    linkTemplateMap: Map<string, Link>;
    /**
     * Gets or sets whether mouse events initiated within the Diagram will be captured.
     * The initial value is true.
     * Setting this property does not notify about any changed event.
     */
    isMouseCaptured: boolean;
    /**
     * Gets or sets the coordinates of this Diagram in the viewport.
     * Value must be of type Point in document coordinates.
     * The default is Point(NaN, NaN), but is typically set to a real value when a Diagram is initialized.
     *
     * Scrolling and panning the Diagram modify the Diagram's position.
     *
     * Setting this property does not notify about any changed event.
     * However you can listen with #addDiagramListener for a DiagramEvent
     * with the name "ViewportBoundsChanged".
     *
     * The #viewportBounds x and y values are always the same as the Diagram's position values.
     *
     * If you set this property any replacement of the #model will result in a layout
     * and a computation of new #documentBounds, which in turn may cause the diagram to be scrolled
     * and zoomed, depending on various Diagram properties named "initial...".
     * You may want to set #initialPosition instead of setting this property around the time
     * that you are loading a model.
     */
    position: Point;
    /**
     * Gets or sets the initial coordinates of this Diagram in the viewport, eventually setting the #position.
     * This value is relevant on initialization of a #model or if #delayInitialization is called.
     * Value must be of type Point in document coordinates.
     * The default is Point(NaN, NaN).
     *
     * Setting this property has the same effect as implementing
     * an "InitialLayoutCompleted" DiagramEvent listener that sets #position.
     *
     * Setting this property does not notify about any changed event.
     * @see #initialDocumentSpot
     * @see #initialViewportSpot
     * @since 1.1
     */
    initialPosition: Point;
    /**
     * Gets or sets the initial scale of this Diagram in the viewport, eventually setting the #scale.
     * This value is relevant on initialization of a #model or if #delayInitialization is called.
     * The default is NaN.
     *
     * Setting this property has the same effect as implementing
     * an "InitialLayoutCompleted" DiagramEvent listener that sets #scale.
     * @since 1.1
     */
    initialScale: number;
    /**
     * Gets or sets a Panel of type Panel.Grid acting as the background grid
     * extending across the whole viewport of this diagram.
     */
    grid: Panel;
    /**
     * This read-only property returns the bounds of the portion of the Diagram in document coordinates that is viewable from its HTML Canvas.
     * Typically when the viewport bounds are smaller than the #documentBounds, the user can scroll or pan the view.
     *
     * The x and y coordinates are equal to the #position of the Diagram,
     * and the width and height are equal to the Diagram's canvas width and height,
     * divided by the #scale.
     */
    readonly viewportBounds: Rect;
    /**
     * Gets or sets a fixed bounding rectangle to be returned by #viewportBounds
     * when #div is null.
     * By default this is (NaN, NaN), and it is not typically set except in DOM-less environments
     * where there will not be a Diagram DIV. Normally, the viewportBounds is sized by the DIV.
     *
     * See the intro page on <a href="../../intro/nodeScript.html">GoJS within Node</a> for a usage example.
     *
     * @since 2.0
     */
    viewSize: Size;
    /**
     * Gets or sets a fixed bounding rectangle to be returned by #documentBounds
     * and #computeBounds.
     * By default this has NaN values, meaning that #computeBounds
     * will compute the union of all of the parts in the Diagram to determine the #documentBounds.
     * If all x/y/width/height values are real numbers, this value is used as the #documentBounds.
     */
    fixedBounds: Rect;
    /**
     * Gets or sets a scrollable area that surrounds the document bounds, allowing the user to scroll into empty space.
     *
     * The margin is only effective in each direction when the document bounds plus margin is greater than the viewport bounds.
     *
     * The default value is a margin of 0, all around the edge of the document.
     * @since 1.5
     */
    scrollMargin: MarginLike;
    /**
     * Gets or sets the scrollMode of the Diagram, allowing the user to either
     * scroll to document bound borders with Diagram.DocumentScroll,
     * or scroll endlessly with Diagram.InfiniteScroll.
     *
     * The default value is Diagram.DocumentScroll.
     * Changing this property value does not raise a Changed event.
     * @since 1.5
     */
    scrollMode: EnumValue;
    /**
     * Gets or sets whether the page may be scrolled when the diagram receives focus.
     * This happens in some browsers when the top-left corner of the diagram's HTMLDivElement
     * is scrolled out of view, the diagram does not have focus, and the user clicks in the diagram.
     *
     * The default value is false.
     * @since 1.8
     */
    scrollsPageOnFocus: boolean;
    /**
     * Gets or sets the function used to determine the position that this Diagram can be scrolled or moved to.
     *
     * By default this function is null and the Diagram's position is bound only by the document bounds.
     *
     * When this property is set the function is given a reference to the diagram and the proposed new position Point.
     * The function must return a new point.
     *
     * An example that disallows decimal position values:
     * ```js
     *   function computeIntegralPosition(diagram, pt) {
     *     return new go.Point(Math.floor(pt.x), Math.floor(pt.y));
     *   }
     * ```
     *
     * The function, if supplied, must not have any side-effects.
     * @since 1.5
     */
    positionComputation: ((thisDiagram: Diagram, newPosition: Point) => Point) | null;
    /**
     * Gets or sets the function used to determine valid scale values for this Diagram.
     * @since 1.5
     */
    scaleComputation: ((thisDiagram: Diagram, newScale: number) => number) | null;
    /**
     * This read-only property returns the bounds of the diagram's contents, in document coordinates.
     *
     * This is normally computed and set by #computeBounds during Diagram updates
     * that can occur for any number of relevant reasons, such as a Part changing size.
     *
     * The Diagram's documentBounds can have an unvarying specific value
     * by setting the #fixedBounds property.
     *
     * If the documentBounds are larger than the #viewportBounds,
     * scrollbars will appear on desktop browsers. You can disable scrolling with the
     * #allowHorizontalScroll and #allowVerticalScroll properties, and
     * you can disable scrollbars themselves with the #hasHorizontalScrollbar
     * and #hasVerticalScrollbar properties.
     */
    readonly documentBounds: Rect;
    /**
     * Undocumented
     */
    isVirtualized: boolean;
    /**
     * Gets or sets the scale transform of this Diagram.
     * Value must be a positive number.
     * The default value is 1.
     * Any new value will be coerced to be between #minScale and #maxScale.
     *
     * Scale can automatically be set by the #autoScale property.
     * There are also #initialScale and #initialAutoScale for
     * setting the scale on (re)initialization of a Diagram.
     *
     * Setting this property does not notify about any changed event.
     * However you can listen with #addDiagramListener for a DiagramEvent
     * with the name "ViewportBoundsChanged".
     *
     * If you set this property any replacement of the #model will result in a layout
     * and a computation of new #documentBounds, which in turn may cause the diagram to be scrolled
     * and zoomed, depending on various Diagram properties named "initial...".
     * You may want to set #initialScale instead of setting this property around the time
     * that you are loading a model.
     */
    scale: number;
    /**
     * Gets or sets the Diagram#scale set by CommandHandler#resetZoom
     * and when computing stretch values, such as when #autoScale or
     * #initialAutoScale are set, or when #zoomToFit is called.
     *
     * The default value is 1.0.
     * The value must be a number larger than 0.
     * Setting this property does not raise any events.
     * @since 2.0
     */
    defaultScale: number;
    /**
     * Gets or sets the autoScale behavior of the Diagram, controlling whether or not the
     * Diagram's bounds automatically scale to fit the view.
     *
     * The only accepted values are the constant properties of Diagram,
     * Diagram.None, Diagram.Uniform, or Diagram.UniformToFill.
     * Setting this will change the Diagram's Diagram#scale and Diagram#position, if appropriate.
     *
     * The default value is Diagram.None - the scale and position are not automatically adjusted
     * according to the area covered by the document.
     * When the value is not None, any value for #initialAutoScale or #initialScale is ignored.
     * <p class="box">
     * When autoScale is set to a non-Diagram.None value,
     * the user will not be able to zoom, and setting #scale will do nothing.
     * If you only want to scale automatically on initialization, use #initialAutoScale.
     *
     * Setting this property to Diagram.Uniform is basically the same as calling #zoomToFit
     * all the time, or just disabling interactive zooming.
     *
     * Note that depending on the values of #maxScale and #minScale, the actual value for #scale
     * might be limited.
     */
    autoScale: EnumValue;
    /**
     * Gets or sets how the scale of the diagram is automatically
     * set at the time of the "InitialLayoutCompleted" DiagramEvent, after the model has been replaced.
     *
     * The only accepted values are listed as constant properties of Diagram,
     * such as Diagram.None, Diagram.Uniform, or Diagram.UniformToFill.
     * Setting this will change the Diagram's Diagram#scale and Diagram#position, if appropriate.
     *
     * If you want to always automatically scale the Diagram, set #autoScale instead.
     * If you want to set the scale to a specific value on initialization (each time the model is replaced),
     * set #initialScale.
     *
     * The default value is Diagram.None -- the scale and position are not automatically adjusted
     * according to the area covered by the document.
     *
     * Setting this property to Diagram.Uniform is basically the same as calling #zoomToFit
     * in an "InitialLayoutCompleted" DiagramEvent listener.
     *
     * Note that depending on the values of #maxScale and #minScale, the actual value for #scale
     * might be limited.
     */
    initialAutoScale: EnumValue;
    /**
     * Gets or sets the spot in the viewport that should be coincident with the
     * #initialDocumentSpot of the document when the document is first initialized.
     * The default value is Spot.TopLeft.
     *
     * If you set this, often you will also want to set #initialDocumentSpot.
     * If you set #initialPosition, it will take precedence over this property.
     *
     * Setting this property and #initialDocumentSpot has the same effect as implementing
     * an "InitialLayoutCompleted" DiagramEvent listener that calls #alignDocument.
     *
     * @see #initialDocumentSpot
     * @see #initialContentAlignment
     */
    initialViewportSpot: Spot;
    /**
     * Gets or sets the spot in the document's area that should be coincident with the
     * #initialViewportSpot of the viewport when the document is first initialized.
     * The default value is Spot.TopLeft.
     *
     * If you set this, often you will also want to set #initialViewportSpot.
     * If you set #initialPosition, it will take precedence over this property.
     *
     * Setting this property and #initialViewportSpot has the same effect as implementing
     * an "InitialLayoutCompleted" DiagramEvent listener that calls #alignDocument.
     *
     * @see #initialViewportSpot
     * @see #initialPosition
     * @see #initialContentAlignment
     */
    initialDocumentSpot: Spot;
    /**
     * Gets or sets the smallest value greater than zero that #scale may take.
     * This property is only used to limit the range of new values of #scale.
     *
     * The default value is 0.0001.
     * Values must be larger than zero and not greater than one.
     * Setting this to a value that is greater than the current #scale
     * will cause the current diagram scale to be set to this new value.
     */
    minScale: number;
    /**
     * Gets or sets the largest value that #scale may take.
     * This property is only used to limit the range of new values of #scale.
     *
     * The default value is 100.0.
     * Values must be no less than one.
     * Setting this to a value that is less than the current #scale
     * will cause the current diagram scale to be set to this new value.
     */
    maxScale: number;
    /**
     * Gets or sets the zoom point of this Diagram, in viewport coordinates.
     * This is used by Tool#standardMouseWheel and scale-setting commands to control where to zoom in or out.
     *
     * Typical usage is to remember the value of this property and
     * to set this property to some point within the viewport (between zero and the canvas width and height).
     * This is commonly accomplished by using the InputEvent#viewPoint of Diagram#lastInput.
     * Then one changes the #scale somehow, perhaps by executing one of the CommandHandler commands,
     * or by rotating the mouse wheel, or just by setting the Diagram#scale property.
     * Finally one restores the original value of this property.
     *
     * The default value is Point(NaN, NaN).
     * Value must be of type Point, in element coordinates, not in document coordinates.
     * Setting this property does not notify about any changed event.
     * @since 1.2
     */
    zoomPoint: Point;
    /**
     * Gets or sets the content alignment Spot of this Diagram, to be used in determining
     * how parts are positioned when the #viewportBounds width or height is smaller than the #documentBounds.
     *
     * For instance a spot of Spot.Center would ensure that the Diagram's
     * contents are always centered in the viewport.
     *
     * If you want the content to be aligned only initially, use #initialContentAlignment instead.
     *
     * The default value is Spot.Default, which causes no automatic scrolling or positioning.
     * When the value is not Default, any value for #initialContentAlignment or #initialPosition is ignored.
     *
     * Setting this property has the same effect as implementing
     * a "LayoutCompleted" DiagramEvent listener that scrolls the viewport to align the content.
     *
     */
    contentAlignment: Spot;
    /**
     * Gets or sets the initial content alignment Spot of this Diagram, to be used in determining
     * how parts are positioned initially relative to the viewport, when the #viewportBounds width or height is smaller than the #documentBounds.
     *
     * For instance a spot of Spot.Center would ensure that the Diagram's contents are initially centered in the viewport.
     *
     * To initially align the document when the documentBounds are larger than the viewport, use #initialDocumentSpot and #initialViewportSpot.
     *
     * If you want the content to be constantly aligned with a spot, use #contentAlignment instead.
     *
     * The default value is Spot.Default, which causes no automatic scrolling or positioning.
     *
     * Setting this property has the same effect as implementing
     * an "InitialLayoutCompleted" DiagramEvent listener that scrolls the viewport to align the content.
     *
     * @see #initialDocumentSpot
     * @see #initialViewportSpot
     */
    initialContentAlignment: Spot;
    /**
     * Gets or sets the Margin that describes the Diagram's padding,
     * which controls how much extra space there is around the area occupied by the document.
     * This keeps nodes from butting up against the side of the diagram (unless scrolled).
     *
     * The default value is a margin of 5, all around the edge of the document.
     */
    padding: MarginLike;
    /**
     * This read-only property returns an iterator of all Nodes and Groups in the Diagram.
     *
     * This includes both data-bound and unbound nodes,
     * and both top-level nodes and nodes inside Groups.
     * All of the simple Parts are accessible via the #parts property.
     * @see #findTopLevelGroups
     * @see #findTreeRoots
     */
    readonly nodes: Iterator<Node>;
    /**
     * This read-only property returns an iterator of all Links in the Diagram.
     *
     * This includes both data-bound and unbound links,
     * and both top-level links and links inside Groups.
     */
    readonly links: Iterator<Link>;
    /**
     * This read-only property returns an iterator of all Parts in the Diagram
     * that are not Nodes or Links or Adornments.
     *
     * This includes both data-bound and unbound parts,
     * and both top-level parts and parts inside Groups.
     * Use the #nodes or #links properties for getting the
     * collection of all Nodes or Links in the diagram.
     */
    readonly parts: Iterator<Part>;
    /**
     * Returns an iterator of all Groups that are at top-level,
     * in other words that are not themselves inside other Groups.
     *
     * This is useful for when you want to traverse the diagram's graph by recursing into Groups.
     * @return {Iterator.<Group>}
     * @since 1.2
     */
    findTopLevelGroups(): Iterator<Group>;
    /**
     * Gets or sets the Layout used to position all of the top-level nodes and links in this Diagram.
     * By default this property is an instance of a simple Layout
     * that assigns positions to all parts that need it.
     * The value cannot be null and must not be shared with other Diagrams.
     */
    layout: Layout;
    /**
     * Perform all invalid layouts.
     * If the optional argument is true,
     * this will perform all of the layouts (Diagram#layout and all Group#layouts),
     * not just the invalid ones.
     *
     * Under normal circumstances you should not need to call this method,
     * because layouts will be performed automatically after they become invalid.
     * However you may have disabled automatic layouts by setting Layout#isInitial
     * and/or Layout#isOngoing to false, or by restricting a Part's Part#layoutConditions.
     * If that is the case you might call this method
     * (perhaps due to a user command) to perform the layout at a time of your choosing.
     * @param {boolean=} invalidateAll If true, this will explicitly set Layout#isValidLayout to false on each Layout in the diagram.
     */
    layoutDiagram(invalidateAll?: boolean): void;
    /**
     * Gets or sets whether the Diagram tree structure is defined by
     * links going from the parent node to their children, or vice-versa.
     * By default this property is true: links go from the parent node to the child node.
     */
    isTreePathToChildren: boolean;
    /**
     * Returns an iterator of all top-level Nodes that have no tree parents.
     *
     * This is useful for when you want to traverse the diagram's graph by starting at the root of each tree,
     * assuming that the diagram consists of one tree or a forest of trees.
     * @return {Iterator.<Node>}
     * @since 1.2
     */
    findTreeRoots(): Iterator<Node>;
    /**
     * Register an event handler that is called when there is a DiagramEvent of a given name.
     *
     * See the DiagramEvent documentation for a complete listing of diagram event names and their purposes.
     * @param {string} name the name is normally capitalized, but this method uses case-insensitive comparison.
     * @param {function(DiagramEvent)} listener a function that takes a DiagramEvent as its argument.
     * @see #removeDiagramListener
     */
    addDiagramListener(name: DiagramEventName, listener: ((e: DiagramEvent) => void)): void;
    /**
     * Unregister a DiagramEvent handler.
     *
     * See the documentation for DiagramEvent for a complete listing of diagram event names and their purposes.
     * @param {string} name the name is normally capitalized, but this method uses case-insensitive comparison.
     * @param {function(DiagramEvent)} listener a function that takes a DiagramEvent as its argument.
     * @see #addDiagramListener
     */
    removeDiagramListener(name: DiagramEventName, listener: ((e: DiagramEvent) => void)): void;
    /**
     * Undocumented.
     * @param {string} name the name is normally capitalized, but this method uses case-insensitive comparison.
     * @param {Object=} obj an optional subject of the event.
     * @param {*=} param an optional parameter describing the change to the subject of the event.
     * @see #addDiagramListener
     * @see #removeDiagramListener
     */
    raiseDiagramEvent(name: DiagramEventName, obj?: ObjectData, param?: any): void;
    /**
     * Gets or sets the Margin that describes the Diagram's autoScrollRegion. The default value is a Margin of 16 on all sides.
     * When the mouse drag point is within this region on the left or right sides,
     * the view will automatically scroll horizontally in that direction.  When the point is within
     * the region on the top or bottom, the view will automatically scroll
     * vertically in that direction.  You can specify a distance of zero to disable autoscrolling
     * in a direction; a value of 0,0,0,0 turns off autoscrolling altogether.
     */
    autoScrollRegion: MarginLike;
    /**
     * Undocumented
     * @param {Point} viewPt
     */
    doAutoScroll(viewPt: Point): void;
    /**
     * Undocumented
     */
    stopAutoScroll(): void;
    /**
     * Create an SVGElement that contains a SVG rendering of the current Diagram.
     *
     * By default this method returns a snapshot of the visible diagram, but optional arguments give more options.
     * <p class="boxread">
     * See the page on <a href="../../intro/makingSVG.html">Making SVG</a> for usage examples.
     * See the <a href="../../samples/minimalSVG.html">Minimal SVG Download sample</a>,
     * which also demonstrates downloading an SVG file without involving a web server.
     *
     * See #makeImageData for an explanation of possible options that are shared by both methods.
     * Additional SVG-specific options for this method:
     *   - **elementFinished:**
     *     A function with two arguments, GraphObject and SVGElement.
     *     As the SVG elements are created representing each graph object, this function is called on them,
     *     allowing you to modify the SVG as it is being built, to assign stylesheets, IDs, etc. Example:
     *     ```js
     *     elementFinished: function(graphObject, SVGElement) {
     *       // set something on every SVG element that represents a GoJS TextBlock
     *       if (graphObject instanceof go.TextBlock) SVGElement.setAttribute(...);
     *     }
     *     ```
     *
     * At the current time methods such as Diagram#makeImage,
     * Diagram#makeImageData and Diagram#makeSvg do not work on Overviews.
     * @param {any=} options a JavaScript object detailing optional arguments for SVG creation.
     * @return {SVGElement}
     * @see #makeImage
     */
    makeSvg(options?: SvgRendererOptions): SVGElement;
    /**
     * Add a renderer to the Diagram. This property is only used when building GoJS from source.
     *
     * When building from source, to include SVG rendering functionality for Diagram#makeSvg,
     * you must add the SVGSurface renderer with the name 'SVG': `myDiagram.addRenderer('SVG', new SVGSurface(myDiagram, document));`
     *
     * @param {string} name Name of the renderer
     * @param {ISurface} surface Rendering class, either SVGSurface or CanvasSurface.
     */
    addRenderer(name: string, surface: ISurface): void;
    /**
     *
     * Create an HTMLImageElement that contains a bitmap of the current Diagram.
     * This method is just a convenience function that creates an image,
     * sets its source to the returned string of #makeImageData,
     * and returns a reference to that Image.
     *
     * See #makeImageData for a complete explanation of possible options.
     *
     * By default this method returns a snapshot of the visible diagram, but optional arguments give more options.
     *
     * At the current time methods such as Diagram#makeImage,
     * Diagram#makeImageData and Diagram#makeSvg do not work on Overviews.
     * @param {any=} options a JavaScript object detailing optional arguments for image creation, to be passed to #makeImageData.
     * @return {HTMLImageElement | null} An HTML Image element, or null if a callback is specified, or null if there is no DOM.
     * @see #makeImageData
     * @see #makeSvg
     */
    makeImage(options?: ImageRendererOptions): HTMLImageElement | null;
    /**
     * Create a bitmap of the current Diagram encoded as a base64 string, or returned as an ImageData object.
     * This method uses the toDataURL method of the HTMLCanvasElement to create the data URL,
     * or the getImageData method of the Canvas Context.
     * Unlike toDataURL, this method will not throw an error if cross-domain images
     * were drawn on the canvas, instead it will return a data URL of a bitmap with those images omitted.
     *
     * A simple example:
     * ```js
     * myDiagram.makeImageData({
     *   scale: 1.5,
     *   size: new go.Size(100,100)
     * });
     * ```
     * <p class="boxread">
     * See the page on <a href="../../intro/makingImages.html">Making Images</a> for more usage examples.
     *
     * At the current time methods such as Diagram#makeImage,
     * Diagram#makeImageData and Diagram#makeSvg do not work on Overviews.
     * @param {any=} options a JavaScript object detailing optional arguments for image creation.
     * Rendering options for both images and SVG:
     *   - **size:**
     *     The size of the created image, as a Size, limited by the **maxSize** property.
     *     If no **scale** or **position** is specified then the diagram will be scaled to fit the given size.
     *     If you set a size, you should also set a **position**. If you are scaling the diagram, you may also want to scale the size.
     *   - **scale:**
     *     The scale of the diagram. If **scale** is specified and **size** is not, the resulting image will be sized to uniformly
     *     fit the space needed for the given scale.
     *     Can be constrained by the **maxSize** property. A scale value of `NaN` will
     *     automatically scale ot fit within the maxSize, but may be smaller, with a maximum computed scale of 1.
     *   - **maxSize:**
     *     The maximum size of the created image, as a Size. The default value is `(Infinity, Infinity)` for SVG and `(2000, 2000)` for images.
     *     This is typically used when **scale** is specified.
     *   - **position:**
     *     The position of the diagram, as a Point.
     *     By default this is the position of Diagram#documentBounds with the Diagram#padding removed.
     *     If a specific parts collection is used, by default this is the top-left diagram position of their collective bounds.
     *     If you set a `position`, you should also set a `size`.
     *   - **parts:**
     *     An iterator of GraphObjects, typically Parts, such as one from Diagram#selection or Layer#parts.
     *     If GraphObjects are specified their containing Part will be drawn. By default all Parts are drawn except temporary parts (see **showTemporary**).
     *   - **padding:**
     *     A Margin (or number) to pad the image with. If a size is specified,
     *     the padding will not increase the image size, it will only offset the Diagram contents within the image.
     *     The default value is a padding of `1`.
     *   - **background:**
     *     A valid CSS color to replace the default (transparent) canvas background. Any padding area is also colored.
     *   - **showTemporary:**
     *     A boolean value, defaulting to false,
     *     that determines whether or not temporary objects such as adornments are included in the image.
     *   - **showGrid:**
     *     A boolean value, defaulting to the value of **showTemporary**,
     *     that determines whether or not the Grid Layer (containing Diagram#grid) is included in the image
     *     regardless of the value of **showTemporary**.
     *     This is useful if you want to include the grid but not adornments, or vice versa.
     *   - **document:**
     *     An HTML Document, defaulting to `window.document` (or the root object in other contexts)
     *     This may be useful to set if you intend your Image or SVG to be opened in a new window.
     *
     * Additional image-specific arguments (not for SVG):
     *   - **type:**
     *     The optional MIME type of the image. Valid values are typically `"image/png"` and `"image/jpeg"`.
     *     Some browsers allow `"image/webp"`. The default value is `"image/png"`, and unrecognized values will defer to the default.
     *   - **returnType:**
     *     The optional return type of the image data. Valid values are `"ImageData"`, `"string"`, and `"blob"`.
     *     The `"string"` option returns a base64 string representation of the image.
     *     The `"ImageData"` option returns an `ImageData` object representation of the image.
     *     The `"Image"` option returns an `HTMLImageElement` using `ImageData` as the `HTMLImageElement.src`.
     *     The `"blob"` option requires that the **callback** property is also defined.
     *     The default value is `"string"`, and unrecognized values will return a string.
     *   - **callback:**
     *     The function to call when an image is finished creation. It has one argument, which is of the type specified by the value of the **returnType**.
     *     If provided, call the callback when finished instead of returning immediately. This can be useful if you need to wait for image assets to load.
     *     This also respects the **callbackTimeout**.
     *     This argument is necessary if the **returnType** is `"blob"`, however a callback can be used with any **returnType**.
     *     See the <a href="../../samples/minimalBlob.html">Minimal Image Blob Download sample</a> for an example usage,
     *     which also demonstrates downloading an image file without involving a web server.
     *   - **callbackTimeout:**
     *     If a **callback** is specified, the additional amount of time in milliseconds a call will wait before completeing. Right now, it will only wait if
     *     image assets in the Diagram are not yet loaded. Default is 300 (milliseconds).
     *   - **details:**
     *     The optional details to pass to the HTMLCanvasElement's toDataURL function.
     *     If the type is `"image/jpeg"` then this can be a number from `0` to `1`, inclusive, describing the desired jpeg quality.
     *
     * @return {ImageData|string|null} An ImageData, or a base64-encoded string describing an image, or null if a callback is specified.
     * @see #makeImage
     */
    makeImageData(options?: ImageRendererOptions): HTMLImageElement | ImageData | string | null;
    /**
     * This static function declares that a class (constructor function) derives from another class --
     * **_but please note that most classes do not support inheritance_**.
     * Do not call this function when your class is defined using an ES2015 or TypeScript "class" declaration.
     *
     * Because you must not modify the prototypes for the **GoJS** classes,
     * in order to override methods you need to define a new class that inherits from a predefined class.
     * You can then modify the prototype of your derived class.
     *
     * Typical usage is:
     * ```js
     * function BaseClass() {
     *   this._total = 0;
     * }
     *
     * public increment(x) {
     *   this._total += x;
     * }
     *
     * function DerivedClass() {
     *   this._anotherProperty = "";
     * }
     * go.Diagram.inherit(DerivedClass, BaseClass);
     *
     * DerivedClass.prototype.someMethod = ...;
     * ```
     *
     * **_Note that most classes do not support inheritance._**
     * Currently you can only inherit from Layout, Tool, CommandHandler, and Link or their subclasses.
     * When you override a method, you must strongly consider calling the base method.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call a base method.
     *
     * The call to `Diagram.inherit` should be made immediately after defining the subclass constructor function
     * and before defining any new methods or overriding any base class methods.
     * You must not call this static function more than once on a derived class, or at all on a class defined using an ES2015 or TypeScript class declaration.
     *
     * The need for subclassing is greatly diminished by the presence of a number of properties that have functional values.
     * Setting such a property to a function will cause that function to be called as if it were
     * an event handler for that particular object.
     * Example properties include: GraphObject#click, GraphObject#mouseEnter, Part#layerChanged,
     * Node#treeExpandedChanged, LinkingBaseTool#linkValidation, CommandHandler#memberValidation,
     * TextEditingTool#textValidation.
     * @param {Function} derivedclass
     * @param {Function} baseclass
     */
    static inherit(derivedclass: Function, baseclass: Function): void;
}
/**
 * This is used in dragging methods to remember initial locations, for relative positioning and for cancellation.
 * You do not typically need to create these unless you are overriding tools.
 *
 * CommandHandler#computeEffectiveCollection returns a Map of Parts and DraggingInfo,
 * which is used as an argument for methods like Diagram#computeMove.
 *
 * Diagram#moveParts constructs a  Map of Parts and DraggingInfo to record each Part's location offset during moves.
 *
 * @unrestricted
 * @since 2.0
 */
export class DraggingInfo {
    /**
     * Optionally takes a Point that sets #point.
     * @param {Point=} pt
     */
    constructor(pt?: Point);
    /**
     * Gets or sets the initial location for a dragged object.
     * This is most commonly the Part.location of the dragged object, or a rounded Part.location if snapping.
     */
    point: Point;
}
/**
 * New in 2.0, this class contains options for dragging and moving parts.
 *
 * This is used by Diagram#moveParts, Diagram#computeMove, and DraggingTool#computeMove.
 *
 * Both the Diagram and its DraggingTool have a DraggingOptions instance.
 * Typically, only the DraggingTool one is used, unless the Diagram has no DraggingTool defined.
 *
 * In typical DraggingTool operation you do not need to create or modify a DraggingOptions instance.
 * Setting properties on the DraggingTool such as DraggingTool#isGridSnapEnabled
 * sets the value on the DraggingTool#dragOptions instance for you.
 *
 * @unrestricted
 * @see DraggingTool#dragOptions
 * @since 2.0
 */
export class DraggingOptions {
    /**
     * Whether or not the dragged parts snap to the Diagram#grid.
     * By default this property is false.
     *
     * In typical DraggingTool operation this is set via DraggingTool#isGridSnapEnabled.
     */
    isGridSnapEnabled: boolean;
    /**
     * If dragging with the DraggingTool, whether or not Parts snap to a grid (if applicable) in real-time,
     * or only at the end of the dragging transaction.
     * By default this property is true.
     *
     * In typical DraggingTool operation this is set via DraggingTool#isGridSnapRealtime.
     */
    isGridSnapRealtime: boolean;
    /**
     * Gets or sets the size of the grid cell used when snapping during a drag.
     * By default this property is the Size(NaN, NaN), which causes this tool to use the Panel#gridCellSize value of the Diagram#grid.
     *
     * In typical DraggingTool operation this is set via DraggingTool#gridSnapCellSize.
     */
    gridSnapCellSize: Size;
    /**
     * Gets or sets the Spot that specifies what point in the grid cell dragged parts snap to,
     * if the value of #isGridSnapEnabled is true. By default this property is Spot.TopLeft.
     *
     * In typical DraggingTool operation this is set via DraggingTool#gridSnapCellSpot.
     */
    gridSnapCellSpot: Spot;
    /**
     * Gets or sets the snapping grid's origin point, in document coordinates,
     * if the value of #isGridSnapEnabled is true. By default this property is the Point(NaN, NaN).
     *
     * In typical DraggingTool operation this is set via DraggingTool#gridSnapOrigin.
     */
    gridSnapOrigin: Point;
    /**
     * Gets or sets whether the user can move a single Link,
     * disconnecting it from connected nodes and possibly connecting it to valid ports.
     * By default this property is false.
     *
     * In typical DraggingTool operation this is set via DraggingTool#dragsLink.
     */
    dragsLink: boolean;
    /**
     * Gets or sets whether moving or copying a node also includes all of the
     * node's tree children and their descendants, along with the links to those additional nodes.
     * By default this property is false.
     *
     * In typical DraggingTool operation this is set via DraggingTool#dragsTree.
     */
    dragsTree: boolean;
    /**
     * Determines whether or not Groups will snap their member Parts in addition to themselves when moving.
     * By default this property is false.
     */
    groupsSnapMembers: boolean;
    /**
     * Determines whether or not Groups will ensure their member Parts always move,
     * even if those member parts have Part#copyable and Part#movable set to false.
     * By default this property is true.
     */
    groupsAlwaysMove: boolean;
    constructor();
}
/**
 * Used for the options arguments to Diagram.makeImage, Diagram.makeImageData, and Diagram.makeSvg.
 */
export interface DiagramRendererOptions {
    size?: Size;
    scale?: number;
    maxSize?: Size;
    position?: Point;
    parts?: Iterable<Part>;
    padding?: MarginLike;
    background?: BrushLike;
    showTemporary?: boolean;
    showGrid?: boolean;
    document?: HTMLDocument;
}
/**
 * Used for the options argument to Diagram.makeSvg.
 */
export interface SvgRendererOptions extends DiagramRendererOptions {
    elementFinished?: ((graphobj: GraphObject, svgelt: SVGElement) => void) | null;
}
/**
 * Used for the options argument to Diagram.makeImage and Diagram.makeImageData.
 */
export interface ImageRendererOptions extends DiagramRendererOptions {
    type?: string;
    returnType?: string;
    callback?: ((result: any) => void) | null;
    callbackTimeout?: number;
    details?: ObjectData;
}
/**
 * Palette extends the Diagram class to allow objects to be dragged and placed onto other Diagrams.
 * Its Diagram#layout is a GridLayout.
 * The Palette is Diagram#isReadOnly but to support drag-and-drop its Diagram#allowDragOut is true.
 * <p class="box">
 * For more discussion, see <a href="../../intro/palette.html">Introduction to Palettes</a>.
 * See samples that make use of Palettes in the <a href="../../samples/index.html#palette">samples index</a>.
 *
 * You can control the order of Parts in the palette in several manners:
 *   - If you want to keep the order given by `myPalette.model.nodeDataArray`,
 *     just set `myPalette.layout.sorting = go.GridLayout.Forward` (GridLayout#sorting).
 *   - If you want to sort based on some text string associated with each Part, just bind Part#text.
 *   - If you want to sort more generally, set `myPalette.layout.comparer` (GridLayout#comparer).
 * @extends Diagram
 * @unrestricted
 */
export class Palette extends Diagram {
    /**
     * @param {HTMLDivElement|string} div A reference to a div or its ID as a string.
     */
    constructor(div?: Element | string);
}
/**
 * An Overview is a Diagram that displays all of a different diagram,
 * with a rectangular box showing the viewport displayed by that other diagram.
 * <p class="box">
 * For more discussion, see <a href="../../intro/overview.html">Introduction to Overviews</a>.
 * See samples that make use of Overviews in the <a href="../../samples/index.html#overview">samples index</a>.
 *
 * All you need to do is set Overview#observed.
 * For example:
 *
 * ```js
 *   var myDiagram = new go.Diagram("myDiagramDIV");
 *   . . . other initialization . . .
 *
 *   // create and initialize the Overview:
 *   new go.Overview("myOverviewDIV").observed = myDiagram;
 * ```
 *
 * The Overview draws what the observed Diagram is displaying,
 * so setting or modifying any diagram templates or template Maps has no effect.
 * At the current time methods such as Diagram#makeImage,
 * Diagram#makeImageData and Diagram#makeSvg do not work on Overviews.
 * @extends Diagram
 * @unrestricted
 */
export class Overview extends Diagram {
    /**
     * @param {Element|string} div A reference to a div or its ID as a string.
     */
    constructor(div?: Element | string);
    /**
     * Gets or sets the Diagram for which this Overview is
     * displaying a model and showing its viewport into that model.
     *
     * The value must be null or another Diagram, but may not be an Overview.
     */
    observed: Diagram | null;
    /**
     * Gets or sets the rectangular Part that represents the
     * viewport of the #observed Diagram.
     * By default the part contains only a magenta Shape.
     * The box's Part#selectionObject is what is resized by the Overview to the apparent size of the observed diagram's viewport.
     */
    box: Part;
    /**
     * Gets or sets whether this overview draws the temporary layers of the observed Diagram.
     * The default value is true.
     * @since 1.2
     */
    drawsTemporaryLayers: boolean;
}
/**
 * The Diagram#commandHandler implements various
 * commands such as CommandHandler#deleteSelection or CommandHandler#redo.
 * The CommandHandler includes keyboard event handling to interpret
 * key presses as commands.
 *
 * CommandHandlers cannot be shared amongst multiple Diagrams.
 *
 * You may define a CommandHandler subclass and override methods.
 * However you must seriously consider calling the base method in order to get its default behavior.
 * There may be situations where not calling the base method may cause subtle bugs,
 * but that depends on the method.
 * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call a base method.
 *
 * There is an example custom CommandHandler in the extensions directory: <a href="../../extensions/DrawCommandHandler.js">DrawCommandHandler.js</a>,
 * which provides alignment commands and additional behaviors for the arrow keys.
 * <p class="boxread">
 * For additional discussion, please read the <a href="../../intro/commands.html">Introduction page on Commands</a>.
 * <h3>Keyboard Shortcuts</h3>
 *
 * The CommandHandler implements the following command bindings for keyboard input in #doKeyDown:
 *   - `Ctrl-X` & `Shift-Del` invoke #cutSelection
 *   - `Ctrl-C` & `Ctrl-Insert` invoke #copySelection
 *   - `Ctrl-V` & `Shift-Insert` invoke #pasteSelection
 *   - `Del` & `Backspace` invoke #deleteSelection
 *   - `Ctrl-A` invokes #selectAll
 *   - `Ctrl-Z` & `Alt-Backspace` invoke #undo
 *   - `Ctrl-Y` & `Alt-Shift-Backspace` invoke #redo
 *   - `Up` & `Down` & `Left` & `Right` (arrow keys) call Diagram#scroll
 *   - `PageUp` & `PageDown` call Diagram#scroll
 *   - `Home` & `End` call Diagram#scroll
 *   - `Space` invokes #scrollToPart
 *   - `Ctrl-- & Keypad--` (minus) invoke #decreaseZoom
 *   - `Ctrl-+ & Keypad-+` (plus) invoke #increaseZoom
 *   - `Ctrl-0` invokes #resetZoom
 *   - `Shift-Z` invokes #zoomToFit; repeat to return to the original scale and position
 *   - `Ctrl-G` invokes #groupSelection
 *   - `Ctrl-Shift-G` invokes #ungroupSelection
 *   - `F2` invokes #editTextBlock
 *   - `Menu Key` invokes #showContextMenu
 *   - `Esc` invokes #stopCommand
 *
 * On a Macintosh the Command key is used as the modifier instead of the Control key.
 *
 * On touch devices there is a default context menu that shows many commonly-used commands
 * when you hold a finger down on the diagram.
 * @unrestricted
 */
export class CommandHandler {
    /**
     * The constructor produces a CommandHandler with the default key bindings.
     */
    constructor();
    /**
     * This read-only property returns the Diagram that is using this CommandHandler,
     * after Diagram#commandHandler has been set to this object.
     */
    readonly diagram: Diagram;
    /**
     * This is called by tools to handle keyboard commands.
     * For most commands, this calls the "can..." predicate; if that returns true it calls the command method.
     * If GoJS handles a key-down event as a keyboard command, the underlying event will not bubble.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * There is an example custom CommandHandler in the extensions directory: <a href="../../extensions/DrawCommandHandler.js">DrawCommandHandler.js</a>,
     * which implements additional behaviors for the arrow keys by overriding this method.
     * For additional discussion, please read the <a href="../../intro/commands.html">Introduction page on Commands</a>.
     * @expose
     */
    doKeyDown(): void;
    /**
     * This is called by tools to handle keyboard commands.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    doKeyUp(): void;
    /**
     * This command cancels the operation of the current tool.
     * This is typically called when the user presses ESCAPE.
     *
     * If the current tool is a ToolManager, this clears the diagram's selection.
     * This then calls Tool#doCancel on the current tool.
     * @expose
     * @see #canStopCommand
     */
    stopCommand(): void;
    /**
     * This predicate controls whether the user may stop the current tool.
     * This just returns true.
     *
     * This method may be overridden, but probably should not be overridden.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean} true.
     * @see #stopCommand
     */
    canStopCommand(): boolean;
    /**
     * This command selects all of the selectable Parts in the diagram by setting Part#isSelected to true on each one.
     * This is normally invoked by the `Ctrl-A` keyboard shortcut.
     *
     * This method raises the "ChangingSelection" and "ChangedSelection" diagram events.
     * This ignores all parts in temporary layers.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canSelectAll
     */
    selectAll(): void;
    /**
     * This predicate controls whether or not the user can invoke the #selectAll command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean} This returns true if Diagram#allowSelect is true.
     * @see #selectAll
     */
    canSelectAll(): boolean;
    /**
     * This command deletes the currently selected parts from the diagram.
     * This is normally invoked by the `Del` keyboard shortcut.
     *
     * This will first start a "Delete" transaction, then
     * raise the "SelectionDeleting" DiagramEvent,
     * call Diagram#removeParts on a perhaps extended collection of selected Parts,
     * raise the "SelectionDeleted" diagram event,
     * and finally commit the transaction.
     *
     * Because this command changes the selection,
     * this method also raises the "ChangingSelection" and "ChangedSelection" diagram events.
     * All changes are performed within a transaction.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canDeleteSelection
     */
    deleteSelection(): void;
    /**
     * This predicate controls whether or not the user can invoke the #deleteSelection command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean} This returns true:
     * if the diagram is not Diagram#isReadOnly,
     * if Diagram#allowDelete is true, and
     * if there are some selected Parts.
     * @see #deleteSelection
     */
    canDeleteSelection(): boolean;
    /**
     * This command copies the currently selected parts, Diagram#selection, from the Diagram into the clipboard.
     * This is normally invoked by the `Ctrl-C` keyboard shortcut.
     *
     * This makes a copy of the current selection by calling #copyToClipboard.
     * This also raises the "ClipboardChanged" diagram event.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canCopySelection
     */
    copySelection(): void;
    /**
     * This predicate controls whether or not the user can invoke the #copySelection command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     * This returns true:
     * if Diagram#allowCopy is true,
     * if Diagram#allowClipboard is true, and
     * if there are some selected Parts.
     * @see #copySelection
     */
    canCopySelection(): boolean;
    /**
     * This command executes a #copySelection followed by a #deleteSelection.
     * This is normally invoked by the `Ctrl-X` keyboard shortcut.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canCutSelection
     */
    cutSelection(): void;
    /**
     * This predicate controls whether or not the user can invoke the #cutSelection command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     * This returns true:
     * if the diagram is not Diagram#isReadOnly,
     * if Diagram#allowCopy is true,
     * if Diagram#allowDelete is true,
     * if Diagram#allowClipboard is true, and
     * if there are some selected Parts.
     * @see #cutSelection
     */
    canCutSelection(): boolean;
    /**
     * Make a copy of the given collection of Parts
     * and stores it in a static variable acting as the clipboard.
     *
     * The clipboard is initially null.
     * It can hold a collection of copied Parts.
     * It also remembers the Model#dataFormat of the diagram
     * from which the parts were copied.
     *
     * This calls Diagram#copyParts in order to make a copy of the Parts for the clipboard.
     * The values of #copiesParentKey and #copiesGroupKey affect whether
     * a copied node data remembers its tree parent node (if in a TreeModel) or
     * its containing group (if in a GraphLinksModel).
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Iterable.<Part>} coll A collection of Parts.
     * If the value is null, the clipboard is cleared of all data.
     * @see #pasteFromClipboard
     */
    copyToClipboard(coll: Iterable<Part>): void;
    /**
     * If the clipboard holds a collection of Parts,
     * and if the Model#dataFormat matches that stored in the clipboard,
     * this makes a copy of the clipboard's parts and adds the copies to this Diagram.
     *
     * This calls Diagram#copyParts in order to make a copy of the Parts in the clipboard and add them to this diagram.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {Set.<Part>} a collection of the newly pasted Parts,
     * or an empty Set if there was no data in the clipboard.
     * @see #copyToClipboard
     */
    pasteFromClipboard(): Set<Part>;
    /**
     * This command copies the contents of the clipboard into this diagram and makes those new parts the new selection.
     * This is normally invoked by the `Ctrl-V` keyboard shortcut.
     *
     * This calls #pasteFromClipboard to add copies of Parts
     * into this diagram, and then selects all of the newly created parts.
     * This also raises the "ClipboardPasted" diagram event.
     * This method raises the "ChangingSelection" and "ChangedSelection" diagram events.
     * All of the changes are performed in a transaction.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Point=} pos Point at which to center the newly pasted parts; if not present the parts are not moved.
     * @see #canPasteSelection
     */
    pasteSelection(pos?: Point): void;
    /**
     * This predicate controls whether or not the user can invoke the #pasteSelection command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     * This returns true:
     * if the diagram is not Diagram#isReadOnly,
     * if Diagram#allowInsert is true,
     * if Diagram#allowClipboard is true, and
     * if the clipboard has parts in it.
     * @see #pasteSelection
     */
    canPasteSelection(): boolean;
    /**
     * This command calls UndoManager#undo.
     * This is normally invoked by the `Ctrl-Z` keyboard shortcut.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canUndo
     */
    undo(): void;
    /**
     * This predicate controls whether or not the user can invoke the #undo command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     * This returns true:
     * if the diagram is not Diagram#isReadOnly,
     * if Diagram#allowUndo is true, and
     * if the UndoManager#canUndo predicate returns true.
     * @see #undo
     */
    canUndo(): boolean;
    /**
     * This command calls UndoManager#redo.
     * This is normally invoked by the `Ctrl-Y` keyboard shortcut.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canRedo
     */
    redo(): void;
    /**
     * This predicate controls whether or not the user can invoke the #redo command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     * This returns true:
     * if the diagram is not Diagram#isReadOnly,
     * if Diagram#allowUndo is true, and
     * if the UndoManager#canRedo predicate returns true.
     * @see #redo
     */
    canRedo(): boolean;
    /**
     * This command decreases the Diagram#scale by a given factor.
     * This is normally invoked by the `Ctrl--` and `Keypad--` keyboard shortcuts.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {number=} factor This defaults to 1/#zoomFactor.  The value should be less than one and greater than zero.
     * @see #canDecreaseZoom
     */
    decreaseZoom(factor?: number): void;
    /**
     * This predicate controls whether or not the user can invoke the #decreaseZoom command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {number=} factor This defaults to 1/#zoomFactor.  The value should be less than one and greater than zero.
     * @return {boolean}
     * This returns true if Diagram#allowZoom is true
     * and if the new scale is within the range of Diagram#minScale and Diagram#maxScale.
     * @see #decreaseZoom
     */
    canDecreaseZoom(factor?: number): boolean;
    /**
     * This command increases the Diagram#scale by a given factor.
     * This is normally invoked by the `Ctrl-+` and `Keypad-+` keyboard shortcuts.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {number=} factor This defaults to #zoomFactor.  The value should be greater than one.
     * @see #canIncreaseZoom
     */
    increaseZoom(factor?: number): void;
    /**
     * This predicate controls whether or not the user can invoke the #increaseZoom command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {number=} factor This defaults to #zoomFactor.  The value should be greater than one.
     * @return {boolean}
     * This returns true if Diagram#allowZoom is true
     * and if the new scale is within the range of Diagram#minScale and Diagram#maxScale.
     * @see #increaseZoom
     */
    canIncreaseZoom(factor?: number): boolean;
    /**
     * This command sets the Diagram#scale to a new scale value, by default 1.
     * This is normally invoked by the `Ctrl-0` keyboard shortcut.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {number=} newscale This defaults to #defaultScale, which is normally 1.0.  The value should be greater than zero.
     * @see #canResetZoom
     */
    resetZoom(newscale?: number): void;
    /**
     * This predicate controls whether or not the user can invoke the #resetZoom command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {number=} newscale This defaults to #defaultScale, which is normally 1.0.  The value should be greater than zero.
     * @return {boolean}
     * This returns true if Diagram#allowZoom is true.
     * and if the new scale is within the range of Diagram#minScale and Diagram#maxScale.
     * @see #resetZoom
     */
    canResetZoom(newscale?: number): boolean;
    /**
     * This command changes the Diagram#scale so that the Diagram#documentBounds fits within the viewport.
     * If this command had been called before without any other zooming since then,
     * the original Diagram scale and position are restored.
     * This is normally invoked by the `Shift-Z` keyboard shortcut.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canZoomToFit
     * @since 1.1
     */
    zoomToFit(): void;
    /**
     * This predicate controls whether or not the user can invoke the #zoomToFit command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     * This returns true if Diagram#allowZoom is true.
     * @see #zoomToFit
     * @since 1.1
     */
    canZoomToFit(): boolean;
    /**
     * This command scrolls the diagram to make a highlighted or selected Part visible in the viewport.
     * Call this command repeatedly to cycle through the Diagram#highlighteds collection,
     * if there are any Parts in that collection, or else in the Diagram#selection collection,
     * scrolling to each one in turn by calling Diagram#centerRect.
     *
     * This is normally invoked by the `Space` keyboard shortcut.
     * If there is no argument and there is no highlighted or selected Part, this command does nothing.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Part=} part This defaults to the first highlighted Part of Diagram#highlighteds,
     *                     or, if there are no highlighted Parts, the first selected Part.
     * @see #canScrollToPart
     * @since 1.6
     */
    scrollToPart(part?: Part): void;
    /**
     * This predicate controls whether or not the user can invoke the #scrollToPart command.
     * This returns false if there is no argument Part and there are no selected Parts.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Part=} part This defaults to the first selected Part of Diagram#selection
     * @return {boolean}
     * This returns true if Diagram#allowHorizontalScroll and Diagram#allowVerticalScroll are true.
     * @see #scrollToPart
     * @since 1.6
     */
    canScrollToPart(part?: Part): boolean;
    /**
     * This command collapses all expanded selected Nodes.
     * This operation is performed within a "Collapse Tree" transaction.
     * Just before the end of the transaction this raises the "TreeCollapsed" DiagramEvent,
     * with a collection of collapsed Nodes as the subject.
     * This currently has no default keyboard shortcut.
     *
     * This calls Node#collapseTree to perform the collapse, which will set Node#isTreeExpanded to false.
     * You may want to save the collapsed/expanded state to the model by using a TwoWay Binding on the "isTreeExpanded" property of your Nodes,
     * and perhaps also on the Node#wasTreeExpanded property.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node=} node if supplied, ignore the selection and collapse this particular Node subtree.
     * @see #canCollapseTree
     */
    collapseTree(node?: Node): void;
    /**
     * This predicate controls whether the user can collapse expanded subtrees of Nodes.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node=} node if supplied, ignore the selection and consider collapsing this particular Node.
     * @return {boolean}
     * @see #collapseTree
     */
    canCollapseTree(node?: Node): boolean;
    /**
     * This command expands all collapsed selected Nodes.
     * This operation is performed within an "Expand Tree" transaction.
     * Just before the end of the transaction this raises the "TreeExpanded" DiagramEvent,
     * with a collection of expanded Nodes as the subject.
     * This currently has no default keyboard shortcut.
     *
     * This calls Node#expandTree to perform the expand, which will set Node#isTreeExpanded to true.
     * You may want to save the collapsed/expanded state to the model by using a TwoWay Binding on the "isTreeExpanded" property of your Nodes,
     * and perhaps also on the Node#wasTreeExpanded property.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node=} node if supplied, ignore the selection and collapse this particular Node subtree.
     * @see #canExpandTree
     */
    expandTree(node?: Node): void;
    /**
     * This predicate controls whether the user can expand collapsed subtrees of Nodes.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node=} node if supplied, ignore the selection and consider expanding this particular Node.
     * @return {boolean}
     * @see #expandTree
     */
    canExpandTree(node?: Node): boolean;
    /**
     * This command adds a copy of #archetypeGroupData to the diagram's model
     * to create a new Group and then adds the selected Parts to that new group.
     * This is normally invoked by the `Ctrl-G` keyboard shortcut.
     *
     * This creates a new Group by adding a copy of the #archetypeGroupData to the model.
     * Each of the selected parts for which Part#canGroup is true
     * and for which #isValidMember is true
     * is made a member of that new group.
     * If all of the selected groupable parts were members of a pre-existing group,
     * the new group also becomes a member of that pre-existing group,
     * if #isValidMember is true for that existing group with the new group.
     * The new group becomes the only selected part.
     * This raises the "SelectionGrouped" diagram event.
     * This method also raises the "ChangingSelection" and "ChangedSelection" diagram events.
     * All of the changes are performed in a "Group" transaction.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @see #canGroupSelection
     */
    groupSelection(): void;
    /**
     * This predicate controls whether or not the user can invoke the #groupSelection command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     * This returns true:
     * if the #archetypeGroupData is not null,
     * if the diagram is not Diagram#isReadOnly,
     * if Diagram#allowInsert is true,
     * if Diagram#allowGroup is true,
     * if Diagram#model is a GraphLinksModel, and
     * if there is any selected Part that can be Part#canGrouped.
     * @see #groupSelection
     */
    canGroupSelection(): boolean;
    /**
     * This predicate is called to determine whether a Node may be added as a member of a Group.
     * This always checks to make sure no group might become a member of itself, either directly or indirectly.
     * If the Group has a Group#memberValidation predicate and if it returns false, this method returns false.
     * If this CommandHandler has a #memberValidation predicate and if it returns false, this method returns false.
     * Otherwise this will return true.
     * <p class="box">
     * For a more general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Group} group this may be null if the node is being added as a top-level node.
     * @param {Part} part a Part, usually a Node, possibly another Group, but not a Link or an Adornment.
     * @return {boolean} true if OK to add the node to the group.
     */
    isValidMember(group: Group, part: Part): boolean;
    /**
     * This command removes selected groups from the diagram without removing their members from the diagram.
     * This is normally invoked by the `Ctrl-Shift-G` keyboard shortcut.
     *
     * For the given group, or if not supplied, each selected Group that is Group#ungroupable,
     * expand the subgraph and change all of its member parts to be members of the group that the selected group node is in.
     * (If the selected group is a top-level node, i.e. not a member of any group node, its members become top-level parts too.)
     * All of those selected groups are deleted.
     * All of the reparented member parts are selected.
     *
     * This raises the "SelectionUngrouped" diagram event.
     * This method also raises the "ChangingSelection" and "ChangedSelection" diagram events.
     * All of the changes are performed in an "Ungroup" transaction.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Group=} group if supplied, ignore the selection and consider ungrouping this particular Group.
     * @see #canUngroupSelection
     */
    ungroupSelection(group?: Group): void;
    /**
     * This predicate controls whether or not the user can invoke the #ungroupSelection command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Group=} group if supplied, ignore the selection and consider ungrouping this particular Group.
     * @return {boolean}
     * This returns true:
     * if the diagram is not Diagram#isReadOnly,
     * if Diagram#allowDelete is true,
     * if Diagram#allowUngroup is true,
     * if Diagram#model is a GraphLinksModel, and
     * if there are any selected Groups that are Group#ungroupable.
     * @see #ungroupSelection
     */
    canUngroupSelection(group?: Group): boolean;
    /**
     * Make sure all of the unnested Parts in the given collection
     * are removed from any containing Groups.
     *
     * This sets Part#containingGroup to null on each Part that is
     * not a member of another Part in the argument collection.
     * If the check argument to this method is supplied and true,
     * this will call #isValidMember on each part, passing null as the first argument.
     *
     * This functionality is similar to #ungroupSelection,
     * except that this is not a command (there is no transaction and this does not raise a DiagramEvent)
     * and the parts are necessarily becoming top-level parts
     * (whereas ungrouping would add them to the Group containing the Group being ungrouped).
     *
     * This function is typically called in a Diagram#mouseDrop event handler
     * in order to remove the selected Parts from whatever Group they had been in.
     *
     * If you want to add Parts to be members of a Group, call Group#addMembers.
     * If you want to remove Parts completely from a Diagram, call Diagram#removeParts.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Iterable.<Part>} coll a collection of Parts.
     * @param {boolean=} check whether to call #isValidMember to confirm that changing the Part to be a top-level Part is valid.
     * @return {boolean} true if all non-Links were changed to be top-level Parts in this Diagram; false if some Parts or Nodes were not able to be added.
     */
    addTopLevelParts(coll: Iterable<Part>, check?: boolean): boolean;
    /**
     * This command collapses all expanded selected Groups.
     * This operation is performed within a "Collapse SubGraph" transaction.
     * Just before the end of the transaction this raises the "SubGraphCollapsed" DiagramEvent,
     * with a collection of collapsed Groups as the subject.
     * This currently has no default keyboard shortcut.
     *
     * This calls Group#collapseSubGraph to perform the collapse, which will set Group#isSubGraphExpanded to false.
     * You may want to save the collapsed/expanded state to the model by using a TwoWay Binding on the "isSubGraphExpanded" property of your Groups,
     * and perhaps also on the Group#wasSubGraphExpanded property.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Group=} group if supplied, ignore the selection and collapse this particular Group.
     * @see #canCollapseSubGraph
     */
    collapseSubGraph(group?: Group): void;
    /**
     * This predicate controls whether the user can collapse expanded Groups.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Group=} group if supplied, ignore the selection and consider collapsing this particular Group.
     * @return {boolean}
     * @see #collapseSubGraph
     */
    canCollapseSubGraph(group?: Group): boolean;
    /**
     * This command expands all collapsed selected Groups.
     * This operation is performed within an "Expand SubGraph" transaction.
     * Just before the end of the transaction this raises the "SubGraphExpanded" DiagramEvent,
     * with a collection of expanded Groups as the subject.
     * This currently has no default keyboard shortcut.
     *
     * This calls Group#expandSubGraph to perform the collapse, which will set Group#isSubGraphExpanded to true.
     * You may want to save the collapsed/expanded state to the model by using a TwoWay Binding on the "isSubGraphExpanded" property of your Groups,
     * and perhaps also on the Group#wasSubGraphExpanded property.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Group=} group if supplied, ignore the selection and expand this particular Group.
     * @see #canExpandSubGraph
     */
    expandSubGraph(group?: Group): void;
    /**
     * This predicate controls whether the user can expand collapsed Groups.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Group=} group if supplied, ignore the selection and consider expanding this particular Group.
     * @return {boolean}
     * @see #expandSubGraph
     */
    canExpandSubGraph(group?: Group): boolean;
    /**
     * This command starts in-place editing of a TextBlock in the selected Part.
     * This is normally invoked by the `F2` keyboard shortcut.
     *
     * This starts the TextEditingTool to have the user enter or modify the text string and finally set the TextBlock#text.
     * You may want to save the new string to the model by using a TwoWay Binding on the "text" property of your TextBlock.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {TextBlock=} textblock the TextBlock to start editing.
     * @see #canEditTextBlock
     */
    editTextBlock(textblock?: TextBlock): void;
    /**
     * This predicate controls whether or not the user can invoke the #editTextBlock command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {TextBlock=} textblock the TextBlock to consider editing.
     * @return {boolean}
     * This returns true:
     * if the diagram is not Diagram#isReadOnly,
     * if Diagram#allowTextEdit is true,
     * if there is a ToolManager#textEditingTool, and
     * if there is any selected Part for which Part#canEdit is true.
     * @see #editTextBlock
     */
    canEditTextBlock(textblock?: TextBlock): boolean;
    /**
     * This command opens the context menu for a selected Part or given GraphObject, or else for the whole Diagram.
     * This is normally invoked by the `Menu` keyboard shortcut.
     *
     * The given GraphObject must have a GraphObject#contextMenu
     * defined in order to show anything.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {GraphObject|Diagram=} obj a GraphObject or Diagram with a contextMenu defined.
     * If none is given, this method will use the first selected object, or else the Diagram.
     * The method will simulate a right-button click at the middle of the GraphObject or,
     * if a Diagram, at the current mouse position if it is in the viewport.
     * @see #canShowContextMenu
     */
    showContextMenu(obj?: GraphObject | Diagram): void;
    /**
     * This predicate controls whether or not the user can invoke the #showContextMenu command.
     *
     * This method may be overridden, but you should consider calling this base method in order to get all of its functionality.
     * This method must not have any side-effects.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {GraphObject|Diagram=} obj a GraphObject or Diagram with a contextMenu defined.
     * If none is given, this method will use the first selected object, or else the Diagram.
     * @return {boolean}
     * @see #showContextMenu
     */
    canShowContextMenu(obj?: GraphObject | Diagram): boolean;
    /**
     * Gets or sets whether #copySelection should also copy Links that connect with selected Nodes.
     * The default value is true.
     * Setting this property does not raise any events.
     *
     * The DraggingTool#copiesEffectiveCollection property serves a similar role for the DraggingTool
     * when the user holds down the control key to modify the drag into a copy operation.
     * @since 1.5
     */
    copiesConnectedLinks: boolean;
    /**
     * Gets or sets whether #deleteSelection should also delete links that are connected to nodes that are deleted.
     * The default value is true.
     * Setting this property does not raise any events.
     * @since 1.6
     */
    deletesConnectedLinks: boolean;
    /**
     * Gets or sets whether #copySelection should also copy subtrees.
     * The default value is false.
     * Setting this property does not raise any events.
     *
     * The DraggingTool#dragsTree property serves a similar role for the DraggingTool
     * for both moving and copying operations.
     */
    copiesTree: boolean;
    /**
     * Gets or sets whether #deleteSelection should also delete subtrees.
     * The default value is false.
     * Setting this property does not raise any events.
     */
    deletesTree: boolean;
    /**
     * Gets or sets whether #copySelection and #copyToClipboard copy the node data property
     * whose value is the tree-parent node data's key.
     *
     * Set this property to true if you want a copy/paste of a node to automatically have the new node be a tree-child of the original tree-parent node.
     * Caution: this only has an effect if the Diagram's Diagram#model is a TreeModel.
     *
     * The default value is false.
     * @since 1.3
     */
    copiesParentKey: boolean;
    /**
     * Gets or sets whether #copySelection and #copyToClipboard copy the node data property
     * whose value is the containing group data's key.
     *
     * Set this property to true if you want a copy/paste of a node to automatically have the new node be a member of the original group.
     * Caution: this only has an effect if the Diagram's Diagram#model is a GraphLinksModel.
     *
     * The default value is false.
     * @since 1.3
     */
    copiesGroupKey: boolean;
    /**
     * Gets or sets a data object that is copied by #groupSelection
     * when creating a new Group.
     *
     * The default value is null.
     * If you set this to an Object, be sure that GraphLinksModel#isGroupForNodeData is true for that object.
     * Setting this property does not raise any events.
     */
    archetypeGroupData: ObjectData;
    /**
     * Gets or sets the predicate that determines whether or not a node may become a member of a group.
     * This predicate is called in addition to any existing group's Group#memberValidation predicate.
     * The default predicate is null, which is equivalent to simply returning true.
     * The predicate may be called passing null as the first argument (the Group) --
     * this asks whether it is OK to make the second argument (the Part, but not a Link) a top-level Part of the diagram.
     * <p class="box">
     * For a more general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
     *
     * The function, if supplied, must not have any side-effects.
     */
    memberValidation: ((aGroup: Group, somePart: Part) => boolean) | null;
    /**
     * Deprecated in favor of Diagram#defaultScale.
     *
     * Gets or sets the Diagram#scale set by #resetZoom.
     *
     * The default value is 1.0.
     * The value must be a number larger than 0.
     * Setting this property does not raise any events.
     * @since 1.3
     */
    defaultScale: number;
    /**
     * Gets or sets the amount by which #decreaseZoom and #increaseZoom change
     * the Diagram#scale.
     *
     * The default value is 1.05 (5%).
     * The value must be a number larger than 1.0.
     * Setting this property does not raise any events.
     * @since 1.1
     */
    zoomFactor: number;
    /**
     * Find the actual collection of nodes and links to be moved or copied,
     * given an initial collection.
     * This includes links that connected at both ends to nodes being moved or copied,
     * members of Groups, and if DraggingTool#dragsTree is true,
     * this includes nodes and links that are "tree" descendants from selected nodes.
     *
     * Note that this does not return a simple collection of Parts,
     * but a Map associating a chosen Part
     * with an Object holding its original location Points as the value of the "point" property.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * This was moved to CommandHandler from DraggingTool in 2.0.
     * @expose
     * @param {Iterable.<Part>} parts A Set or List of Parts.
     * @param {DraggingOptions=} options Potential options for the collection computation. If not specified, this uses the DraggingTool's drag options.
     * @return {Map.<Part,DraggingInfo>}  a Map mapping Parts to DraggingInfo Objects
     *         that have a "point" property remembering the original location of that Part.
     * @since 2.0
     */
    computeEffectiveCollection(parts: Iterable<Part>, options?: DraggingOptions): Map<Part, DraggingInfo>;
}
/**
 * A type that describes a constructor.  Used in the declaration of GraphObject.make.
 */
export type ConstructorType<T extends new (...args: Array<any>) => InstanceType<T>> = new (...args: Array<any>) => InstanceType<T>;
/**
 * A type that declares which types may be passed as arguments, depending on the type of instance being built.
 * Used in the declaration of GraphObject.make.
 */
export type MakeAllow<CT extends ConstructorType<CT>, C, E> = (InstanceType<CT> extends C ? E : never);
/**
 * This is the abstract base class for all graphical objects.
 * Classes inheriting from GraphObject include:
 * Shape, TextBlock, Picture, and Panel.
 * From the Panel class the Part class is derived, from which the
 * Node and Link classes derive.
 *
 * It is very common to make use of the static function GraphObject.make in order to build up
 * a visual tree of GraphObjects.  You can see many examples of this throughout the
 * Introduction, starting at <a href="../../intro/buildingObjects.html">Building Objects</a>,
 * and the Samples, starting with <a href="../../samples/minimal.html">Minimal Sample</a>.
 *
 * Since GraphObject is an abstract class, programmers do not create GraphObjects themselves,
 * but this class defines many properties used by all kinds of GraphObjects.
 *
 * The only visual properties on GraphObject are #background and #areaBackground.
 * However one can control whether the GraphObject is drawn at all by setting #visible,
 * or by setting #opacity to zero if you still want the GraphObject to occupy space.
 * Also, if you want to control whether any mouse or touch events "see" the GraphObject,
 * you can set #pickable to false.
 *
 * For more information about specifying how things get drawn, see the properties on the
 * Shape, TextBlock, and Picture classes.
 * <h3>GraphObject Sizing</h3>
 *
 * GraphObject defines most of the properties that cause objects to size themselves differently.
 * The most prominent ones include:
 *   - The #desiredSize, #minSize, and #maxSize properties are used to
 *     explicitly set or limit the size of visual elements. #width and #height are
 *     convenience properties that set the #desiredSize width and height, respectively.
 *   - The #angle and #scale properties are used to transform visual elements.
 *   - The #stretch property determines how a GraphObject will fill its visual space,
 *     contextually granted to it by its containing Panel. Top-level (Part)
 *     GraphObjects are not affected by this property because they are always granted infinite space.
 *
 * All GraphObjects in a Diagram are measured and then arranged by their containing Panels in a tree-like fashion.
 * After measuring and arranging, a GraphObject will have valid values for the read-only
 * properties #naturalBounds, #measuredBounds, and #actualBounds.
 *
 *   - The #naturalBounds of a GraphObject describe its local size,
 *     without any transformations (#scale, #angle) affecting it.
 *   - The #measuredBounds of a GraphObject describe its size within its containing Panel.
 *   - The #actualBounds of a GraphObject describe its position and given size inside of its panel.
 *     This size may be smaller than #measuredBounds, for instance if a GraphObject with a large #desiredSize
 *     is placed in a Panel of a smaller #desiredSize. Smaller #actualBounds than #measuredBounds
 *     typically means an object will be cropped.
 *
 * <p class="boxread">
 * See <a href="../../intro/sizing.html">the Introduction page on sizing</a>
 * for usage information and examples.
 * <h3>GraphObject Size and Position within Panel</h3>
 * Several GraphObject properties guide the containing Panel for how to size and position the object within the panel.
 *   - The #alignment specifies where the object should be relative to some area of the panel.
 *     For example, an alignment value of Spot.BottomRight means that the GraphObject should be at the bottom-right corner of the panel.
 *   - The #alignmentFocus specifies precisely which point of the GraphObject should be aligned at the #alignment spot.
 *   - The #column and #row properties are only used by Panel.Table panels, to indicate where the GraphObject should be.
 *   - The #columnSpan and #rowSpan properties tell the Panel.Table panel how large the GraphObject should be.
 *   - The #isPanelMain property indicates to some kinds of Panels that the GraphObject is the "primary" object
 *     that other panel children should be measured with or positioned in.
 *   - The #margin property tells the containing Panel how much extra space to put around this GraphObject.
 *   - The #position property is used to determine the relative position of GraphObjects when they are elements of a Panel.Position panel.
 *
 * <p class="boxread">
 * See <a href="../../intro/panels.html">the Introduction page on Panels</a>
 * and <a href="../../intro/tablePanels.html">Table Panels</a> for an overview of the capabilities.
 * <h3>Top-level GraphObjects are Parts</h3>
 *
 * A Part is a derived class of GraphObject representing a top-level object.
 * All top-level GraphObjects must be Parts, and Node, Link, Group, and Adornment derive from Part.
 * The position of a Part determines the point of the Part's top-left corner in document coordinates.
 * See also Part#location, which supports an way to specify the position based on a different
 * spot of a different element within the Part.
 *
 * There are several read-only properties that help navigate up the visual tree.
 *   - #panel returns the Panel that directly contains this GraphObject
 *   - #part returns the Part that this GraphObject is in, perhaps via intervening Panels;
 *     this is frequently used in order to get to the model data, Panel#data
 *   - #layer returns the Layer that this GraphObject's Part is in
 *   - #diagram returns the Diagram that this GraphObject's Part's Layer is in
 *
 * <p class="boxrun">
 * See <a href="../../samples/visualTree.html">the Visual Tree sample</a>
 * for a diagram displaying the visual tree of a simple diagram.
 * <h3>User Interaction</h3>
 *
 * GraphObjects have several properties enabling dynamic customizable interaction.
 * There are several definable functions that execute on input events: #mouseDragEnter,
 * #mouseDragLeave, #mouseDrop, #mouseEnter, #mouseHold,
 * #mouseHover, #mouseLeave, and #mouseOver.
 * For example, you could define mouse enter-and-leave event handlers to modify the appearance of a link
 * as the mouse passes over it:
 * ```js
 * myDiagram.linkTemplate =
 *   $(go.Link,
 *     $(go.Shape,
 *       { strokeWidth: 2, stroke: "gray" },  // default color is "gray"
 *       { // here E is the InputEvent and OBJ is this Shape
 *         mouseEnter: function(e, obj) { obj.strokeWidth = 4; obj.stroke = "dodgerblue"; },
 *         mouseLeave: function(e, obj) { obj.strokeWidth = 2; obj.stroke = "gray"; }
 *       }));
 * ```
 *
 * There are #click, #doubleClick, and #contextClick functions
 * that execute when a user appropriately clicks the GraphObject.
 * These click functions are called with the InputEvent as the first argument
 * and this GraphObject as the second argument.
 * For example, you could define a click event handler on a Node that goes to another page:
 * ```js
 * myDiagram.nodeTemplate =
 *   $(go.Node, "Auto",
 *     $(go.Shape, "RoundedRectangle",
 *       new go.Binding("fill", "color")),
 *     $(go.TextBlock,
 *       { name: "TB", margin: 3 },
 *       new go.Binding("text", "key")),
 *     { // second arg will be this GraphObject, which in this case is the Node itself:
 *       click: function(e, node) {
 *         window.open("https://en.wikipedia.org/Wiki/" + node.data.key);
 *       }
 *     });
 * ```
 *
 * Note: you may prefer defining DiagramEvent listeners on the Diagram
 * rather than on individual GraphObjects.  DiagramEvents also include more general events that
 * do not necessarily correspond to input events.
 *
 * The properties #actionCancel, #actionDown, #actionMove,
 * and #actionUp define functions to execute when the GraphObject's #isActionable property
 * is set to true (default false). See the ActionTool for more detail.
 * <p class="boxread">
 * See <a href="../../intro/events.html">the Introduction page on Events</a> for a more general discussion.
 * <h3>GraphObjects as Ports</h3>
 *
 * In GoJS, Links can only connect to elements within a Node
 * that are specified as "ports", and by default the only port is the Node itself.
 * Setting the #portId of a GraphObject inside a Node allows that object to act as a port.
 * Note: the only kind of model that can save which port a link is connected with, i.e. portIds that are not an empty string,
 * is a GraphLinksModel whose GraphLinksModel#linkFromPortIdProperty and
 * GraphLinksModel#linkToPortIdProperty have been set to name properties on the link data objects.
 *
 * GraphObjects have several properties that are only relevant when they are acting as ports.
 * These port-related properties are:
 *   - #portId, which must be set to a string that is unique within the Node,
 *     in order for this GraphObject to be treated as a "port", rather than the whole node
 *   - #fromSpot and #toSpot, where a link should connect with this port
 *   - #fromEndSegmentLength and #toEndSegmentLength, the length of the link segment adjacent to this port
 *   - #fromShortLength and #toShortLength, the distance the link should terminate before touching this port
 *   - #fromLinkable and #toLinkable, whether the user may draw links connecting with this port
 *   - #fromLinkableDuplicates and #toLinkableDuplicates, whether the user may draw multiple links between the same pair of ports
 *   - #fromLinkableSelfNode and #toLinkableSelfNode, whether the user may draw a link between ports on the same node
 *   - #fromMaxLinks and #toMaxLinks, to limit the number of links connecting with this port in a particular direction
 *
 * <p class="boxread">
 * See <a href="../../intro/ports.html">the Introduction page on ports</a>
 * and <a href="../../intro/links.html">link routing</a>
 * and <a href="../../intro/connectionPoints.html">link connection points</a>
 * for port usage information and examples.
 * <h3>GraphObjects as labels on a Link</h3>
 *
 * GraphObjects can also be used as "labels" on a Link.
 * In addition to the #alignmentFocus property, these properties direct a Link Panel
 * to position a "label" at a particular point along the route of the link, in a particular manner:
 *   - #segmentIndex, which segment the label should be on
 *   - #segmentFraction, how far along the segment the label should be
 *   - #segmentOffset, where the label should be positioned relative to the segment
 *   - #segmentOrientation, how the label should be rotated relative to the angle of the segment
 *
 * <p class="boxread">
 * See <a href="../../intro/linkLabels.html">the Introduction page on link labels</a>
 * for examples of how to make use of labels on Links.
 * <h3>Interactive Behavior</h3>
 *
 * There are several properties that specify fairly high-level interactive behavior:
 *   - #cursor, a CSS string specifying a cursor
 *   - #contextMenu, an Adornment
 *   - #toolTip, an Adornment
 *
 * <p class="boxread">
 * For more information, please read <a href="../../intro/contextMenus.html">the Introduction page about Context Menus</a>
 * and <a href="../../intro/toolTips.html">the page about ToolTips</a>.
 * <p class="boxrun">
 * Also see <a href="../../samples/basic.html">the Basic sample</a>
 * for examples of how to show context menus and tooltips.
 * @unrestricted
 */
export abstract class GraphObject {
    /**
     * This is an abstract class, so you should not use this constructor.
     */
    constructor();
    /**
     * Copies properties from this object to the given object, which must be of the same class.
     * This is called by #copy.
     * This method may be overridden.
     *
     * For every property that you add to a subclass of a GraphObject-inheriting class,
     * in this method you should assign its value to the copied object.
     * For performance reasons you should set all of the same properties to <code>this</code>
     * that the constructor does, and in the same order.
     *
     * For example, let us define a custom Link class and add two properties:
     *
     * ```js
     * function CustomLink() {
     *   go.Link.call(this);
     *   this._someNewProperty = 17;
     *   this._someNewProperty2 = [];  // an Array
     * }
     * go.Diagram.inherit(CustomLink, go.Link);
     *
     * CustomLink.prototype.cloneProtected = function() {
     *   // Always call the base method in an override
     *   // In TypeScript you would use: super.cloneProtected(copy);
     *   go.Link.prototype.cloneProtected.call(this, copy);
     *   // assign every new property to the copy:
     *   copy._someNewProperty = this._someNewProperty;
     *   copy._someNewProperty2 = this._someNewProperty2.slice(0);  // make a copy of the Array
     * }
     * ```
     *
     * This ensures that copies of GraphObjects and their subclasses are faithful reproductions.
     * Consider for properties that are references to objects whether the reference should be shared
     * or whether that object value should be copied as well, resulting in a less shallow copy.
     * This is demonstrated above by making a copy of the property value that is an Array,
     * so that modifications to the Array will not be shared by copies of the CustomLink.
     * Further copies of the Array items might be warranted, depending on their purpose.
     *
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {GraphObject} copy
     */
    protected cloneProtected(copy: this): void;
    /**
     * Creates a deep copy of this GraphObject and returns it.
     * This method is the same as a clone for simple GraphObjects
     * such as Shape, TextBlock, and Picture.
     * For Panel this method copies the visual tree of GraphObjects that it contains.
     * @expose
     * @return {GraphObject}
     */
    copy(): this;
    /**
     * GraphObjects with this enumeration as the value of GraphObject#stretch
     * are not automatically scaled to fit in the given bounds;
     * there may be clipping in one or both directions if the available dimensions are too small.
     * @constant
     */
    static None: EnumValue;
    /**
     * GraphObjects with this enumeration as the value of GraphObject#stretch
     * are stretched depending on the context they are used. For instance a 'Circle' figure might
     * be uniformly stretched whereas an 'Ellipse' figure might be non-uniformly stretched.
     * @constant
     */
    static Default: EnumValue;
    /**
     * GraphObjects with this enumeration as the value of GraphObject#stretch
     * are scaled as much as possible in the y-axis. In another context, can be used as a value of PanelLayout, so type is "any".
     * @constant
     */
    static Vertical: any;
    /**
     * GraphObjects with this enumeration as the value of GraphObject#stretch
     * are scaled as much as possible in the x-axis. In another context, can be used as a value of PanelLayout, so type is "any".
     * @constant
     */
    static Horizontal: any;
    /**
     * GraphObjects with this enumeration as the value of GraphObject#stretch
     * are scaled in both directions so as to fit exactly in the given bounds;
     * there is no clipping but the aspect ratio may change, causing the object to appear stretched.
     * @constant
     */
    static Fill: EnumValue;
    /**
     * Pictures with this enumeration as the value of Picture#imageStretch are drawn with equal
     * scale in both directions to fit the arranged (actual) bounds;
     * Panels of type Viewbox with this as the value of Panel#viewboxStretch
     * are scaled equally in both directions to fit in the given bounds.
     * @constant
     */
    static Uniform: EnumValue;
    /**
     * Pictures with this enumeration as the value of Picture#imageStretch are drawn with equal
     * scale in both directions to fit the larger side of the image bounds;
     * Panels of type Viewbox with this as the value of Panel#viewboxStretch
     * are scaled equally in both directions to fit the larger side of the element's bounds.
     * there may be clipping in one dimension.
     * @constant
     */
    static UniformToFill: EnumValue;
    /**
     * GraphObjects with this enumeration as the value of Picture#flip or TextBlock#flip
     * are drawn upside-down, with Y coordinate points increasing upwards.
     * @constant
     * @since 2.0
     */
    static FlipVertical: EnumValue;
    /**
     * GraphObjects with this enumeration as the value of Picture#flip or TextBlock#flip
     * are drawn mirror-image, with X coordinate points increasing towards the left.
     * @constant
     * @since 2.0
     */
    static FlipHorizontal: EnumValue;
    /**
     * GraphObjects with this enumeration as the value of Picture#flip or TextBlock#flip
     * are drawn with both X and Y coordinates reversed.
     * @constant
     * @since 2.0
     */
    static FlipBoth: EnumValue;
    /**
     * Returns the Point in document coordinates for a given Spot in this object's bounds
     * or for a Point in local coordinates.
     *
     * For example, for an instance of a Node like this:
     * ```js
     * myDiagram.nodeTemplate =
     *   $(go.Node, "Auto",
     *     $(go.Shape, "RoundedRectangle",
     *       new go.Binding("fill", "color")),
     *     $(go.TextBlock,
     *       { name: "TB", margin: 3 },
     *       new go.Binding("text", "key")));
     * ```
     * where the Node is positioned at `100, 200`,
     * ```js
     *   node.findObject("TB").getDocumentPoint(go.Spot.Center)
     * ```
     * could return a Point that is approximately at `122, 213`.
     * @param {Spot|Point} local a real Spot describing a relative location in or near this GraphObject, or a real Point in local coordinates.
     * @param {Point=} result an optional Point that is modified and returned.
     * @return {Point} in document coordinates.
     * @see #getDocumentBounds
     * @see #getLocalPoint
     */
    getDocumentPoint(local: Spot | Point, result?: Point): Point;
    /**
     * Returns the Rect in document coordinates for this object's bounds.
     * If this GraphObject is a Part, the rect will be identical to its #actualBounds.
     * @param {Rect=} result an optional Rect that is modified and returned.
     * @return {Rect} in document coordinates.
     * @see #getDocumentPoint
     * @since 2.0
     */
    getDocumentBounds(result?: Rect): Rect;
    /**
     * Returns the effective angle that the object is drawn at, in document coordinates, normalized to between 0 and 360.
     *
     * Basically this adds together all of the rotation declared by this #angle and the angles of all of its containing Panels, including the Part.
     * @return {number}
     */
    getDocumentAngle(): number;
    /**
     * Returns the total scale that the object is drawn at, in document coordinates.
     *
     * Basically this multiplies together this #scale with the scales of all of its containing Panels, including the Part.
     * @return {number}
     */
    getDocumentScale(): number;
    /**
     * Given a Point in document coordinates, returns a new Point in local coordinates.
     *
     * For example, if you have a mouse event whose InputEvent#documentPoint is at `122, 213`,
     * and if you have a Node whose #position is at `100, 200`,
     * `node.getLocalPoint(e.documentPoint)` could return a Point that is at `22, 13`.
     * For a GraphObject within the Node named "TB",
     * ```js
     *   node.findObject("TB").getLocalPoint(e.documentPoint)
     * ```
     * could return a Point that is at `15.7, 6.7`,
     * if that "TB" object is positioned somewhat inside the bounds of the Node.
     * @param {Point} p a real Point in document coordinates.
     * @param {Point=} result an optional Point that is modified and returned.
     * @return {Point} The corresponding Point in local coordinates.
     * @see #getDocumentPoint
     */
    getLocalPoint(p: Point, result?: Point): Point;
    /**
     * Gets or sets whether or not this GraphObject will be shadowed inside a Part that has Part#isShadowed set to true.
     *
     * The default is null, which means this GraphObject will obey the default shadow rules (see Part#isShadowed).
     *
     * A value of true or false will ensure that this part is shadowed or not regardless of the default shadow rules,
     * but this GraphObject's shadowed status will not affect other GraphObjects in the Part.
     *
     * Typically this property does not need to be set, but you may need to set this value to false
     * on GraphObjects inside a Part that you do not wish to be shadowed.
     * @see Part#isShadowed
     * @since 1.6
     */
    shadowVisible: boolean | null;
    /**
     * This predicate is true if this object is an element, perhaps indirectly,
     * of the given panel.
     *
     * For example, if this GraphObject is inside a Part but is not itself the Part,
     * `obj.isContainedBy(obj.part)` should be true.
     * @param {GraphObject} panel
     * @return {boolean} true if this object is contained by the given panel,
     * or if it is contained by another panel that is contained by the given panel,
     * to any depth; false if the argument is null or is not a Panel.
     */
    isContainedBy(panel: GraphObject): boolean;
    /**
     * This predicate is true if this object is #visible
     * and each of its visual containing panels are also visible.
     * This ignores the actual location or appearance (except visibility)
     * of the panel that this object is part of, as well as ignoring all
     * properties of the Layer or Diagram.
     *
     * For Parts, you can call the Part#isVisible predicate to determine
     * if not only the Part is visible but also any containing Group or Link or Layer.
     * @return {boolean}
     */
    isVisibleObject(): boolean;
    /**
     * This predicate is false if this object is inside any Panel that is not Panel#isEnabled,
     * or if this is itself a disabled panel.
     * This ignores the #visible and #pickable properties.
     * @return {boolean}
     * @see #enabledChanged
     * @see Panel#isEnabled
     * @since 1.7
     */
    isEnabledObject(): boolean;
    /**
     * Gets or sets the function to execute when some containing Panel changes the value of Panel#isEnabled.
     * It is typically used to modify the appearance of the object.
     * This function must not change the value of any panel Panel#isEnabled.
     *
     * If this property value is a function, it is called with two arguments,
     * this GraphObject and the new value.
     * By default this property is null -- no function is called.
     * @see #isEnabledObject
     * @see Panel#isEnabled
     * @since 1.7
     */
    enabledChanged: ((thisObj: GraphObject, enabled: boolean) => void) | null;
    /**
     * Gets or sets the orientation of a GraphObject that is in a Link or Panel.Graduated.
     * This controls the automatic rotation of the object by the Link Panel or Graduated Panel.
     * The only accepted values are the Link "Orient..." values of Link
     * and the default value: Link.None.
     *
     * When the value is Link.None, the #angle of this object is unchanged as the link is routed.
     * Setting this to a value of Link.OrientAlong will cause routing to set the #angle
     * to be the angle of the segment that this object is on.
     * Other values compute the angle somewhat differently.
     * If the value is changed back to Link.None, the #angle of this object is set to zero.
     *
     * Note that when this property is not Link.None, this property takes precedence
     * over any setting or binding of the #angle property.
     * Changes to the angle caused by orientation might not result in Changed events,
     * and any original value for the angle may be lost.
     *
     * For examples of how to use this property, see <a href="../../intro/linkLabels.html">Link Labels</a>.
     * @see #segmentFraction
     * @see #segmentIndex
     * @see #segmentOffset
     */
    segmentOrientation: EnumValue;
    /**
     * Gets or sets the segment index of a GraphObject that is in a Link.
     * Non-negative numbers count up from zero, which is the first segment, at the "from" end of the Link.
     * Negative numbers count segments from the "to" end of the Link, where -1 means the last segment
     * and -2 means the next-to-last segment.
     * The value should be an integer.
     *
     * If you do not set this property, the Link will choose a place that is approximately at the
     * mid-point of the link's route.
     *
     * For examples of how to use this property, see <a href="../../intro/linkLabels.html">Link Labels</a>.
     * @see #segmentFraction
     * @see #segmentOffset
     * @see #segmentOrientation
     */
    segmentIndex: number;
    /**
     * Gets or sets the fractional distance along a segment of a GraphObject that is in a Link.
     * The value should be between zero and one, where zero is at the point at the start of the segment,
     * and where one is at the point at the end of the segment.
     * The default value is zero.
     *
     * For examples of how to use this property, see <a href="../../intro/linkLabels.html">Link Labels</a>.
     * @see #segmentIndex
     * @see #segmentOffset
     * @see #segmentOrientation
     */
    segmentFraction: number;
    /**
     * Gets or sets the offset of a GraphObject that is in a Link from a point on a segment
     * or in a Panel.Graduated from a point along the main element.
     * The X component of the Point indicates the distance along the route,
     * with positive values going further toward the "to" end of the link or panel.
     * The Y component of the Point indicates the distance away from the route,
     * with positive values towards the right as seen when facing further towards the "to" end of the link or panel.
     * The value defaults to the Point (0, 0).
     * You cannot modify the x or y of the value of this property --
     * if you want to change the segmentOffset you must set this property to a different Point.
     *
     * For labels that are near either end of a link, it may be convenient to set the segmentOffset
     * to Point(NaN, NaN). This causes the offset to be half the width and half the height of the label object.
     *
     * For examples of how to use this property, see <a href="../../intro/linkLabels.html">Link Labels</a>.
     * @see #segmentFraction
     * @see #segmentIndex
     * @see #segmentOrientation
     */
    segmentOffset: Point;
    /**
     * Gets or sets the stretch of the GraphObject.
     * This controls whether the width and/or height of this object automatically adjusts to fill
     * the area allotted by the containing Panel.
     *
     * The only accepted values are listed as constant properties of GraphObject,
     * such as GraphObject.None, GraphObject.Fill, GraphObject.Horizontal, or GraphObject.Vertical.
     * The default value is GraphObject.Default, which allows the Panel to decide how to treat this object, depending on the type of Panel.
     *
     * Objects with an #angle that are stretched may look incorrect unless the angle is a multiple of 90.
     *
     * Stretch will have have different effects based upon the Panel containing this object. Elements of:
     * - Auto panels will not stretch, except the main element growing to fill the panel or being made uniform
     * - Horizontal panels will only stretch vertically
     * - Vertical panels will only stretch horizontally
     * - Spot panels will stretch to the size of the main element
     * - Table panels will stretch to the size of their cell, defined by their row and column, which is usually determined by other GraphObjects in that cell that are not stretching
     * - Grid panels, Link panels, and Graduated panels will not stretch
     * @see Panel#defaultStretch
     * @see #desiredSize
     * @see #minSize
     * @see #maxSize
     * @see #measuredBounds
     * @see #actualBounds
     * @see #scale
     * @see Picture#imageStretch
     */
    stretch: EnumValue;
    /**
     * Gets or sets the name for this object.
     * The default value is the empty string.
     * The name should be unique within a Panel, although if it isn't,
     * it reduces the usefulness of methods such as Panel#findObject.
     *
     * You must not modify the name of a GraphObject once it is in the visual tree of a Part.
     *
     * This is frequently needed to identify a particular GraphObject in the visual tree of a Part,
     * for example as the value of the Part#locationObjectName or
     * Part#selectionObjectName properties.
     */
    name: string;
    /**
     * Gets or sets the multiplicative opacity for this GraphObject and (if a Panel) all elements.
     * The value must be between 0.0 (fully transparent) and 1.0 (no additional transparency).
     *
     * Unlike #visible, Opacity only affects drawing, it does not cause objects to be resized or remeasured.
     * Opacity settings do not change the shape of the object or exclude it from object-picking
     * (does not change whether any objects are found by the "find..." methods).
     *
     * This value is multiplicative with any existing transparency,
     * for instance from Layer#opacity or a GraphObject's opacity higher in the visual tree,
     * or from a Brush or image transparency.
     * The default value is 1.
     * @see #visible
     * @see #pickable
     * @see Layer#opacity
     * @since 1.4
     */
    opacity: number;
    /**
     * Gets or sets whether a GraphObject is visible.
     * The default value is true.
     * A not visible object takes no space in the Panel that it is in.
     * Toggling visibility may cause elements in the visual tree to re-measure and re-arrange.
     * Making a Panel not visible causes all of its elements not to be seen or receive input events.
     * Changing a Panel to become visible causes all of its elements to be seen and be active,
     * unless those elements are themselves not visible.
     *
     * This object does not get any mouse/touch events if it is not #visible
     * or if it is not #pickable.
     *
     * One can have a visible Shape that is not drawn by setting its Shape#fill
     * and Shape#stroke to null or to `"transparent"`.
     * Similarly, one can set TextBlock#stroke to null or to `"transparent"`.
     * It is also possible make a GraphObjects transparent by setting GraphObject#opacity to 0.
     * Finally, one can make a whole Layer-full of Parts invisible by setting Layer#visible to false.
     *
     * Use the #isVisibleObject predicate to see if this GraphObject is visible and is inside a Panel
     * that is #isVisibleObject, and so forth up the chain of panels until reaching the Part.
     *
     * For Parts, you can call the Part#isVisible predicate to determine
     * if not only the Part is visible but also any containing Group or Link,
     * and whether the Layer it is in is visible.
     * @see #opacity
     * @see #pickable
     * @see Layer#visible
     */
    visible: boolean;
    /**
     * Gets or sets whether or not this GraphObject can be chosen by visual "find" or "hit-test" methods such
     * as Diagram#findObjectAt.
     *
     * This object does not get any mouse/touch events if it is not #visible
     * or if it is not #pickable.
     *
     * The default value is true -- mouse events on this object will be noticed.
     * If this value is false and this object is a Panel, not only is this Panel not "hittable",
     * but all of the elements inside the Panel will be ignored.
     * @see #visible
     * @see #opacity
     * @see Layer#pickable
     * @see Panel#isEnabled
     * @since 1.2
     */
    pickable: boolean;
    /**
     * Gets or sets whether the user may draw duplicate Links from this port.
     * This property is used by LinkingBaseTool#isValidLink.
     * The default value is false.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see #fromLinkable
     * @see #fromLinkableSelfNode
     * @see #toLinkableDuplicates
     */
    fromLinkableDuplicates: boolean;
    /**
     * Gets or sets whether the user may draw Links that connect from this port's Node.
     * This property is used by LinkingBaseTool#isValidLink.
     * The default value is false.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see #fromLinkable
     * @see #fromLinkableDuplicates
     * @see #toLinkableSelfNode
     */
    fromLinkableSelfNode: boolean;
    /**
     * Gets or sets whether the user may draw duplicate Links to this port.
     * This property is used by LinkingBaseTool#isValidLink.
     * The default value is false.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see #toLinkable
     * @see #toLinkableSelfNode
     * @see #fromLinkableDuplicates
     */
    toLinkableDuplicates: boolean;
    /**
     * Gets or sets whether the user may draw Links that connect to this port's Node.
     * This property is used by LinkingBaseTool#isValidLink.
     * The default value is false.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see #toLinkable
     * @see #toLinkableDuplicates
     * @see #fromLinkableSelfNode
     */
    toLinkableSelfNode: boolean;
    /**
     * Gets or sets whether a GraphObject is the "main" object for some types of Panel.
     * Panels that use a "main" object include Panel.Auto, Panel.Spot, and Panel.Link.
     *
     * Panels that use a "main" object will use the first object that has this property set to true,
     * or else just the first object, if none have the property set.
     *
     * Do not modify this property once this object is an element of a panel.
     */
    isPanelMain: boolean;
    /**
     * This property determines whether or not this GraphObject's events occur
     * before all other events, including selection. This enables the #actionDown,
     * #actionMove, #actionUp, and #actionCancel events,
     * which are all handled by the ActionTool.
     *
     * This object does not get any mouse/touch events if it is not #visible
     * or if it is not #pickable.
     *
     * This property is infrequently used -- typically only when implementing objects
     * that act as buttons or knobs or sliders.
     * The default value is false.
     * @see #actionDown
     * @see #actionMove
     * @see #actionUp
     * @see #actionCancel
     * @see #pickable
     * @see Panel#isEnabled
     */
    isActionable: boolean;
    /**
     * Gets or sets the areaBackground Brush of this GraphObject.
     * The areaBackground fills the rectangle described by this GraphObject's containing panel's coordinates.
     * If the object is rotated, the area background will expand to fill the entire measured bounds of the object,
     * without rotating the brush.
     *
     * The value may be either a Brush object or a string that is a CSS color.
     * The default value is null -- no area background is drawn.
     * More information about the syntax of CSS color strings is available at:
     * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/color">CSS colors (mozilla.org)</a>.
     * @see #background
     * @see Shape#fill
     */
    areaBackground: BrushLike;
    /**
     * Gets or sets the background Brush of this GraphObject,
     * filling the rectangle of this object's local coordinate space.
     * If the object is rotated, the background will rotate with it.
     *
     * The value may be either a Brush object or a string that is a CSS color.
     * The default value is null -- no background is drawn.
     * More information about the syntax of CSS color strings is available at:
     * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/color">CSS colors (mozilla.org)</a>.
     * @see #areaBackground
     * @see Shape#fill
     */
    background: BrushLike;
    /**
     * This read-only property returns the Part containing this object, if any.
     * The Part will be the root GraphObject in this GraphObject's visual tree.
     *
     * It is common to refer to the containing Part of a GraphObject
     * in order to refer to the Panel#data to which it is bound.
     *
     * This property is not settable.
     * If you want this GraphObject to belong to a Part, you will need to add it to a Part,
     * or else add it to some visual tree structure that is added to a Part using Panel#add.
     *
     * Note that for objects such as buttons that are in Adornments such as tooltips or context menus,
     * this property will return that Adornment, not the Node or Link that is adorned.
     *
     * If you want to find a Group that contains a Part, use the Part#containingGroup property:
     * `someObj.part.containingGroup`
     */
    readonly part: Part | null;
    /**
     * This read-only property returns the GraphObject's containing Panel, or null if this object is not in a Panel.
     *
     * Although Part inherits from this class, a Part will never belong to a Panel,
     * so this property will always be null for every Node or Link.
     *
     * This property is not settable.
     * Instead, call Panel#add in order to put a GraphObject in a Panel.
     */
    readonly panel: Panel | null;
    /**
     * This read-only property returns the GraphObject's containing Layer, if there is any.
     * A plain GraphObject cannot belong directly to a Layer -- only a Part can belong directly to a Layer.
     *
     * This property is not settable.
     * Normally one changes which Layer that a GraphObject is in by setting Part#layerName.
     * Adding a Part to a Diagram will automatically add that Part to a Layer in that Diagram based on the layerName.
     */
    readonly layer: Layer | null;
    /**
     * This read-only property returns the Diagram that this GraphObject is in, if it is.
     *
     * This property is not settable.
     * Although you cannot add any plain GraphObject to a Diagram, you can call Diagram#add
     * to add a Part to a Diagram.
     */
    readonly diagram: Diagram | null;
    /**
     * Gets or sets the position of this GraphObject in container coordinates
     * (either a Panel or the document).
     * Value must be of type Point.
     * You cannot modify the x or y of the value of this property --
     * if you want to change the position you must set this property to a different Point.
     * Default is Point(NaN, NaN).
     *
     * For Parts, see also Part#location.
     */
    position: Point;
    /**
     * This read-only property returns the bounds of this GraphObject in container coordinates. This means that
     * the actualBounds are in the coordinate space of the GraphObject's Panel,
     * unless this is a Part, in which case they are in
     * the Diagram's coordinate system.
     *
     * You must not modify any of the properties of the Rect that is the value of this property.
     *
     * If this GraphObject is a Part,
     * then the x and y values of the actualBounds are identical to that Part's #position,
     * and the width and height values of the actualBounds represent the rectangular space occupied
     * by the Part in Diagram#documentBounds coordinates.
     *
     * If this GraphObject is not a top-level object (not a Part), then the actualBounds
     * x and y values represent that GraphObject's position within its Panel. In a Panel of type Panel.Position
     * this is identical to the GraphObject's #position, but in other cases it is dependent on
     * the unique workings of each Panel type. The actualBounds width and height
     * of a GraphObject are the final size after the #scale and #angle are applied.
     *
     * It is possible for a GraphObject (be it an GraphObject or a Panel containing several more GraphObjects)
     * to have no containing Part, in which case these GraphObjects cannot possibly be in a Diagram.
     * These GraphObjects are unlikely to have real-number values for their actualBounds, as they may
     * never have had the chance to be measured and arranged.
     *
     * As with all read-only properties, using this property as a binding source is unlikely to be useful.
     * @see #measuredBounds
     * @see #desiredSize
     */
    readonly actualBounds: Rect;
    /**
     * Gets or sets the scale transform of this GraphObject.
     * Value must be a number; larger values will make this object appear bigger.
     * Default is 1.
     * @see #angle
     * @see #stretch
     */
    scale: number;
    /**
     * Gets or sets the angle transform, in degrees, of this GraphObject.
     * Value must be a number. If the value is not between (0 &lt;= value &lt; 360),
     * it will be normalized to be in that range.
     * Zero is along the positive X-axis (rightwards); 90 is along the positive Y-axis (downwards).
     * Default is 0.
     * @see #scale
     * @see #stretch
     */
    angle: number;
    /**
     * Gets or sets the desired size of this GraphObject in local coordinates.
     * Value must be of type Size.
     * Default is Size(NaN, NaN).
     * You cannot modify the width or height of the value of this property --
     * if you want to change the desiredSize you must set this property to a different Size.
     *
     * Getting or setting #width or #height is equivalent to getting or setting the
     * width or height of this property.
     *
     * The size does not include any transformation due to #scale or #angle,
     * nor any pen thickness due to Shape#strokeWidth if this is a Shape.
     * If there is a containing Panel the Panel will determine the actual size.
     * If the desiredSize is greater than the allowed size that the GraphObject's Panel determines,
     * then the GraphObject may be visually clipped. If the desiredSize does not meet the constraints
     * of #minSize and #maxSize, the GraphObject will be resized to meet them.
     * @see #minSize
     * @see #maxSize
     * @see #naturalBounds
     * @see #measuredBounds
     * @see #actualBounds
     */
    desiredSize: Size;
    /**
     * Gets or sets the desired width of this GraphObject in local coordinates.
     * This just gets or sets the width component of the #desiredSize.
     * Default is NaN.
     *
     * Size can also be constrained by setting #minSize and #maxSize.
     *
     * The width does not include any transformation due to #scale or #angle,
     * nor any pen thickness due to Shape#strokeWidth if this is a Shape.
     * If there is a containing Panel the Panel will determine the actual size.
     */
    width: number;
    /**
     * Gets or sets the desired height of this GraphObject in local coordinates.
     * This just gets or sets the height component of the #desiredSize.
     * Default is NaN.
     *
     * Size can also be constrained by setting #minSize and #maxSize.
     *
     * The height does not include any transformation due to #scale or #angle,
     * nor any pen thickness due to Shape#strokeWidth if this is a Shape.
     * If there is a containing Panel the Panel will determine the actual size.
     */
    height: number;
    /**
     * Gets or sets the minimum size of this GraphObject in container coordinates
     * (either a Panel or the document).
     * Any new value must be of type Size; NaN values are treated as 0.
     *
     * You cannot modify the width or height of the value of this property --
     * if you want to change the minSize you must set this property to a different Size.
     * The default value is zero by zero.
     * A containing Panel will determine the actual size of this object.
     * @see #maxSize
     * @see #desiredSize
     */
    minSize: Size;
    /**
     * Gets or sets the maximum size of this GraphObject in container coordinates
     * (either a Panel or the document).
     * Any new value must be of type Size; NaN values are treated as Infinity. If you want no maximum width or height, use NaN or Infinity.
     *
     * You cannot modify the width or height of the value of this property --
     * if you want to change the maxSize you must set this property to a different Size.
     * The default value is Infinity by Infinity.
     * A containing Panel will determine the actual size of this object.
     * @see #minSize
     * @see #desiredSize
     */
    maxSize: Size;
    /**
     * This read-only property returns the measuredBounds of the GraphObject in container coordinates
     * (either a Panel or the document).
     * This describes the transformed bounds with margins excluded.
     *
     * You must not modify any of the properties of the Rect that is the value of this property.
     *
     * As with all read-only properties, using this property as a binding source is unlikely to be useful.
     * @see #naturalBounds
     * @see #desiredSize
     * @see #actualBounds
     */
    readonly measuredBounds: Rect;
    /**
     * This read-only property returns the natural bounding rectangle of this GraphObject in local coordinates,
     * before any transformation by #scale or #angle,
     * and before any resizing due to #minSize or #maxSize or #stretch.
     * Defaults to unknown (NaN,NaN).
     *
     * You must not modify any of the properties of the Rect that is the value of this property.
     *
     * The value can only be changed by changing properties of the particular GraphObject,
     * such as GraphObject#desiredSize, Shape#geometry, or TextBlock#font.
     *
     * As with all read-only properties, using this property as a binding source is unlikely to be useful.
     * @see #desiredSize
     * @see #measuredBounds
     * @see #actualBounds
     */
    readonly naturalBounds: Rect;
    /**
     * Gets or sets the size of empty area around this GraphObject, as a Margin,
     * in the containing Panel coordinates.
     *
     * Negative values are permitted but may cause overlaps with adjacent
     * objects in a Panel.
     * You cannot modify the top or left or right or bottom of the value of this property --
     * if you want to change the margin you must set this property to a different Margin.
     * Default margin is Margin(0,0,0,0).
     *
     * The property setter accepts a number instead of a Margin object: providing a
     * number N will result in using a Margin(N, N, N, N).
     * The property getter will always return a Margin.
     * @see #desiredSize
     * @see #measuredBounds
     * @see #actualBounds
     * @see Panel#padding
     */
    margin: MarginLike;
    /**********************************************************
    Panel-specific properties:
    ***********************************************************/
    /**
     * Gets or sets the alignment Spot of this GraphObject used in Panel layouts,
     * to determine where in the area allocated by the panel this object should be placed.
     *
     * The default value is Spot.Default, which lets the Panel determine the Spot using
     * Panel#defaultAlignment. If that property is also Spot.Default,
     * then the alignment spot will be different depending on the Panel type.
     *
     * The #alignmentFocus is often used along with this property to specify
     * where this object should be positioned in a Panel.
     *
     * A Spot.Default is equivalent to Spot.Center in Spot, Auto, Horizontal, and Vertical panels.
     * For examples of alignments in different panels, see the <a href="../../intro/panels.html">Introduction page on Panels</a>.
     *
     * @see #alignmentFocus
     * @see Panel#defaultAlignment
     */
    alignment: Spot;
    /**
     * Gets or sets the column of this GraphObject if it is in a Table Panel.
     * The value must be a small non-negative integer. The default is 0.
     */
    column: number;
    /**
     * Gets or sets the number of columns spanned by this GraphObject if it is in a Table Panel.
     * The value must be a small positive integer. The default is 1.
     */
    columnSpan: number;
    /**
     * Gets or sets the row of this GraphObject if it is in a Table Panel.
     * The value must be a small non-negative integer. The default is 0.
     */
    row: number;
    /**
     * Gets or sets the number of rows spanned by this GraphObject if it is in a Table Panel.
     * The value must be a small positive integer. The default is 1.
     */
    rowSpan: number;
    /**
     * Undocumented
     */
    spanAllocation: ((a: GraphObject, b: RowColumnDefinition, c: number) => number) | null;
    /**
     * Gets or sets the spot on this GraphObject to be used as the alignment point
     * in Spot and Fixed Panels.
     * Value must be of the Spot.
     *
     * The default value is Spot.Default, which means that the Panel type can decide the effective alignment spot.
     *
     * The #alignment is often used along with this property to specify
     * where this object should be positioned in a Panel.
     *
     * For Panel.Graduated, the alignmentFocus spot determines the spot on a child element to be aligned with some
     * point along the main element.
     *
     * When you want a link label Node to be positioned by its location spot rather than by this alignmentFocus spot,
     * you can set this property to Spot.None, only on Nodes.
     * @see Panel#alignmentFocusName
     *
     * For examples of alignments in different panels, see the <a href="../../intro/panels.html">Introduction page on Panels</a>.
     *
     * WARNING: Since 2.0, for Spot Panels, the offsetX/offsetY of #alignmentFocus has been reversed.
     * The offsetX/Y now describes offset distance from the alignmentFocus point to the alignment point, rather than the opposite.
     * This is what it has always described when using #alignmentFocus with Link Labels.
     */
    alignmentFocus: Spot;
    /**********************************************************
    Port-specific properties:
    ***********************************************************/
    /**
     * Gets or sets an identifier for an object acting as a port on a Node.
     * The default value is null -- this object is not a port.
     *
     * A value that is the empty string is used by convention to mean the primary
     * (and usually only) port of the node.
     *
     * If a Node has no named ports, then the Node itself is the sole port.
     *
     * Note: the only kind of model that can save port information, i.e. portIds that are not an empty string,
     * for links is a GraphLinksModel whose GraphLinksModel#linkFromPortIdProperty and
     * GraphLinksModel#linkToPortIdProperty have been set to name properties on the link data objects.
     *
     * The value should be unique within the Node.
     * You must not modify this property once this GraphObject is in the visual tree of a Node.
     * <p class="boxread">
     * See <a href="../../intro/ports.html">the Introduction page on ports</a>
     * for usage information and examples.
     * @see #fromLinkable
     * @see #toLinkable
     * @see #fromSpot
     * @see #toSpot
     * @see Link#fromSpot
     * @see Link#toSpot
     */
    portId: string;
    /**
     * Gets or sets where a link should connect to this port.
     * The default value is Spot.None, meaning that the link routing
     * must consider the shape of the port and connect to the closest point.
     *
     * The value of Link#toSpot, if not Spot.Default, takes precedence over the value at this port
     * when determining the route of the link.
     * A number of the predefined Layouts automatically set Link#fromSpot and Link#toSpot,
     * thereby causing this property and #fromSpot on the port element to be ignored.
     * Depending on the layout, you may be able to disable that behavior, such as by setting ForceDirectedLayout#setsPortSpots,
     * TreeLayout#setsPortSpot, TreeLayout#setsChildPortSpot, or LayeredDigraphLayout#setsPortSpots to false.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see Link#toSpot
     * @see Link#computeSpot
     * @see #fromSpot
     * @see #portId
     */
    toSpot: Spot;
    /**
     * Gets or sets the length of the last segment of a link going to this port.
     * This value is used when the computed "to spot" is not Spot.None.
     * The default value is 10.
     *
     * The value of Link#toEndSegmentLength, if not NaN, takes precedence over the value at this port
     * when determining the route of the link.
     * This value also limits how short the Link#toShortLength may be drawn.
     *
     * For examples of how to use this property, see <a href="../../intro/links.html#EndSegmentLengths">Link End Segment Lengths</a>.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see Link#toEndSegmentLength
     * @see Link#computeEndSegmentLength
     * @see #fromEndSegmentLength
     * @see #portId
     */
    toEndSegmentLength: number;
    /**
     * Gets or sets how far the end segment of a link going to this port stops short of the actual port.
     * Positive values are limited by the #toEndSegmentLength or Link#toEndSegmentLength.
     * Negative values cause the link to extend into the port.
     * The default value is zero.
     *
     * This property is useful when you have a thick link and a pointy arrowhead.
     * Normally the link Shape extends all the way to the end of the arrowhead.
     * If the link Shape is wide, its edges will be seen behind the arrowhead.
     * By setting this property to a small positive value, the link Shape can end within the
     * body of the arrowhead, leaving only the point of the arrowhead visible at the end of the link.
     *
     * A negative value for this property can also be useful when you want the link Shape to continue
     * into the port, perhaps because a portion of the port is transparent and you want the link to
     * appear to connect visually with a different point on the node.
     *
     * The value of Link#toShortLength, if not NaN, takes precedence over the value at this port
     * when determining the route of the link.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see #fromShortLength
     * @see Link#toShortLength
     * @see #portId
     */
    toShortLength: number;
    /**
     * Gets or sets whether the user may draw Links to this port.
     * This property is used by LinkingBaseTool#isValidTo.
     *
     * The default value is null, which indicates that the real value is inherited from
     * the parent Panel, or false if there is no containing panel.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node,
     * or unless you are disabling the "linkability" of a particular GraphObject
     * inside a Panel whose toLinkable has been set or bound to true.
     * @see #fromLinkable
     * @see #toMaxLinks
     * @see #portId
     * @see #cursor
     */
    toLinkable: boolean | null;
    /**
     * Gets or sets the maximum number of links that may go into this port.
     * This property is used by LinkingBaseTool#isValidTo.
     *
     * The value must be non-negative.
     * The default value is Infinity.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see #toLinkable
     * @see #fromMaxLinks
     * @see #portId
     */
    toMaxLinks: number;
    /**
     * Gets or sets where a link should connect from this port.
     * The default value is Spot.None, meaning that the link routing
     * must consider the shape of the port and connect at the closest point.
     *
     * The value of Link#fromSpot, if not Spot.Default, takes precedence over the value at this port
     * when determining the route of the link.
     * A number of the predefined Layouts automatically set Link#fromSpot and Link#toSpot,
     * thereby causing this property and #toSpot on the port element to be ignored.
     * Depending on the layout, you may be able to disable that behavior, such as by setting ForceDirectedLayout#setsPortSpots,
     * TreeLayout#setsPortSpot, TreeLayout#setsChildPortSpot, or LayeredDigraphLayout#setsPortSpots to false.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see Link#fromSpot
     * @see Link#computeSpot
     * @see #toSpot
     * @see #portId
     */
    fromSpot: Spot;
    /**
     * Gets or sets the length of the first segment of a link coming from this port.
     * This value is used when the computed "from spot" is not Spot.None.
     * The default value is 10.
     * This value also limits how short the Link#fromShortLength may be drawn.
     *
     * The value of Link#fromEndSegmentLength, if not NaN, takes precedence over the value at this port
     * when determining the route of the link.
     *
     * For examples of how to use this property, see <a href="../../intro/links.html#EndSegmentLengths">Link End Segment Lengths</a>.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see Link#fromEndSegmentLength
     * @see Link#computeEndSegmentLength
     * @see #toEndSegmentLength
     * @see #portId
     */
    fromEndSegmentLength: number;
    /**
     * Gets or sets how far the end segment of a link coming from this port stops short of the actual port.
     * Positive values are limited by the #fromEndSegmentLength or Link#fromEndSegmentLength.
     * Negative values cause the link to extend into the port.
     * The default value is zero.
     *
     * This property is useful when you have a thick link and a pointy arrowhead.
     * Normally the link Shape extends all the way to the end of the arrowhead.
     * If the link Shape is wide, its edges will be seen behind the arrowhead.
     * By setting this property to a small positive value, the link Shape can end within the
     * body of the arrowhead, leaving only the point of the arrowhead visible at the end of the link.
     *
     * A negative value for this property can also be useful when you want the link Shape to continue
     * into the port, perhaps because a portion of the port is transparent and you want the link to
     * appear to connect visually with a different point on the node.
     *
     * The value of Link#fromShortLength, if not NaN, takes precedence over the value at this port
     * when determining the route of the link.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see #toShortLength
     * @see Link#fromShortLength
     * @see #portId
     */
    fromShortLength: number;
    /**
     * Gets or sets whether the user may draw Links from this port.
     * This property is used by LinkingBaseTool#isValidFrom.
     *
     * The default value is null, which indicates that the real value is inherited from
     * the parent Panel, or false if there is no containing panel.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node,
     * or unless you are disabling the "linkability" of a particular GraphObject
     * inside a Panel whose fromLinkable has been set or bound to true.
     * @see #toLinkable
     * @see #fromMaxLinks
     * @see #portId
     * @see #cursor
     */
    fromLinkable: boolean | null;
    /**
     * Gets or sets the maximum number of links that may come out of this port.
     * This property is used by LinkingBaseTool#isValidFrom.
     *
     * The value must be non-negative.
     * The default value is Infinity.
     *
     * You must set this property on a GraphObject whose #portId is non-null,
     * unless the whole Node is acting as a single port,
     * in which case this property should be set on the Node.
     * @see #fromLinkable
     * @see #toMaxLinks
     * @see #portId
     */
    fromMaxLinks: number;
    /**********************************************************
    Event Handling:
    ***********************************************************/
    /**
     * Gets or sets the mouse cursor to use when the mouse is over this object with no mouse buttons pressed.
     * The value is null when no particular cursor is specified for this object;
     * the actual cursor is determined by any containing Panel.
     *
     * The default value is the empty string, which means the
     * current mouse cursor is determined by the Diagram.
     * Other strings should be valid CSS strings that specify a cursor.
     * This provides some more information about cursor syntax:
     * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/cursor">CSS cursors (mozilla.org)</a>.
     * @see Diagram#defaultCursor
     * @see Diagram#currentCursor
     */
    cursor: string;
    /**
     * Gets or sets the function to execute when the user single-primary-clicks on this object.
     * This typically involves a mouse-down followed by a prompt mouse-up
     * at approximately the same position using the left (primary) mouse button.
     * This property is used by the ClickSelectingTool
     * when the user clicks on a GraphObject.
     * The function is called in addition to the DiagramEvent
     * that is raised with the name `"ObjectSingleClicked"`.
     *
     * If this property value is a function, it is called with an InputEvent
     * and this GraphObject.
     * The InputEvent#targetObject provides the GraphObject that was found
     * at the mouse point before looking up the visual tree of GraphObject#panels
     * to get to this object.
     *
     * From the second argument, *obj*, you can get to the Node or Link via the #part property.
     * From there you can access the bound data via the Panel#data property.
     * So from an event handler you can get the bound data by `obj.part.data`.
     *
     * By default this property is null.
     *
     * Objects in Layers that are Layer#isTemporary do not receive click events.
     * If you do want such objects to respond to clicks, set #isActionable to true.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call Diagram#startTransaction and
     * Diagram#commitTransaction.
     * <p class="boxrun">
     * An example of a click event handler is shown in the
     * <a href="../../samples/arrowheads.html">Arrowheads sample</a>.
     * @see #doubleClick
     * @see #contextClick
     * @see Diagram#click
     */
    click: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the user double-primary-clicks on this object.
     * This typically involves a mouse-down/up/down/up in rapid succession
     * at approximately the same position using the left (primary) mouse button.
     * This property is used by the ClickSelectingTool
     * when the user clicks on a GraphObject.
     * The function is called in addition to the DiagramEvent
     * that is raised with the name `"ObjectDoubleClicked"`.
     *
     * If this property value is a function, it is called with an InputEvent
     * and this GraphObject.
     * The InputEvent#targetObject provides the GraphObject that was found
     * at the mouse point before looking up the visual tree of GraphObject#panels
     * to get to this object.
     *
     * From the second argument, *obj*, you can get to the Node or Link via the #part property.
     * From there you can access the bound data via the Panel#data property.
     * So from an event handler you can get the bound data by `obj.part.data`.
     *
     * By default this property is null.
     *
     * Objects in Layers that are Layer#isTemporary do not receive click events.
     * If you do want such objects to respond to clicks, set #isActionable to true.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call Diagram#startTransaction and
     * Diagram#commitTransaction.
     *
     * The <a href="../../samples/classHierarchy.html">Class Hierarchy sample</a>
     * demonstrates the definition of a double-click event handler that opens up
     * a web page with the documentation for that class:
     * ```js
     * diagram.nodeTemplate =
     *   $(go.Node, . . .,
     *     {
     *       doubleClick:  // here the second argument is this object, which is this Node
     *         function(e, node) { window.open("../api/symbols/" + node.data.key + ".html"); }
     *     },
     *     . . .
     *     );
     * ```
     * @see #click
     * @see #contextClick
     * @see Diagram#doubleClick
     */
    doubleClick: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the user single-secondary-clicks on this object.
     * This typically involves a mouse-down followed by a prompt mouse-up
     * at approximately the same position using the right (secondary) mouse button.
     * This property is used by the ClickSelectingTool
     * when the user clicks on a GraphObject.
     * The function is called in addition to the DiagramEvent
     * that is raised with the name `"ObjectContextClicked"`.
     *
     * If this property value is a function, it is called with an InputEvent
     * and this GraphObject.
     * The InputEvent#targetObject provides the GraphObject that was found
     * at the mouse point before looking up the visual tree of GraphObject#panels
     * to get to this object.
     *
     * From the second argument, *obj*, you can get to the Node or Link via the #part property.
     * From there you can access the bound data via the Panel#data property.
     * So from an event handler you can get the bound data by `obj.part.data`.
     *
     * By default this property is null.
     *
     * Objects in Layers that are Layer#isTemporary do not receive click events.
     * If you do want such objects to respond to clicks, set #isActionable to true.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call Diagram#startTransaction and
     * Diagram#commitTransaction.
     * @see #click
     * @see #doubleClick
     * @see Diagram#contextClick
     */
    contextClick: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the user moves the mouse
     * into this object without holding down any buttons.
     * This property is used by the ToolManager.
     *
     * If this property value is a function, it is called with an InputEvent,
     * this GraphObject that the mouse is now in,
     * and any previous GraphObject that the mouse was in.
     * The InputEvent#targetObject provides the GraphObject that was found
     * at the mouse point before looking up the visual tree of GraphObject#panels
     * to get to this object.
     * By default this property is null.
     *
     * This function is called with Diagram#skipsUndoManager temporarily set to true,
     * so that any changes to GraphObjects are not recorded in the UndoManager.
     * You do not need to start and commit any transaction in this function.
     * After calling this function the diagram will be updated immediately.
     *
     * For example, consider the situation where one wants to display buttons that the user can click
     * whenever the user passes the mouse over a node, and the buttons automatically disappear when the
     * mouse leaves the node.  This can be implemented by showing an Adornment holding the buttons.
     * ```js
     * var nodeContextMenu =
     *   $(go.Adornment, "Spot",
     *     { background: "transparent" },  // to help detect when the mouse leaves the area
     *     $(go.Placeholder),
     *     $(go.Panel, "Vertical",
     *       { alignment: go.Spot.Right, alignmentFocus: go.Spot.Left },
     *       $("Button",
     *         $(go.TextBlock, "Command 1"),
     *         {
     *           click: function(e, obj) {
     *             var node = obj.part.adornedPart;
     *             alert("Command 1 on " + node.data.text);
     *             node.removeAdornment("ContextMenuOver");
     *           }
     *         }),
     *       $("Button",
     *         $(go.TextBlock, "Command 2"),
     *         {
     *           click: function(e, obj) {
     *             var node = obj.part.adornedPart;
     *             alert("Command 2 on " + node.data.text);
     *             node.removeAdornment("ContextMenuOver");
     *           }
     *         })
     *       ));
     * ```
     * Then in the definition of the Node we can implement a mouseEnter event handler:
     * ```js
     * myDiagram.nodeTemplate =
     *   $(go.Node,
     *     . . .
     *     {
     *       mouseEnter: function(e, node) {
     *         nodeContextMenu.adornedObject = node;
     *         nodeContextMenu.mouseLeave = function(ev, cm) {
     *           node.removeAdornment("ContextMenuOver");
     *         }
     *         node.addAdornment("ContextMenuOver", nodeContextMenu);
     *       }
     *     });
     * ```
     * Note how it automatically defines a #mouseLeave event handler too.
     * The context menu Adornment is removed either when the mouse leaves the area of the Adornment
     * or when the user executes a button click event handler.
     * @see #mouseLeave
     * @see #mouseOver
     * @see #mouseHover
     * @see #mouseDragEnter
     */
    mouseEnter: ((e: InputEvent, thisObj: GraphObject, prevObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the user moves the mouse
     * out of this object without holding down any buttons.
     * This property is used by the ToolManager.
     *
     * If this property value is a function, it is called with an InputEvent,
     * this GraphObject that the mouse has left,
     * and any next GraphObject that the mouse is now in.
     * The InputEvent#targetObject provides the GraphObject that was found
     * at the mouse point before looking up the visual tree of GraphObject#panels
     * to get to this object.
     * By default this property is null.
     *
     * This function is called with Diagram#skipsUndoManager temporarily set to true,
     * so that any changes to GraphObjects are not recorded in the UndoManager.
     * You do not need to start and commit any transaction in this function.
     * After calling this function the diagram will be updated immediately.
     *
     * For example, the <a href="../../samples/flowchart.html">Flow Chart sample</a>
     * automatically shows and hides the ports as the mouse passes over a node.
     * The node template includes the following settings:
     * ```js
     * myDiagram.nodeTemplate =
     *   $(go.Node,
     *     . . .
     *     {
     *       . . .
     *       // handle mouse enter/leave events to show/hide the ports
     *       mouseEnter: function(e, obj) { showPorts(obj.part, true); },
     *       mouseLeave: function(e, obj) { showPorts(obj.part, false); }
     *       . . .
     *     });
     * ```
     * where the `showPorts` function is defined to set the #visible
     * property of each of the port elements of the node.
     * @see #mouseEnter
     * @see #mouseOver
     * @see #mouseHover
     * @see #mouseDragLeave
     */
    mouseLeave: ((e: InputEvent, thisObj: GraphObject, nextObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the user moves the mouse
     * over this object without holding down any buttons.
     * This property is used by the ToolManager.
     * This property is infrequently used -- it is more common to implement #mouseEnter
     * and #mouseLeave functions.
     *
     * If this property value is a function, it is called with an InputEvent
     * and this GraphObject.
     * The InputEvent#targetObject provides the GraphObject that was found
     * at the mouse point before looking up the visual tree of GraphObject#panels
     * to get to this object.
     * By default this property is null.
     *
     * This function is called with Diagram#skipsUndoManager temporarily set to true,
     * so that any changes to GraphObjects are not recorded in the UndoManager.
     * You do not need to start and commit any transaction in this function.
     * After calling this function the diagram will be updated immediately.
     * @see #mouseHover
     * @see #mouseEnter
     * @see #mouseLeave
     */
    mouseOver: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the user holds the mouse still for a while
     * over this object without holding down any buttons.
     * This property is used by the ToolManager.
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call Diagram#startTransaction and
     * Diagram#commitTransaction.
     *
     * You can control how long the user must wait with a motionless mouse before
     * a "mouse hover" event occurs, by setting ToolManager#hoverDelay.
     * For example:
     * ```js
     * myDiagram =
     *   $(go.Diagram, "myDiagramDiv",
     *     { "toolManager.hoverDelay": 500 });  // 500 milliseconds
     * ```
     * or:
     * ```js
     * myDiagram.toolManager.hoverDelay = 500;  // 500 milliseconds
     * ```
     * @see #mouseOver
     * @see #mouseEnter
     * @see #mouseLeave
     */
    mouseHover: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the user holds the mouse still for a while
     * over this object while holding down a button.
     * This property is used by the ToolManager.
     *
     * If this property value is a function, it is called with an InputEvent.
     * By default this property is null.
     *
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call Diagram#startTransaction and
     * Diagram#commitTransaction.
     *
     * You can control how long the user must wait during a drag with a motionless mouse before
     * a "mouse hold" event occurs, by setting ToolManager#holdDelay.
     * For example:
     * ```js
     * myDiagram =
     *   $(go.Diagram, "myDiagramDiv",
     *     { "toolManager.holdDelay": 500 });  // 500 milliseconds
     * ```
     * or:
     * ```js
     * myDiagram.toolManager.holdDelay = 500;  // 500 milliseconds
     * ```
     * @see #mouseDragEnter
     * @see #mouseDragLeave
     * @see #mouseHover
     */
    mouseHold: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the user moves the mouse
     * into this stationary object during a DraggingTool drag;
     * this allows you to provide feedback during a drag based on where it might drop.
     *
     * If this property value is a function, it is called with an InputEvent,
     * this GraphObject, and any previous GraphObject.
     * The InputEvent#targetObject provides the GraphObject that was found
     * at the mouse point before looking up the visual tree of GraphObject#panels
     * to get to this object.
     * By default this property is null.
     *
     * Note that for a drag-and-drop that originates in a different diagram, the target diagram's
     * selection collection will not be the parts that are being dragged.
     * Instead the temporary parts being dragged can be found as the source diagram's DraggingTool#copiedParts.
     *
     * This function is called with Diagram#skipsUndoManager temporarily set to true,
     * so that any changes to GraphObjects are not recorded in the UndoManager.
     * You do not need to start and commit any transaction in this function,
     * because the DraggingTool will be conducting one already.
     * After calling this function the diagram will be updated immediately.
     *
     * For an example of a mouseDragEnter event handler, see the node template in the
     * <a href="../../samples/orgChartEditor.html">Org Chart Editor sample</a>.
     * @see #mouseDragLeave
     * @see #mouseHold
     * @see #mouseDrop
     * @see #mouseEnter
     * @see Group#handlesDragDropForMembers
     */
    mouseDragEnter: ((e: InputEvent, thisObj: GraphObject, prevObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the user moves the mouse
     * out of this stationary object during a DraggingTool drag;
     * this allows you to provide feedback during a drag based on where it might drop.
     *
     * If this property value is a function, it is called with an InputEvent,
     * this GraphObject, and any new GraphObject that the mouse is in.
     * The InputEvent#targetObject provides the GraphObject that was found
     * at the mouse point before looking up the visual tree of GraphObject#panels
     * to get to this object.
     * By default this property is null.
     *
     * Note that for a drag-and-drop that originates in a different diagram, the target diagram's
     * selection collection will not be the parts that are being dragged.
     * Instead the temporary parts being dragged can be found as the source diagram's DraggingTool#copiedParts.
     *
     * This function is called with Diagram#skipsUndoManager temporarily set to true,
     * so that any changes to GraphObjects are not recorded in the UndoManager.
     * You do not need to start and commit any transaction in this function,
     * because the DraggingTool will be conducting one already.
     * After calling this function the diagram will be updated immediately.
     *
     * For an example of a mouseDragLeave event handler, see the node template in the
     * <a href="../../samples/orgChartEditor.html">Org Chart Editor sample</a>.
     * @see #mouseDragEnter
     * @see #mouseHold
     * @see #mouseDrop
     * @see #mouseLeave
     * @see Group#handlesDragDropForMembers
     */
    mouseDragLeave: ((e: InputEvent, thisObj: GraphObject, nextObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when a user drops the selection on this object
     * at the end of a DraggingTool drag;
     * this allows you to customize the behavior when a drop occurs on an object.
     *
     * If this property value is a function, it is called with an InputEvent,
     * this GraphObject.
     * The InputEvent#targetObject provides the GraphObject that was found
     * at the mouse point before looking up the visual tree of GraphObject#panels
     * to get to this object.
     * The function is called within the transaction performed by the DraggingTool,
     * so you do not need to conduct one.
     * By default this property is null.
     *
     * For an example of a mouseDrop event handler, see the node template in the
     * <a href="../../samples/orgChartEditor.html">Org Chart Editor sample</a>.
     * @see #mouseDragEnter
     * @see #mouseDragLeave
     * @see #mouseHold
     * @see Group#handlesDragDropForMembers
     */
    mouseDrop: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute on a mouse-down event when this GraphObject's #isActionable
     * is set to true.
     * This property is infrequently set.
     * By default this property is null.
     *
     * This functional property is only set on objects such as buttons, knobs, or sliders that want to handle all events,
     * in conjunction with ActionTool, pre-empting the normal tool mechanisms.
     *
     * The ActionTool does not conduct any transaction, so if this property has a value,
     * the function will not be called within a transaction.
     * @see #actionMove
     * @see #actionUp
     * @see #actionCancel
     */
    actionDown: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute on a mouse-move event when this GraphObject's #isActionable
     * is set to true.
     * This property is infrequently set.
     * By default this property is null.
     *
     * This functional property is only set on objects such as buttons, knobs, or sliders that want to handle all events,
     * in conjunction with ActionTool, pre-empting the normal tool mechanisms.
     *
     * The ActionTool does not conduct any transaction, so if this property has a value,
     * the function will not be called within a transaction.
     * @see #actionDown
     * @see #actionUp
     * @see #actionCancel
     */
    actionMove: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute on a mouse-up event when this GraphObject's #isActionable
     * is set to true.
     * This property is infrequently set.
     * By default this property is null.
     *
     * This functional property is only set on objects such as buttons, knobs, or sliders that want to handle all events,
     * in conjunction with ActionTool, pre-empting the normal tool mechanisms.
     *
     * The ActionTool does not conduct any transaction, so if this property has a value,
     * the function will not be called within a transaction.
     * If you do provide a function that makes changes to the diagram or to its model,
     * you should do so within a transaction -- call Diagram#startTransaction and
     * Diagram#commitTransaction.
     * @see #actionDown
     * @see #actionMove
     * @see #actionCancel
     */
    actionUp: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * Gets or sets the function to execute when the ActionTool is cancelled and this GraphObject's #isActionable
     * is set to true.
     * This property is infrequently set.
     * By default this property is null.
     *
     * This functional property is only set on objects such as buttons, knobs, or sliders that want to handle all events,
     * in conjunction with ActionTool, pre-empting the normal tool mechanisms.
     *
     * The ActionTool does not conduct any transaction, so if this property has a value,
     * the function will not be called within a transaction.
     * @see #actionDown
     * @see #actionMove
     * @see #actionUp
     */
    actionCancel: ((e: InputEvent, thisObj: GraphObject) => void) | null;
    /**
     * This Adornment or HTMLInfo is shown when the mouse hovers over this object.
     * The default value is null, which means no tooltip is shown.
     *
     * A typical tooltip is defined in the following manner, as taken from
     * the <a href="../../samples/kittenMonitor.html">Kitten Monitor sample</a>:
     * ```js
     * myDiagram.nodeTemplate =
     *   $(go.Node,
     *     . . .
     *     { // this tooltip shows the name and picture of the kitten
     *       toolTip:
     *         $("ToolTip",
     *           $(go.Panel, "Vertical",
     *             $(go.Picture,
     *               new go.Binding("source", "src", function(s) { return "images/" + s + ".png"; })),
     *             $(go.TextBlock, { margin: 3 },
     *               new go.Binding("text", "key"))))
     *     });
     * ```
     * Note that this Adornment depends on having the same data binding as the adorned Part
     * (i.e. the same value for Panel#data).
     *
     * Tooltips are not copied by #copy, so that tooltips may be shared by all instances of a template.
     *
     * Tooltips are shown after a timed delay given by the ToolManager#hoverDelay.
     * You can change the delay time by:
     * ```js
     * myDiagram =
     *   $(go.Diagram, "myDiagramDiv",
     *     { "toolManager.hoverDelay": 500 });  // 500 milliseconds
     * ```
     * or:
     * ```js
     * myDiagram.toolManager.hoverDelay = 500;  // 500 milliseconds
     * ```
     *
     * Tooltips are normally positioned by ToolManager#positionToolTip.
     * However, if there is a Placeholder in the tooltip, the tooltip (i.e. an Adornment)
     * will be positioned so that the Placeholder is at the same position as this adorned GraphObject.
     *
     * Replacing this value will not modify or remove any existing tooltip that is being shown for this object.
     *
     * Read more about tooltips at <a href="../../intro/toolTips.html">ToolTips</a>.
     */
    toolTip: Adornment | HTMLInfo | null;
    /**
     * This Adornment or HTMLInfo is shown upon a context click on this object.
     * The default value is null, which means no context menu is shown.
     *
     * Changing this value will not modify or remove any existing menu that is being shown for this object.
     *
     * Context menus may also depend on having the same data binding as the adorned Part
     * (i.e. the same value for Panel#data).
     *
     * Context menus are not copied by #copy, so that context menus may be shared by all instances of a template.
     *
     * A typical context menu is implemented as an Adornment with several buttons in it.
     * For example, this context menu is defined in the
     * <a href="../../samples/dynamicPorts.html">Dynamic Port sample</a>:
     * ```js
     * var nodeMenu =  // context menu for each Node
     *  $("ContextMenu",
     *     $("ContextMenuButton",
     *       $(go.TextBlock, "Add top port"),
     *       { click: function(e, obj) { addPort("top"); } }),
     *     $("ContextMenuButton",
     *       $(go.TextBlock, "Add left port"),
     *       { click: function(e, obj) { addPort("left"); } }),
     *     $("ContextMenuButton",
     *       $(go.TextBlock, "Add right port"),
     *       { click: function(e, obj) { addPort("right"); } }),
     *     $("ContextMenuButton",
     *       $(go.TextBlock, "Add bottom port"),
     *       { click: function(e, obj) { addPort("bottom"); } }));
     * ```
     * and is used in the node template:
     * ```js
     * myDiagram.nodeTemplate =
     *   $(go.Node, "Table",
     *     { . . .
     *       contextMenu: nodeMenu
     *     },
     *     . . .);
     * ```
     *
     * Context menus are normally positioned by ContextMenuTool#positionContextMenu.
     * However, if there is a Placeholder in the context menu, the context menu (i.e. an Adornment)
     * will be positioned so that the Placeholder is at the same position as this adorned GraphObject.
     *
     * The <a href="../../samples/basic.html">Basic sample</a> also shows how
     * to make context menu items invisible when the command is disabled.
     *
     * Replacing this value will not modify or remove any existing context menu that is being shown for this object.
     *
     * Read more about context menus at <a href="../../intro/contextMenus.html">Context Menus</a>.
     */
    contextMenu: Adornment | HTMLInfo | null;
    /**
     * Add a data-binding of a property on this GraphObject to a property on a data object.
     * It is unusual to call this method -- GraphObject.make will call this method for you
     * when you pass it an instance of Binding.
     *
     * Do not add, modify, or remove any Bindings after this object has been copied.
     *
     * Read more about Bindings at <a href="../../intro/dataBinding.html">the Introduction page about Data Bindings</a>.
     * @param {Binding} binding
     */
    bind(binding: Binding): void;
    /**
     * Undocumented
     * @return {Panel}
     */
    findTemplateBinder(): Panel | null;
    /**
     * This method sets a collection of properties according to the property/value pairs that have been set on the given Object,
     * in the same manner as GraphObject.make does when constructing a GraphObject with an argument that is a simple JavaScript Object.
     *
     * If this is a Panel, you can set properties on named elements within the panel by using a *name.property* syntax for the property name.
     * For example, if a Node has a Picture that is named "ICON" (because its #name property has been set to "ICON")
     * and a TextBlock whose name is "TB", one could set properties on the Node and on each of those named elements by:
     * ```js
     * aNode.setProperties({
     *     background: "red",
     *     "ICON.source": "https://www.example.com/images/alert.jpg",
     *     "TB.font": "bold 12pt sans-serif"
     * });
     * ```
     *
     * At the current time only a single dot is permitted in the property "name".
     * Note that the use of all-upper-case object names is simply a convention.
     *
     * Calling this method is much less efficient than setting properties directly.
     * @param {Object} props a plain JavaScript object with various property values to be set on this GraphObject.
     * @since 1.5
     */
    setProperties(props: ObjectData): void;
    /**
     * This static function builds an object given its class and additional arguments
     * providing initial properties or GraphObjects that become Panel elements.
     *
     * The first argument must be the class type or the name of a class or the name of a predefined kind of Panel.
     * This function will construct a new instance of that type and use the rest of the arguments to initialize the object.
     * The first argument cannot be a regular Object (such as a GraphObject) that you are trying to initialize;
     * for that you can call #setProperties or Diagram#setProperties, although that would
     * be less efficient than setting properties directly.
     *
     * If an initializer argument is an enumerated value, this tries to set the property that seems most appropriate.
     *
     * If an initializer argument is a string, this sets a particular property depending on the type of object being built.
     *   - If the object is a TextBlock, it sets TextBlock#text.
     *   - If the object is a Shape, it sets Shape#figure.
     *   - If the object is a Picture, it sets Picture#source.
     *   - If the object is a Panel (including Part, Node, or Group), it sets Panel#type.
     *
     * If an initializer argument is a particular kind of object, this can add that object to the object being built.
     *   - GraphObjects and RowColumnDefinitions can only be added as elements of Panels.
     *   - Bindings can only be applied to GraphObjects and RowColumnDefinitions.
     *   - PathFigures can only be added to Geometry objects.
     *   - PathSegments can only be added to PathFigure objects.
     *   - Regular JavaScript Arrays provide a sequence of initializer arguments.
     *   - Regular JavaScript objects provide property/value pairs that are set on the object being built.
     *
     * When the initializer argument is a plain JavaScript Object, there are several ways that that object's properties are applied.
     * If the property name is a string with a period inside it, this has a special meaning if the object is a Panel or a Diagram.
     * At the current time only a single period separator is valid syntax for a property string, and it is valid only on Panels and Diagrams.
     *
     * For Panels, the substring before the period is used as the name passed to Panel#findObject
     * to get the actual object on which to set the property, which is the substring after the period.
     * This is normally useful only on the predefined Panels:
     *   - a **"Button"** has a Shape named "ButtonBorder" surrounding the content of the Panel.
     *   - a **"TreeExpanderButton"** has a "ButtonBorder" Shape and a "ButtonIcon" Shape that is the plus-or-minus sign.
     *   - a **"SubGraphExpanderButton"** has a "ButtonBorder" Shape and a "ButtonIcon" Shape that is the plus-or-minus sign.
     *   - a **"ContextMenuButton"** has a Shape named "ButtonBorder" surrounding the content of the Panel.
     *
     * But you can define your own names that GraphObject.make can build by calling the static function GraphObject.defineBuilder.
     *
     * For Diagrams, the substring before the period is used as the name of a property on the Diagram itself
     * to get the actual object on which to set the property.
     * As a special case, if such a property value does not exist on the Diagram, it looks on the Diagram#toolManager.
     * See some examples below.
     *
     * Also for Diagrams, and only for Diagrams, if the property name is the name of a DiagramEvent,
     * the property value must be a DiagramEvent listener function, and Diagram#addDiagramListener is called
     * using that DiagramEvent name and that function.
     * Note that all DiagramEvent names are capitalized and do not contain any periods,
     * so there cannot be any name conflicts with any properties on Diagram or ToolManager.
     * Although you can register multiple listeners for the same DiagramEvent names, due to JavaScript limitations
     * those need to be declared using separate JavaScript objects, because JavaScript does not permit duplicate property names
     * in an Object literal.
     *
     * Furthermore for Diagrams, if the property name is `"Changed"` or `"ModelChanged"`,
     * the property value must be a ChangedEvent listener function, which is called with a ChangedEvent argument.
     * When the property name is `"Changed"`, it calls Diagram#addChangedListener,
     * notifying about changes to the Diagram or its Layers or GraphObjects.
     * When the property name is `"ModelChanged"`, it calls Model#addChangedListener on the Diagram#model,
     * resulting in notifications about changes to the Model or its data.
     * This is handy because the Diagram#model property setter will automatically call
     * Model#removeChangedListener on the old model, thereby avoiding any overhead if there are any
     * more changes to the old model and also avoiding a reference to the listener which might cause garbage collection retention.
     * It also will call Model#addChangedListener on the new model, helping implement the same behavior with the new model.
     *
     * If the property name is a number and if the object being constructed is a Brush,
     * the number and value are added to the Brush by calling Brush#addColorStop.
     *
     * Otherwise the property name is used as a regular property name on the object being built.
     * This tries to do some property name and value checking:
     * when a property is not defined on the object being built, it will signal an error.
     * Many typos can be found this way that would be ignored by JavaScript code.
     *
     * If the property name begins with an underscore, this will not complain about the property being undefined.
     * Not only is that underscore property set on the object being built, but calls to #copy
     * will also copy the values of such named properties to the new objects.
     *
     * In the samples and in the intro pages this function is called using the alias `$`.
     * You can use a different short name if you would like to preserve the use of `$` for another JavaScript library.
     * ```js
     *  var $ = go.GraphObject.make;
     *
     *  var diagram =
     *    $(go.Diagram, "myDiagramDiv",
     *      {
     *         // don't initialize some properties until after a new model has been loaded
     *         "InitialLayoutCompleted": loadDiagramProperties,
     *        allowZoom: false,  // don't allow the user to change the diagram's scale
     *        "grid.visible": true,  // display a background grid for the whole diagram
     *        "grid.gridCellSize": new go.Size(20, 20),
     *        // allow double-click in background to create a new node
     *        "clickCreatingTool.archetypeNodeData": { text: "Node" },
     *        // allow Ctrl-G to call the groupSelection command
     *        "commandHandler.archetypeGroupData":
     *            { text: "Group", isGroup: true, color: "blue" },
     *        "toolManager.hoverDelay": 100,  // how quickly tooltips are shown
     *        // mouse wheel zooms instead of scrolls
     *        "toolManager.mouseWheelBehavior": go.ToolManager.WheelZoom,
     *        "commandHandler.copiesTree": true,  // for the copy command
     *        "commandHandler.deletesTree": true, // for the delete command
     *        "draggingTool.dragsTree": true,  // dragging for both move and copy
     *        "draggingTool.isGridSnapEnabled": true,
     *        layout: $(go.TreeLayout,
     *                  { angle: 90, sorting: go.TreeLayout.SortingAscending })
     *      });
     *
     *  diagram.nodeTemplate =
     *    $(go.Node, "Auto",  // or go.Panel.Auto
     *      new go.Binding("location", "loc", go.Point.parse).makeTwoWay(go.Point.stringify),
     *      $(go.Shape, "RoundedRectangle",
     *        {
     *          fill: $(go.Brush, "Linear", { 0: "#FEC901", 1: "#FEA200" }),
     *          stroke: "gray",
     *          strokeWidth: 2,
     *          strokeDashArray: [3, 3]
     *        }),
     *      $(go.TextBlock,
     *        { margin: 5, font: "bold 12pt sans-serif" },
     *        new go.Binding("text", "key"))
     *    );
     * ```
     * <p class="boxread">
     * See <a href="../../intro/buildingObjects.html">the Introduction page on building objects</a>
     * for usage information and examples of GraphObject.make.
     * @param {Function|string} type a class function or the name of a class in the `go` namespace,
     * or one of several predefined kinds of Panels: `"Button"`, `"TreeExpanderButton"`,
     * `"SubGraphExpanderButton"`, or `"ContextMenuButton"`.
     * @param {...*} initializers zero or more values that initialize the new object,
     * typically an Object with properties whose values are set on the new object,
     * or a JavaScript Array with additional initializer arguments,
     * or a GraphObject that is added to a Panel,
     * or a Binding for one of the new object's properties,
     * or a constant value as the initial value of a single property of the new object that
     * is recognized to take that value,
     * or a string that is used as the value of a commonly set property.
     * @return {Object}
     */
    static make<T extends Adornment>(// for specific named builders
    cls: ('ToolTip' | 'ContextMenu'), ...initializers: Array<string | (Partial<GraphObject> & {
        [p: string]: any;
    }) | Binding | EnumValue | RowColumnDefinition | PanelLayout | Array<string | (Partial<GraphObject> & {
        [p: string]: any;
    }) | Binding | EnumValue | RowColumnDefinition | PanelLayout>>): T;
    static make<T extends Panel>(// for specific named Panel builders
    cls: ('Button' | 'TreeExpanderButton' | 'SubGraphExpanderButton' | 'ContextMenuButton' | 'PanelExpanderButton' | 'CheckBoxButton' | 'CheckBox'), ...initializers: Array<string | (Partial<GraphObject> & {
        [p: string]: any;
    }) | Binding | EnumValue | RowColumnDefinition | PanelLayout | Array<string | (Partial<GraphObject> & {
        [p: string]: any;
    }) | Binding | EnumValue | RowColumnDefinition | PanelLayout>>): T;
    static make<T extends GraphObject>(cls: string, // for named Panel builders
    ...initializers: Array<string | (Partial<GraphObject> & {
        [p: string]: any;
    }) | Binding | EnumValue | RowColumnDefinition | PanelLayout | Array<string | (Partial<GraphObject> & {
        [p: string]: any;
    }) | Binding | EnumValue | RowColumnDefinition | PanelLayout>>): T;
    static make<CT extends ConstructorType<CT>>(cls: CT, ...initializers: Array<string | (Partial<InstanceType<CT>> & {
        [p: string]: any;
    } & (InstanceType<CT> extends Diagram ? DiagramEventsInterface & {
        Changed?: ChangedEventHandler;
        ModelChanged?: ChangedEventHandler;
    } : {})) | MakeAllow<CT, GraphObject, Binding> | MakeAllow<CT, Panel, GraphObject> | MakeAllow<CT, Panel, RowColumnDefinition> | MakeAllow<CT, Panel, PanelLayout> | MakeAllow<CT, RowColumnDefinition, Binding> | MakeAllow<CT, Geometry, PathFigure> | MakeAllow<CT, PathFigure, PathSegment> | EnumValue | HTMLDivElement | Array<string | (Partial<InstanceType<CT>> & {
        [p: string]: any;
    } & (InstanceType<CT> extends Diagram ? DiagramEventsInterface & {
        Changed?: ChangedEventHandler;
        ModelChanged?: ChangedEventHandler;
    } : {})) | MakeAllow<CT, GraphObject, Binding> | MakeAllow<CT, Panel, GraphObject> | MakeAllow<CT, Panel, RowColumnDefinition> | MakeAllow<CT, Panel, PanelLayout> | MakeAllow<CT, RowColumnDefinition, Binding> | MakeAllow<CT, Geometry, PathFigure> | MakeAllow<CT, PathFigure, PathSegment> | EnumValue>>): InstanceType<CT>;
    /**
     * This static function defines a named function that GraphObject.make can use to build objects.
     * Once this is called one can use the name as the first argument for GraphObject.make.
     * Names are case sensitive.
     *
     * The second argument must be a function that returns a newly created object, typically a GraphObject.
     * It is commonplace for that object to be a Panel holding a newly created visual tree of GraphObjects.
     * The function receives as its only argument an Array that is holds all of the arguments that are being
     * passed to GraphObject.make, which it may modify in order to change the arguments that GraphObject.make receives.
     *
     * Predefined builder names include: `"Button"`, `"TreeExpanderButton"`, `"SubGraphExpanderButton"`,
     * `"PanelExpanderButton"`, and `"ContextMenuButton"`.
     * The implementation of these builders is provided by <a href="../../extensions/Buttons.js">Buttons.js</a>
     * in the Extensions directory.
     * @param {string} name a capitalized name; must not be `""` or `"None"`
     * @param {function(Array.<*>):Object} func that takes an Array of `GraphObject.make` arguments and returns a new object
     * @since 1.5
     */
    static defineBuilder(name: string, func: ((a: Array<any>) => ObjectData)): void;
    /**
     * This static function returns the first argument from the arguments array passed
     * to a GraphObject.defineBuilder function by GraphObject.make.
     * By default this requires the first argument to be a string,
     * but you can provide a predicate to determine whether the argument is suitable.
     * @param {Array.<*>} args the arguments Array passed to the builder function;
     *        this may be modified if an acceptable argument is found and returned
     * @param {*=} defval the default value to return if the argument is optional and not present as the first argument;
     *        otherwise throw an error when the argument is not there
     * @param {function(*):boolean|null=} pred a predicate to determine the acceptability of the argument;
     *        the default predicate checks whether the argument is a string
     * @return {*}
     * @since 1.5
     */
    static takeBuilderArgument(args: Array<any>, defval?: any, pred?: ((a: any) => boolean) | null): any;
}
/**
 * @unrestricted
 */
export class SGradient {
    /**
     * @param {string} aType
     */
    constructor(aType: string);
}
/**
 * The BrushLike type is the same as `Brush | string | null`.
 * A string in place of a Brush object is treated as a Solid Brush of that color.
 * @category Type
 */
export type BrushLike = Brush | string | null;
/**
 * A Brush holds color information and describes how to draw the inside
 * of a Shape or the stroke of a shape or a TextBlock or the
 * background of any GraphObject.
 *
 * A Brush must not be modified once it has been assigned to a GraphObject,
 * such as the Shape#fill or TextBlock#stroke
 * or GraphObject#background.
 * However, a Brush may be shared by multiple GraphObjects.
 * @unrestricted
 * @category Geometry
 */
export class Brush {
    /**
     * Construct a Brush class that holds the given color information.
     * @param {EnumValue|string=} type Optional, one of the values Brush.Solid, Brush.Linear, Brush.Radial, Brush.Pattern,
     * or a well-formed CSS string describing a solid color brush. No parameter
     * defaults to a Brush.Solid with a color description of 'black'.
     */
    constructor(type?: EnumValue | string);
    /**
     * Create a copy of this Brush with the same values.
     * @return {Brush}
     */
    copy(): Brush;
    /**
     * Specify a particular color at a particular fraction of the distance.
     * If the #type is Brush.Solid, change the type to Brush.Linear.
     * You should have a color stop at 0 and a color stop at 1.
     * You should not have duplicate color stop values at the same fractional distance.
     * @param {number} loc A number between 0 and 1 (inclusive).
     * @param {string} color A valid CSS color string.
     * @return {Brush} this Brush
     */
    addColorStop(loc: number, color: string): Brush;
    /**
     * Gets or sets the type of brush.
     * The default value is Brush.Solid.
     * The value must be one of: Brush.Solid, Brush.Linear, Brush.Radial, Brush.Pattern.
     * If the new value is a linear or radial brush type,
     * and if the #start or #end spots are not specific spots,
     * they are changed to be specific spots, depending on the type of brush.
     */
    type: EnumValue;
    /**
     * Gets or sets the color of a solid Brush.
     * The default value is 'black'.
     * The value must be a valid CSS color string.
     */
    color: string;
    /**
     * Gets or sets the starting location for a linear or radial gradient.
     * A Spot value specifies a relative point in the object's GraphObject#naturalBounds.
     * The default value is Spot.TopCenter for linear gradients and Spot.Center for radial gradients.
     */
    start: Spot;
    /**
     * Gets or sets the ending location for a linear or radial gradient.
     * A Spot value specifies a relative point in the object's GraphObject#naturalBounds.
     * The default value is Spot.BottomCenter for linear gradients and Spot.Center for radial gradients.
     */
    end: Spot;
    /**
     * Gets or sets the radius of a radial brush at the start location.
     * The default value is 0.
     */
    startRadius: number;
    /**
     * Gets or sets the radius of a radial brush at the end location.
     * The default value is NaN.
     */
    endRadius: number;
    /**
     * Gets or sets a Map holding all of the color stops used in this gradient,
     * where the key is a number, the fractional distance between 0 and 1 (inclusive),
     * and where the corresponding value is a color string.
     *
     * Call #addColorStop in order to add color stops to this brush.
     * This property value may be null if no gradient stops have been defined.
     */
    colorStops: Map<number, string> | null;
    /**
     * Gets or sets the pattern of a brush of type Brush.Pattern.
     */
    pattern: HTMLCanvasElement | HTMLImageElement | null;
    /**
     * This static function can be used to generate a random color.
     * @param {number=} min A number between 0 and 255, defaults to 128.
     * @param {number=} max A number between 0 and 255, defaults to 255.
     * @return {string} A color value in # hexadecimal format.
     */
    static randomColor(min?: number, max?: number): string;
    /**
     * This static function returns true if a given color string is well-formed for drawing.
     * @param {string} color A color string to validate.
     * @return {boolean}
     * @since 1.7
     */
    static isValidColor(color: string): boolean;
    /**
     * This static function takes a color and lightens it by 20% in the Lab color space.
     * This is a convenience function which calls Brush.lightenBy.
     * @param {string} color A valid CSS color string.
     * @return {string} A CSS string for the lightened color in RGBA.
     * @since 1.7
     */
    static lighten(color: string): string;
    /**
     * Modifies all colors within this Brush, lightening them by some fraction.
     * @param {number=} fraction Fraction to lighten the colors by. Defaults to 0.2, must be between 0 and 1 (inclusive).
     * @param {EnumValue=} mode Color space to use for adjusting. Must be Brush.Lab or Brush.HSL, defaults to Brush.Lab.
     * @return {Brush} This Brush with modified color values.
     * @since 1.7
     */
    lightenBy(fraction?: number, mode?: EnumValue): Brush;
    /**
     * This static function takes a color and lightens it.
     * @param {string} color A valid CSS color string.
     * @param {number=} fraction Fraction to lighten the colors by. Defaults to 0.2, must be between 0 and 1 (inclusive).
     * @param  {EnumValue=} mode Color space to use for adjusting. Must be Brush.Lab or Brush.HSL, defaults to Brush.Lab.
     * @return {string} A CSS string for the lightened color in RGBA or HSLA.
     * @since 1.7
     */
    static lightenBy(color: string, fraction?: number, mode?: EnumValue): string;
    /**
     * This static function takes a color and darkens it by 20% in the Lab color space.
     * This is a convenience function which calls Brush.darkenBy.
     * @param {string} color A valid CSS color string.
     * @return {string} A CSS string for the darkened color in RGBA.
     * @since 1.7
     */
    static darken(color: string): string;
    /**
     * Modifies all colors within this Brush, darkening them by some fraction.
     * @param {number=} fraction Fraction to darken the colors by. Defaults to 0.2, must be between 0 and 1 (inclusive).
     * @param {EnumValue=} mode Color space to use for adjusting. Must be Brush.Lab or Brush.HSL, defaults to Brush.Lab.
     * @return {Brush} This Brush with modified color values.
     * @since 1.7
     */
    darkenBy(fraction?: number, mode?: EnumValue): Brush;
    /**
     * This static function takes a color and darkens it.
     * @param {string} color A valid CSS color string
     * @param {number=} fraction Fraction to darken the color by. Defaults to 0.2, must be between 0 and 1 (inclusive).
     * @param {EnumValue=} mode Color space to use for adjusting. Must be Brush.Lab or Brush.HSL, defaults to Brush.Lab.
     * @return {string} A CSS string for the darkened color in RGBA or HSLA.
     * @since 1.7
     */
    static darkenBy(color: string, fraction?: number, mode?: EnumValue): string;
    /**
     * This static function takes two colors and mixes them together, using the (optionally) specified amount of the second color.
     * @param {string} color1 A valid CSS color string.
     * @param {string} color2 Another valid CSS color string to mix.
     * @param {number=} fraction Fraction specifying how much color2 to mix into color1. Defaults to .5, must be between 0 and 1 (inclusive).
     * @return {boolean}
     * @since 2.0
     */
    static mix(color1: string, color2: string, fraction?: number): string;
    /**
     * This function determines whether this Brush is "dark."
     * @return {boolean}
     * @since 2.0
     */
    isDark(): boolean;
    /**
     * This static function takes a color and determines whether it is "dark." Does not account for transparency.
     *
     * Example usage:
     * ```js
     * myDiagram.nodeTemplate =
     *   $(go.Node, "Auto",
     *     $(go.Shape, "RoundedRectangle", { strokeWidth: 0 },
     *       new go.Binding("fill", "color")),
     *     $(go.TextBlock,
     *       { margin: 8 },
     *       new go.Binding("stroke", "color", function (c) {
     *         // Dark nodes use white text, light nodes use black text
     *         return go.Brush.isDark(c) ? "white" : "black";
     *       }),
     *       new go.Binding("text", "key")
     *     )
     *   );
     * ```
     * @param {Brush|string} color A valid CSS color string or a Brush.
     * @return {boolean}
     * @since 2.0
     */
    static isDark(color: BrushLike): boolean;
    /**
     * For simple, solid color brushes, used as the value for Brush#type.
     * @constant
     */
    static Solid: EnumValue;
    /**
     * For linear gradient brushes, used as the value for Brush#type.
     * @constant
     */
    static Linear: EnumValue;
    /**
     * For radial gradient brushes, used as the value for Brush#type.
     * @constant
     */
    static Radial: EnumValue;
    /**
     * For pattern brushes, used as the value for Brush#type.
     * @constant
     */
    static Pattern: EnumValue;
    /**
     * For lightening and darkening, used as a color-space value.
     * @constant
     */
    static Lab: EnumValue;
    /**
     * For lightening and darkening, used as a color-space value.
     * @constant
     */
    static HSL: EnumValue;
}
/**
 * This is the abstract base class for all Panel Layouts, which inform the possible Panel types.
 * It is possible to create your own Panel type by creating a subclass of PanelLayout,
 * though this is not common and not recommended for beginners.
 *
 * By default, GoJS has 12 panel types, each corresponding to a PanelLayout subclass:
 *   - `'Position', PanelLayoutPosition`
 *   - `'Horizontal', PanelLayoutHorizontal`
 *   - `'Vertical', PanelLayoutVertical`
 *   - `'Spot', PanelLayoutSpot`
 *   - `'Auto', PanelLayoutAuto`
 *   - `'Table', PanelLayoutTable`
 *   - `'Viewbox', PanelLayoutViewbox`
 *   - `'TableRow', PanelLayoutTableRow`
 *   - `'TableColumn', PanelLayoutTableColumn`
 *   - `'Link', PanelLayoutLink`
 *   - `'Grid', PanelLayoutGrid`
 *   - `'Graduated', PanelLayoutGraduated`
 *
 * These are included by default in builds of `go.js` and `go-debug.js`.
 * When building from source, you can optionally exclude all of them except `Position`,
 * `Vertical`, `Auto`, `Link`, and `Grid`.
 * This is demonstrated in `minimalSource` and `maximalSource`, in the `/projects` folder.
 *
 * Adding a new Layout is done by calling the static function, Panel.definePanelLayout:
 *
 * ```js
 * Panel.definePanelLayout('Table', new PanelLayoutTable());
 * ```
 *
 * Each PanelLayout defines a #measure and #arrange routine.
 * The measure routine must call #measureElement with each element of the Panel,
 * and the arrange routine must similarly call #arrangeElement with each element of the Panel.
 *
 * There is an example PanelLayout in the <a href="../../samples/panelLayout.html">PanelLayout sample</a>.
 * @since 2.0
 * @unrestricted
 */
export abstract class PanelLayout {
    name: string;
    constructor();
    /**
     */
    readonly classType: Function;
    /**
     * Given the available size, measure the Panel and
     * determine its expected drawing size. Sets the measuredBounds of the object.
     *
     * This must call #measureElement with each Panel element.
     *
     * This must also construct the union.width and union.height of the passed in union Rect argument.
     * This union must reflect the measured size of the panel.
     *
     * @expose
     * @param {Panel} panel Panel which called this layout
     * @param {number} width expected width of the panel
     * @param {number} height expected width of the panel
     * @param {Array.<GraphObject>} elements Array of Panel elements
     * @param {Rect} union rectangle to contain the expected union bounds of every element in the Panel. Useful for arrange.
     * @param {number} minw minimum width of the panel
     * @param {number} minh minimum height of the panel
     */
    measure(panel: Panel, width: number, height: number, elements: Array<GraphObject>, union: Rect, minw: number, minh: number): void;
    /**
     * Given the available size, measure one element of the Panel and
     * determine its expected drawing size. Sets the measuredBounds of the object.
     *
     * @param {GraphObject} obj Panel which called this layout
     * @param {number} width expected width of the GraphObject
     * @param {number} height expected width of the GraphObject
     * @param {number} minw minimum width of the GraphObject
     * @param {number} minh minimum height of the GraphObject
     */
    protected measureElement(obj: GraphObject, width: number, height: number, minw: number, minh: number): void;
    /**
     * Given the panel and its list of elements, arrange each element.
     *
     * This must call #arrangeElement with each Panel element, which will set that element's GraphObject#actualBounds.
     *
     * For arranging some elements, it is useful to know the total unioned area of every element.
     * This Rect can be used to right-align or center-align, etc, elements within an area.
     *
     * @expose
     * @param {Panel} panel Panel which called this layout
     * @param {Array.<GraphObject>} elements Array of Panel elements
     * @param {Rect} union rectangle, if properly constructed in #measure, that contains the expected union bounds of every element in the Panel.
     */
    arrange(panel: Panel, elements: Array<GraphObject>, union: Rect): void;
    /**
     * Arranges the GraphObject onto its parent Panel.
     * The passed-in numbers typically account for GraphObject#margin and other offsets.
     * This sets GraphObject#actualBounds.
     *
     * @param {GraphObject} obj GraphObject to be arranged.
     * @param {number} fx The final x value of actualBounds that the Panel computes for the GraphObject.
     * @param {number} fy The final y value of actualBounds that the Panel computes for the GraphObject.
     * @param {number} fw The final width value of actualBounds that the Panel computes for the GraphObject.
     * @param {number} fh The final height value of actualBounds that the Panel computes for the GraphObject.
     * @param {Rect=} clipRect an optional area to constrain this actualBounds to when picking and drawing.
     * By default, this is only used with Fixed/Table panels element, provided as a Rect.
     */
    protected arrangeElement(obj: GraphObject, fx: number, fy: number, fw: number, fh: number, clipRect?: Rect): void;
}
/**
 * A Panel is a GraphObject that holds other GraphObjects as its elements.
 * A Panel is responsible for sizing and positioning its elements.
 * The elements of a panel are drawn in the order in which they appear in the #elements collection.
 *
 * The Part class inherits from Panel; Part in turn is the base class of Node and Link.
 *
 * Every Panel has a #type and establishes its own coordinate system. The type of a Panel
 * determines how it will size and arrange its elements:
 *   - Panel.Position is used to arrange elements based on their absolute positions within the Panel's local coordinate system.
 *   - Panel.Vertical and Panel.Horizontal are used to create linear "stacks" of elements.
 *   - Panel.Auto is used to size the main element to fit around other elements in the Panel.
 *   - Panel.Spot is used to arrange elements based on the Spot properties GraphObject#alignment
 *     and GraphObject#alignmentFocus, relative to a main element of the panel.
 *     Spot panels can align relative to other elements by using Panel#alignmentFocusName.
 *   - Panel.Table is used to arrange elements into rows and columns, typically employing the different
 *     elements' GraphObject#row, GraphObject#rowSpan, GraphObject#column,
 *     and GraphObject#columnSpan properties. This Panel type also makes use of RowColumnDefinition.
 *   - Panel.TableRow and Panel.TableColumn can only be used immediately within a Panel.Table Panel
 *     to organize a collection of elements as a row or as a column in a table.
 *   - Panel.Viewbox is used to automatically resize a single element to fit inside the panel's available area.
 *   - Panel.Grid is not used to house typical elements, but is used only to draw regular patterns of lines.
 *     The elements must be Shapes used to describe the repeating lines.
 *   - Panel.Link is only used by Link parts and Link Adornments.
 *   - Panel.Graduated is used to draw regular tick marks and text along the main Shape element.
 *
 * Using <a href="../../intro/buildingObjects.html">GraphObject.make</a>, the second argument can be used to declare the Panel type:
 * ```js
 * // Either:
 * $(go.Panel, go.Panel.Horizontal, ...
 * // Or:
 * $(go.Panel, "Horizontal", ...
 *
 * // Full example:
 * $(go.Panel, "Horizontal",
 *   { width: 60, height: 60 }, // panel properties
 *   // elements in the panel:
 *   $(go.Shape, "Rectangle", { stroke: "lime" }),
 *   $(go.TextBlock, "Some Text")
 * ) // end of panel definition
 * ```
 *
 * <p class="boxread">
 * For an overview of most Panel types, please read the <a href="../../intro/panels.html">Introduction page on Panels</a>.
 *
 * Panel.Vertical and Panel.Horizontal panels are frequently used to position two or more GraphObjects
 * vertically above each other or horizontally next to each other.  Use the GraphObject#alignment or
 * GraphObject#stretch properties on the individual elements to control their position and size.
 * Set #isOpposite to true if you want the elements arranged from right-to-left in Horizontal Panels
 * or from bottom-to-top in Vertical Panels.
 *
 * Panel.Spot and Panel.Auto panels have a "main" element, signified by the Panel's first element
 * with GraphObject#isPanelMain set to true.  If there is no such element, it uses the first element as the "main" one.
 * Use the GraphObject#alignment property to position elements with respect to the main element.
 * Use the GraphObject#alignmentFocus property to further specify the position within Spot Panels.
 * "Spot" and "Auto" Panels should have two or more elements in them.
 *
 * In Panel.Table panels you will want to set the GraphObject#row and GraphObject#column
 * properties on each element.  The GraphObject#alignment and GraphObject#stretch properties are also
 * useful when an element's table cell is larger than that element.
 * <p class="boxrun">Please read the <a href="../../intro/tablePanels.html">Introduction page on Table Panels</a>
 * for more examples and explanation.
 *
 * Panel.TableRow and Panel.TableColumn panels can only be used as elements within a Panel.Table Panel.
 * They are typically only used in item templates, e.g. for automatically creating rows in a Table Panel based on model data
 * provided in an #itemArray. You will still need to specify properties on the individual elements within a TableRow
 * or TableColumn as if they were immediate elements of the containing Table panel.
 * <p class="boxrun">For an example that uses TableRow Panels, see <a href="../../samples/records.html">Records sample</a>.
 *
 * Panel.Grid panels are often used for the Diagram's Diagram#grid.
 * ```js
 * $(go.Diagram, "myDiagramDiv",
 *   { . . .
 *     grid: $(go.Panel, "Grid",
 *             { gridCellSize: new go.Size(40, 40) },
 *             $(go.Shape, "LineH", { stroke: "lightgray" }),
 *             $(go.Shape, "LineV", { stroke: "lightgray" })
 *           ),
 *     . . .
 *   });
 * ```
 * Or to get a green bar effect:
 * ```js
 * $(go.Diagram, "myDiagramDiv",
 *   { . . .
 *     grid: $(go.Panel, "Grid",
 *             { gridCellSize: new go.Size(100, 100) },
 *             $(go.Shape, "BarH", { fill: "lightgreen", height: 50 })
 *           ),
 *     . . .
 *   });
 * ```
 * But Grid Panels can also be stand alone objects:
 * ```js
 * $(go.Node, go.Panel.Grid,
 *   { gridCellSize: new go.Size(6, 6), width: 60, height: 60 },
 *   $(go.Shape, "LineH", { stroke: "gray" }),
 *   $(go.Shape, "LineV", { stroke: "gray" }))
 * ```
 *
 * A Grid Panel's elements do not participate in object picking.
 * <p class="boxrun">Please read the <a href="../../intro/grids.html">Introduction page on Grid Patterns</a>
 * for more examples and explanation.
 *
 * Panel.Graduated panels, like Spot and Auto Panels have a "main" element. The other elements within a Graduated Panel
 * are used to define ticks and labels to draw along the main shape's path.
 * ```js
 * $(go.Part, "Graduated",
 *   $(go.Shape, { geometryString: "M0 0 H400" }),
 *   $(go.Shape, { geometryString: "M0 0 V10" }),
 *   // offset to display below ticks
 *   $(go.TextBlock, { segmentOffset: new go.Point(0, 12) })
 * );
 * ```
 *
 * Only the main shape of a Graduated Panel participates in object picking, but a `background`
 * can be set if the entire panel needs to be pickable.
 * You cannot set or bind the Panel#itemArray of a Graduated Panel.
 * Events on the tick Shapes and TextBlock labels of a Graduated Panel will be ignored.
 * Graduated Panel TextBlock labels cannot be edited.
 *
 * Rotating the main shape will not rotate the ticks, just as rotating a Spot Panel's main element
 * won't rotate its children. Rotation should generally be done at the Panel level. Another similarity
 * to Spot Panels is that resizing of a Graduated Panel should generally be done on the main shape.
 * <p class="boxrun">Please read the <a href="../../intro/graduatedPanels.html">Introduction page on Graduated Panels</a>
 * for more examples and explanation.
 * <h3>Changing and accessing elements of a Panel</h3>
 *
 * You can change the collection of #elements by calling #add, #insertAt, #remove, or #removeAt.
 * You can get direct access to a particular element by calling #elt.
 *
 * You can search the visual tree of a Panel for GraphObjects that given a GraphObject#name using #findObject.
 * <h3>Panel Size and Appearance</h3>
 *
 * Panels typically compute their own size based on their elements and Panel #type,
 * but can also be sized by setting GraphObject#desiredSize, GraphObject#minSize, and GraphObject#maxSize.
 * Setting an explicit size on a Panel may cause nested elements of that panel to size themselves differently,
 * especially in the cases of nested elements having a GraphObject#stretch value or TextBlock's having word wrap.
 *
 * Panels have no visual components of their own unless a GraphObject#background or GraphObject#areaBackground is specified
 * or separators are specified either as defaults for the whole Table Panel or on individual RowColumnDefinitions.
 * Panels can specify #padding, to make the Panel larger including its background. Setting a padding
 * when the Panel is constrained in size will reduce the total area that it has to arrange its elements. Setting a #margin
 * will not do this -- instead the Panel will expand in size.
 *
 * In addition to the GraphObject properties on elements that are only used by certain types of panels,
 * several Panel properties only apply to specific Panel types.
 *   - Panels of #type Panel.Table use the #rowCount, #rowSizing, #columnCount,
 *     #columnSizing, #leftIndex, #topIndex, and all of the "default" separator properties.
 *   - Panels of #type Panel.TableRow and Panel.TableColumn do not act like regular GraphObjects,
 *     instead they are only to be used immediately within a Panel.Table. They are pass-through containers
 *     that hold elements for their parent table, and ignore their own scale and angle.
 *   - Panels of #type Panel.Grid use the #gridCellSize and #gridOrigin properties.
 *   - Panels of #type Panel.Viewbox use the #viewboxStretch property.
 *   - Panels of #type Panel.Graduated use the #graduatedMin, #graduatedMax,
 *     #graduatedTickUnit, and #graduatedTickBase properties.
 *
 * <p class="boxread">
 * For live examples of all Panel types, see the <a href="../../intro/panels.html">Introduction page on Panels.</a>
 * <h3>Data Binding</h3>
 *
 * Panels also provide fundamental support for data binding.
 * When a diagram gets a new model or when a diagram's model is augmented with additional data,
 * the diagram automatically creates a new Node or Link whose #data property refers to the corresponding
 * node data or link data object.
 * <p class="boxread">
 * For more discussion of data binding, please read the <a href="../../intro/usingModels.html">Introduction page on Models</a>
 * and <a href="../../intro/dataBinding.html">Data Binding</a>.
 *
 * Panels provide support for automatically creating elements within the Panel based on items in a JavaScript Array.
 * This is achieved by setting or binding the #itemArray property, which acts in a manner similar
 * to the Model#nodeDataArray property.
 * You can supply an #itemTemplate, which must itself be a simple Panel, which is copied to create the element
 * in this container Panel corresponding to an item in the itemArray.
 * This property is analogous to the Diagram#nodeTemplate property, although for the diagram the
 * template must be a Node, Group, or simple Part.
 *
 * Much like the Diagram#nodeTemplateMap, Panel's #itemTemplateMap supports having multiple templates,
 * so that the actual structure of the element created for a data item can be chosen dynamically.
 * Just as the Model#nodeCategoryProperty determines which template in the Diagram#nodeTemplateMap
 * is copied to create a Node, the #itemCategoryProperty names the data property whose value
 * chooses the Panel in the itemTemplateMap to copy for the item.
 *
 * When binding the #itemArray property, it is commonplace to set Model#copiesArrays and
 * Model#copiesArrayObjects properties to true, so that when a node is copied, the item Array and its
 * contents are copied, not shared.  Or more generally, to customize the model's copying processes, you
 * can supply a custom Model#copyNodeDataFunction.
 * <p class="boxread">
 * For more discussion and examples of item arrays, please read the <a href="../../intro/itemArrays.html">Introduction page on Item Arrays</a>.
 * @extends GraphObject
 * @unrestricted
 */
export class Panel extends GraphObject {
    /**
     * Constructs an empty Panel of the given #type. Default type is Panel.Position.
     * The panel type must be one of the enumerated values defined on the Panel class, including:
     * Panel.Position, Panel.Vertical, Panel.Horizontal, Panel.Table,
     * Panel.Auto, Panel.Spot, or Panel.Viewbox.
     * @param {EnumValue=} type If not supplied, the default Panel type is Panel.Position.
     */
    constructor(type?: PanelLayout);
    /**
     * Creates a deep copy of this Panel and returns it.
     * @return {Panel}
     */
    copy(): this;
    /**
     * Gets or sets the type of the Panel, which controls how the Panel's elements are measured and arranged.
     * The only accepted values are listed as constant properties of Panel, including:
     *   - Panel.Position
     *   - Panel.Vertical
     *   - Panel.Horizontal
     *   - Panel.Auto
     *   - Panel.Spot
     *   - Panel.Table
     *   - Panel.Viewbox
     *   - Panel.Link (see also Links, which are all Panels of type Link)
     *   - Panel.TableRow
     *   - Panel.TableColumn
     *   - Panel.Grid
     *   - Panel.Graduated
     *
     * The default value is Panel.Position.
     *
     * You must not modify this property once the Panel has been measured or the Panel has been added to another Panel.
     */
    type: PanelLayout;
    /**
     * This read-only property returns an iterator over the collection of the GraphObjects that this panel manages.
     *
     * You can change the collection by calling #add, #insertAt, #remove, or #removeAt.
     *
     * You can also get direct access to individual elements by calling #elt.
     */
    readonly elements: Iterator<GraphObject>;
    /**
     * Gets or sets the space between this Panel's border and its content.
     * Unlike GraphObject#margin, padding expands the area inside of the Panel's border.
     * If this Panel's size is unconstrained, this will increase the size of the panel.
     * If this Panel's size is constrained, this will decrease the total area
     * for the Panel elements to arrange themselves.
     *
     * Unlike margin, increases in size due to padding are visually covered by the
     * GraphObject#background and GraphObject#areaBackground.
     *
     * Padding cannot contain negative numbers.
     * The default value is a Margin of zero.
     * @see GraphObject#margin
     */
    padding: MarginLike;
    /**
     * Gets or sets the default alignment spot of this Panel, used as
     * the alignment for an element when its GraphObject#alignment value is Spot.Default.
     * The default value is Spot.Default, which is interpreted by the Panel in whatever manner seems reasonable,
     * depending on the Panel type.
     */
    defaultAlignment: Spot;
    /**
     * Gets or sets the default stretch of this Panel, used as
     * the stretch for an element when its GraphObject#stretch value is GraphObject.Default.
     * The default value is GraphObject.Default, which typically resolves to GraphObject.None.
     */
    defaultStretch: EnumValue;
    /**
     * For Panel.Table|Table Panels: Gets or sets the additional padding for rows and columns.
     * Padding is applied both before and after a row or column's contents.
     * @see RowColumnDefinition#separatorPadding
     * @see #defaultColumnSeparatorStrokeWidth
     * @see #defaultRowSeparatorStrokeWidth
     * @since 1.2
     */
    defaultSeparatorPadding: MarginLike;
    /**
     * For Panel.Table|Table Panels: Gets or sets the default stroke (color) for rows
     * provided a given row has a nonzero RowColumnDefinition#separatorStrokeWidth.
     * RowColumnDefinition#separatorStroke can override this default value.
     * The default value is null -- no line is drawn.
     * @TablePanel
     * @see #defaultRowSeparatorStrokeWidth
     * @see #defaultRowSeparatorDashArray
     * @see RowColumnDefinition#separatorStroke
     * @see #defaultColumnSeparatorStroke
     * @since 1.2
     */
    defaultRowSeparatorStroke: BrushLike;
    /**
     * For Panel.Table|Table Panels: Gets or sets the default stroke width for a row's separator.
     * RowColumnDefinition#separatorStrokeWidth can override this default value.
     * The default value is 1.  Any new value must be a real, non-negative number.
     * @see RowColumnDefinition#separatorStrokeWidth
     * @see #defaultColumnSeparatorStrokeWidth
     * @since 1.2
     */
    defaultRowSeparatorStrokeWidth: number;
    /**
     * For Panel.Table|Table Panels: Gets or sets the default dash array for a row's separator.
     * RowColumnDefinition#separatorDashArray can override this default value.
     *
     * Must be an array of positive numbers and zeroes,
     * or else null to indicate a solid line.
     *
     * For example, the array [5, 10] would create dashes of 5 pixels and spaces of 10 pixels.
     *
     * Setting an array with all zeroes will set the value to null.
     *
     * Default is null.
     * @see RowColumnDefinition#separatorDashArray
     * @see #defaultColumnSeparatorDashArray
     * @since 1.2
     */
    defaultRowSeparatorDashArray: Array<number> | null;
    /**
     * For Panel.Table|Table Panels: Gets or sets the default stroke (color) for columns
     * provided a given column has a nonzero RowColumnDefinition#separatorStrokeWidth.
     * RowColumnDefinition#separatorDashArray can override this default value.
     * The default value is null -- no line is drawn.
     * @see #defaultColumnSeparatorStrokeWidth
     * @see #defaultColumnSeparatorDashArray
     * @see RowColumnDefinition#separatorStroke
     * @see #defaultRowSeparatorStroke
     * @since 1.2
     */
    defaultColumnSeparatorStroke: BrushLike;
    /**
     * For Panel.Table|Table Panels: Gets or sets the default stroke width for a column's separator.
     * RowColumnDefinition#separatorStrokeWidth can override this default value.
     * The default value is 1.  Any new value must be a real, non-negative number.
     * @see RowColumnDefinition#separatorStrokeWidth
     * @see #defaultRowSeparatorStrokeWidth
     * @since 1.2
     */
    defaultColumnSeparatorStrokeWidth: number;
    /**
     * For Panel.Table|Table Panels: Gets or sets the default dash array for a column's separator.
     * RowColumnDefinition#separatorStrokeWidth can override this default value.
     *
     * Must be an array of positive numbers and zeroes,
     * or else null to indicate a solid line.
     *
     * For example, the array [5, 10] would create dashes of 5 pixels and spaces of 10 pixels.
     *
     * Setting an array with all zeroes will set the value to null.
     *
     * Default is null.
     * @see RowColumnDefinition#separatorDashArray
     * @see #defaultRowSeparatorDashArray
     * @since 1.2
     */
    defaultColumnSeparatorDashArray: Array<number> | null;
    /**
     * For Panel.Viewbox|Viewbox Panels: Gets or sets how the panel will resize its content.
     *
     * Possible values are GraphObject.Uniform and GraphObject.UniformToFill.
     * The default is GraphObject.Uniform.
     */
    viewboxStretch: EnumValue;
    /**
     * For Panel.Grid|Grid Panels: Gets or sets the distance between lines.
     * The units are in local coordinates.
     * The default is 10x10.  Any new width or height must be a positive real number.
     */
    gridCellSize: Size;
    /**
     * For Panel.Grid|Grid Panels: Gets or sets an origin point for the grid cells.
     * The units are in local coordinates.
     * The default is (0,0).  Any new value must use real numbers.
     */
    gridOrigin: Point;
    /**
     * For Panel.Graduated|Graduated Panels: Gets or sets the minimum value represented.
     * Must be less than #graduatedMax. The default is 0.
     * @since 1.7
     */
    graduatedMin: number;
    /**
     * For Panel.Graduated|Graduated Panels: Gets or sets the maximum value represented.
     * Must be greater than #graduatedMin. The default is 100.
     * @since 1.7
     */
    graduatedMax: number;
    /**
     * For Panel.Graduated|Graduated Panels: This read-only property returns the range of values represented by the Panel.
     *
     * For example, a #graduatedMin of 25 and #graduatedMax of 75 would return 50.
     * @since 1.7
     */
    readonly graduatedRange: number;
    /**
     * For Panel.Graduated|Graduated Panels: Gets or sets the difference between two consecutive values marked by ticks.
     * Must be positive. The default is 10.
     * @since 1.7
     */
    graduatedTickUnit: number;
    /**
     * For Panel.Graduated|Graduated Panels: Gets or sets the base value which is marked with a tick.
     * The default is 0.
     * @since 1.7
     */
    graduatedTickBase: number;
    /**
     * Return an immediate child element whose GraphObject#isPanelMain is true,
     * or else just return the first child element.
     * @return {GraphObject}  this may return null if there are no child elements
     * @since 1.5
     */
    findMainElement(): GraphObject | null;
    /**
     * Search the visual tree starting at this Panel for a GraphObject
     * whose GraphObject#name is the given name.
     *
     * This does not recurse into the elements inside a Panel that holds elements for an #itemArray.
     * @param {string} name The name to search for, using a case-sensitive string comparison.
     * @return {GraphObject} If no such named object can be found, this returns null.
     */
    findObject(name: string): GraphObject | null;
    /**
     * Adds a GraphObject to the end of this Panel's list of elements, visually in front of all of the other elements.
     *
     * If the element to be added is already in this Panel's list of elements,
     * the object is moved to the end of the list.
     * You cannot add a GraphObject to a Panel if that GraphObject is already in a different Panel.
     * @param {GraphObject} element A GraphObject.
     */
    add(element: GraphObject): void;
    /**
     * Returns the GraphObject in this Panel's list of elements at the specified index.
     * @param {number} idx
     * @return {GraphObject}
     */
    elt(idx: number): GraphObject;
    /**
     * Adds a GraphObject to the Panel's list of elements at the specified index.
     *
     * If the element to be added is already in this Panel's list of elements,
     * the object is moved to the specified index.
     * You cannot add a GraphObject to a Panel that if that GraphObject is already in a different Panel.
     * @param {number} index
     * @param {GraphObject} element A GraphObject.
     */
    insertAt(index: number, element: GraphObject): void;
    /**
     * Removes a GraphObject from this Panel's list of elements.
     * @param {GraphObject} element A GraphObject.
     */
    remove(element: GraphObject): void;
    /**
     * Removes an GraphObject from this Panel's list of elements at the specified index.
     * @param {number} idx
     */
    removeAt(idx: number): void;
    /**
     * For Panel.Table|Table Panels: This read-only property returns the number of rows.
     * This value is only valid after the Panel has been measured.
     */
    readonly rowCount: number;
    /**
     * For Panel.Table|Table Panels: Gets the RowColumnDefinition for a particular row.
     * If you ask for the definition of a row at or beyond the #rowCount,
     * it will automatically create one and return it.
     *
     * If this Panel is not a Table Panel, this method returns null.
     * @expose
     * @param {number} idx the non-negative zero-based integer row index.
     * @return {RowColumnDefinition}
     */
    getRowDefinition(idx: number): RowColumnDefinition;
    /**
     * For Panel.Table|Table Panels: Removes the RowColumnDefinition for a particular row.
     *
     * If this Panel is not a Table Panel, this method does nothing.
     * @param {number} idx the non-negative zero-based integer row index.
     */
    removeRowDefinition(idx: number): void;
    /**
     * For Panel.Table|Table Panels: This read-only property returns the number of columns.
     * This value is only valid after the Panel has been measured.
     */
    readonly columnCount: number;
    /**
     * For Panel.Table|Table Panels: Gets the RowColumnDefinition for a particular column.
     * If you ask for the definition of a column at or beyond the #columnCount,
     * it will automatically create one and return it.
     *
     * If this Panel is not a Table Panel, this method returns null.
     * @expose
     * @param {number} idx the non-negative zero-based integer column index.
     * @return {RowColumnDefinition}
     */
    getColumnDefinition(idx: number): RowColumnDefinition;
    /**
     * For Panel.Table|Table Panels: Removes the RowColumnDefinition for a particular row.
     *
     * If this Panel is not a Table Panel, this method does nothing.
     * @param {number} idx the non-negative zero-based integer row index.
     */
    removeColumnDefinition(idx: number): void;
    /**
     * For Panel.Table|Table Panels: Gets or sets how this Panel's rows deal with extra space.
     * Valid values are RowColumnDefinition.ProportionalExtra and RowColumnDefinition.None.
     * The default is RowColumnDefinition.ProportionalExtra.
     * @see RowColumnDefinition#sizing
     */
    rowSizing: EnumValue;
    /**
     * For Panel.Table|Table Panels: Gets or sets how this Panel's columns deal with extra space.
     * Valid values are RowColumnDefinition.ProportionalExtra and RowColumnDefinition.None.
     * The default is RowColumnDefinition.ProportionalExtra.
     * @see RowColumnDefinition#sizing
     * @see #rowSizing
     */
    columnSizing: EnumValue;
    /**
     * For Panel.Table|Table Panels: Gets or sets the first row that this Panel displays.
     * The default value is 0.
     * @see #leftIndex
     */
    topIndex: number;
    /**
     * For Panel.Table|Table Panels: Gets or sets the first column that this Panel displays.
     * The default value is 0.
     * @see #topIndex
     */
    leftIndex: number;
    /**
     * For Panel.Table|Table Panels: Returns the row at a given y-coordinate in local coordinates.
     * Call GraphObject#getLocalPoint to convert a Point in document coordinates into
     * a Point in local coordinates.
     *
     * @param {number} y
     * @return {number} a zero-based integer
     * @see #findColumnForLocalX
     * @since 1.2
     */
    findRowForLocalY(y: number): number;
    /**
     * For Panel.Table|Table Panels: Returns the cell at a given x-coordinate in local coordinates.
     * Call GraphObject#getLocalPoint to convert a Point in document coordinates into
     * a Point in local coordinates.
     *
     * @param {number} x
     * @return {number} a zero-based integer
     * @see #findRowForLocalY
     * @since 1.2
     */
    findColumnForLocalX(x: number): number;
    /**
     * For Panel.Graduated|Graduated Panels: Returns the point that corresponds with a value, in the panel's coordinates.
     *
     * If the value provided is not within the #graduatedMin and #graduatedMax,
     * it will be constrained to within those values.
     *
     * If this Panel is not a Graduated Panel, this method returns Point(NaN, NaN).
     * @param {number} val a value between #graduatedMin and #graduatedMax
     * @param {Point=} result an optional Point that is modified and returned
     * @return {Point}
     * @see #graduatedValueForPoint
     * @since 1.7
     */
    graduatedPointForValue(val: number, result?: Point): Point;
    /**
     * For Panel.Graduated|Graduated Panels: Returns the value that corresponds with the given Point.
     * The Point must be in the panel's coordinates.
     * The value returned will be in the Graduated Panel's range.
     *
     * If this Panel is not a Graduated Panel, this method returns NaN.
     * @param {Point} pt a Point in the Graduated Panel's coordinates
     * @return {number}
     * @see #graduatedPointForValue
     * @since 1.7
     */
    graduatedValueForPoint(pt: Point): number;
    /**
     * Gets or sets the optional model data to which this panel is data-bound.
     * The data must be a JavaScript Object if this is a Part.
     * The data can be any JavaScript value if this is a Panel created for
     * an item in an Array that was data-bound by the #itemArray property.
     * The default value is null.
     *
     * Setting it to a new value automatically calls #updateTargetBindings
     * in order to assign new values to all of the data-bound GraphObject properties.
     *
     * Once you set this property you cannot add, remove, or modify any data bindings
     * on any of the GraphObjects in the visual tree of this Panel, including on
     * this panel itself.
     *
     * You should not modify this property on a Part that is created automatically
     * to represent model data, nor on a Panel that is created automatically for a data item
     * in the containing Panel's Panel#itemArray.
     * Call Model#removeNodeData and Model#addNodeData if you want to replace
     * this Part with another one, or call Model#removeArrayItem and Model#insertArrayItem
     * if you want to replace this Panel with another one.
     *
     * Although you might not be able to replace this data value if this Part was created
     * automatically by the Diagram, you can still modify that data object's properties.
     * Call the appropriate Model method for changing properties that affect the structure of the diagram.
     * Call Model#setDataProperty for changing other properties that may be the sources
     * of Bindings on GraphObject properties that are in the visual tree of this panel/part.
     */
    data: any | null;
    /**
     * Gets the index of this Panel's data if it was created to represent an item in its containing Panel's Panel#itemArray.
     * The default value is NaN.
     *
     * This is only set internally by code such as #rebuildItemElements or Model#insertArrayItem
     * when building or shifting Panel representing items
     * in the Panel whose Panel#itemArray was set or bound to an Array of value.
     *
     * This property can be used in data bindings within the item template to produce values
     * that depend on its position in the item Array.
     * For example:
     * ```js
     *  $(go.Panel, // the item Panel
     *    . . .,
     *    new go.Binding("itemArray", "someProperty"),
     *    {
     *      itemTemplate:
     *        $(go.Panel,
     *          // set Panel.background to a color based on the Panel.itemIndex
     *          new go.Binding("background", "itemIndex",
     *                         // using this conversion function
     *                         function(i) { return (i%2 === 0) ? "lightgreen" : "lightyellow"; })
     *                   // bound to this Panel itself, not to the Panel.data item
     *                   .ofObject(),
     *          $(go.TextBlock,  // a trivial item template, just showing some text
     *            new go.Binding("text"))  // sets TextBlock.text = data.text
     *        )
     *    }
     *  )
     * ```
     *
     * The main element of a Spot or Auto or Link Panel, or the first TableRow or TableColumn element of a Table Panel
     * whose #isPanelMain property is true, will not have this property set to a number,
     * because it will not have been created by #rebuildItemElements.
     * @since 1.4
     */
    itemIndex: number;
    /**
     * Undocumented
     */
    freezeBindings(): void;
    /**
     * Undocumented.
     * Make a deep copy of this Panel and allow it to be a template.
     * This makes copies of Bindings, unlike the regular copy().
     * @return {Panel}
     */
    copyTemplate(): Panel;
    /**
     * Re-evaluate all data bindings on this panel,
     * in order to assign new property values to the GraphObjects in this visual tree
     * based on this object's #data property values.
     *
     * It is better to call Model#setDataProperty to modify data properties,
     * because that will both record changes for undo/redo and will update all bindings
     * that may depend on that property.
     *
     * This method does nothing if #data is null.
     * @param {string=} srcprop An optional source data property name:
     *   when provided, only evaluates those Bindings that use that particular property;
     *   when not provided or when it is the empty string, all bindings are evaluated.
     * @see Model#updateTargetBindings
     */
    updateTargetBindings(srcprop?: string): void;
    /**
     * Gets or sets a JavaScript Array of values or objects, each of which will be represented
     * by a Panel as elements in this Panel.
     * Replacing this array results all of this panel's child objects being replaced with
     * a copy of the Panel found in #itemTemplateMap for each particular
     * item in the Array.
     *
     * Because the software does not receive any notifications when an Array is modified,
     * any insertions or removals or replacements of data in the Array will not be noticed
     * unless you call Model#insertArrayItem or Model#removeArrayItem.
     * You may also reset this property to its current value (the modified Array) or
     * call #updateTargetBindings, if there is a Binding whose target is this property.
     *
     * When binding this property, it is commonplace to set Model#copiesArrays and
     * Model#copiesArrayObjects properties to true, so that when a node is copied,
     * the item Array and its contents are copied, not shared.
     * Or more generally, to customize the model's copying processes, you can supply a custom
     * Model#copyNodeDataFunction and perhaps a GraphLinksModel#copyLinkDataFunction.
     *
     * Any JavaScript Object that is in this Array must only appear once in the array
     * and must not appear in any other Panel#itemArrays.
     * Use #findItemPanelForData to find the data-bound Panel created for an Object in this panel's item Array.
     *
     * Non-Object values in an item Array may appear multiple times.
     * An item Array may be shared by multiple Panels.
     *
     * Item Arrays should not be used with Grid Panels or Graduated Panels as they may not have nested Panels.
     */
    itemArray: Array<any> | null;
    /**
     * Create and add new GraphObjects corresponding to and bound to the data in the #itemArray,
     * after removing all existing elements from this Panel.
     * This method is automatically called when replacing the #itemArray value,
     * or when changing the value of #itemTemplate or #itemTemplateMap.
     *
     * This uses #itemCategoryProperty to determine the category for an item data.
     * That string is used to look up a template in #itemTemplateMap.
     * The resulting template (which is also a Panel) is copied, added to this panel,
     * and its #itemIndex is set to its index in that Array.
     * That new child Panel is then data-bound to that Array item by setting its #data.
     *
     * If #itemArray is null, this method just removes all elements from this panel.
     * Actually, if this Panel #type is "Spot", "Auto", or "Link", the very first element
     * is always kept by this method.
     * Also, if this Panel type is "Table", and if the first element is a "TableRow" or "TableColumn" Panel
     * whose #isPanelMain property is set to true, that first element will be kept too.
     * That is useful for defining literal TableRow headers in Table panels, when the header information
     * is not kept as the first item in the itemArray.
     *
     * It is wasteful to call this method after making some model data changes.
     * It is better to call Model#setDataProperty, Model#addArrayItem,
     * Model#insertArrayItem, or Model#removeArrayItem, or other model methods.
     * Not only do those methods update efficiently, they also preserve unbound state and support undo/redo.
     */
    rebuildItemElements(): void;
    /**
     * Return the Panel that was made for a particular data object in this panel's #itemArray.
     * If this returns a Panel, its #data property will be the argument data object,
     * and its containing GraphObject#panel will be this panel.
     * @param {Object} data must be an Object, not a string or a number or a boolean or a function
     * @return {Panel} or null if not found
     * @since 1.6
     */
    findItemPanelForData(data: ObjectData): Panel | null;
    /**
     * Gets or sets the default Panel template used as the archetype
     * for item data that are in #itemArray.
     *
     * Setting this property just modifies the #itemTemplateMap
     * by replacing the entry named with the empty string.
     * Any new value must be a Panel but not a Part.
     * By default this property is null.
     *
     * GraphObject#copy when copying a panel will share the #itemTemplateMap
     * between the original panel and the copied panel.
     */
    itemTemplate: Panel;
    /**
     * Gets or sets a Map mapping template names to Panels.
     * One of these Panels is copied for each item data that is in the #itemArray.
     * Replacing this map will automatically rebuild all of the elements in this Panel.
     *
     * By default this property is null.
     * All values in the Map must be Panels but not Parts.
     *
     * If you modify this Map, by replacing a Panel or by adding or
     * removing a map entry, you need to explicitly call #rebuildItemElements afterwards.
     *
     * GraphObject#copy when copying a panel will share the #itemTemplateMap
     * between the original panel and the copied panel.
     */
    itemTemplateMap: Map<string, Panel> | null;
    /**
     * Gets or sets the name of the item data property that returns a string describing that data's category,
     * or a function that takes an item data object and returns that string;
     * the default value is the name 'category'.
     * This is used to distinguish between different
     * kinds of items in the #itemArray.
     *
     * The name must not be null.
     * If the value is an empty string, the category is assumed to be an empty string,
     * the default category name, for all item data objects.
     * You must not change this property when the #itemArray already has a value.
     */
    itemCategoryProperty: string | ((a: any) => string);
    /**
     * For Spot Panels: Gets or sets whether this Panel's main element clips instead of fills.
     * The main element will not paint its stroke, if it has any.
     * This assumes that the main element is a Shape.
     *
     * When this property is `true`, the Spot panel will size itself to be the *intersection* of the main element bounds and
     * all other elements' bounds, rather than the union of these bounds.
     * @since 1.7
     */
    isClipping: boolean;
    /**
     * For Panel.Horizontal|Horizontal and Panel.Vertical|Vertical Panels: gets or sets whether this Panel arranges its contents from the
     * typical side (left and top, respectively), or the opposite side (right and bottom, respectively).
     *
     * The default value is false.
     * @since 1.7
     */
    isOpposite: boolean;
    /**
     * Gets or sets whether this Panel or any GraphObject inside the panel actually responds to user click events.
     * It may be used as a Binding target.
     * See how this property is used in <a href="../../extensions/Buttons.js">Buttons.js</a>.
     *
     * This property does not have any effect on picking ("hit-testing") -- that behavior is implemented
     * by the GraphObject#pickable property.
     * When this property is false, non-click events may still occur on this panel or on objects within this panel.
     * This property is normally only used for Panels that are GraphObject#isActionable.
     *
     * Call GraphObject#isEnabledObject to decide whether a particular object can be clicked.
     * @see GraphObject#isEnabledObject
     * @see GraphObject#enabledChanged
     * @see GraphObject#isActionable
     * @see GraphObject#pickable
     * @since 1.7
     */
    isEnabled: boolean;
    /**
     * For Panels which are elements of Spot Panels: Gets or sets the name
     * of this Panel's element that should be used as the alignment object instead of this Panel.
     *
     * This allows Spot Panels to align objects that are nested in the element tree of its own elements.
     * @since 1.7
     */
    alignmentFocusName: string;
    /**
     * Register a PanelLayout. This is called when making new Panel types.
     * See the <a href="../../samples/panelLayout.html">PanelLayout sample</a> for an example.
     * @param {string} layoutName Panel name
     * @param {PanelLayout} layout instance of the PanelLayout
     */
    static definePanelLayout(layoutName: string, layout: PanelLayout): void;
    /**
     * The default #type, arranges each element according to their GraphObject#position.
     * @constant
     */
    static readonly Position: PanelLayout;
    /**
     * This value for #type lays out the elements horizontally with
     * their GraphObject#alignment property dictating their alignment on the Y-axis.
     * @constant
     */
    static readonly Horizontal: PanelLayout;
    /**
     * This value for #type lays out the elements vertically with
     * their GraphObject#alignment property dictating their alignment on the X-axis.
     * @constant
     */
    static readonly Vertical: PanelLayout;
    /**
     * This value for #type arranges GraphObjects about a main element using the
     * GraphObject#alignment and GraphObject#alignmentFocus properties;
     * the main element is the first GraphObject with GraphObject#isPanelMain set to true,
     * or else the first GraphObject if none have that property set to true.
     * @constant
     */
    static readonly Spot: PanelLayout;
    /**
     * This value for #type resizes the main element to fit around the other elements;
     * the main element is the first GraphObject with GraphObject#isPanelMain set to true,
     * or else the first GraphObject if none have that property set to true.
     * @constant
     */
    static readonly Auto: PanelLayout;
    /**
     * This value for #type arranges GraphObjects into rows and columns;
     * set the GraphObject#row and GraphObject#column properties on each element.
     * @constant
     */
    static readonly Table: PanelLayout;
    /**
     * This value for #type rescales a single GraphObject to fit inside the panel
     * depending on the element's GraphObject#stretch property.
     * @constant
     */
    static readonly Viewbox: PanelLayout;
    /**
     * Organizational Panel type that is only valid inside of a Table panel;
     * This Panel ignores its angle and scale, and does not  have a meaningful size on its own,
     * it is only an organizational container for other elements of a Panel.
     * @constant
     * @since 1.1
     */
    static readonly TableRow: PanelLayout;
    /**
     * Organizational Panel type that is only valid inside of a Table panel;
     * This Panel ignores its angle and scale, and does not  have a meaningful size on its own,
     * it is only an organizational container for other elements of a Panel.
     * @constant
     * @since 1.1
     */
    static readonly TableColumn: PanelLayout;
    /**
     * This value for #type is used for Links and adornments that act as Links.
     * @constant
     */
    static readonly Link: PanelLayout;
    /**
     * This value for #type is used to draw regular patterns of lines.
     * @constant
     */
    static readonly Grid: PanelLayout;
    /**
     * This value for #type is used to draw regular tick marks and labels along some shape.
     * The main element is the first GraphObject with GraphObject#isPanelMain set to true,
     * or else the first GraphObject if none have that property set to true.
     * @constant
     * @since 1.7
     */
    static readonly Graduated: PanelLayout;
}
/**
 * The RowColumnDefinition class describes constraints on a row or a column
 * in a Panel of type Panel.Table.
 * It also provides information about the actual layout after the
 * Table Panel has been arranged.
 * @unrestricted
 */
export class RowColumnDefinition {
    /**
     * You should not use this constructor, because calls to
     * Panel#getRowDefinition or Panel#getColumnDefinition
     * will automatically create and remember a RowColumnDefinition for you.
     */
    constructor();
    /**
     * The default #sizing, which resolves to RowColumnDefinition.None or else
     * the Table Panel's rowSizing and columnSizing if present.
     * @constant
     */
    static Default: EnumValue;
    /**
     * The default #sizing if none is specified on the Table Panel's rowSizing and columnSizing.
     * @constant
     */
    static None: EnumValue;
    /**
     * If a Table Panel is larger than all the rows then this #sizing
     * grants this row and any others with the same value the extra space, apportioned proportionally between them
     * @constant
     */
    static ProportionalExtra: EnumValue;
    /**
     * Undocumented
     * @return {number}
     * @see #computeEffectiveSpacing
     */
    computeEffectiveSpacingTop(): number;
    /**
     * Undocumented
     * @return {number}
     */
    computeEffectiveSpacing(): number;
    /**
     * This read-only property returns the Panel that this row or column definition is in.
     */
    readonly panel: Panel | null;
    /**
     * This read-only property is true when this describes a row instead of a column in the #panel.
     * When this is true, the #height, #minimum, and #maximum
     * all describe the row height.
     * Otherwise #width and the other two properties describe the column width.
     */
    isRow: boolean;
    /**
     * This read-only property returns which row or column this RowColumnDefinition describes in the #panel.
     * The value is a zero-based integer.
     * @see Panel#findColumnForLocalX
     * @see Panel#findRowForLocalY
     * @see Panel#leftIndex
     * @see Panel#topIndex
     * @see #position
     */
    index: number;
    /**
     * Gets or sets the row height, in local coordinates.
     * This describes the row height if #isRow is true;
     * otherwise this property is meaningless.
     * The value must be non-negative and finite.
     *
     * The default value is NaN, which means this row will get a height
     * that is just big enough to hold all of the objects in the row.
     *
     * Setting this value to a number will mean that all of the objects
     * of this Panel in this row will be allocated that
     * amount of row height.
     * Whether an object in the row is actually arranged to have that height
     * depends on whether the GraphObject#stretch stretches vertically.
     * @see #width
     * @see #minimum
     * @see #maximum
     * @see #sizing
     */
    height: number;
    /**
     * Gets or sets the column width, in local coordinates.
     * The size describes the column width if #isRow is false;
     * otherwise this property is meaningless.
     * The value must be non-negative and finite.
     *
     * The default value is NaN, which means this column will get a width
     * that is just big enough to hold all of the objects in the column.
     *
     * Setting this value to a number will mean that all of the objects
     * of this Panel in this column will be allocated that
     * amount of column width.
     * Whether an object in the column is actually arranged to have that width
     * depends on whether the GraphObject#stretch stretches horizontally.
     * @see #height
     * @see #minimum
     * @see #maximum
     * @see #sizing
     */
    width: number;
    /**
     * Gets or sets the minimum row height or column width, in local coordinates.
     * The minimum describes the row height if #isRow is true;
     * otherwise this describes the column width.
     *
     * The value must be non-negative and finite.
     * The default value is zero.
     * The arranged height of all objects in this row,
     * or the arranged width of all objects in this column,
     * will be no less than this value.
     * @see #width
     * @see #height
     * @see #maximum
     * @see #sizing
     */
    minimum: number;
    /**
     * Gets or sets the maximum row height or column width, in local coordinates.
     * The maximum describes the row height if #isRow is true;
     * otherwise this describes the column width.
     *
     * The value must be non-negative.
     * The default value is Infinity.
     * The arranged height of all objects in this row,
     * or the arranged width of all objects in this column,
     * will be no greater than this value.
     * @see #width
     * @see #height
     * @see #minimum
     * @see #sizing
     */
    maximum: number;
    /**
     * Gets or sets a default alignment for elements that are in this row or column.
     * The value must be a Spot.
     * The default value is Spot.Default, so that this RowColumnDefinition
     * does not supply any alignment information for the row or column.
     *
     * When an element's GraphObject#alignment property is Spot.Default,
     * it gets the horizontal alignment from the element's column's RowColumnDefinition
     * and the vertical alignment from the element's row's RowColumnDefinition.
     * When that RowColumnDefinition#alignment property is also Default,
     * it takes the value from the table panel's Panel#defaultAlignment property.
     * @see GraphObject#alignment
     * @see Panel#defaultAlignment
     */
    alignment: Spot;
    /**
     * Gets or sets the default stretch for elements that are in this row or column.
     * The only accepted values are listed as constant properties of GraphObject:
     * GraphObject.None, GraphObject.Fill, GraphObject.Horizontal, GraphObject.Vertical, or GraphObject.Default.
     * The default value is GraphObject.Default, so that this RowColumnDefinition
     * does not supply any stretch information for the row or column.
     *
     * When an element's GraphObject#stretch property is GraphObject.Default,
     * it gets the horizontal stretch from the element's column's RowColumnDefinition
     * and the vertical stretch from the element's row's RowColumnDefinition.
     * When that RowColumnDefinition#stretch property is also Default,
     * it takes the value from the table panel's Panel#defaultStretch property.
     * @see GraphObject#stretch
     * @see Panel#defaultStretch
     * @see #sizing
     * @see #width
     * @see #height
     */
    stretch: EnumValue;
    /**
     * Gets or sets the additional padding for a particular row or column, in local coordinates.
     * Padding is applied on two sides - before and after a row or column's contents.
     * The #separatorStrokeWidth comes before any padding Margin#top or Margin#left.
     *
     * The default value is null, so that this RowColumnDefinition
     * does not supply any padding information for the row or column.
     * There may still be some padding between rows and/or columns if Panel#defaultSeparatorPadding is non-zero.
     * @see Panel#defaultSeparatorPadding
     * @see #separatorStroke
     * @see #separatorStrokeWidth
     * @see #separatorDashArray
     * @see #background
     * @since 1.2
     */
    separatorPadding: MarginLike;
    /**
     * Gets or sets the stroke (color) for the separator line that is drawn before a particular row or column,
     * provided that row or column has a nonzero #separatorStrokeWidth.
     *
     * The default value is null, so that this RowColumnDefinition
     * does not specify any brush for the separator line to draw before the row or column.
     * The line may still be drawn if the value of Panel#defaultRowSeparatorStroke or
     * Panel#defaultColumnSeparatorStroke is non-null.
     *
     * However, if no stroke color is specified, then no stroke width will be drawn,
     * even if the #separatorStrokeWidth value is non-zero.
     * @see Panel#defaultColumnSeparatorStroke
     * @see Panel#defaultRowSeparatorStroke
     * @see #separatorPadding
     * @see #separatorStrokeWidth
     * @see #separatorDashArray
     * @see #background
     * @since 1.2
     */
    separatorStroke: BrushLike;
    /**
     * Gets or sets the stroke width for a particular row or column's separator line, in local coordinates.
     *
     * The default value is NaN, so that this RowColumnDefinition
     * does not supply any stroke width information for what is drawn before the row or column.
     * The separator line may still be drawn if Panel#defaultRowSeparatorStroke
     * or Panel#defaultColumnSeparatorStroke is a real number (they default to 1).
     * @see Panel#defaultColumnSeparatorStrokeWidth
     * @see Panel#defaultRowSeparatorStrokeWidth
     * @see #separatorPadding
     * @see #separatorStroke
     * @see #separatorDashArray
     * @see #background
     * @since 1.2
     */
    separatorStrokeWidth: number;
    /**
     * Gets or sets the dash array for dashing the separator line, provided this
     * row or column has a nonzero RowColumnDefinition#separatorStrokeWidth and non-null RowColumnDefinition#separatorStroke.
     *
     * Must be an array of positive numbers and zeroes,
     * or else null to indicate a solid line.
     *
     * For example, the array [5, 10] would create dashes of 5 pixels and spaces of 10 pixels.
     *
     * Setting an array with all zeroes will set the value to null.
     *
     * Default is null, so that this RowColumnDefinition
     * does not supply any stroke dash array information for what is drawn before the row or column.
     * The separator line may still be drawn using dashes if Panel#defaultRowSeparatorDashArray
     * or Panel#defaultColumnSeparatorDashArray is non-null.
     * @see Panel#defaultColumnSeparatorDashArray
     * @see Panel#defaultRowSeparatorDashArray
     * @see #separatorPadding
     * @see #separatorStroke
     * @see #separatorStrokeWidth
     * @see #background
     * @since 1.2
     */
    separatorDashArray: Array<number>;
    /**
     * Gets or sets the background color for a particular row or column,
     * which fills the entire span of the row or column, including any separatorPadding.
     *
     * The default value is null, which means nothing is drawn in the background of the row or column.
     * @see GraphObject#background
     * @since 1.2
     */
    background: BrushLike;
    /**
     * Determines whether or not the #background, if there is one, is in front of or behind the separators.
     *
     * The default value is false -- any background is drawn behind any separator lines.
     * @since 1.2
     */
    coversSeparators: boolean;
    /**
     * Gets or sets how this row or column deals with a Table Panel's extra space.
     * The value must be one of: RowColumnDefinition.None, RowColumnDefinition.ProportionalExtra,
     * or RowColumnDefinition.Default.
     * The default value is RowColumnDefinition.Default.
     * @see Panel#columnSizing
     * @see Panel#rowSizing
     * @see #actual
     * @see #total
     * @see #height
     * @see #width
     */
    sizing: EnumValue;
    /**
     * This read-only property returns the usable row height or column width, after arrangement, in local coordinates,
     * that objects in this row or column can be arranged within.
     * This does not include #separatorPadding or #separatorStrokeWidth, as #total does.
     *
     * This value gives the row height if #isRow is true;
     * otherwise this gives the column width.
     * The value is meaningless until after the Table Panel using this
     * RowColumnDefinition has been arranged.
     * @see #total
     * @see #height
     * @see #width
     * @see #sizing
     */
    actual: number;
    /**
     * This read-only property returns the total arranged row height or column width, after arrangement, in local coordinates.
     * This value gives the #actual size plus the #separatorPadding and #separatorStrokeWidth.
     *
     * This value gives the vertical space occupied by the row if #isRow is true;
     * otherwise this gives the horizontal space occupied by the column.
     * The value is meaningless until after the Table Panel using this
     * RowColumnDefinition has been arranged.
     * @see #actual
     * @see #height
     * @see #width
     * @see #sizing
     * @since 1.2
     */
    total: number;
    /**
     * This read-only property returns the actual arranged row or column starting position, after arrangement, in local coordinates.
     * This value gives the Y position if #isRow is true;
     * otherwise this gives the X position.
     * The value is meaningless until after the Table Panel using this
     * RowColumnDefinition has been arranged.
     * @see Panel#findColumnForLocalX
     * @see Panel#findRowForLocalY
     * @see Panel#leftIndex
     * @see Panel#topIndex
     * @see #index
     */
    position: number;
    /**
     * Add a data-binding of a property on this object to a property on a data object.
     * @param {Binding} binding
     */
    bind(binding: Binding): void;
}
/**
 * A Shape is a GraphObject that shows a geometric figure.
 * The Geometry determines what is drawn;
 * the properties #fill and #stroke
 * (and other stroke properties) determine how it is drawn.
 *
 * There are generally two types of shapes: Those that use a custom Geometry by setting
 * Shape#geometry, and those that receive an automatically generated Geometry using the value of
 * #figure, #toArrow, or #fromArrow. An explicitly set Geometry always supersedes
 * the figure and arrowhead properties.
 *
 * Some created Shapes:
 * ```js
 * var $ = go.GraphObject.make;  // for conciseness in defining GraphObjects
 *
 * // A shape with the figure set to RoundedRectangle:
 * $(go.Shape, { figure: "RoundedRectangle", fill: "lightgreen" })
 * // Alternatively:
 * $(go.Shape, "RoundedRectangle", { fill: "lightgreen" })
 *
 * // A shape with a custom geometry, using geometryString:
 * $(go.Shape,
 *     { geometry: go.Geometry.parse("M120 0 L80 80 0 50z") })
 *
 * // A shape with a custom geometry, using geometryString:
 * $(go.Shape,
 *    { geometryString: "F M120 0 L80 80 0 50z",
 *      fill: "lightgreen" })
 *
 * // A common link template, using two shapes,
 * // the first for the link path and the second for the arrowhead
 * myDiagram.linkTemplate =
 *   $(go.Link,
 *     // The first shape in a link is special, its geometry is set by the Link's routing,
 *     // so it does not need a geometry or figure set manually
 *     $(go.Shape,
 *       { strokeWidth: 2, stroke: 'gray' }),
 *     $(go.Shape,
 *       { toArrow: "Standard", fill: 'gray', stroke: null })
 *   );
 * ```
 * <p class="box">
 * You can see more custom geometry examples and read about geometryString
 * on the <a href="../../intro/geometry.html">Geometry Path Strings Introduction page.</a>
 *
 * When automatically generating a Shape Geometry, the value of #toArrow takes precedence,
 * then #fromArrow, then #figure. If the value of #toArrow or #fromArrow is "None"
 * then it is ignored, and the "None" value of #figure is identical to "Rectangle".
 *
 * All of the predefined figures are shown in the <a href="../../samples/shapes.html">Shapes sample</a>.
 * You can define your own named figures by calling the static function Shape.defineFigureGenerator.
 * Get a Map of named figures by calling the static function Shape.getFigureGenerators.
 *
 * All of the predefined arrowheads are shown in the <a href="../../samples/arrowheads.html">Arrowheads sample</a>.
 * You can define your own named arrowheads by calling the static function Shape.defineArrowheadGeometry.
 * Get a Map of named arrowheads by calling the static function Shape.getArrowheadGeometries.
 *
 * You can see a copy of all of the built-in arrowhead definitions in this file: <a href="../../extensions/Arrowheads.js">Arrowheads.js</a>.
 *
 * The Shape properties #parameter1, and #parameter2 determine details of the
 * construction of some #figure geometries.
 * Specifically, they often set the #spot1, #spot2 for the Shape.
 * These spots determine the "inner area" of an Auto panel when a Shape is the main object.
 * See the <a href="../../intro/panels.html">Auto Panels section of the Panels Introduction page</a> for more details.
 *
 * Shapes use their geometric bounds when determining hit testing,
 * but use rectangular bounds when participating in (panel) layouts.
 * @extends GraphObject
 * @unrestricted
 */
export class Shape extends GraphObject {
    /**
     * A newly constructed Shape has a default #figure of "None",
     * which constructs a rectangular geometry, and is filled and stroked with a black brush.
     */
    constructor();
    /**
     * Gets or sets the Shape's Geometry that defines the Shape's figure.
     * Setting a geometry is not necessary if a #figure is specified,
     * as that will construct a geometry instead.
     *
     * Setting this geometry property will freeze the supplied Geometry.
     *
     * Setting this geometry property always overrides any set figure.
     * The default value is null.
     * @see #geometryString
     */
    geometry: Geometry | null;
    /**
     * When set, creates a Geometry and normalizes it from a given path string,
     * then sets the #geometry property on this Shape and
     * set the GraphObject#position to the amount computed by normalization.
     * The property getter simply returns the toString value of the #geometry,
     * or the empty string if there is no Geometry value.
     *
     * See the <a href="../../intro/geometry.html">Geometry Path Strings Introduction page</a> for examples.
     * @since 1.1
     */
    geometryString: string;
    /**
     * Gets or sets the whether the GraphObject#position of this shape denotes
     * the top-left corner of this shape in panel coordinates or the origin of this geometry's coordinate system.
     * Basically, this determines whether the #strokeWidth affects the rendered location.
     * A true value allows multiple shapes to be positioned precisely in a "Position" Panel independent of the stroke width.
     * The default is false.
     * @since 1.1
     */
    isGeometryPositioned: boolean;
    /**
     * Gets or sets the Brush or string that describes how the geometry is filled when drawn.
     *
     * The default value is "black", causing the shape to be filled with solid black.
     * Any valid CSS string can specify a solid color, and the Brush
     * class can be used to specify a gradient or pattern.
     * A null fill will mean no fill is drawn and the filled portion
     * of the Shape will not be pickable.
     * A "transparent" fill is useful when wanting to allow a shape to be pickable
     * without obscuring any other objects behind it.
     * More information about the syntax of CSS color strings is available at:
     * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/color">CSS colors (mozilla.org)</a>.
     *
     * The geometry is filled before the #stroke is drawn.
     */
    fill: BrushLike;
    /**
     * Gets or sets the Brush or string that describes how the geometry is drawn as if by a pen.
     *
     * The default value is "black", causing the shape to be outlined in black.
     * Any valid CSS string can specify a solid color, and the Brush
     * class can be used to specify a gradient or pattern.
     * A null stroke will mean no stroke is drawn.
     * A "transparent" stroke is useful when wanting to allow a shape to be pickable
     * without obscuring any other objects behind it.
     * More information about the syntax of CSS color strings is available at:
     * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/color">CSS colors (mozilla.org)</a>.
     *
     * The stroke is drawn after the geometry is filled with the #fill Brush.
     */
    stroke: BrushLike;
    /**
     * Gets or sets the thickness of the stroke's pen.
     *
     * Value must be a real number greater than or equal to zero.
     * The default value is 1.0.
     * A value of zero will cause the stroke not to be drawn.
     *
     * The stroke width will affect the GraphObject#measuredBounds and GraphObject#actualBounds of this shape.
     * The stroke is drawn centered on the path of the geometry.
     */
    strokeWidth: number;
    /**
     * Gets or sets the style for how the ends of the stroke's line are drawn.
     * The value must be one of "butt", "round", or "square". The default is "butt".
     *
     * For more information, see <a href="https://www.w3.org/TR/2dcontext/#dom-context-2d-linecap">Stroke Line Cap (w3.org)</a>.
     */
    strokeCap: ('butt' | 'round' | 'square');
    /**
     * Gets or sets the type of corner that will be drawn for a stroke at the intersection of two straight segments of the geometry.
     * The value must be one of "miter", "bevel", or "round". The default is "miter".
     *
     * For more information, see <a href="https://www.w3.org/TR/2dcontext/#dom-context-2d-linejoin">Stroke Line Join (w3.org)</a>.
     */
    strokeJoin: ('miter' | 'bevel' | 'round');
    /**
     * Gets or sets the style for the stroke's mitre limit ratio.
     * The value must be a real number greater than or equal to one.
     * The default is 10.0.
     *
     * For more information, see <a href="https://www.w3.org/TR/2dcontext/#dom-context-2d-miterlimit">Stroke Miter Limit (w3.org)</a>.
     */
    strokeMiterLimit: number;
    /**
     * Gets or sets the dash array for creating dashed or dotted lines.
     * The value must be an array of positive numbers and zeroes,
     * or else null to indicate a solid line.
     * For example, the array [5, 10] would create dashes of 5 pixels and spaces of 10 pixels.
     * For more information, see <a href="https://www.w3.org/TR/2dcontext/#dom-context-2d-setlinedash">Stroke Line Dash Array (w3.org)</a>.
     *
     * The default value is null, resulting in a line without dashes or dots.
     * Setting an array with all zeroes will set the value to null.
     * @since 1.1
     */
    strokeDashArray: Array<number> | null;
    /**
     * Gets or sets the offset for dashed lines, used to start the drawing of the dash pattern with some space.
     * The value must be a real non-negative number. The default is zero.
     *
     * For more information, see <a href="https://www.w3.org/TR/2dcontext/#dom-context-2d-linedashoffset">Stroke Line Dash Offset (w3.org)</a>.
     * @since 1.1
     */
    strokeDashOffset: number;
    /**
     * Gets or sets the figure name, used to construct a Geometry.
     * The value must be a string. The default value is "None".
     *
     * The name can be any case but will always be canonicalized when set. For instance,
     * setting "roundedrectangle" will set the value of figure to "RoundedRectangle".
     * All of the predefined figures are shown in the <a href="../../samples/shapes.html">Shapes sample</a>.
     *
     * At most one of the following three properties may be set to a non-"None" value at the same time on the same shape:
     * #figure, #toArrow, #fromArrow.
     *
     * You can define your own named figures by calling the static function Shape.defineFigureGenerator.
     */
    figure: string;
    /**
     * Gets or sets the name of the kind of arrowhead that this shape should take
     * when this shape is an element of a Link.
     * Value must be a string.
     *
     * The default is "None", which means that this Shape is not an arrowhead, causing it to be the default Shape, a large filled Rectangle.
     * If you want to have an arrowhead Shape but sometimes not show an arrowhead, you can set or bind
     * the GraphObject#visible property, or you can set or bind this "toArrow" property to
     * be the empty string.
     * The arrowhead named "", an empty string, will display as nothing.
     *
     * The name can be any case but will always be canonicalized when set. For instance,
     * setting "opentriangle" will set the value of the arrowhead to "OpenTriangle".
     * All of the predefined arrowheads are shown in the <a href="../../samples/arrowheads.html">Arrowheads sample</a>.
     *
     * Setting this property may also set the GraphObject#segmentIndex,
     * GraphObject#segmentOrientation, and GraphObject#alignmentFocus properties.
     * This shape should be an element of a Link.
     *
     * At most one of the following three properties may be set to a non-"None" value at the same time on the same shape:
     * #figure, #toArrow, #fromArrow.
     *
     * You can define your own named arrowheads by calling the static function Shape.defineArrowheadGeometry.
     *
     * You can see a copy of all of the built-in arrowhead definitions in this file: <a href="../../extensions/Arrowheads.js">Arrowheads.js</a>.
     */
    toArrow: string;
    /**
     * Gets or sets the name of the kind of arrowhead that this shape should take
     * when this shape is an element of a Link.
     * Value must be a string.
     * For bi-directional links the arrowhead name often starts with "Backward...".
     *
     * The default is "None", which means that this Shape is not an arrowhead, causing it to be the default Shape, a large filled Rectangle.
     * If you want to have an arrowhead Shape but sometimes not show an arrowhead, you can set or bind
     * the GraphObject#visible property, or you can set or bind this "toArrow" property to
     * be the empty string.
     * The arrowhead named "", an empty string, will display as nothing.
     *
     * The name can be any case but will always be canonicalized when set. For instance,
     * setting "opentriangle" will set the value of the arrowhead to "OpenTriangle".
     * All of the predefined arrowheads are shown in the <a href="../../samples/arrowheads.html">Arrowheads sample</a>.
     *
     * Setting this property may also set the GraphObject#segmentIndex,
     * GraphObject#segmentOrientation, and GraphObject#alignmentFocus properties.
     * This shape should be an element of a Link.
     *
     * At most one of the following three properties may be set to a non-"None" value at the same time on the same shape:
     * #figure, #toArrow, #fromArrow.
     *
     * You can define your own named arrowheads by calling the static function Shape.defineArrowheadGeometry.
     *
     * You can see a copy of all of the built-in arrowhead definitions in this file: <a href="../../extensions/Arrowheads.js">Arrowheads.js</a>.
     */
    fromArrow: string;
    /**
     * Gets or sets the top-left Spot used by some Panels for determining where in the shape other objects may be placed.
     * The value is normally Spot.Default, but you may want to set it to override the value that many #figures use.
     */
    spot1: Spot;
    /**
     * Gets or sets the bottom-right Spot used by some Panels for determining where in the shape other objects may be placed.
     * The value is normally Spot.Default, but you may want to set it to override the value that many #figures use.
     */
    spot2: Spot;
    /**
     * Gets or sets a property for parameterizing the construction of a Geometry from a figure.
     * The meaning of this property depends on the particular figure.
     * The value must be a number; the default value is NaN.
     */
    parameter1: number;
    /**
     * Gets or sets a property for parameterizing the construction of a Geometry from a figure.
     * The meaning of this property depends on the particular figure.
     * The value must be a number; the default value is NaN.
     */
    parameter2: number;
    /**
     * This read-only property returns the natural bounds of this Shape as determined by its #geometry's bounds.
     * The bounds will always include the (0,0) point.
     * If there is no geometry available, it returns a Rect with the #desiredSize,
     * which may have NaN values for the width and height.
     */
    readonly naturalBounds: Rect;
    /**
     * Gets or sets a GraphObject that is drawn repeatedly along the path of the stroke of this shape.
     * This property may be set to a shared GraphObject; the GraphObject should not belong to any Panel.
     * Note that data bindings do <em>not</em> work in such shared GraphObjects, because they are not part of the visual tree.
     * The default value is null, causing no object to be drawn repeatedly.
     *
     * Typically the object is a small Shape or a Picture.
     * The larger the object is the worse the results will be, especially if the stroke has short segments or sharp curves.
     * The pathPattern object is not part of the measured bounds of the Shape, it is a cosmetic element only.
     *
     * The stroke is always drawn normally -- having a value for this property will draw the value along the stroke as well,
     * so it is commonplace to set the #stroke to "transparent" and the #strokeWidth to be as wide as
     * the height of the GraphObject being drawn along the stroke.
     *
     * Examples of path patterns can be seen in the <a href="../../samples/relationships.html">Relationships sample</a>.
     * @since 1.6
     */
    pathPattern: GraphObject | null;
    /**
     * Gets or sets how the shape's geometry is proportionally created given its computed size.
     * Possible values are GraphObject.None, GraphObject.Fill, GraphObject.Uniform, and GraphObject.Default.
     * The default is GraphObject.Default, which resolves to GraphObject.Fill for most figures, though
     * some regular figures such as "Circle" and "Square" default to GraphObject.Uniform.
     */
    geometryStretch: EnumValue;
    /**
     * Gets or sets how frequently this shape should be drawn within a "Grid" or "Graduated" Panel,
     * in multiples of the Panel#gridCellSize or Panel#graduatedTickUnit.
     * The default is 1.  Any new value must be a positive integer.
     */
    interval: number;
    /**
     * Gets or sets the fractional distance along the main shape of a "Graduated" Panel at which this kind of tick should start.
     * The default is 0. Any new value should range from 0 to 1.
     * @since 1.7
     */
    graduatedStart: number;
    /**
     * Gets or sets the fractional distance along the main shape of a "Graduated" Panel at which this kind of tick should end.
     * The default is 1. Any new value should range from 0 to 1.
     * @since 1.7
     */
    graduatedEnd: number;
    /**
     * Gets or sets the function to determine which values along a "Graduated" Panel will be skipped.
     * The default is null and doesn't skip any ticks.
     *
     * The function takes a number argument, a value between Panel#graduatedMin and Panel#graduatedMax.
     * The function will return a boolean, whether the tick will be skipped at the value of the argument.
     * @since 2.0
     */
    graduatedSkip: ((val: number) => boolean) | null;
    /**
     * This static function returns a read-only Map of named geometry generators.
     * The keys are figure names.
     * The values are either string synonyms for other figure names, or functions
     * that take a Shape and a width and a height and return a Geometry.
     *
     * The predefined shape figures can be seen in the <a href="../../samples/shapes.html">Shapes sample</a>.
     * @return {Map.<string,string|function(Shape, number, number):Geometry>}
     * @since 1.5
     */
    static getFigureGenerators(): Map<string, string | ((a: Shape, b: number, c: number) => Geometry)>;
    /**
     * This static function defines a named figure geometry generator for Shapes.
     * Once this is called one can use the name as a value for Shape#figure.
     *
     * The first argument is the new figure name and must be a non-empty string that starts with a capital letter and that is not "None".
     *
     * If the second argument is a string this call defines a synonym for an existing figure generator.
     * Do not define cycles of synonyms -- the behavior will be undefined.
     *
     * If the second argument is a function,
     * the Geometry generator function's first argument is the Shape for which the function is producing a Geometry.
     * But this Shape argument may be null in some circumstances.
     * The second and third arguments are the desired width and height.
     * These will always be finite non-negative numbers.
     * The function may look at the Shape#parameter1 and Shape#parameter2 properties, which may be NaN,
     * to decide what geometry to create for the figure given its intended width and height.
     *
     * The function must return a Geometry; you may want to set Geometry#spot1 and Geometry#spot2 on it
     * to indicate where content should be placed within the figure when using an "Auto" Panel.
     * For some figures you may also want to set Geometry#defaultStretch to GraphObject.Uniform
     * in order to maintain the geometry's aspect ratio within the Shape.
     *
     * Generated figures must create a Geometry that is not larger than the supplied with and height. Doing so will signal an error.
     * @param {string} name a capitalized figure name ("RoundedRectangle"); must not be "" or "None"
     * @param {string|function(Shape,number,number):Geometry} func A function that takes (Shape,width,height) and returns a Geometry,
     *    or an existing figure generator name for which the new name will be a synonym.
     * @since 1.5
     */
    static defineFigureGenerator(name: string, func: string | ((shape: Shape, width: number, height: number) => Geometry)): void;
    /**
     * This static function returns a read-only Map of named arrowhead geometries.
     * The keys are arrowhead names.
     * The values are Geometry objects.
     *
     * The predefined arrowheads can be seen in the <a href="../../samples/arrowheads.html">Arrowheads sample</a>.
     * @return {Map.<string,Geometry>}
     * @since 1.5
     */
    static getArrowheadGeometries(): Map<string, Geometry>;
    /**
     * This static function defines a named arrowhead geometry.
     * Once this is called one can use the name as a value for Shape#toArrow or Shape#fromArrow.
     *
     * The first argument is the new arrowhead name and must be a non-empty string that starts with a capital letter and that is not "None".
     *
     * If the second argument is a string, it is converted into a Geometry by calling
     * `go.Geometry.parse(pathstr, false)`,
     * which may throw an error if there are problems with the syntax of the string.
     * @param {string} name a capitalized arrowhead name ("OpenTriangle"); must not be "" or "None"
     * @param {Geometry|string} pathstr a Geometry or a Geometry path string, e.g. "m 0,0 l 8,4 -8,4"
     * @since 1.5
     */
    static defineArrowheadGeometry(name: string, pathstr: Geometry | string): void;
}
/**
 * A TextBlock is a GraphObject that displays a #text string in a given #font.
 *
 * The size and appearance of the text is specified by #font,
 * which takes a well-formed CSS string as its value.
 * The order of the CSS properties given is important for cross-browser compatibility,
 * and should be given in this order:
 *
 * <em>"font-style font-variant font-weight font-size font-family"</em>
 *
 * For example, "Italic small-caps bold 32px Georgia, Serif" is a valid font string
 * using every CSS font property. Note that not all browsers may support every property.
 *
 * Text is drawn using the #stroke brush, which may be any CSS color string or a Brush.
 *
 * Some created TextBlocks:
 * ```js
 * var $ = go.GraphObject.make;  // for conciseness in defining GraphObjects
 *
 * // A TextBlock with text and stroke properties set:
 * $(go.TextBlock, { text: "Hello World", stroke: "gray" })
 * // Alternatively:
 * $(go.TextBlock, "Hello World", { stroke: "gray" })
 * ```
 *
 * TextBlocks typically receive a natural size based on their text and font strings,
 * but often a width is given in order to cause the text to wrap at a certain place.
 * In order for wrapping to occur, the #wrap property must not be TextBlock.None.
 *
 * TextBlocks can be edited by users using the TextEditingTool.
 * The HTMLInfo that a given TextBlock uses as its text editor can be customized
 * by setting the #textEditor property. For an example of custom text editing tool use,
 * see the <a href="../../samples/customTextEditingTool.html">Custom TextEditingTool Sample</a>.
 * <p class="boxread">
 * For examples of TextBlock possibilities and functionality,
 * see the <a href="../../intro/textBlocks.html">Introduction page on TextBlocks</a>.
 * @extends GraphObject
 * @unrestricted
 */
export class TextBlock extends GraphObject {
    /**
     * A newly constructed TextBlock has no string to show; if it did,
     * it would draw the text, wrapping if needed, in the default font using a black stroke.
     */
    constructor();
    /**
     * Undocumented
     */
    static getEllipsis(): string;
    /**
     * Undocumented
     * @param {string} val
     */
    static setEllipsis(val: string): void;
    /**
     * Gets the function that, given the TextBlock and numerical text height, computes the position to draw the baseline of a line of text in all TextBlocks.
     * By default this is null and default behavior returns `(textHeight * 0.75)`.
     *
     * Note: This affects drawing only, and does not change TextBlock measurement calculations.
     * @return {(textBlock: TextBlock, textHeight: number) => number | null}
     * @since 2.0
     */
    static getBaseline(): (textBlock: TextBlock, textHeight: number) => number | null;
    /**
     * Sets the function that, given the TextBlock and numerical text height, computes the position to draw the baseline of a line of text in all TextBlocks.
     *
     * Note: This affects drawing only, and does not change TextBlock measurement calculations.
     * @param {(textBlock: TextBlock, textHeight: number) => number | null} value
     * @since 2.0
     */
    static setBaseline(value: (textBlock: TextBlock, textHeight: number) => number | null): void;
    /**
     * Gets the function that, given the TextBlock and numerical text height, computes the position to draw the underline of a line of text in all TextBlocks.
     * By default this is null and default behavior returns `(textHeight * 0.75)`.
     *
     * Note: This affects drawing only, and does not change TextBlock measurement calculations.
     * @return {(textBlock: TextBlock, textHeight: number) => number | null}
     * @since 2.0
     */
    static getUnderline(): (textBlock: TextBlock, textHeight: number) => number | null;
    /**
     * Sets the function that, given the TextBlock and numerical text height, computes the position to draw the underline of a line of text in all TextBlocks.
     *
     * Note: This affects drawing only, and does not change TextBlock measurement calculations.
     * @param {(textBlock: TextBlock, textHeight: number) => number | null} value
     * @since 2.0
     */
    static setUnderline(value: (textBlock: TextBlock, textHeight: number) => number | null): void;
    /**
     * Used as a value for TextBlock#wrap, the TextBlock will not wrap its text.
     * @constant
     */
    static None: EnumValue;
    /**
     * Used as a value for TextBlock#wrap, the TextBlock will wrap text, making the width of
     * the TextBlock equal to the width of the longest line.
     * @constant
     */
    static WrapFit: EnumValue;
    /**
     * Used as the default value for TextBlock#wrap, the TextBlock will wrap text and the width of
     * the TextBlock will be the desiredSize's width, if any.
     * @constant
     */
    static WrapDesiredSize: EnumValue;
    /**
     * Used a a value for TextBlock#wrap, the TextBlock will attempt to wrap at each character, allowing
     * breaks within "words."
     * @since 2.0
     * @constant
     */
    static WrapBreakAll: EnumValue;
    /**
     * Used as the default value for TextBlock#overflow: if the width is too small to display all text,
     * the TextBlock will clip.
     * @constant
     * @since 1.4
     */
    static OverflowClip: EnumValue;
    /**
     * Used as a value for TextBlock#overflow: if the width is too small to display all text,
     * the TextBlock will display an ellipsis.
     * @constant
     * @since 1.4
     */
    static OverflowEllipsis: EnumValue;
    /**
     * Gets or sets the current font settings.
     * The font property must be a valid CSS string describing a font.
     * The font string can accept several CSS properties but they must be
     * in a specific order in order to render correctly across all browsers:
     *
     * `"font-style font-variant font-weight font-size font-family"`
     *
     * For example, `"Italic small-caps bold 32px Georgia, Serif"` is a valid font string
     * using every CSS font property. Not every browser can render every font option.
     * For more information about CSS font syntax, see <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/font">CSS fonts (mozilla.org)</a>.
     *
     * If your Node sizes depend on TextBlocks, it is best to ensure any custom fonts you are using are finished loading before you load your Diagram.
     * This will ensure nodes are sized appropriately for the initial Diagram layout.
     *
     * The default font is "13px sans-serif".
     */
    font: string;
    /**
     * Undocumented
     * @param {string} font
     * @return {boolean}
     */
    static isValidFont(font: string): boolean;
    /**
     * Gets or sets the TextBlock's text string. The default is an empty string.
     * The text of a TextBlock, along with the values of #font, #wrap,
     * #isMultiline and sizing restrictions are what naturally determine
     * the size of the TextBlock.
     *
     * The text in textblocks can include manual line-breaks by using the character escape, `\n`.
     *
     * Leading and trailing whitespace is eliminated in each line of TextBlock text.
     *
     * If #editable is set to true, users can edit textblocks with the TextEditingTool.
     */
    text: string;
    /**
     * Gets or sets the alignment location in the TextBlock's given space.
     * The only possible values are `"start"`, `"end"`, `"left"`, `"right"`, and `"center"`.
     * Any other value is invalid.
     *
     * This property is most pertinent when the TextBlock has multiple lines of text,
     * or when the TextBlock is given a size that differs from the text's natural size (such as with #desiredSize).
     *
     * In left-to-right writing systems, `"start"` and `"left"` are synonymous, as are `"end"` and `"right"`.
     *
     * The default is `"start"`.
     * @see #verticalAlignment
     */
    textAlign: ('start' | 'end' | 'left' | 'right' | 'center');
    /**
     * Gets or sets how the TextBlock is displayed: Either normally or with a Horizontal or Vertical flip or both.
     *
     * Possible values are GraphObject.None, GraphObject.FlipHorizontal, GraphObject.FlipVertical, or GraphObject.FlipBoth.
     * The default is GraphObject.None.
     * @since 2.0
     */
    flip: EnumValue;
    /**
     * Gets or sets the vertical alignment Spot of this TextBlock, used when
     * the TextBlock has more available vertical space than it needs to draw all lines.
     *
     * The default value is Spot.Top, which aligns the TextBlock to the top of its available space.
     *
     * The #textAlign is often used along with this property to specify
     * where the should be positioned in its available space.
     *
     * This does not affect TextBlock coordinates or bounds, it only affects where text is drawn within the given area.
     * @see #textAlign
     * @since 1.7
     */
    verticalAlignment: Spot;
    /**
     * This read-only property returns the natural bounds of this TextBlock in local coordinates,
     * as determined by its #font and #text string, and optionally its #desiredSize.
     */
    readonly naturalBounds: Rect;
    /**
     * Gets or sets whether or not the text displays multiple lines or embedded newlines.
     * If this is false, all characters including and after the first newline will be omitted.
     * The default is true.
     * @see #maxLines
     */
    isMultiline: boolean;
    /**
     * Gets or sets whether or not the text is underlined.
     * The default is false.
     * @see #isStrikethrough
     * @since 1.2
     */
    isUnderline: boolean;
    /**
     * Gets or sets whether or not the text has a strikethrough line (line-through).
     * The default is false.
     * @see #isUnderline
     * @since 1.2
     */
    isStrikethrough: boolean;
    /**
     * Gets or sets whether the text should be wrapped if it is too long to fit on one line.
     *
     * Possible values are TextBlock.WrapDesiredSize, TextBlock.WrapFit, TextBlock.WrapBreakAll,
     * and TextBlock.None.
     *
     * The default value is TextBlock.WrapDesiredSize.
     * @see #overflow
     */
    wrap: EnumValue;
    /**
     * Gets or sets how text that is too long to display should be handled.
     *
     * Possible values are TextBlock.OverflowClip and TextBlock.OverflowEllipsis.
     * For OverflowEllipsis to work, you must constrain the available size of the TextBlock in some way,
     * such as setting #wrap to TextBlock.None,
     * or limiting the number of lines with #maxLines or a height constraint.
     *
     * The default value is TextBlock.OverflowClip.
     * @see #wrap
     * @since 1.4
     */
    overflow: EnumValue;
    /**
     * Gets or sets the Brush or string that describes the stroke (color) of the text that is drawn.
     *
     * The default value is `"black"`.
     * Any valid CSS string can specify a solid color, and the Brush
     * class can be used to specify a gradient or pattern.
     * More information about the syntax of CSS color strings is available at:
     * <a href="https://developer.mozilla.org/en-US/docs/Web/CSS/color">CSS colors (mozilla.org)</a>.
     */
    stroke: BrushLike;
    /**
     * This read-only property returns the total number of lines in this TextBlock, including lines created
     * from embedded newlines (`\n`), #wrapping, and #maxLines.
     *
     * This value may be meaningless before the TextBlock is measured.
     */
    readonly lineCount: number;
    /**
     * Gets or sets whether or not this TextBlock allows in-place editing of the #text
     * string by the user with the help of the TextEditingTool.
     * The default is false.
     *
     * See also Part#textEditable.
     */
    editable: boolean;
    /**
     * Gets or sets the HTMLInfo that this TextBlock uses as its text editor in the TextEditingTool.
     * If null, the TextBlock will use the default text editor of the TextEditingTool.
     * The default is null.
     * The value should be set to an instance of HTMLInfo.
     *
     * As of 2.0 setting this to an HTML Element is no longer supported.
     *
     * For example usage, see the <a href="../../samples/customTextEditingTool.html">Custom TextEditingTool Sample</a>.
     */
    textEditor: HTMLInfo | null;
    /**
     * Gets or sets the function to call if a text edit made with the TextEditingTool is invalid.
     * The default is null.
     */
    errorFunction: ((tool: TextEditingTool, oldString: string, newString: string) => void) | null;
    /**
     * Gets or sets how frequently this text should be drawn within a "Graduated" Panel,
     * in multiples of the Panel#graduatedTickUnit.
     * The default is 1.  Any new value must be a positive integer.
     * @since 1.7
     */
    interval: number;
    /**
     * Gets or sets the fractional distance along the main shape of a "Graduated" Panel at which this text should start.
     * The default is 0; the value should range from 0 to 1.
     * @since 1.7
     */
    graduatedStart: number;
    /**
     * Gets or sets the fractional distance along the main shape of a "Graduated" Panel at which this kind of tick text should end.
     * The default is 1; the value should range from 0 to 1.
     * @since 1.7
     */
    graduatedEnd: number;
    /**
     * Gets or sets the function to convert from a value along a "Graduated" Panel to a string.
     * The default returns a string representing the value rounded to at most 2 decimals.
     *
     * The function takes a number argument, a value between Panel#graduatedMin and Panel#graduatedMax.
     * The function will return a string, the text that will appear at the value of the argument.
     * @since 1.7
     */
    graduatedFunction: ((val: number) => string) | null;
    /**
     * Gets or sets the function to determine which values along a "Graduated" Panel will be skipped.
     * The default is null and doesn't skip any text labels.
     *
     * The function takes a number argument, a value between Panel#graduatedMin and Panel#graduatedMax.
     * The function will return a boolean, whether the text label will be skipped at the value of the argument.
     * @since 2.0
     */
    graduatedSkip: ((val: number) => boolean) | null;
    /**
     * Gets or sets the predicate that determines whether or not a user-edited string of text is valid.
     * If this is non-null, the predicate is called in addition to any TextEditingTool#textValidation predicate.
     * See TextEditingTool#isValidText for more details.
     *
     * ```js
     * function(textBlock, oldString, newString)
     * ```
     *
     * The default predicate is null, which is equivalent to simply returning true.
     *
     * The function, if supplied, must not have any side-effects, and must return true or false.
     * @see TextEditingTool#textValidation
     */
    textValidation: ((thisTextBlock: TextBlock, oldString: string, newString: string) => boolean) | null;
    /**
     * Gets or sets the function that is called after the TextBlock's text has been edited by the TextEditingTool.
     *   - The first argument is a reference to this TextBlock.
     *   - The second argument is the previous text, before editing.
     *   - The third argument is the current text, which is also TextBlock.text.
     *
     * ```js
     * function(textBlock, previousText, currentText)
     * ```
     *
     * The default value is null -- no function is called.
     * @since 1.7
     */
    textEdited: ((thisTextBlock: TextBlock, oldString: string, newString: string) => void) | null;
    /**
     * Undocumented
     */
    spacingAbove: number;
    /**
     * Undocumented
     */
    spacingBelow: number;
    /**
     * Gets or sets the maximum number of lines that this TextBlock can display.
     * Value must be a greater than zero whole number or `Infinity`.
     * The default is `Infinity`.
     *
     * Modifying this value may modify the computed height of the TextBlock.
     * If maxLines is set, the value of #lineCount will never be larger than maxLines.
     * @see #isMultiline
     * @since 1.5
     */
    maxLines: number;
    /**
     * Gets or sets the an array of possible choices for a custom TextEditingTool.
     * The value must be an array of strings.
     *
     * The default value is null.
     * For example usage, see the <a href="../../samples/customTextEditingTool.html">Custom TextEditingTool Sample</a>.
     * @since 1.7
     */
    choices: Array<string> | null;
}
/**
 * A Picture is a GraphObject that shows an image, video-frame, or Canvas element.
 *
 * You can specify what to show by either setting the #source URL property
 * to a URL string or the #element property to an HTMLImageElement,
 * HTMLCanvasElement, or HTMLVideoElement.
 *
 * If a #source URL is set, the Picture will automatically create a corresponding
 * HTMLImageElement and retain a reference to it in memory. If multiple Pictures specify
 * the same #source URL then they will all refer to the same HTMLImageElement.
 *
 * Some created Pictures:
 * ```js
 * var $ = go.GraphObject.make;  // for conciseness in defining GraphObjects
 *
 * // A shape with the figure set to "example.png". It will show a gray area until the image is loaded:
 * $(go.Picture, { source: "example.png", background: "gray", width: 50, height: 50 })
 * // Alternatively:
 * $(go.Picture, "example.png", { background: "gray", width: 50, height: 50 })
 * ```
 *
 * If an element is not completely loaded during Diagram initialization, a redraw may occur,
 * and if an image's size is not known before loading, the containing Part of this Picture
 * may be resized, causing side effects such as layouts.
 * This can be avoided by knowing the size of the image beforehand,
 * and setting the Picture's GraphObject#desiredSize.
 *
 * With some images (notably sprite sheets) only a portion of the image is expected to be drawn.
 * The #sourceRect property allows the programmer to specify a rectangular area of
 * the source image that the Picture should display.
 *
 * The #imageStretch property allows an image to be resized inside of its bounding box.
 * This property does not change the size of the Picture element, it only resizes or re-scales
 * the image to fit (or not) in its bounds.
 * <p class="boxread">
 * For examples of sizing and #imageStretch,
 * see the <a href="../../intro/pictures.html">Introduction page on Pictures</a>.
 *
 * The #errorFunction property allows one to set a function to call when a source fails to load.
 * This is useful in instances where images cannot be guaranteed to work, such as with
 * user specified input. The error function can set the #source to a known good value,
 * but care should be taken to avoid error infinite loops when doing so.
 * @extends GraphObject
 * @unrestricted
 */
export class Picture extends GraphObject {
    /**
     * The constructor creates a picture that shows nothing until the
     * #source or #element is specified. It is also common to specify the
     * GraphObject#desiredSize to make sure that the picture's size is known
     * before the image is loaded asynchronously, so that layouts do not need to be recomputed.
     */
    constructor();
    /**
     * Undocumented
     * @param {string=} url optional, clear only a specified URL from the image cache. If relative urls are used as Picture sources, this must be in the same format.
     */
    static clearCache(url?: string): void;
    /**
     * Gets or sets the Picture's HTML element that provides some kind of visual image.
     *
     * The default value is null.
     * This can be a HTMLImageElement, HTMLVideoElement, or HTMLCanvasElement.
     * If an image, this element must have its source (src) attribute defined.
     * Setting this does not set the Picture#source attribute and that attribute may be unknowable.
     */
    element: HTMLImageElement | HTMLVideoElement | HTMLCanvasElement | null;
    /**
     * Gets or sets the Picture's source URL, which can be any valid image (png, jpg, gif, etc) URL.
     *
     * The default value is the empty string, which specifies no image source.
     * Setting this attribute creates an HTMLImageElement and sets the Picture#element
     * attribute to that element.
     * Setting the source of multiple Pictures to the same URL will cause only one HTMLImageElement
     * to be created and shared.
     * It is commonplace to either specify a constant URL or to data bind this property to some data property,
     * perhaps using a conversion function in order to produce a proper URL.
     *
     * To avoid remeasuring and rearranging Parts as images load asynchronously,
     * be sure to set the GraphObject#desiredSize
     * (or GraphObject#width and GraphObject#height) to fixed values
     *
     * For cross-browser support of SVG sources additional care is needed.
     * See the final section of the <a href="../../intro/pictures.html">Introduction page on Pictures</a>.
     */
    source: string;
    /**
     * Undocumented
     */
    reloadSource(): void;
    /**
     * Undocumented
     */
    redraw(): void;
    /**
     * Gets or sets a function that returns a value for image.crossOrigin.
     *
     * The default value is null, which will not set a value for image.crossOrigin.
     * Set this property to a function that returns "anonymous" or "use-credentials"
     * if you want images to complete a cross-origin request.
     * Example:
     * ```js
     *    $(go.Picture,
     *      { width: 64, height: 64 },
     *      { sourceCrossOrigin: function(pict) { return "use-credentials"; } },
     *      new go.Binding("source", "path"))
     * ```
     * @since 1.5
     */
    sourceCrossOrigin: ((a: Picture) => string) | null;
    /**
     * Gets or sets the rectangular area of the source image that this picture should display.
     * This is only common with sprite maps and image tables.
     *
     * The value must be of type Rect.
     * The default value is Rect(NaN, NaN, NaN, NaN), which means the whole source image should be used.
     */
    sourceRect: Rect;
    /**
     * Gets or sets how the Picture's image is stretched within its bounding box.
     * Some images will be a different aspect ratio than their given size or #desiredSize.
     * This property will never change the size of the Picture itself,
     * only the size of the image that is drawn in the Picture's #actualBounds.
     *
     * Possible values are GraphObject.Fill, GraphObject.Uniform,
     * GraphObject.UniformToFill, and GraphObject.None.
     * The default is GraphObject.Fill.
     * @see #imageAlignment
     */
    imageStretch: EnumValue;
    /**
     * Gets or sets how the Picture is displayed: Either normally or with a Horizontal or Vertical flip or both.
     *
     * Possible values are GraphObject.None, GraphObject.FlipHorizontal, GraphObject.FlipVertical, or GraphObject.FlipBoth.
     * The default is GraphObject.None.
     * @since 2.0
     */
    flip: EnumValue;
    /**
     * Gets or sets the Spot to align the source image to, when the source image
     * is smaller than the Picture. This is only relevant when
     * the #imageStretch property value is not go.GraphObject.Fill.
     *
     * This does not affect Picture coordinates or bounds, it only affects what is drawn within the given area.
     * The default value is Spot.Center.
     * @see #imageStretch
     * @since 1.7
     */
    imageAlignment: Spot;
    /**
     * Gets or sets the function to call if an image fails to load.
     * The arguments to this function are this Picture and the HTMLImageElement's "error" (onerror) Event.
     * The default value is null, meaning that no specific action occurs when there is an error loading an image.
     * @see #successFunction
     */
    errorFunction: ((a: Picture, b: Event) => void) | null;
    /**
     * Gets or sets the function to call when an image loads successfully.
     * The arguments to this function are this Picture and the HTMLImageElement's "load" Event.
     * The default value is null, meaning that no specific action occurs when an image finishes loading.
     * @see #errorFunction
     * @since 1.7
     */
    successFunction: ((a: Picture, b: Event) => void) | null;
    /**
     * This read-only property returns the natural size of this picture as determined by its source's width and height.
     * The value is initially NaN x NaN until the picture has been measured.
     * It will use the #element's `naturalWidth` and `naturalHeight` if available.
     */
    readonly naturalBounds: Rect;
}
/**
 * This is the base class for all user-manipulated top-level objects.
 * Because it inherits from Panel, it is automatically a visual container
 * of other GraphObjects.
 * Because it thus also inherits from GraphObject, it also has properties such as
 * GraphObject#actualBounds, GraphObject#contextMenu, and GraphObject#visible.
 *
 * If you just want an object that users can select and manipulate,
 * you can create an instance of this class.
 *
 * If you want an object that also supports being connected by links to other objects,
 * use the Node class, which inherits from Part.
 * Create those connections by using instances of the Link class.
 *
 * If you want a node that logically contains a subgraph of nodes and links,
 * use the Group class, which inherits from Node.
 *
 * If you want an object that decorates another Part, without having to modify that Part,
 * use the Adornment class.
 * Adornments do not support linking or grouping or being selected.
 *
 * You can construct a Part, add GraphObjects to it programmatically,
 * and then add the part to a diagram by calling Diagram#add.
 * However it is commonplace to add data to a model by setting its Model#nodeDataArray or calling Model#addNodeData,
 * or for Links, setting the GraphLinksModel#linkDataArray or calling GraphLinksModel#addLinkData.
 * Such actions will cause a diagram that is displaying the model to copy a template,
 * which is a Part that may have data Bindings, and add the new part to the diagram.
 * The Panel#data property will refer to that data object in the model.
 *
 * Some examples of adding Parts to a Diagram:
 * ```js
 * // A simple Part template
 * myDiagram.nodeTemplate =
 *   $(go.Part, "Horizontal",
 *     $(go.Shape, "Circle", { width: 20, height: 20 }),
 *     $(go.TextBlock, "Hello World")
 *   );
 * // Node templates can be either Nodes, or simple Parts
 * // (But not Groups, Adornments, or Links)
 *
 * // Adds copies of the nodeTemplate bound to the specified node data:
 * myDiagram.model.nodeDataArray =
 * [
 *   { key: "Alpha" },
 *   { key: "Beta" }
 * ];
 *
 * // Adds one copy of the nodeTemplate bound to the given node data:
 * myDiagram.model.addNodeData( { key: "Gamma" } );
 * ```
 * <p class="boxread">
 * See the <a href="../../intro/usingModels.html">Introduction on using Models</a>
 * for examples and more information.
 * <h3>Layers and Z-ordering</h3>
 *
 * Parts added to a Diagram exist in one of the Diagram's Layers.
 * You can specify which layer the part should be in by setting #layerName.
 * Parts cannot be nested in the visual tree -- they cannot be added to other Parts of Panels.
 *
 * Parts can be individually z-ordered within a layer by setting #zOrder.
 * Parts within the same layer that have a higher zOrder number will be drawn above parts with a lower number.
 * <h3>Size and Position</h3>
 *
 * The size and position of a part are given by its GraphObject#actualBounds.
 * The size is determined by the GraphObjects that are elements inside this part.
 * You can change the position by setting GraphObject#position or Part#location.
 *
 * The "location" of a part is commonly the same as its "position".
 * The "position" is always the point that is at the top-left corner of the area occupied by the part.
 * But the "location" may be different from the "position" if you want to think of the part as being "at" a different spot in the part.
 * For example, you might want the "location" to be at the center of a Picture that has a TextBlock title of arbitrary size.
 * In this case you would set the #locationSpot to be Spot.Center and the #locationObjectName
 * to be the name of the Picture element in your Part.
 *
 * A part may be selected or de-selected by setting its #isSelected property.
 * This may also happen due to a call to Diagram#select or other operations that change the selection.
 * The user may change this property as part of the operation of the ClickSelectingTool,
 * due to the user's mouse click, if the part is #selectable.
 * <h3>Ability Properties (Permissions)</h3>
 *
 * There are many properties named "...able", that control what operations the user may perform on this part.
 * These properties correspond to the similarly named properties on Diagram and Layer that govern the
 * behavior for all parts in all layers or for all parts in the given layer.
 * For example, the Part#copyable property corresponds to the properties Diagram#allowCopy and Layer#allowCopy.
 *
 * For each of these "ability" properties there is a corresponding "can..." predicate.
 * For example, the Part#canCopy predicate is false if any of the three previously named properties is false.
 * Commands and tools will normally call these predicates rather than just looking at Part properties.
 * <p class="boxread">
 * For more discussion about permissions, please read: <a href="../../intro/permissions.html">Permissions</a>.
 *
 * As previously mentioned, each Diagram supports the notion of selected parts.
 * One way of displaying that a part is selected is by modifying the part.
 * You can set the #selectionChanged property to be a function that is called when the value of #isSelected has changed;
 * it is passed the Part as the first argument.
 * The function can modify the color of one or more GraphObjects in the visual tree of that Part.
 * Or perhaps it could toggle the GraphObject#visible property of an object that is normally hidden when the part is not selected.
 *
 * The Part class also supports showing separate visual objects for a part when it gets selected.
 * These visuals are typically used to show that the part is selected ("selection handles")
 * or are used to allow the user to manipulate or modify the part with a tool ("tool handles").
 * These handles are instances of Adornments.
 * The #updateAdornments method is responsible for showing or hiding adornments,
 * normally depending on whether the part is selected.
 *
 * When the #selectionAdorned property is true, a selected part automatically gets an Adornment created for it.
 * By default the selection adornment is just a simple blue box around the Part, and a blue shape following the route of a selected Link.
 * However you can set the #selectionAdornmentTemplate to an arbitrarily complex Adornment.
 * This way it can show more information or buttons for executing various commands when the user selects a Part.
 *
 * Tool handles are shown for those mode-less mouse-down tools that need it.
 * The process of updating adornments for a part will call Tool#updateAdornments on each tool in ToolManager#mouseDownTools.
 * Most tools might not need special tool handles.
 * But, for example, ResizingTool naturally will want to create an adornment with eight resize handles
 * positioned at the corners and at the middles of the sides of the selected node's visual element,
 * if the node has its #canResize function returning true.
 *
 * One may not always want the whole Part to get the selection handle or all tool handles.
 * Sometimes one wants to emphasize selection by highlighting a particular element within the part's visual tree.
 * This can be achieved by setting the #selectionObjectName property,
 * and making sure the desired element has the same GraphObject#name property value.
 * <p class="boxread">
 * For more discussion about selection, see <a href="../../intro/selection.html">Selection</a>.
 *
 * Similarly the #resizeObjectName and #rotateObjectName properties direct the
 * corresponding ResizingTool and RotatingTool to operate on the particular GraphObject
 * in the Part's visual tree with the given name.
 * That includes both providing tool handles and actually modifying properties on that object.
 *
 * Parts are not resizable or rotatable by default: you need to set #resizable and/or #rotatable to true.
 * <p class="boxread">
 * For more discussion about tools, see <a href="../../intro/tools.html">Tools</a>.
 *
 * A Part may be positioned (or a Link may be routed) by a Layout.
 * This will happen automatically if Diagram#layout or Group#layout are set.
 * The default Diagram#layout will position any nodes that were not given explicit positions or location.
 *
 * If you set #isLayoutPositioned to false, this part will not participate in any of the standard layouts,
 * so it will not be moved by a layout or affect other parts in a layout.
 * In order for the part to get a #location or position you will need to supply it explicitly.
 *
 * As parts are added to or removed from a diagram, the Layout responsible for positioning the part is invalidated.
 * This will cause the layout to be performed again in the near future, at the end of the transaction.
 * This automatic layout invalidation also occurs as parts change their visibility (GraphObject#visible)
 * or their size (GraphObject#actualBounds).
 * If you do want there to be a Diagram#layout but you do not want an automatic layout to happen
 * after removing parts (for example), you can set #layoutConditions not to include the Part.LayoutRemoved flag.
 * In this particular case, you could set #layoutConditions to:
 * `go.Part.LayoutStandard & ~go.Part.LayoutRemoved`.
 * It may also reasonable for your application to set it to Part.LayoutNone.
 * Do not forget to consider applying the same conditions to links as well as to nodes and groups.
 *
 * If you want to save the locations/positions of the parts in a diagram, it is commonplace to
 * data bind the #location to a property on your node data with a TwoWay Binding
 * (call Binding#makeTwoWay). For example:
 * ```js
 * $(go.Part, "Horizontal",
 *   new go.Binding("location", "loc", go.Point.parse).makeTwoWay(go.Point.stringify),
 *   ...
 * ```
 *
 * Then as the nodes are moved, whether manually by the user or automatically by a Layout,
 * the model data is automatically updated with the location.
 * <p class="boxread">
 * For more discussion about related topics, see <a href="../../intro/selection.html">Selection</a>,
 * <a href="../../intro/tools.html">Tools</a>, and
 * <a href="../../intro/permissions.html">Permissions</a>.
 *
 * Parts that are templates should have no relationships with other Parts.
 * Only real Parts that are in a Diagram can belong to Groups or have any Adornments.
 * Only real Nodes in a Diagram can be connected with Links.
 * @extends Panel
 * @unrestricted
 */
export class Part extends Panel {
    /**
     * The constructor builds an empty Part.
     * @param {PanelLayout=} type if not supplied, the default Panel type is Panel.Position.
     */
    constructor(type?: PanelLayout);
    /**
     * This value may be used as the value of the Part#layoutConditions property to indicate that
     * no operation on this Part causes invalidation of the Layout responsible for this Part.
     * @constant
     */
    static LayoutNone: number;
    /**
     * This flag may be combined with other "Layout" flags as the value of the Part#layoutConditions property to indicate that
     * when a Part is added to a Diagram or Group, it invalidates the Layout responsible for the Part.
     * @constant
     */
    static LayoutAdded: number;
    /**
     * This flag may be combined with other "Layout" flags as the value of the Part#layoutConditions property to indicate that
     * when a Part is removed from a Diagram or Group, it invalidates the Layout responsible for the Part.
     * @constant
     */
    static LayoutRemoved: number;
    /**
     * This flag may be combined with other "Layout" flags as the value of the Part#layoutConditions property to indicate that
     * when a Part's GraphObject#visible becomes true, it invalidates the Layout responsible for the Part.
     * @constant
     */
    static LayoutShown: number;
    /**
     * This flag may be combined with other "Layout" flags as the value of the Part#layoutConditions property to indicate that
     * when a Part's GraphObject#visible becomes false, it invalidates the Layout responsible for the Part.
     * @constant
     */
    static LayoutHidden: number;
    /**
     * This flag may be combined with other "Layout" flags as the value of the Part#layoutConditions property to indicate that
     * when a Node or simple Part's GraphObject#actualBounds changes size, it invalidates the Layout responsible for the Part;
     * this flag is ignored for Parts that are Links.
     * @constant
     */
    static LayoutNodeSized: number;
    /**
     * This flag may be combined with other "Layout" flags as the value of the Part#layoutConditions property to indicate that
     * when a Group has been laid out, it invalidates the Layout responsible for that Group;
     * this flag is ignored for Parts that are not Groups.
     * @constant
     */
    static LayoutGroupLayout: number;
    /**
     * This flag may be combined with other "Layout" flags as the value of the Part#layoutConditions property to indicate that
     * when a Node or simple Part's #category changes, it invalidates the Layout responsible for the Part;
     * this flag is ignored for Parts that are Links.
     * @constant
     */
    static LayoutNodeReplaced: number;
    /**
     * This is the default value for the Part#layoutConditions property, basically a combination of all of the conditions:
     * the Layout responsible for the Part is invalidated when the Part is added or removed or replaced
     * from the Diagram or Group, or when it changes visibility or size, or when a Group's layout has been performed.
     * @constant
     */
    static LayoutStandard: number;
    /**
     * Re-evaluate all data bindings in this Part,
     * in order to assign new property values to the GraphObjects in this visual tree
     * based on this this object's #data property values.
     * This method does nothing if #data is null.
     *
     * It is better to call Model#setDataProperty to modify data properties,
     * because that will both record changes for undo/redo and will update all bindings
     * that make depend on that property.
     *
     * To update relationships between nodes, call #updateRelationshipsFromData.
     * @param {string=} srcprop An optional source data property name:
     *   when provided, only evaluates those Bindings that use that particular property;
     *   when not provided or when it is the empty string, all bindings are evaluated.
     * @see #updateRelationshipsFromData
     */
    updateTargetBindings(srcprop?: string): void;
    /**
     * Update all of the references to nodes in case they had been modified in the model without
     * properly notifying the model by calling GraphLinksModel#setGroupKeyForNodeData or
     * GraphLinksModel#setToKeyForLinkData or other similar methods.
     * This method does not conduct a transaction, so you need to start and commit one yourself.
     *
     * This only updates the relationships between nodes, to have them reflect what is now declared in the model data.
     * For example, in a GraphLinksModel if code has changed the value of the "to" property of a link data,
     * calling this method on the corresponding Link would cause the link to connect with
     * the Node whose data has the new key.
     *
     * To update GraphObject properties that are data bound, call #updateTargetBindings.
     * @expose
     * @see #updateTargetBindings
     * @since 1.5
     */
    updateRelationshipsFromData(): void;
    /**
     * This read-only property returns the Part's Model data key if it is in a Diagram and is backed by Model data.
     * Otherwise this returns undefined.
     * @since 1.8
     * @see Model#getKeyForNodeData
     */
    readonly key: Key;
    /**
     * This read-only property returns an iterator over all of the Adornments associated with this part.
     * After each call to the iterator's next() method that returns true,
     * the iterator's key will be the category and the iterator's value will
     * be an Adornment.
     *
     * Templates should not have any adornments.
     */
    readonly adornments: Iterator<Adornment>;
    /**
     * Find an Adornment of a given category associated with this Part.
     *
     * Templates should not have any adornments.
     * @param {string} category
     * @return {Adornment} Returns null if no such Adornment was found.
     */
    findAdornment(category: string): Adornment | null;
    /**
     * Associate an Adornment with this Part, perhaps replacing any existing adornment of the same category.
     * Don't forget to set Adornment#adornedObject before calling this method.
     * This adds the Adornment to the Layer named by #layerName, normally "Adornment".
     *
     * Adornments are also data bound to the same data that this Part has, if any.
     * If the Adornment was already associated with a Part, it is unassociated with that old Part.
     *
     * This method should not be called on templates.
     * @param {string} category a string identifying the kind or role of the given adornment for this Part.
     * @param {Adornment} ad the new Adornment.
     */
    addAdornment(category: string, ad: Adornment): void;
    /**
     * Remove any Adornment of the given category that may be associated with this Part.
     * @param {string} category a string identifying the kind or role of the given adornment for this Part.
     */
    removeAdornment(category: string): void;
    /**
     * Remove all adornments associated with this part.
     */
    clearAdornments(): void;
    /**
     * This is responsible for creating any selection Adornment (if this Part #isSelected) and
     * any tool adornments for this part.
     * @expose
     */
    updateAdornments(): void;
    /**
     * This read-only property returns the Layer that this Part is in.
     * The value is the Layer that is named with the value of #layerName.
     * If you want to change what Layer this Part is in, change the value of #layerName to refer to a different Layer.
     *
     * This will be null if it has not yet been added to a Diagram,
     * or if it has already been removed from a Diagram.
     * @see #layerName
     */
    readonly layer: Layer | null;
    /**
     * This read-only property returns the Diagram that this Part is in.
     *
     * This will be null if it is not in a Layer.
     */
    readonly diagram: Diagram | null;
    /**
     * Gets or sets the layer name for this part.
     * The initial value is an empty string, which is the name of the default layer.
     * The value of this property determines the value of #layer.
     *
     * If this part is not yet in a Diagram,
     * this value is used by Diagram#add
     * to determine which Layer this part should go in.
     * If no layer can be found with this name, it uses the default layer.
     *
     * Changing the value of this property while it is already in a layer
     * causes it to change layers if needed.
     * @see #layerChanged
     */
    layerName: string;
    /**
     * Gets or sets the function to execute when this part changes layers.
     * It is typically used to modify the appearance of the part.
     * This function must not change the layer of this part by setting #layerName.
     *
     * If this property value is a function, it is called with three arguments,
     * this Part, the old Layer (may be null), and the new Layer (may be null).
     * By default this property is null -- no function is called.
     * @see #layerName
     */
    layerChanged: ((thisPart: Part, oldLayer: Layer | null, newLayer: Layer | null) => void) | null;
    /**
     * Gets or sets the Z-ordering position of this Part within its Layer.
     *
     * Within the same layer, nodes with larger zOrder values are placed in front of nodes with smaller zOrder values.
     * When the value is NaN the ordering is not specified.
     * The default value is NaN.
     *
     * When a Group has a zOrder value of NaN, it is automatically placed behind its member nodes and links
     * that also have no zOrder.
     * Such automatic ordering is not guaranteed if any nodes including the groups have a numeric zOrder.
     * If you do want to specify the zOrder of nodes, you should also specify the zOrder of their containing groups
     * unless those groups are in different layers.
     * @since 1.6
     */
    zOrder: number;
    /**
     * This read-only property returns the GraphObject that determines the location of this Part.
     * The value will be in the visual tree of this Part and is usually named with
     * the value of #locationObjectName.
     */
    readonly locationObject: GraphObject;
    /**
     * Gets or sets the minimum location of this Part to which the user may drag using the DraggingTool.
     *
     * Value must be of type Point.
     * The initial value is (-Infinity, -Infinity), which imposes no position constraint.
     * A X value of NaN causes Diagram#computeMove to use the part's current location's X value as the minimum,
     * and similarly for NaN as the Y value.
     * @see #location
     * @see #maxLocation
     * @see #dragComputation
     * @see #movable
     */
    minLocation: Point;
    /**
     * Gets or sets the maximum location of this Part to which the user may drag using the DraggingTool.
     *
     * Value must be of type Point.
     * The initial value is (Infinity, Infinity), which imposes no position constraint.
     * A X value of NaN causes Diagram#computeMove to use the part's current location's X value as the maximum,
     * and similarly for NaN as the Y value.
     * @see #location
     * @see #minLocation
     * @see #dragComputation
     * @see #movable
     */
    maxLocation: Point;
    /**
     * Gets or sets the name of the GraphObject that provides the location of this Part.
     * This name determines the value of #locationObject.
     * The actual #location also depends on the #locationSpot.
     *
     * The initial value is an empty string, meaning the whole Part itself determines the location.
     * If you want to use a particular GraphObject in the visual tree of this Part,
     * set this property to be the GraphObject#name of the element that you want to be the #locationObject.
     * @see #location
     * @see #locationSpot
     */
    locationObjectName: string;
    /**
     * Gets or sets the location Spot of this Node, the spot on the #locationObject
     * that is used in positioning this part in the diagram.
     *
     * Value must be of the type Spot.
     * The initial value is `Spot.TopLeft`.
     * The value must be a specific spot -- i.e. one for which Spot#isSpot is true.
     *
     * It is commonplace to set this property to `Spot.Center`, so that the #location
     * has a value corresponding to the point at the center of this Part's #locationObject element.
     * But the GraphObject#position of a Part is always at the top-left corner
     * point of the GraphObject#actualBounds.
     * @see #location
     * @see #locationObjectName
     */
    locationSpot: Spot;
    /**
     * Move this part and any parts that are owned by this part to a new position.
     *
     * If this part is a Group, it also moves all of its members, recursively.
     * If this part is a Link, it also moves all of its label nodes.
     * @expose
     * @param {Point} newpos a new Point in document coordinates.
     * @param {boolean=} useLocation true if you want to set the #location instead of the position. False by default.
     */
    move(newpos: Point, useLocation?: boolean): void;
    /**
     * Move this part and any parts that are owned by this part to a new position.
     * This just calls #move without the caller having to allocate a new Point.
     * @param {number} newx a new X value in document coordinates.
     * @param {number} newy a new Y value in document coordinates.
     * @param {boolean=} useLocation true if you want to set the #location instead of the position. False by default.
     * @since 1.4
     */
    moveTo(newx: number, newy: number, useLocation?: boolean): void;
    /**
     * This predicate is true if this Part can be seen.
     * Parts that can be seen can be manipulated by the user, can take space in the document, or can take part in a layout,
     * among many possibilities.
     * Note that the value of this predicate can often be false even while GraphObject#visible is true.
     *
     * A Part is not seen if it is not GraphObject#visible or if it is in a Layer that is not Layer#visible.
     *
     * If a Part is a member of a Group and the Group is not Group#isSubGraphExpanded, the part is not seen.
     * (The containing Group might still be visible.)
     *
     * If a Node is a "tree child" of a Node that is not Node#isTreeExpanded, the node is not seen.
     * (The parent Node might still be visible.)
     *
     * If a Link is connected to or from a Node that is not `isVisible()`
     * and is not a member of a Group that `isVisible()`, the link is not seen.
     *
     * If a Node is a "link label" of a Link and that Link is not `isVisible()`, the node is not seen.
     *
     * This is different from GraphObject#isVisibleObject,
     * which ignores whether the Layer is visible and just checks GraphObject#visible
     * up the chain of containing Panels.
     *
     * If you want to know whether a Part is in the Diagram's viewport, try:
     * ```js
     *   diagram.viewportBounds.containsRect(part.actualBounds);
     * ```
     *
     * or:
     * ```js
     *   diagram.viewportBounds.intersectsRect(part.actualBounds);
     * ```
     * @expose
     * @return {boolean} true if GraphObject#visible is true and if Layer#visible is true.
     */
    isVisible(): boolean;
    /**
     * Undocumented
     * @param {GraphObject} elt
     * @param {Spot} spot
     * @param {Point=} result an optional Point that is modified and returned
     * @return {Point}
     */
    getRelativePoint(elt: GraphObject, spot: Spot, result?: Point): Point;
    /**
     * Returns the Rect in document coordinates for this object's bounds.
     * If this GraphObject is a Part, the rect will be identical to its #actualBounds.
     * @param {Rect=} result an optional Rect that is modified and returned.
     * @return {Rect} in document coordinates.
     * @see #getDocumentPoint
     * @since 2.0
     */
    getDocumentBounds(result?: Rect): Rect;
    /**
     * Measures if needed to make sure the GraphObject#measuredBounds and GraphObject#naturalBounds are all real numbers,
     * primarily to get the actual width and height.
     * GraphObject#actualBounds will get a real width and height, but the x and y values may continue to be `NaN`
     * if they were that way beforehand.
     *
     * This is sometimes necessary to call when defining custom layouts or implementing virtualization,
     * so that it can work with the actual size of the nodes.
     *
     * For efficiency, do not call this method unnecessarily.
     * @since 1.6
     */
    ensureBounds(): void;
    /**
     * Gets or sets the position of this part in document coordinates,
     * based on the #locationSpot in this part's #locationObject.
     *
     * Value must be of type Point.
     * The initial value is Point(NaN, NaN).
     * It is commonplace to data bind this property to some property on your model node data.
     *
     * The value is related to the GraphObject#position.
     * For Parts, both are in document coordinates; setting one property will set the other property.
     * By default both will have the same value.
     * However, by setting either or both of #locationSpot and #locationObjectName,
     * the location will be determined by a spot in the #locationObject,
     * a GraphObject that is in the visual tree of this Part.
     * The GraphObject#position will always refer to the point at the top-left corner of the whole part.
     *
     * The #minLocation and #maxLocation limit the location of a part, not its position.
     * Grid snapping will normally locate the location to be on grid points.
     * @see #locationObjectName
     * @see #locationSpot
     * @see #movable
     * @see #maxLocation
     * @see #minLocation
     * @see #dragComputation
     */
    location: Point;
    /**
     * Gets or sets the category of this part,
     * typically used to distinguish different kinds of nodes or links.
     *
     * The initial value is an empty string, which is the default category.
     * Any new value must be a string.
     * This should not be set in templates.
     *
     * When building Parts for node data or link data in a model,
     * the Diagram will call Model#getCategoryForNodeData or
     * GraphLinksModel#getCategoryForLinkData to get the category string for the data object.
     * The diagram uses this value to look up a template in Diagram#nodeTemplateMap,
     * Diagram#linkTemplateMap or Diagram#groupTemplateMap.
     * That template is copied to create the actual Part that is added to the diagram.
     * The diagram will set this property to remember the category it used.
     *
     * Note that the class of the new Part must be the same as the class of the original Part.
     * For example, a Node cannot be replaced by a simple Part or vice-versa.
     * Nor can a Link be replaced by a subclass of Link or vice-versa.
     *
     * To change the category for a Part created for model data, call Model#setCategoryForNodeData or
     * GraphLinksModel#setCategoryForLinkData.
     *
     * This property is also used to distinguish Adornments on a Part.
     * In this scenario you create the Adornment, often indirectly by specifying a template,
     * and set this property explicitly.
     * For example, ResizingTool#updateAdornments creates a resizing Adornment from the
     * Part#resizeAdornmentTemplate and sets its category to be "Resizing".
     * Changing the category of an existing Adornment will update any adorned part's association.
     */
    category: string;
    /**
     * This predicate returns true if #copyable is true,
     * if the layer's Layer#allowCopy is true, and
     * if the diagram's Diagram#allowCopy is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may copy this part.
     */
    canCopy(): boolean;
    /**
     * This predicate returns true if #deletable is true,
     * if the layer's Layer#allowDelete is true, and
     * if the diagram's Diagram#allowDelete is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may delete this part.
     */
    canDelete(): boolean;
    /**
     * This predicate returns true if #textEditable is true,
     * if the layer's Layer#allowTextEdit is true, and
     * if the diagram's Diagram#allowTextEdit is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may edit this part.
     */
    canEdit(): boolean;
    /**
     * This predicate returns true if #groupable is true,
     * if the layer's Layer#allowGroup is true, and
     * if the diagram's Diagram#allowGroup is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may group this part.
     */
    canGroup(): boolean;
    /**
     * This predicate returns true if #movable is true,
     * if the layer's Layer#allowMove is true, and
     * if the diagram's Diagram#allowMove is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may move this part.
     */
    canMove(): boolean;
    /**
     * This predicate returns true if #reshapable is true,
     * if the layer's Layer#allowReshape is true, and
     * if the diagram's Diagram#allowReshape is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may reshape this part.
     */
    canReshape(): boolean;
    /**
     * This predicate returns true if #resizable is true,
     * if the layer's Layer#allowResize is true, and
     * if the diagram's Diagram#allowResize is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may resize this part.
     */
    canResize(): boolean;
    /**
     * This predicate returns true if #rotatable is true,
     * if the layer's Layer#allowRotate is true, and
     * if the diagram's Diagram#allowRotate is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may rotate this part.
     */
    canRotate(): boolean;
    /**
     * This predicate returns true if #selectable is true,
     * if the layer's Layer#allowSelect is true, and
     * if the diagram's Diagram#allowSelect is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may select this part.
     */
    canSelect(): boolean;
    /**
     * Gets or sets whether the user may copy this part.
     * The initial value is true.
     * @see #canCopy
     */
    copyable: boolean;
    /**
     * Gets or sets whether the user may delete this part.
     * The initial value is true.
     * @see #canDelete
     */
    deletable: boolean;
    /**
     * Gets or sets whether the user may do in-place text editing on TextBlocks in this part
     * that have TextBlock#editable set to true.
     * The initial value is true.
     * @see #canEdit
     * @see TextBlock#editable
     */
    textEditable: boolean;
    /**
     * Gets or sets whether the user may group this part to be a member of a new Group.
     * The initial value is true.
     *
     * The grouping command is implemented by CommandHandler#groupSelection
     * and depends on CommandHandler#archetypeGroupData having been set to a node data object.
     * A Group can be ungrouped by the user if you set Group#ungroupable to true.
     * @see #canGroup
     */
    groupable: boolean;
    /**
     * Gets or sets whether the user may move this part.
     * The initial value is true.
     * @see #canMove
     * @see #dragComputation
     * @see #maxLocation
     * @see #minLocation
     */
    movable: boolean;
    /**
     * Gets or sets whether a selection adornment is shown for this part when it is selected.
     * The initial value is true.
     * @see #selectionChanged
     * @see #selectionAdornmentTemplate
     */
    selectionAdorned: boolean;
    /**
     * Gets or sets whether this Part is part of the document bounds.
     *
     * The initial value is true.
     * A value of false causes Diagram#computeBounds to ignore this part.
     * If the value is false, it is possible that user will not be able to scroll far enough to see this part,
     * if the part's GraphObject#actualBounds are outside of the Diagram#documentBounds.
     */
    isInDocumentBounds: boolean;
    /**
     * Gets or sets whether a Layout positions this Node or routes this Link.
     * This property affects the value of #canLayout.
     *
     * The initial value is true, meaning that this part is laid out by the layout responsible for this Part.
     * If this part is a member of a Group, it is the Group#layout,
     * otherwise it is the Diagram#layout.
     *
     * A value of false means that this part is not affected by and does not affect any automatic layout,
     * so the #layoutConditions property is ignored.
     * You will need to make sure that it has a real #location or GraphObject#position value,
     * or else the Part might not be visible anywhere in the diagram.
     *
     * Another way of controlling when layouts are invalidated is by setting
     * Part#layoutConditions or Layout#isInitial or Layout#isOngoing.
     */
    isLayoutPositioned: boolean;
    /**
     * Gets or sets whether the user may select this part.
     * The initial value is true.
     *
     * If you set this to true you may also want to set #selectionObjectName
     * to the GraphObject#named element that you want to be adorned
     * when the Part is selected.
     * @see #canSelect
     * @see #isSelected
     * @see #selectionAdorned
     * @see #selectionObjectName
     * @see #selectionAdornmentTemplate
     * @see #selectionChanged
     */
    selectable: boolean;
    /**
     * Gets or sets whether the user may reshape this part.
     * The initial value is false.
     * @see #canReshape
     * @see LinkReshapingTool
     */
    reshapable: boolean;
    /**
     * Gets or sets whether the user may resize this part.
     * The initial value is false.
     *
     * If you set this to true you may also want to set #resizeObjectName
     * to the GraphObject#named element that you want the user to resize.
     * It is also commonplace to add a TwoWay Binding of that named element's
     * GraphObject#desiredSize in order to save to the model data the value that the user
     * set via the ResizingTool.
     * @see #canResize
     * @see #resizeObjectName
     * @see #resizeCellSize
     * @see #resizeAdornmentTemplate
     */
    resizable: boolean;
    /**
     * Gets or sets whether the user may rotate this part.
     * The initial value is false.
     *
     * If you set this to true you may also want to set #rotateObjectName
     * to the GraphObject#named element that you want the user to rotate.
     * It is also commonplace to add a TwoWay Binding of that named element's
     * GraphObject#angle in order to save to the model data the value that the user
     * set via the RotatingTool.
     * @see #canRotate
     * @see #rotateObjectName
     * @see #rotateAdornmentTemplate
     */
    rotatable: boolean;
    /**
     * Gets or sets whether this Part is selected.
     * The initial value is false.
     *
     * Selected parts typically are shown either with an Adornment
     * or with a different appearance by changing the brush or visibility of one or
     * more of the GraphObjects within the part.
     *
     * Changing this value does not by itself raise any "ChangingSelection" and "ChangedSelection" DiagramEvents.
     * Tools and the CommandHandler and methods such as Diagram#select do raise those DiagramEvents
     * because they want to surround changes to this property with a single "ChangingSelection" DiagramEvent beforehand and a single "ChangedSelection" afterwards.
     * @see #selectable
     * @see #selectionAdorned
     * @see #selectionChanged
     * @see #selectionObjectName
     * @see #selectionAdornmentTemplate
     */
    isSelected: boolean;
    /**
     * Gets or sets whether this Part is highlighted.
     * The initial value is false.
     *
     * Highlighted parts may be shown with a different appearance by changing the brush or visibility of one or
     * more of the GraphObjects within the part.
     * One way of doing that is by using binding.
     * Consider part of the definition of a Node template:
     * ```js
     *   $(go.Shape,
     *     . . .,
     *     // Shape.fill is bound to Node.data.color
     *     new go.Binding("fill", "color"),
     *     // Shape.stroke is red when Node.isHighlighted is true, black otherwise
     *     new go.Binding("stroke", "isHighlighted",
     *                    function(h) { return h ? "red" : "black"; }).ofObject()),
     * ```
     * @see #highlightedChanged
     * @see Diagram#highlighteds
     * @see Diagram#highlight
     * @see Diagram#highlightCollection
     * @see Diagram#clearHighlighteds
     * @since 1.4
     */
    isHighlighted: boolean;
    /**
     * Gets or sets whether this part will draw shadows.
     * The initial value is false.
     *
     * By default, setting this property to true will attempt to draw shadows only
     * on the GraphObjects in this Part that appear to act as background objects, and
     * not on GraphObjects that appear to be in front of other GraphObjects in the Part.
     *
     * To finely control shadows, you may need to set GraphObject#shadowVisible on
     * elements of this Part, so that they explicitly do or do not get shadowed accordingly.
     * @see #shadowOffset
     * @see #shadowColor
     * @see #shadowBlur
     * @see GraphObject#shadowVisible
     */
    isShadowed: boolean;
    /**
     * Gets or sets whether this part may be animated.
     * The initial value is true.
     * @since 1.4
     */
    isAnimated: boolean;
    /**
     * Gets or sets the function to execute when this #isHighlighted changes.
     * It is typically used to modify the appearance of the part.
     * This function must not highlight or unhighlight any parts.
     *
     * If this property value is a function, it is called with one argument,
     * this Part that whose #isHighlighted value changed.
     * By default this property is null.
     * @see #isHighlighted
     * @since 1.7
     */
    highlightedChanged: ((thisPart: Part) => void) | null;
    /**
     * Gets or sets the name of the GraphObject that should get a selection handle
     * when this part is selected.
     * The value of this property affects the value of #selectionObject.
     * The initial value is an empty string, meaning the whole Part itself gets any selection handle.
     *
     * If no GraphObject has a GraphObject#name that is this name, #selectionObject returns the whole Part.
     * @see #selectable
     * @see #selectionObject
     */
    selectionObjectName: string;
    /**
     * Gets or sets the Adornment template used to create a selection handle for this Part.
     *
     * If this is null, depending on the class of this Part, the value of Diagram#nodeSelectionAdornmentTemplate, Diagram#groupSelectionAdornmentTemplate,
     * or Diagram#linkSelectionAdornmentTemplate is used instead.
     *
     * It is commonplace to make use of a Placeholder in an Adornment for a Node, Group, or simple Part.
     * The Placeholder represents the Adornment#adornedObject of the adorned Part.
     * For Links, the Adornment must be of Panel#type Panel.Link.
     * @see #selectable
     * @see #selectionObjectName
     */
    selectionAdornmentTemplate: Adornment | null;
    /**
     * This read-only property returns the GraphObject that should get a selection handle when this part is selected.
     * The value will be in the visual tree of this Part and is usually named with
     * the value of #selectionObjectName.
     * When the #selectionObjectName is unspecified, this whole Part is used as the "selection object".
     * @see #selectable
     * @see #selectionObjectName
     */
    readonly selectionObject: GraphObject;
    /**
     * Gets or sets the function to execute when this part is selected or deselected.
     * It is typically used to modify the appearance of the part.
     * This function must not select or deselect any parts.
     *
     * If this property value is a function, it is called with one argument,
     * this Part that was selected or that became unselected.
     * When it is called, the value of Diagram#skipsUndoManager is temporarily set to true.
     * By default this property is null.
     *
     * This function is called with Diagram#skipsUndoManager temporarily set to true,
     * so that any changes to GraphObjects are not recorded in the UndoManager.
     * You do not need to start and commit any transaction in this function.
     * @see #isSelected
     * @see #selectionAdorned
     * @see #selectionAdornmentTemplate
     */
    selectionChanged: ((thisPart: Part) => void) | null;
    /**
     * Gets or sets the adornment template used to create a resize handle Adornment for this part.
     * This is used by the ResizingTool, ToolManager#resizingTool.
     *
     * If an Adornment is supplied, it is normally a Panel.Spot panel that contains a Placeholder
     * with some number of resize handles at the four corners or at the four side midpoints.
     * @see #resizable
     * @see #resizeObjectName
     * @see ResizingTool
     */
    resizeAdornmentTemplate: Adornment | null;
    /**
     * Gets or sets the name of the GraphObject that should get a resize handle
     * when this part is selected.
     * The value of this property affects the value of #resizeObject.
     * The initial value is an empty string, meaning the whole Part itself gets any resize handle.
     * @see #resizable
     * @see #resizeObject
     * @see ResizingTool
     */
    resizeObjectName: string;
    /**
     * This read-only property returns the GraphObject that should get resize handles when this part is selected.
     * The value will be in the visual tree of this Part and is usually named with
     * the value of #resizeObjectName.
     * @see #resizable
     * @see #resizeObjectName
     * @see ResizingTool
     */
    readonly resizeObject: GraphObject;
    /**
     * Gets or sets the width and height multiples used when resizing.
     * By default this property is the Size(NaN, NaN).
     * @see #resizable
     * @see #resizeObjectName
     * @see ResizingTool
     */
    resizeCellSize: Size;
    /**
     * Gets or sets the adornment template used to create a rotation handle Adornment for this part.
     * This is used by the RotatingTool, ToolManager#rotatingTool.
     *
     * This Adornment should not have a Placeholder in it, because the RotatingTool will position it away from the
     * #rotateObject at its GraphObject#angle.
     * @see #rotatable
     * @see #rotateObjectName
     * @see RotatingTool
     */
    rotateAdornmentTemplate: Adornment | null;
    /**
     * Gets or sets the name of the GraphObject that should get a rotate handle
     * when this part is selected.
     * The value of this property affects the value of #rotateObject.
     * The initial value is an empty string, meaning the whole Part itself gets any rotate handle.
     * @see #rotatable
     * @see #rotateObject
     * @see RotatingTool
     */
    rotateObjectName: string;
    /**
     * This read-only property returns the GraphObject that should get rotate handles when this part is selected.
     * The value will be in the visual tree of this Part and is usually named with
     * the value of #rotateObjectName.
     * @see #rotatable
     * @see #rotateObjectName
     * @see RotatingTool
     */
    readonly rotateObject: GraphObject;
    /**
     * Gets or sets the spot on the #rotateObject
     * that is used in rotating this part with the RotatingTool.
     *
     * Value must be of the type Spot.
     * The value must be a specific spot -- i.e. one for which Spot#isSpot is true, or else `Spot.Default`.
     *
     * If the value is `Spot.Default`, the RotatingTool uses
     * the locationSpot if the #rotateObject is equal to the #locationObject, otherwise
     * it uses `Spot.Center`.
     *
     * The initial value is `Spot.Default`.
     *
     * @since 2.0
     * @see #rotateObjectName
     */
    rotationSpot: Spot;
    /**
     * Gets or sets a text string that is associated with this part.
     *
     * The initial value is an empty string.
     * This value is often used for sorting.
     */
    text: string;
    /**
     * Gets or sets the Group of which this Part or Node is a member.
     * This will be null if this is a top-level part.
     *
     * You cannot set this property on a Link; it is set for you automatically
     * based on the group memberships of the connected nodes.
     * You cannot set this property on an Adornment at all.
     *
     * A template should not be a member of any group.
     * @see Group#memberParts
     * @see #findTopLevelPart
     * @see #isTopLevel
     */
    containingGroup: Group | null;
    /**
     * Gets or sets the function that is called after this Part has changed which Group it belongs to, if any.
     * It is typically used to modify the appearance of the part.
     * The first argument will be this Part.
     * The second argument will be the old Group, or null if it had been a top-level part.
     * The third argument will be the new Group, or null if it is now a top-level part.
     *
     * If the value is a function, that function must not modify the part's containing Group.
     * The containing Group has already been changed -- trying to change it again may produce undefined behavior.
     *
     * The initial value is null -- no function is called.
     */
    containingGroupChanged: ((thisPart: Part, oldGroup: Group | null, newGroup: Group | null) => void) | null;
    /**
     * Return how deep this part is in the hierarchy of nested Groups.
     * For parts that have no #containingGroup this returns zero.
     *
     * If you want to know how deep a Node is in a tree structure, call Node#findTreeLevel.
     * @return {number}
     * @since 1.5
     */
    findSubGraphLevel(): number;
    /**
     * Gets the top-level Part for this part, which is itself when #isTopLevel is true.
     * If this Part is a member of a Group, this returns the top-level Part for that Group.
     * If this is a Node that is a label node for a labeled Link, this returns the top-level Part for that Link.
     *
     * If this is a Node and you are searching for the root of the tree that this node is in, use Node#findTreeRoot.
     * @return {Part} This will not return null.
     * @since 1.1
     */
    findTopLevelPart(): Part;
    /**
     * This read-only property is true when this part is not member of any Group node
     * nor is it a label node for a Link.
     * @see #containingGroup
     * @see Node#labeledLink
     */
    readonly isTopLevel: boolean;
    /**
     * This predicate is true if this part is a member of the given Part, perhaps indirectly.
     *
     * If the given part is a Group and this part is a member of the given group, this returns true.
     * If this part is a Node and it is a label node for the given link, this returns true.
     * Otherwise this searches recursively any Part#containingGroup of the given part.
     *
     * A part cannot be contained by itself.
     * A template should not be a member of any group.
     *
     * If this is a Node and you want to find whether it is in a subtree whose root is a given Node, use Node#isInTreeOf.
     * @param {Part} part
     * @return {boolean}
     */
    isMemberOf(part: Part): boolean;
    /**
     * Find the Group that perhaps indirectly contains both this part and another one.
     * If this is a Group and it contains the OTHER Part, return this.
     * If the OTHER Part is a Group and it contains this Part, return that OTHER Part.
     *
     * This returns null if the two parts are unrelated in the hierarchy of part membership.
     * If non-null, the result is a Group.
     *
     * If you want to find the Node that is the tree parent of two Nodes,
     * call Node#findCommonTreeParent.
     * @param {Part} other
     * @return {Group} may be null
     */
    findCommonContainingGroup(other: Part): Group | null;
    /**
     * Gets or sets flags that control when the Layout that is responsible for this Part is invalidated.
     * The initial value is Part.LayoutStandard,
     * which causes the layout for this part to be invalidated when the part is added or removed or changes visibility or size.
     *
     * Individual layout conditions include: Part.LayoutAdded, Part.LayoutRemoved,
     * Part.LayoutShown, Part.LayoutHidden, and Part.LayoutNodeSized.
     *
     * This property is ignored when #isLayoutPositioned is false -- no operation on this Part
     * will by itself cause the responsible Layout to be invalidated.
     *
     * You can also control when layouts are invalidated is by setting Layout#isInitial or Layout#isOngoing.
     */
    layoutConditions: number;
    /**
     * This predicate is called by Layout implementations to decide whether this Part
     * should be positioned and might affect the positioning of other Parts.
     *
     * This is false if #isLayoutPositioned is false, if #isVisible returns false,
     * or if the part is in a temporary Layer.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean}
     */
    canLayout(): boolean;
    /**
     * Invalidate the Layout that is responsible for positioning this Part.
     * If this part is in a Group, invalidate its Group#layout, if it has one.
     * Otherwise invalidate the Diagram#layout.
     *
     * But note that if #isLayoutPositioned is false, or if it is in a temporary Layer,
     * or if it is not in a diagram or group, no layout is invalidated.
     * @param {number=} condition the reason that the layout should be invalidated;
     * if this argument is not supplied, any value of #layoutConditions other than Part.LayoutNone
     * will allow the layout to be invalidated.
     */
    invalidateLayout(condition?: number): void;
    /**
     * Gets or sets the function used to determine the location that this Part can be dragged to.
     * The first argument is a reference to the Part being dragged, the second argument is a Point describing
     * the proposed location, and the third argument is a snapped location, if one was determined during dragging.
     * It should return a Point that is the proposed new location.
     *
     * By default this function is null and the DraggingTool uses the snapped location,
     * if one was determined and if DraggingTool#isGridSnapEnabled is true,
     * or the proposed location (the second argument) if not snapping to a grid.
     *
     * In either case the DraggingTool will limit the proposed new location by #minLocation and #maxLocation.
     *
     * The function, if supplied, must not have any side-effects.
     *
     * An example that limits moving a Node to the current viewport:
     * ```js
     *   function stayInViewport(part, pt, gridpt) {
     *     var diagram = part.diagram;
     *     if (diagram === null) return pt;
     *     // compute the area inside the viewport
     *     var v = diagram.viewportBounds.copy();
     *     v.subtractMargin(diagram.padding);
     *     // get the bounds of the part being dragged
     *     var b = part.actualBounds;
     *     var loc = part.location;
     *     // now limit the location appropriately
     *     var x = Math.max(v.x+1, Math.min(pt.x, v.right-b.width-2)) + (loc.x-b.x);
     *     var y = Math.max(v.y+1, Math.min(pt.y, v.bottom-b.height-2)) + (loc.y-b.y);
     *     return new go.Point(x, y);
     *   }
     * ```
     * Note that for this functionality you will also probably want to set Diagram#autoScrollRegion to be a zero margin.
     * ```js
     *   myDiagram.nodeTemplate =
     *     $(go.Node, . . .,
     *       { dragComputation: stayInViewport },
     *       . . .
     *     );
     * ```
     * @see #maxLocation
     * @see #minLocation
     */
    dragComputation: ((thisPart: Part, newLoc: Point, snappedLoc: Point) => Point) | null;
    /**
     * Gets or sets the X and Y offset of this part's shadow. This is only relevant if #isShadowed is true.
     * The initial value is (6, 6).
     * @see #isShadowed
     * @see #shadowColor
     * @see #shadowBlur
     */
    shadowOffset: Point;
    /**
     * Gets or sets the CSS string that describes a shadow color. Default is 'gray'.
     * Brushes cannot be used for this property -- only strings.
     * @see #isShadowed
     * @see #shadowOffset
     * @see #shadowBlur
     */
    shadowColor: string;
    /**
     * Gets or sets the numerical value that describes the shadow's blur. Number must be a non-negative non-infinity float.
     * A value of 0 would mean the shadow does not blur and larger numbers represent increasingly more blur.
     * The total blur area is independent of the Part's area and can become quite large as this number is increased.
     *
     * This value is not affected by scale. Default value is 4.
     * @see #isShadowed
     * @see #shadowOffset
     * @see #shadowColor
     */
    shadowBlur: number;
}
/**
 * An Adornment is a special kind of Part that is associated with another Part,
 * the Adornment#adornedPart.
 *
 * Adornments are normally associated with a particular GraphObject in the adorned Part --
 * that is the value of #adornedObject.
 * However, the #adornedObject may be null, in which case the #adornedPart will also be null.
 *
 * The area occupied by the adorned object is represented in the Adornment's visual tree by a Placeholder.
 * The placeholder is always the Part#locationObject,
 * although you may specify any Spot as the Part#locationSpot.
 * An adornment need not have a placeholder, but it may have at most one.
 *
 * Adornments can be distinguished by their Part#category.
 * This property can be an arbitrary string value determined by the code creating the adornment,
 * typically a tool that wants to be able to tell various adornments apart from each other.
 * Use the Part#findAdornment method to find an adornment for a part of a given category.
 *
 * For example, one of the Adornments created by Part#updateAdornments when the part
 * Part#isSelected has the Part#category of "Selection".
 * Those created by ResizingTool#updateAdornments have a category of "Resize"
 * and normally contain eight resize handles.
 *
 * Besides the selection Adornment and tool Adornments, Adornments are also used for
 * context menus and tooltips.
 * The #adornedObject in such cases refers to the GraphObject to which the
 * the context menu or tooltip applies.
 *
 * There cannot be any links connected to an Adornment,
 * nor can an Adornment have members or be a member of a group.
 *
 * An Adornment cannot have its own Adornments.
 * An Adornment cannot be selected.
 *
 * Adornments are not positioned by a Layout because they are normally positioned
 * according to the Part that they adorn.
 *
 * For more discussion and examples, see <a href="../../intro/selection.html">Selection</a>,
 * <a href="../../intro/toolTips.html">ToolTips</a>,
 * <a href="../../intro/contextMenus.html">Context Menus</a>, and
 * <a href="../../intro/tools.html">Tools</a>.
 * @extends Part
 * @unrestricted
 */
export class Adornment extends Part {
    /**
     * @param {PanelLayout=} type if not supplied, the default Panel type is Panel.Position.
     */
    constructor(type?: PanelLayout);
    /**
     * This read-only property returns a Placeholder that this Adornment may contain in its visual tree.
     * This may be null if there is no such placeholder object.
     */
    readonly placeholder: Placeholder | null;
    /**
     * Gets or sets the GraphObject that is adorned.
     * Setting this property updates the adorned object's Part by calling Part#addAdornment.
     * This may be null if the Adornment does not adorn a particular object.
     */
    adornedObject: GraphObject | null;
    /**
     * This read-only property returns the Part that contains the adorned object.
     * This will be null if the #adornedObject is null.
     */
    readonly adornedPart: Part | null;
}
/**
 * A Node is a Part that may connect to other nodes with Links,
 * or that may be a member of a Group.
 *
 * Group inherits from Node,
 * enabling nodes to logically contain other nodes and links.
 * <p class="boxread">
 * For a more general discussion of how to define nodes, see <a href="../../intro/nodes.html">Introduction to Nodes</a>.
 *
 * Although you can create a Node and Diagram#add it to a Diagram, this does not update the Model.
 * It is more common to create a node by adding a node data object to the model
 * by calling Model#addNodeData. For example:
 * ```js
 *   myDiagram.startTransaction("make new node");
 *   myDiagram.model.addNodeData({ key: "Omega" });
 *   myDiagram.commitTransaction("make new node");
 * ```
 *
 * This will cause a Node or simple Part to be created (copying the template found in Diagram#nodeTemplateMap),
 * added to the Diagram in some Layer (based on Part#layerName), and bound to the node data
 * (resulting in Panel#data referring to that node data object).
 * If you do not keep a reference to that JavaScript object, as the above code does not,
 * you can retrieve it later by calling Model#findNodeDataForKey.
 *
 * It is very common to initialize a Diagram by setting Model#nodeDataArray to a JavaScript Array
 * of JavaScript objects holding the properties that you need in your model.
 * Nearly all of the samples do this kind of initialization.
 *
 * You can delete a Node by either calling Diagram#remove or by calling Model#removeNodeData.
 * The latter obviously will modify the Model; the former does so if the Node was created from model data.
 * Commands such as CommandHandler#deleteSelection call these methods within a transaction.
 *
 * You can find all of the Links that are connected with a Node by calling #findLinksConnected.
 * Because links normally have a direction, you can find all of the links that have their Link#toNode
 * be a given Node by calling #findLinksInto.
 * Similarly, you can call #findLinksOutOf to find all of the links coming out from a node;
 * such links have their Link#fromNode be that node.
 * For tree-structured graphs, use #findTreeChildrenLinks or #findTreeParentLink.
 *
 * If you are not so interested in the links but are interested in the nodes at the other end of the links
 * connecting with a node, there are other methods that you can call.
 * #findNodesConnected returns all of the nodes that are at the other end of the links that
 * connect with a given node.
 * #findNodesInto and #findNodesOutOf return the subsets of those nodes considering
 * only those links that go into or come out of the given node.
 * For tree-structured graphs, use #findTreeChildrenNodes or #findTreeParentNode.
 *
 * For example, to operate on the data of all of the destination nodes:
 * ```js
 * var it = somenode.findNodesOutOf();
 * while (it.next()) {
 *   var child = it.value;
 *   if (child.data.text.indexOf("special") >= 0) { ... }
 * }
 * ```
 *
 * You can link two nodes by creating a new Link, setting its Link#toNode and Link#fromNode
 * (in either order), and Diagram#adding it to the diagram.
 * But it is more common to add a link data object to the Diagram#model by calling GraphLinksModel#addLinkData.
 * Just creating and adding a Link will not update the model.
 *
 * Thus to add a link when using a GraphLinksModel you should do something like:
 * ```js
 *   myDiagram.startTransaction("make new link");
 *   myDiagram.model.addLinkData({ from: "Alpha", to: "Beta" });
 *   myDiagram.commitTransaction("make new link");
 * ```
 *
 * Where you would substitute the keys of the actual nodes that you want to connect with a link.
 * If you are using a TreeModel, there are no link data objects, so you just need to call
 * TreeModel#setParentKeyForNodeData to specify the "parent" node's key for a "child" node data.
 *
 * To find a Link given a link data object in the GraphLinksModel,
 * call Diagram#findLinkForData.  When using a TreeModel, call either Diagram#findNodeForData
 * or Diagram#findNodeForKey to get a Node, and then call #findTreeParentLink to get the Link, if any exists.
 *
 * To find a link that connects two nodes, call #findLinksTo or #findLinksBetween.
 * With the former method, the direction matters; with the latter method it returns links in either direction.
 *
 * As links connect with a node or are disconnected, you may want to update the appearance of the node.
 * You can set the #linkConnected and #linkDisconnected properties to be functions that are called.
 * These functions must not modify any link relationships -- the properties just exist to update the appearance of the node.
 * A typical usage would be to change the color or figure of a shape.
 *
 * You can control whether the user may draw a new link or reconnect a link between a pair of Nodes
 * by affecting the result of LinkingBaseTool#isValidLink.
 * You can override that predicate on LinkingTool and RelinkingTool,
 * but it is easier to set the #linkValidation or
 * LinkingBaseTool#linkValidation functional property.
 * <p class="boxread">
 * For a more general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
 *
 * Nodes also support the ability to provide logical and physical distinctions in the connection points
 * that links use at a node. These connection objects are called "ports".
 * By default the port object will be the whole Node.
 * However, you can set the GraphObject#portId property on any GraphObject in the visual tree of a node
 * to cause that element to be treated as a "port".
 * The "port id" is just a string that ought to be unique amongst all of the port elements in the node.
 *
 * In the case of a node only having a single port, you should set the GraphObject#portId as an empty string.
 * When there is no such element declared as the default port, it uses the whole node.
 * You can use the #port property to get the only port element.
 *
 * When a node should have multiple ports,
 * i.e. multiple GraphObjects acting as separate connection points for links,
 * you should set each port's GraphObject#portId to a string value that is unique for the node.
 * When there may be multiple ports on a node, you can get a collection of elements representing ports
 * by using the #ports property.
 * Use the #findPort method to find a particular port element by name.
 *
 * Note: the only kind of model that can save port information, i.e. portIds that are not an empty string,
 * for links is a GraphLinksModel whose GraphLinksModel#linkFromPortIdProperty and
 * GraphLinksModel#linkToPortIdProperty have been set to name properties on the link data objects.
 * <p class="boxread">
 * For a more general discussion of ports, see <a href="../../intro/ports.html">Introduction to Ports</a>.
 *
 * All of the "findLinks..." and "findNodes..." methods mentioned above take an optional port id argument.
 * When no argument is passed, these methods consider all links connecting with the node.
 * When a port id argument is provided, these methods only consider links that connect with that port
 * in the given node.
 * Thus when navigating through the diagram, you can easily look at all of the nodes that links coming out of
 * a given node go to.  Or you can just look at those nodes at the ends of links coming out of a particular port.
 *
 * You can also control the default connecting behavior of Links at each port.
 * Because a port can be any GraphObject, they are all properties on GraphObject.
 * The properties are duplicated so that you can guide the "from" ends of links differently from the "to" ends of links.
 * The properties include:
 *   - GraphObject#fromSpot, GraphObject#toSpot
 *   - GraphObject#fromEndSegmentLength, GraphObject#toEndSegmentLength
 *   - GraphObject#fromShortLength, GraphObject#toShortLength
 *   - GraphObject#fromLinkable, GraphObject#toLinkable
 *   - GraphObject#fromLinkableDuplicates, GraphObject#toLinkableDuplicates
 *   - GraphObject#fromLinkableSelfNode, GraphObject#toLinkableSelfNode
 *   - GraphObject#fromMaxLinks, GraphObject#toMaxLinks
 *
 * The "...Spot" and "...Length" and "...Direction" properties control the position and routing of links at a port.
 * The "...Linkable..." and "...MaxLinks" properties control whether or not users can draw a new link
 * or reconnect an existing link from or to a port.
 * (The "...Spot" and "...Length" and "...Direction" properties also exist on Link, to override for a particular
 * link the default values that come from a port element.)
 * <p class="boxread">
 * For a more general discussion of link points, see <a href="../../intro/connectionPoints.html">Introduction to Link Connection Points</a>.
 *
 * When the graph is tree-structured, you can use several functions for traversing the tree:
 *   - #findTreeParentNode
 *   - #findTreeChildrenNodes
 *   - #findTreeParentLink
 *   - #findTreeChildrenLinks
 *   - #findTreeRoot
 *   - #findTreeParentChain
 *   - #findTreeParts
 *   - #findCommonTreeParent
 *   - #isInTreeOf
 *   - #findTreeLevel
 *
 * Determining whether a tree grows from the root via links that go out to the children or vice-versa
 * is controlled for the whole diagram by the Diagram#isTreePathToChildren property.
 * However an individual link will be ignored by the above functions if Link#isTreeLink is false.
 *
 * The Node class also supports the notion of expanding and collapsing a subtree of nodes and links,
 * causing those nodes and links to be shown or hidden.
 * Principally this is a matter of setting Node#isTreeExpanded.
 * Of course if the diagram's graph is not tree-structured, these concepts and properties might not apply.
 *
 * If you want to change the appearance of the node you can do so in a function that you assign to
 * the #treeExpandedChanged property.
 * This function must not modify any link relationships or expand or collapse any subtrees -- the property just exists
 * to update the appearance of the node.
 *
 * There is an option for link routing to try to avoid crossing over nodes:
 * Link#routing = Link.AvoidsNodes.
 * You can control whether such links should avoid or ignore a node by setting #avoidable.
 * Set #avoidableMargin to control the area beyond the GraphObject#actualBounds
 * where AvoidsNodes links should not go.
 * <p class="boxread">
 * For more discussion and examples, see <a href="../../intro/nodes.html">Nodes</a>,
 * <a href="../../intro/ports.html">Ports</a>, and
 * <a href="../../intro/connectionPoints.html">Link Points</a>.
 * <p class="boxread">
 * For more about trees, see <a href="../../intro/trees.html">Trees</a>, and
 * <a href="../../intro/subtrees.html">SubTrees</a>.
 * <p class="boxread">
 * To customize user-resizing behavior, please read
 * <a href="../../intro/tools.html#ResizingTool">Introduction to the ResizingTool</a>.
 * To customize user-rotating behavior, please read
 * <a href="../../intro/tools.html#RotatingTool">Introduction to the RotatingTool</a>.
 *
 * Only Nodes that are in Diagrams can have connections via Links.
 * Templates should not be connected with Links, be labels of Links, be members of Groups, or have any Adornments.
 * @extends Part
 * @unrestricted
 */
export class Node extends Part {
    /**
     * Constructs an empty Node.
     * The panel type must be one of the values permitted by Panel#type.
     * @param {PanelLayout=} type if not supplied, the default Panel type is Panel.Position.
     */
    constructor(type?: PanelLayout);
    /**
     * This value for Node#portSpreading indicates that links connecting with a port
     * should all connect at a single point on the side(s) indicated by a Spot that is Spot#isSide.
     * @constant
     */
    static SpreadingNone: EnumValue;
    /**
     * This default value for Node#portSpreading indicates that links connecting with a port
     * should be distributed evenly along the side(s) indicated by a Spot that is Spot#isSide.
     * @constant
     */
    static SpreadingEvenly: EnumValue;
    /**
     * This value for Node#portSpreading indicates that links connecting with a port
     * should packed together based on the link's shape's width
     * on the side(s) indicated by a Spot that is Spot#isSide.
     * @constant
     */
    static SpreadingPacked: EnumValue;
    /**
     * Undocumented
     * @param {Node} other
     * @param {string=} thisportid
     * @param {string=} otherportid
     */
    invalidateLinkBundle(other: Node, thisportid?: string, otherportid?: string): void;
    /**
     * Undocumented
     * @param {Set.<Part>=} ignore An optional set of Links to ignore.
     */
    invalidateConnectedLinks(ignore?: Set<Part>): void;
    /**
     * Gets or sets how link points are computed when the port spot is a "side" spot.
     * The default value is Node.SpreadingEvenly.
     * @since 1.5
     */
    portSpreading: EnumValue;
    /**
     * Gets or sets whether this Node is to be avoided by Links
     * whose Link#routing is Link.AvoidsNodes.
     *
     * The default value is true.
     */
    avoidable: boolean;
    /**
     * Gets or sets the margin around this Node in which avoidable links will not be routed.
     *
     * You may need to increase the #fromEndSegmentLength and #toEndSegmentLength in
     * order to prevent link routes from turning within the avoidable area around the Node.
     *
     * Value must be of type Margin.
     * The default margin is Margin(2,2,2,2)
     */
    avoidableMargin: MarginLike;
    /**
     * Undocumented
     * @expose
     * @param {Rect} result  a Rect to be modified and returned.
     * @return {Rect} the area in document coordinates.
     */
    getAvoidableRect(result: Rect): Rect;
    /**
     * Undocumented
     * @expose
     * @return {Node}
     */
    findVisibleNode(): Node | null;
    /**
     * This read-only property returns an iterator over all of the Links that are connected with this node.
     * This includes both links that are coming out of this node as well as links that are going into this node.
     * Setting Link#fromNode or Link#toNode to refer to this Node
     * will add that Link to this collection.
     *
     * Use the #findLinksConnected, #findLinksOutOf, or #findLinksInto methods
     * to get different subsets of the links, depending on direction or depending on connecting to a particular port.
     *
     * A template should not have any links connected with it.
     */
    readonly linksConnected: Iterator<Link>;
    /**
     * Returns an iterator over all of the Links that connect with this node in either direction,
     * perhaps limited to the given port id on this node.
     * @param {string|null=} pid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @return {Iterator.<Link>}
     */
    findLinksConnected(pid?: string | null): Iterator<Link>;
    /**
     * Returns an iterator over all of the Links that come out of this node,
     * perhaps limited to the given port id on this node.
     * @param {string|null=} pid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @return {Iterator.<Link>}
     */
    findLinksOutOf(pid?: string | null): Iterator<Link>;
    /**
     * Returns an iterator over all of the Links that go into this node,
     * perhaps limited to the given port id on this node.
     * @param {string|null=} pid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @return {Iterator.<Link>}
     */
    findLinksInto(pid?: string | null): Iterator<Link>;
    /**
     * Returns an iterator over the Nodes that are connected with this node
     * in either direction, perhaps limited to the given port id on this node.
     *
     * The results may include this node itself if there is a reflexive link connecting this node with itself.
     * @param {string|null=} pid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @return {Iterator.<Node>}
     */
    findNodesConnected(pid?: string | null): Iterator<Node>;
    /**
     * Returns an iterator over the Nodes that are connected with this node
     * by links coming out of this node, perhaps limited to the given port id on this node.
     * @param {string|null=} pid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @return {Iterator.<Node>}
     */
    findNodesOutOf(pid?: string | null): Iterator<Node>;
    /**
     * Returns an iterator over the Nodes that are connected with this node
     * by links going into this node, perhaps limited to the given port id on this node.
     * @param {string|null=} pid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @return {Iterator.<Node>}
     */
    findNodesInto(pid?: string | null): Iterator<Node>;
    /**
     * Returns an iterator over all of the Links that go from this node to another node or vice-versa,
     * perhaps limited to a given port id on this node and a port id on the other node.
     *
     * If you want all of the links between two nodes in just one direction, use #findLinksTo.
     * @param {Node} othernode
     * @param {string|null=} pid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @param {string|null=} otherpid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @return {Iterator.<Link>}
     * @since 1.2
     */
    findLinksBetween(othernode: Node, pid?: string | null, otherpid?: string | null): Iterator<Link>;
    /**
     * Returns an iterator over all of the Links that go from this node to another node,
     * perhaps limited to a given port id on this node and a port id on the other node.
     *
     * If you want all of the links between two nodes in both directions, use #findLinksBetween.
     * @param {Node} othernode
     * @param {string|null=} pid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @param {string|null=} otherpid A port identifier string; if null the link's portId is ignored and all links are included in the search.
     * @return {Iterator.<Link>}
     * @since 1.2
     */
    findLinksTo(othernode: Node, pid?: string | null, otherpid?: string | null): Iterator<Link>;
    /**
     * Gets or sets the function that is called after a Link has been connected with this Node.
     * It is typically used to modify the appearance of the node.
     * The first argument will be this Node.
     * The second argument will be a Link that is now connected with this node.
     * The third argument will be a GraphObject port indicating which port the link was connected with.
     *
     * If the value is a function, that function must not modify what this Node is connected with.
     * The Link has already been added -- trying to remove it or another link may produce undefined behavior.
     * However, the other end of the link may not yet have been connected with a node (and might never be),
     * so you cannot depend on looking at what the link connects with.
     *
     * The default value is null -- no function is called.
     */
    linkConnected: ((thisNode: Node, newLink: Link, thisPort: GraphObject) => void) | null;
    /**
     * Gets or sets the function that is called after a Link has been disconnected from this Node.
     * It is typically used to modify the appearance of the node.
     * The first argument will be this Node.
     * The second argument will be a Link that had been connected with this node.
     * The third argument will be a GraphObject port indicating which port the link had been connected with.
     *
     * If the value is a function, that function must not modify what this Node is connected with.
     * The Link has already been removed -- trying to add it or another link may produce undefined behavior.
     * The other end of the link may not yet have been disconnected from a node (and might never be),
     * so you cannot depend on looking at what the link connects with.
     *
     * The default value is null -- no function is called.
     */
    linkDisconnected: ((thisNode: Node, oldLink: Link, thisPort: GraphObject) => void) | null;
    /**
     * Gets or sets a predicate that determines whether or not a Link may be connected with this node.
     * If this is non-null, the predicate is called in addition to the predicate that is LinkingBaseTool#linkValidation
     * on the LinkingTool and RelinkingTool.
     * See LinkingBaseTool#isValidLink for more details.
     *
     * The default predicate is null, which is equivalent to simply returning true.
     * The first argument will be the proposed "from" Node (may be null).
     * The second argument will be the proposed "from" GraphObject port (may be null).
     * The third argument will be the proposed "to" Node (may be null).
     * The fourth argument will be the proposed "to" GraphObject port (may be null).
     * The fifth argument may be null when asking about creating a new link,
     * or may be a Link when asking about reconnecting an existing link.
     *
     * The function, if supplied, must not have any side-effects.
     * @since 1.3
     */
    linkValidation: ((fromNode: Node, fromPort: GraphObject, toNode: Node, toPort: GraphObject, link: Link) => boolean) | null;
    /**
     * This read-only property is true when this Node is a label node for a Link.
     *
     * If this is true, then `n.labeledLink` will be a Link and `n.labeledLink.isLabeledLink` will be true.
     * @see #labeledLink
     */
    readonly isLinkLabel: boolean;
    /**
     * Gets or sets the Link for which this Node is acting as a smart label.
     * Most nodes do not act as link labels, so this property will be null.
     *
     * A template should not be a label node for a link.
     * @see #isLinkLabel
     */
    labeledLink: Link | null;
    /**
     * Find a GraphObject with a given GraphObject#portId.
     * If no such GraphObject is found, search for one with the empty string as its port identifier.
     * Finally, when failing to find a port with either the given name or the empty string,
     * this method returns this whole node itself.
     * @param {string} pid
     * @return {GraphObject}
     */
    findPort(pid: string): GraphObject;
    /**
     * This read-only property returns the primary GraphObject representing a port in this node.
     * If there is a GraphObject whose GraphObject#portId is the empty string, return it.
     * If there is no such element, just return this whole Node.
     */
    readonly port: GraphObject;
    /**
     * This read-only property returns an iterator over all of the GraphObjects in this node that act as ports.
     */
    readonly ports: Iterator<GraphObject>;
    /**
     * This predicate is true if this node is a child of the given Node, perhaps indirectly as a descendant.
     *
     * If this node is a child of the given node according to Diagram#isTreePathToChildren, this returns true.
     * Otherwise this searches recursively the chain of tree parents of this node,
     * ignoring links for which Link#isTreeLink is false.
     * A node cannot be in its own subtree.
     *
     * If you what to find out whether this Node is (perhaps indirectly) contained by a Group, use Part#isMemberOf.
     * @param {Node} node the Node that might be a parent or ancestor of this node.
     * @return {boolean}  true if the given node is an ancestor of this node, but false otherwise, including false if it is the same node.
     * @see #findTreeRoot
     * @see #findTreeParentChain
     */
    isInTreeOf(node: Node): boolean;
    /**
     * Return the Node that is at the root of the tree that this node is in, perhaps this node itself.
     *
     * This node will be #isInTreeOf the resulting node, unless the resulting node is this node itself.
     * The graph traversal will ignore links for which Link#isTreeLink is false.
     *
     * If you want to search up the containment hierarchy of Groups, use Part#findTopLevelPart.
     * @return {Node} If this Node has no "tree parent", this returns itself.
     * @see Diagram#isTreePathToChildren
     * @see #findTreeParentChain
     * @since 1.2
     */
    findTreeRoot(): Node;
    /**
     * Find the Node that is the perhaps indirect tree parent of both this node and another one,
     * or this node if it is an ancestor of the other node, or vice-versa.
     *
     * If you want to find the Group that contains two Parts, call Part#findCommonContainingGroup.
     * @param {Node} other
     * @return {Node} may be null if in different trees,
     *   or may be itself if the OTHER argument is THIS node,
     *   or may be itself if the OTHER node is a descendant of THIS node,
     *   or may be the OTHER node if THIS node is in the tree of the OTHER node.
     * @see #findTreeRoot
     * @see #findTreeParentChain
     * @since 1.5
     */
    findCommonTreeParent(other: Node): Node | null;
    /**
     * Returns the Link that connects with the tree parent Node of this node if the graph is tree-structured,
     * if there is such a link and Link#isTreeLink is true.
     * @return {Link} The Link to the parent Node, or null if there is no parent node.
     * @see #findTreeParentNode
     * @see #findTreeChildrenLinks
     * @see Diagram#isTreePathToChildren
     * @see #findTreeRoot
     * @see #findTreeParentChain
     * @since 1.1
     */
    findTreeParentLink(): Link | null;
    /**
     * Returns the Node that is the tree parent of this node if the graph is tree-structured, if there is a parent.
     * Links for which Link#isTreeLink is false are ignored.
     * @return {Node} The parent Node, or null if there is no parent node.
     * @see #findTreeParentLink
     * @see #findTreeChildrenNodes
     * @see Diagram#isTreePathToChildren
     * @see #findTreeRoot
     * @see #findTreeParentChain
     * @since 1.1
     */
    findTreeParentNode(): Node | null;
    /**
     * Return a collection of Parts including this Node,
     * its tree parent link and node, and so on up the chain to the root node.
     *
     * This calls #findTreeParentLink and #findTreeParentNode.
     * Links for which Link#isTreeLink is false are ignored.
     *
     * This may result in undefined behavior if there are cycles of Links that are Link#isTreeLink.
     *
     * The result will include this node and the "root" node and all nodes and links in between.
     * The root node is also accessible directly via #findTreeRoot.
     * If any of the nodes are Groups, their member parts are not included.
     * @return {Set.<Part>} A Set of Nodes and Links.
     * @see #findTreeRoot
     * @see #findTreeParts
     * @since 1.7
     */
    findTreeParentChain(): Set<Part>;
    /**
     * Return how deep this node is in a tree structure.
     * For tree root nodes, this returns zero.
     * This calls #findTreeParentNode to find any tree parent node,
     * so this respects Diagram#isTreePathToChildren and Link#isTreeLink
     * to know which way to traverse links and to know to ignore non-tree links.
     *
     * This may result in undefined behavior if there are cycles of Links that are Link#isTreeLink.
     *
     * If you want to know how deep a Part is nested inside Groups, call Part#findSubGraphLevel.
     * @return {number}
     * @see #findTreeRoot
     * @see #findTreeParentChain
     * @since 1.5
     */
    findTreeLevel(): number;
    /**
     * Returns an Iterator for the collection of Links that connect with the immediate tree children of this node.
     * Links for which Link#isTreeLink is false are ignored.
     *
     * This basically returns either #findLinksOutOf or #findLinksInto,
     * depending on Diagram#isTreePathToChildren,
     * but the results excludes links for which Link#isTreeLink is false.
     * @return {Iterator.<Link>}
     * @see #findTreeParentLink
     * @see #findTreeChildrenNodes
     * @see #findTreeParts
     * @since 1.1
     */
    findTreeChildrenLinks(): Iterator<Link>;
    /**
     * Returns an Iterator for the collection of Nodes that are the immediate tree children of this node.
     * Nodes only connected by links for which Link#isTreeLink is false are ignored.
     *
     * This basically returns either #findNodesOutOf or #findNodesInto,
     * depending on Diagram#isTreePathToChildren.
     * @return {Iterator.<Node>}
     * @see #findTreeParentNode
     * @see #findTreeChildrenLinks
     * @see #findTreeParts
     * @since 1.1
     */
    findTreeChildrenNodes(): Iterator<Node>;
    /**
     * Return a collection of Parts including this Node,
     * all of the Links going to child Nodes, and all of their
     * tree child nodes and links.
     * Links for which Link#isTreeLink is false are ignored.
     *
     * Whether child nodes are found for a parent node by following links out of the parent node or
     * by links coming into the parent node is determined by the value of Diagram#isTreePathToChildren.
     *
     * The result will include this, the "root" node.
     * If any of the nodes are Groups, their member parts are not included.
     *
     * If you want to find the collection of Parts that are contained by a Group, use Group#findSubGraphParts.
     * @param {number=} level How many levels of the tree, starting at this node, to include;
     *    the default is Infinity, including all tree children of this node.  Values less than 1 are treated as 1.
     * @return {Set.<Part>} A Set of Nodes and Links.
     * @see #findTreeChildrenNodes
     * @see #findTreeChildrenLinks
     * @see #findTreeParentChain
     */
    findTreeParts(level?: number): Set<Part>;
    /**
     * Hide each child node and the connecting link,
     * and recursively collapse each child node.
     * This changes the value of Part#isVisible of the whole subtree and the parts owned by
     * those nodes and links.
     * However, this root node's visibility is unchanged.
     *
     * Links are assumed to go from the parent node to the children nodes,
     * unless Diagram#isTreePathToChildren is false.
     * Links for which Link#isTreeLink is false are ignored.
     *
     * This sets #isTreeExpanded to false on this node and on all of the children nodes.
     * For those child nodes that were expanded when they were collapsed,
     * #wasTreeExpanded is set to true.
     *
     * You can also pass in a number of levels to hide nodes beyond a certain level starting at this node.
     * If you want to make sure that all nodes are expanded up to a particular level, call #expandTree.
     * If you want to do both, call `expandTree` before calling `collapseTree` to
     * collapse nodes expanded due to the #wasTreeExpanded flag.
     *
     * This method does not perform a transaction.
     * You may want to call the CommandHandler#collapseTree command, which does perform a transaction
     * and raise a DiagramEvent.
     *
     * To collapse a Group's subgraph of Nodes and Links, use Group#collapseSubGraph.
     * @param {number=} level How many levels of the tree, starting at this node, to keep expanded if already expanded;
     *    the default is 1, hiding all tree children of this node.  Values less than 1 are treated as 1.
     * @see #expandTree
     * @see #findTreeParts
     */
    collapseTree(level?: number): void;
    /**
     * Show each child node and the connecting link,
     * and perhaps recursively expand their child nodes.
     * This may change the value of Part#isVisible of the whole subtree and the parts owned by
     * those nodes and links.
     * However, this root node's visibility is unchanged.
     *
     * This sets #isTreeExpanded to true on this node and on all of the children nodes.
     * Links are assumed to go from the parent node to the children nodes,
     * unless Diagram#isTreePathToChildren is false.
     * Links for which Link#isTreeLink is false are ignored.
     *
     * This will expand a tree child node only if its #wasTreeExpanded property was true.
     *
     * You can also pass in a number of levels in order to be sure that all nodes
     * starting at this node and up through that number of levels are visible.
     * If you want to make sure that there are no nodes expanded after a particular level, call #collapseTree.
     * If you want to do both, call `expandTree` before calling `collapseTree` to
     * collapse nodes expanded due to the #wasTreeExpanded flag.
     *
     * This method does not perform a transaction.
     * You may want to call the CommandHandler#expandTree command, which does perform a transaction
     * and raise a DiagramEvent.
     *
     * To expand a Group's subgraph of Nodes and Links, use Group#expandSubGraph.
     * @param {number=} level How many levels of the tree should be expanded;
     *    the default is 2, showing all tree children of this node and potentially more.
     *    Values less than 2 are treated as 2.
     * @see #collapseTree
     * @see #findTreeParts
     */
    expandTree(level?: number): void;
    /**
     * Gets or sets whether the subtree graph starting at this node is expanded.
     * Changing this property's value will call #collapseTree or #expandTree,
     * and also will call the value of #treeExpandedChanged if it is a function.
     *
     * The initial value is true -- "tree-child" nodes, and the links to them, are shown.
     *
     * There is an analogous property for expanded/collapsed Groups: Group#isSubGraphExpanded.
     */
    isTreeExpanded: boolean;
    /**
     * Gets or sets whether the subtree graph starting at this node
     * had been collapsed by a call to #expandTree on the parent node.
     * The initial value is false.
     * @see #isTreeExpanded
     */
    wasTreeExpanded: boolean;
    /**
     * Gets or sets the function that is called when #isTreeExpanded has changed value.
     * The argument to that function will be this Node.
     *
     * If the value is a function, that function must not expand or collapse any trees of nodes and links.
     * The Node has already been expanded or collapsed -- trying to change it again may produce undefined behavior.
     *
     * The default value is null -- no function is called.
     */
    treeExpandedChanged: ((thisNode: Node) => void) | null;
    /**
     * Gets whether this node has no tree children.
     *
     * The initial value is true, meaning that there are no links connected with child nodes
     * in the direction given by Diagram#isTreePathToChildren.
     * This value changes automatically as link connections are added to or removed from this node.
     * Links for which Link#isTreeLink is false are ignored.
     */
    isTreeLeaf: boolean;
}
/**
 * A Group is a Node that can contain a subgraph of Nodes and Links,
 * which are members of the group.
 * <p class="box">
 * For more discussion, see <a href="../../intro/groups.html">Introduction to Groups</a>.
 * See samples that make use of Groups in the <a href="../../samples/index.html#groups">samples index</a>.
 *
 * Although you can create a Group and Diagram#add it to a Diagram, this does not update the Model.
 * It is more common to create a group by adding a node data object to the model
 * by calling Model#addNodeData. For example:
 * ```js
 *   myDiagram.startTransaction("make new group");
 *   myDiagram.model.addNodeData({ key: "Omega", isGroup: true });
 *   myDiagram.commitTransaction("make new group");
 * ```
 *
 * This will cause a Group to be created (copying the template found in Diagram#groupTemplateMap),
 * added to the Diagram in some Layer (based on Part#layerName), and bound to the group data
 * (resulting in Panel#data referring to that group data object).
 * Note that the JavaScript object includes setting `isGroup` to true,
 * to indicate that the object represents a Group rather than a regular Node or simple Part.
 *
 * The member Parts of a Group, which you can access as the #memberParts collection,
 * belong to the group but are not in the visual tree of the group.
 * All Parts are directly in Layers -- they cannot be inside a Panel.
 * This allows group member parts to be in layers different from the group's layer.
 *
 * You can change the membership of a Node or a simple Part in a Group by setting
 * its Part#containingGroup property.
 * This is done automatically for you by the diagram if you initialize the `group` property on the node data
 * in the model to be the key of the containing group node data.
 * Thus you should do something like:
 * ```js
 *   myDiagram.startTransaction("add new member");
 *   myDiagram.model.addNodeData({ group: someexistinggroup.data.key, ... });
 *   myDiagram.commitTransaction("add new member");
 * ```
 *
 * where you would make sure the node data object included all of the properties you need.
 * You can also change the relationship dynamically by calling GraphLinksModel#setGroupKeyForNodeData.
 *
 * The membership of Links is computed automatically for you by the diagram based on the membership of
 * the connected Nodes.
 * For example, if the Link#fromNode is a top-level node but the Link#toNode is a member of a group,
 * the link is a top-level link.
 * If the two connected nodes both belong to the same group, the link is a member of that group.
 * If the two connected nodes belong to different groups, the link belongs to the common container group, if there is any.
 * Note that if a link connects a member of a group with the group itself, the link is a member of that group.
 *
 * All of the group-member relationships effectively form a tree structure.
 * These properties and methods are useful in navigating these relationships:
 *   - Part#containingGroup
 *   - Part#isTopLevel
 *   - Part#findTopLevelPart
 *   - Part#findSubGraphLevel
 *   - Part#findCommonContainingGroup
 *   - #memberParts
 *   - #findSubGraphParts
 *   - #findExternalLinksConnected
 *   - #findExternalNodesConnected
 *
 * As the membership of a group changes, you may want to update the appearance of the group.
 * You can set the #memberAdded and #memberRemoved properties to be functions that are called.
 * These functions must not modify any membership relationships -- these function properties just exist to update the appearance of the Group.
 *
 * You can control whether certain Nodes are added to a Group by CommandHandler#groupSelection or
 * #addMembers or CommandHandler#addTopLevelParts by affecting the result of CommandHandler#isValidMember,
 * which is responsible for deciding whether it is OK to add a Node to a Group or to remove a Node from a Group to be a top-level node.
 * You can override that predicate on CommandHandler, but it is easier to set the #memberValidation or
 * CommandHandler#memberValidation functional property.
 * <p class="box">
 * For a more general discussion of validation, see <a href="../../intro/validation.html">Introduction to Validation</a>.
 *
 * The area occupied by the subgraph is represented in the group's visual tree by a Placeholder.
 * As the group #placeholder grows and shrinks based on the sizes and positions of the member nodes and links,
 * the group will grow and shrink accordingly.
 * The placeholder is always the Part#locationObject,
 * although you may specify any Spot as the Part#locationSpot.
 * A Group need not have a placeholder, but it may have at most one.
 *
 * A group has its own #layout property that is used to position the member nodes and route the member links.
 *
 * The Group class also supports the notion of expanding and collapsing the subgraph,
 * causing the member nodes and links to be shown or hidden.
 * Principally this is a matter of setting #isSubGraphExpanded.
 * Changes to this property will result in calls to #collapseSubGraph or #expandSubGraph, as appropriate.
 *
 * If you want to change the appearance of the group you can do so in a function that you assign to
 * the #subGraphExpandedChanged property.
 * This function must not modify any member relationships or expand or collapse any groups -- the functional property just exists
 * to update the appearance of the Group.
 * <p class="box">
 * For more discussion and examples, see <a href="../../intro/subgraphs.html">SubGraphs</a>.
 *
 * If you want the user to be able to create a Group out of the currently
 * selected Parts using the CommandHandler#groupSelection command,
 * you need to first set the CommandHandler#archetypeGroupData property
 * to a data object with `isGroup` set to true.
 * If you want the user to be able to ungroup a Group,
 * using the CommandHandler#ungroupSelection command,
 * you need to set #ungroupable to true.
 * <p class="box">
 * For more discussion and examples, see <a href="../../intro/groups.html">Groups</a>,
 * <a href="../../intro/subgraphs.html">SubGraphs</a>, and
 * <a href="../../intro/sizedGroups.html">Sized Groups</a>.
 *
 * Only Groups that are in Diagrams can have member Parts or connections via Links.
 * Templates should not be connected with Links, be labels of Links, be members of Groups, have any member Parts, or have any Adornments.
 * @extends Node
 * @unrestricted
 */
export class Group extends Node {
    /**
     * Constructs an empty Group with no visual elements and no member parts;
     * normally a Group will have some visual elements surrounding a Placeholder.
     * @param {PanelLayout=} type if not supplied, the default Panel type is Panel.Position.
     */
    constructor(type?: PanelLayout);
    /**
     * This read-only property returns a Placeholder that this group may contain in its visual tree.
     */
    readonly placeholder: Placeholder | null;
    /**
     * Gets or sets whether the size of the area of the Group's #placeholder
     * should remain the same during a DraggingTool move until a drop occurs.
     * Groups within temporary layers (such as new Groups during a drag-copy) are unaffected by this property.
     *
     * In other words, when the value is true, re-computing the bounds of the
     * members is suspended until a drop occurs, at which time the border is recomputed,
     * perhaps not including some members that had been dragged out and reparented.
     * The initial value is false.
     */
    computesBoundsAfterDrag: boolean;
    /**
     * Gets or sets whether a #placeholder's bounds includes the bounds of member Links.
     * The default value is true.
     * If this is false, only non-Link member Parts are used to compute the Placeholder's bounds in document coordinates.
     */
    computesBoundsIncludingLinks: boolean;
    /**
     * Gets or sets whether a #placeholder's bounds includes the previous Group.location.
     * The default value is false.
     */
    computesBoundsIncludingLocation: boolean;
    /**
     * Gets or sets whether drag-and-drop events may be bubbled up to this Group if not handled by member Parts.
     * The default value is false -- each Node or Link that is a member of the Group needs to define
     * its own GraphObject#mouseDragEnter, GraphObject#mouseDragLeave, and GraphObject#mouseDrop
     * event handlers if you want dragging/dropping on a member part to act as if the user were acting on the group.
     *
     * This is currently restricted to only call the mouseDragEnter, mouseDragLeave, and mouseDrop event handlers
     * defined on the whole Group, not on any element inside the Group's visual tree.
     * @since 1.5
     */
    handlesDragDropForMembers: boolean;
    /**
     * This read-only property returns an iterator over the member Parts of this Group.
     * Setting Part#containingGroup to refer to this Group
     * will add that part to this collection.
     * The Parts can be Nodes, Links, Groups, or simple Parts.
     *
     * A template should not have any member parts.
     */
    readonly memberParts: Iterator<Part>;
    /**
     * Gets or sets the Layout used to position all of the immediate member nodes and links in this group.
     * By default this property is an instance of Layout -- no special layout is used, which just makes
     * sure each member node has a valid location.
     */
    layout: Layout | null;
    /**
     * Gets or sets the function that is called after a member Part has been added to this Group.
     * It is typically used to modify the appearance of the group.
     * The first argument will be this Group.
     * The second argument will be a Part, typically a Node, but may be a simple Part or a Link.
     *
     * If the value is a function, that function must not modify any membership relationships.
     * The member Part has already been added -- trying to remove it or adding or removing another member or the Group itself may produce undefined behavior.
     *
     * The default value is null -- no function is called.
     */
    memberAdded: ((thisGroup: Group, newPart: Part) => void) | null;
    /**
     * Gets or sets the function that is called after a member Part has been removed from this Group.
     * It is typically used to modify the appearance of the group.
     * The first argument will be this Group.
     * The second argument will be a Part, typically a Node, but may be a simple Part or a Link.
     *
     * If the value is a function, that function must not modify any membership relationships.
     * The member Part has already been removed -- trying to add it or adding or removing another member or the Group itself may produce undefined behavior.
     *
     * The default value is null -- no function is called.
     */
    memberRemoved: ((thisGroup: Group, oldPart: Part) => void) | null;
    /**
     * Gets or sets the predicate that determines whether or not a Part may become a member of this group.
     * If this is non-null, the predicate is called in addition to any CommandHandler#memberValidation predicate.
     *
     * The default predicate is null, which is equivalent to simply returning true.
     * The first argument will be this Group.
     * The second argument will be a Part, typically a Node, but will not be a Link or an Adornment.
     *
     * The function, if supplied, must not have any side-effects.
     */
    memberValidation: ((thisGroup: Group, part: Part) => boolean) | null;
    /**
     * See if the given collection of Parts contains non-Links all for which
     * CommandHandler#isValidMember returns true.
     *
     * The CommandHandler predicate will use CommandHandler#memberValidation
     * and #memberValidation, if either or both are defined.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @param {Iterable.<Part>} coll
     * @return {boolean} true.
     */
    canAddMembers(coll: Iterable<Part>): boolean;
    /**
     * Add the Parts in the given collection as members of this Group
     * for those Parts for which CommandHandler#isValidMember returns true.
     * If the check argument to this method is not supplied or false,
     * this will set Part#containingGroup on each part unconditionally, not calling CommandHandler#isValidMember.
     *
     * The CommandHandler predicate will use CommandHandler#memberValidation
     * and #memberValidation, if either or both are defined.
     *
     * At this time there is no "removeMembers" method.
     * If you want to make a collection of Parts to be top-level parts, not members of any Group but still in the Diagram, call CommandHandler#addTopLevelParts.
     * If you want to remove a collection of Parts not only from a Group but from the whole Diagram, call Diagram#removeParts.
     * @expose
     * @param {Iterable.<Part>} coll
     * @param {boolean=} check whether to call CommandHandler#isValidMember to confirm that it is valid to add the Part to be a member of this Group.
     * @return {boolean} true if all non-Links were added to this Group; false if some Parts or Nodes were not able to be added.
     */
    addMembers(coll: Iterable<Part>, check?: boolean): boolean;
    /**
     * Gets or sets whether the user may ungroup this group.
     * The initial value is false.
     * @see #canUngroup
     */
    ungroupable: boolean;
    /**
     * This predicate returns true if #ungroupable is true,
     * if the layer's Layer#allowUngroup is true, and
     * if the diagram's Diagram#allowUngroup is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may ungroup this object.
     */
    canUngroup(): boolean;
    /**
     * Returns an iterator over all of the Links that connect with this group or any node contained by this group,
     * in either direction, but that are not internal to this group.
     *
     * Links that are contained by this group (even in nested groups) are not included in the result collection.
     * @return {Iterator.<Link>}
     * @see Node#findLinksConnected
     * @since 1.3
     */
    findExternalLinksConnected(): Iterator<Link>;
    /**
     * Returns an iterator over all of the Nodes that are connected with this group or any node contained by this group,
     * by a link in either direction, but that are not internal to this group.
     *
     * Nodes that are contained by this group (even in nested groups) are not included in the result collection.
     * However this group itself might be in the results if there is a reflexive link connected to this group.
     * @return {Iterator.<Node>}
     * @see Node#findNodesConnected
     * @since 1.3
     */
    findExternalNodesConnected(): Iterator<Node>;
    /**
     * Return a collection of Parts that are all of the nodes and links
     * that are members of this group, including inside nested groups,
     * but excluding this group itself.
     *
     * For member nodes that are Groups, this will include its members recursively.
     *
     * If you want only the immediate members of this group, use the #memberParts property.
     *
     * If you want to find the collection of Nodes and Links that are in the subtree of a given Node, use Node#findTreeParts.
     * @return {Set.<Part>}
     */
    findSubGraphParts(): Set<Part>;
    /**
     * Hide each of the member nodes and links of this group,
     * and recursively collapse any member groups.
     * This changes the value of Part#isVisible of the whole subgraph and the parts owned by
     * those member nodes and links.
     * However, this group's visibility is unchanged.
     *
     * This sets #isSubGraphExpanded to false on this group and on all of the nested Groups.
     * For those nested Groups that were expanded,
     * #wasSubGraphExpanded is set to true.
     *
     * To collapse trees made of Nodes and Links, use Node#collapseTree.
     */
    collapseSubGraph(): void;
    /**
     * Show each member node and link,
     * and perhaps recursively expand nested subgraphs.
     * This may change the value of Part#isVisible of the whole subgraph and the parts owned by
     * those member nodes and links.
     * However, this group's visibility is unchanged.
     *
     * This sets #isSubGraphExpanded to true on this group and on all of the nested Groups.
     * This will expand a nested group only if its #wasSubGraphExpanded property was true.
     *
     * To expand trees made of Nodes and Links, use Node#expandTree.
     */
    expandSubGraph(): void;
    /**
     * Gets or sets whether the subgraph contained by this group is expanded.
     * Changing this property's value will call #collapseSubGraph or #expandSubGraph,
     * and also will call the value of #subGraphExpandedChanged if it is a function.
     *
     * The initial value is true -- this group's member parts are shown.
     *
     * There is an analogous property for expanded/collapsed trees of Nodes and Links: Node#isTreeExpanded.
     */
    isSubGraphExpanded: boolean;
    /**
     * Gets or sets whether the subgraph starting at this group
     * had been collapsed by a call to #expandSubGraph on the containing Group.
     * The initial value is false.
     * @see #isSubGraphExpanded
     */
    wasSubGraphExpanded: boolean;
    /**
     * Gets or sets the function that is called when #isSubGraphExpanded has changed value.
     * The argument to that function will be this Group.
     *
     * If the value is a function, that function must not expand or collapse any groups.
     * The Group has already been expanded or collapsed -- trying to change it again may produce undefined behavior.
     *
     * The default value is null -- no function is called.
     */
    subGraphExpandedChanged: ((thisGroup: Group) => void) | null;
    /**
     * Move this Group and all of its member parts, recursively.
     * @param {Point} newpos a new Point in document coordinates.
     * @param {boolean=} useLocation true if you want to set the #location instead of the position. False by default.
     */
    move(newpos: Point, useLocation?: boolean): void;
}
/**
 * If a Placeholder is in the visual tree of a Group, it represents the area of all of the member Parts of that Group.
 * If a Placeholder is in the visual tree of an Adornment, it represents the area of the Adornment#adornedObject.
 * Unlike other GraphObjects, the Placeholder measures itself based on these criteria, and not on its own #desiredSize or #scale.
 * It will set its own natural bounds and scale, and in the case of Groups will set the Group's location.
 *
 * It can only be used in the visual tree of a Group node or an Adornment.
 * There can be at most one Placeholder in a Group or an Adornment.
 *
 * See the documentation for Group and Adornment for descriptions of the Placeholder roles in each.
 * @extends GraphObject
 * @unrestricted
 */
export class Placeholder extends GraphObject {
    /**
     * This normally does not need any initialization,
     * except for maybe setting #padding.
     */
    constructor();
    /**
     * This is only called when the Placeholder is inside a Group.
     * Normally this just returns the result of #computeMemberBounds expanded by the #padding.
     * However, if Group#computesBoundsAfterDrag is true,
     * and if the Diagram#currentTool is the DraggingTool,
     * and if this Group is not being dragged,
     * this method returns the last value of #computeBorder before dragging began.
     * @expose
     * @param {Rect} result
     * @return {Rect} in document coordinates.
     */
    protected computeBorder(result: Rect): Rect;
    /**
     * Compute the union of the Bounds of this Placeholder's parent's Group.memberParts.
     * If there are no members, this returns a Rect with Width and Height of zero
     * and an X and Y that are this panel's original location in document coordinates.
     * This is only called when the Placeholder is inside a Group.
     * @expose
     * @param {Rect} result
     * @return {Rect} in document coordinates.
     */
    protected computeMemberBounds(result: Rect): Rect;
    /**
     * Gets or sets the padding around the members of the Group or around the Adornment#adornedObject GraphObject.
     * The initial value is a Margin of zero on all sides.
     */
    padding: MarginLike;
}
/**
 * A Link is a Part that connects Nodes.
 * The link relationship is directional, going from Link#fromNode to Link#toNode.
 * A link can connect to a specific port element in a node, as named by the Link#fromPortId
 * and Link#toPortId properties.
 * <p class="boxread">
 * For more discussion, see <a href="../../intro/links.html">Introduction to Links</a>.
 *
 * To add a Link to a Diagram when using a GraphLinksModel you should do something like:
 * ```js
 *   myDiagram.startTransaction("make new link");
 *   myDiagram.model.addLinkData({ from: "Alpha", to: "Beta" });
 *   myDiagram.commitTransaction("make new link");
 * ```
 *
 * where you would substitute the keys of the actual nodes that you want to connect with a link.
 * This will cause a Link to be created (copying the template found in Diagram#linkTemplateMap),
 * added to the Diagram in some Layer (based on Part#layerName), and bound to the link data
 * (resulting in Panel#data referring to that link data object).
 * Note that link data objects, unlike Node data, do not have their own unique keys or identifiers,
 * because other Parts do not refer to them.
 *
 * If you are using a TreeModel, there are no link data objects, so you just need to call
 * TreeModel#setParentKeyForNodeData to specify the "parent" node's key for a "child" node data.
 *
 * To find a Link given a link data object in the GraphLinksModel,
 * call Diagram#findLinkForData.  When using a TreeModel, call either Diagram#findNodeForData
 * or Diagram#findNodeForKey to get a Node, and then call Node#findTreeParentLink to get the Link, if any exists.
 *
 * To find a link that connects two nodes, call Node#findLinksTo or Node#findLinksBetween.
 * With the former method, the direction matters; with the latter method it returns links in either direction.
 *
 * A link's position and size are determined by the two nodes that it connects.
 * Normally there should be a Shape as the main element in this Link.
 * This shape is what users will see as the "line" or "wire" --
 * you can set its Shape#stroke and other "stroke..." properties to control its appearance.
 *
 * The link will compute a route (a sequence of points) going from the #fromNode's port element
 * to the #toNode's port element.
 * That route is used to generate the path of the main shape.
 * Properties that affect the nature of the route and the geometry of the path include:
 *   - #curve
 *   - #curviness
 *   - #corner
 *   - #routing
 *   - #smoothness
 *   - #adjusting
 *
 * <p class="boxread">
 * For more discussion and examples, see <a href="../../intro/links.html">Links</a>.
 *
 * There are additional properties that affect how the end of the link connects to a port element of a node.
 * There are duplicate properties, ones for the "to" end and ones for the "from" end:
 *   - #fromSpot, #toSpot
 *   - #fromEndSegmentLength, #toEndSegmentLength
 *   - #fromShortLength, #toShortLength
 *
 * These properties normally have "default" values, causing the link's routing and path-geometry generating
 * to get the corresponding values from the connected port element.
 * This scheme permits an individual link to have its own specific connection to a port,
 * taking precedence over how the port normally expects links to connect to it.
 * For example, several of the Layout classes sets these properties on each Link
 * as part of their route computation for links.
 * <p class="boxread">
 * For more discussion and examples, see <a href="../../intro/connectionPoints.html">Link Points</a>.
 *
 * Elements other than the main Shape in the Link may act as decorations on the link, including arrowheads and labels.
 * You can control where they are located along the link route and how they are oriented.
 * Because these decorations may be any GraphObject, they are all properties of that class.
 * The properties include:
 *   - GraphObject#segmentIndex
 *   - GraphObject#segmentFraction
 *   - GraphObject#segmentOffset
 *   - GraphObject#segmentOrientation
 *
 * If you do not set the GraphObject#segmentIndex property, the object is positioned
 * to be at the middle of the link.
 * <p class="boxread">
 * For more discussion and examples, see <a href="../../intro/linkLabels.html">Link Labels</a>.
 *
 * **GoJS** makes it easy to add arrowheads to your link template.
 * Just add a Shape with the appearance properties that you want,
 * and also set the Shape#toArrow or Shape#fromArrow property
 * to the name of the kind of arrowhead that you want.
 * Doing so automatically sets the "segment..." properties that are appropriate for the chosen arrowhead.
 *
 * More than one shape may automatically get the route geometry.
 * This is useful when you want to have multiple link shapes with different thicknesses
 * to create a gradient effect across the path of the link or to produce parallel lines along the path.
 * Just set GraphObject#isPanelMain to true on each such Shape.
 *
 * If you want the user to be able to reconnect a link, using the RelinkingTool,
 * you need to set one or both of #relinkableFrom and #relinkableTo to true.
 * The RelinkingTool shows a RelinkingTool#fromHandleArchetype and/or a
 * RelinkingTool#toHandleArchetype when the link is selected.
 * Such a relink handle can be dragged by the user to start a relinking operation.
 *
 * If you want the user to be able to change the path of the link, using the LinkReshapingTool,
 * set Part#reshapable to true.
 * The LinkReshapingTool shows reshape handles that the user can drag to shift the position
 * of a point in the link's route.
 * The LinkReshapingTool#handleArchetype is copied for each reshape handle.
 *
 * Often if a Link is reshapable, you will want to save the route in the model so that it can be restored
 * upon load.  To save the route automatically, add a TwoWay Binding on the #points property:
 * `new go.Binding("points").makeTwoWay()`.  Model#toJson will automatically
 * convert the List of Points into an Array of numbers in the JSON representation,
 * if the property is named "points".
 *
 * If a Link is Part#reshapable, it is also possible to allow the user to add and remove segments
 * from the link's route by setting #resegmentable to true.
 * This causes the LinkReshapingTool to add resegmenting handles at the midpoints of each segment.
 * The LinkReshapingTool#midHandleArchetype is copied for each resegment handle.
 * When the user drags such a resegmenting handle, a new segment is inserted into the route.
 * Also, when the user drags a reshape handle such that two adjacent segments end up in a straight line,
 * a segment is removed from the route.
 * <p class="boxread">
 * For more discussion and examples, see <a href="../../intro/links.html">Links</a>,
 * <a href="../../intro/linkLabels.html">Link Labels</a>, and
 * <a href="../../intro/connectionPoints.html">Link Points</a>.
 * <p class="boxread">
 * To control what links a user may draw or reconnect, please read about
 * <a href="../../intro/validation.html">Validation</a>.
 * <p class="boxread">
 * To customize linking and relinking behavior, please read
 * <a href="../../intro/tools.html#LinkingToolAndRelinkingTool">Introduction to the Linking Tools</a>
 * and <a href="../../intro/tools.html#RelinkingTool">Introduction to the RelinkingTool</a>.
 * For customizing the reshaping of Links,
 * see <a href="../../intro/tools.html#LinkReshapingTool">Introduction to the LinkReshapingTool</a>.
 *
 * Only Links that are in Diagrams can have connections with Nodes.
 * Templates should not be connected with Nodes, be members of Groups, or have any Adornments.
 * @extends Part
 * @unrestricted
 */
export class Link extends Part {
    /**
     * Constructs an empty link that does not connect any nodes.
     * If you want the link to be seen, you must provide a Shape as an element
     * to be used as the visual path for the link.
     */
    constructor();
    /**
     * Used as the default value for Link#routing:
     * the route goes fairly straight between ports.
     * @constant
     */
    static Normal: EnumValue;
    /**
     * Used as a value for Link#routing:
     * each segment is horizontal or vertical.
     * @constant
     */
    static Orthogonal: EnumValue;
    /**
     * Used as a value for Link#routing:
     * each segment is horizontal or vertical, but the route tries to avoid crossing over nodes.
     * @constant
     */
    static AvoidsNodes: EnumValue;
    /**
     * This is the default value for Link#curve and Link#adjusting,
     * to indicate that the path geometry consists of straight line segments and
     * to indicate that the link route computation does not depend on any previous route points;
     * this can also be used as a value for GraphObject#segmentOrientation
     * to indicate that the object is never rotated along the link route -- its angle is unchanged.
     * @constant
     */
    static None: EnumValue;
    /**
     * Used as a value for Link#curve, to indicate that
     * the link path uses Bezier curve segments.
     * @constant
     */
    static Bezier: EnumValue;
    /**
     * Used as a value for Link#curve, to indicate that
     * orthogonal link segments will be discontinuous where they cross over
     * other orthogonal link segments that have a Link#curve or JumpOver or JumpGap.
     * @constant
     */
    static JumpGap: EnumValue;
    /**
     * Used as a value for Link#curve, to indicate that
     * orthogonal link segments will veer around where they cross over
     * other orthogonal link segments that have a Link#curve or JumpOver or JumpGap.
     * @constant
     */
    static JumpOver: EnumValue;
    /**
     * Used as a value for Link#adjusting,
     * to indicate that the link route computation should keep the
     * intermediate points of the previous route, just modifying the first and/or last points;
     * if the routing is orthogonal, it will only modify the first two and/or last two points.
     * @constant
     */
    static End: EnumValue;
    /**
     * Used as a value for Link#adjusting,
     * to indicate that the link route computation should scale and rotate the intermediate points
     * so that the link's shape looks approximately the same;
     * if the routing is orthogonal, this value is treated as if it were Link.End.
     * @constant
     */
    static Scale: EnumValue;
    /**
     * Used as a value for Link#adjusting,
     * to indicate that the link route computation should linearly interpolate the intermediate points
     * so that the link's shape looks stretched;
     * if the routing is orthogonal, this value is treated as if it were Link.End.
     * @constant
     */
    static Stretch: EnumValue;
    /**
     * This value for GraphObject#segmentOrientation results in
     * the GraphObject turned to have the same angle as the route:
     * the GraphObject's angle is always the same as the angle of the link's route
     * at the segment where the GraphObject is attached;
     * use this orientation for arrow heads.
     * @constant
     */
    static OrientAlong: EnumValue;
    /**
     * This value for GraphObject#segmentOrientation results in
     * the GraphObject is turned clockwise to be perpendicular to the route:
     * the GraphObject's angle is always 90 degrees more than the angle of the link's route
     * at the segment where the GraphObject is attached.
     * @constant
     */
    static OrientPlus90: EnumValue;
    /**
     * This value for GraphObject#segmentOrientation results in
     * the GraphObject being turned counter-clockwise to be perpendicular to the route:
     * the GraphObject's angle is always 90 degrees less than the angle of the link's route
     * at the segment where the GraphObject is attached.
     * @constant
     */
    static OrientMinus90: EnumValue;
    /**
     * This value for GraphObject#segmentOrientation results in
     * the GraphObject's angle always being 180 degrees opposite from the angle of the link's route
     * at the segment where the GraphObject is attached.
     * @constant
     */
    static OrientOpposite: EnumValue;
    /**
     * This value for GraphObject#segmentOrientation results in
     * the GraphObject turned to have the same angle as the route, just like Link.OrientAlong, but is never upside down:
     * the GraphObject's angle always following the angle of the link's route
     * at the segment where the GraphObject is attached;
     * this is typically only used for TextBlocks or Panels that contain text.
     * @constant
     */
    static OrientUpright: EnumValue;
    /**
     * This value for GraphObject#segmentOrientation results in
     * the GraphObject turned clockwise to be perpendicular to the route, just like Link.OrientPlus90, but is never upside down:
     * the GraphObject's angle always being 90 degrees more than the angle of the link's route
     * at the segment where the GraphObject is attached;
     * this is typically only used for TextBlocks or Panels that contain text.
     * @constant
     */
    static OrientPlus90Upright: EnumValue;
    /**
     * This value for GraphObject#segmentOrientation results in
     * the GraphObject turned counter-clockwise to be perpendicular to the route, just like Link.OrientMinus90, but is never upside down:
     * the GraphObject's angle always being 90 degrees less than the angle of the link's route
     * at the segment where the GraphObject is attached;
     * this is typically only used for TextBlocks or Panels that contain text.
     * @constant
     */
    static OrientMinus90Upright: EnumValue;
    /**
     * This value for GraphObject#segmentOrientation results in
     * the GraphObject's angle always following the angle of the link's route
     * at the segment where the GraphObject is attached, but never upside down
     * and never angled more than +/- 45 degrees:
     * when the route's angle is within 45 degrees of vertical (90 or 270 degrees),
     * the GraphObject's angle is set to zero;
     * this is typically only used for TextBlocks or Panels that contain text.
     * @constant
     */
    static OrientUpright45: EnumValue;
    /**
     * Gets or sets the Node that this link comes from.
     * The #fromPortId specifies which port the link comes from.
     * The default value is null -- this link is not coming from any node.
     *
     * A template should not be connected with any node.
     */
    fromNode: Node | null;
    /**
     * Gets or sets the identifier of the port that this link comes from.
     * The default value is the empty string.
     */
    fromPortId: string;
    /**
     * This read-only property returns a GraphObject that is the "from" port that this link is connected from.
     * The #fromNode provides the node that the link is coming from.
     * The #fromPortId provides the identifier for which port this link is coming from;
     * you can set that property in order to change the value of this property.
     * This method may return null.
     */
    readonly fromPort: GraphObject | null;
    /**
     * Gets or sets the function that is called after this Link changes which Node or port it connects from.
     * The first argument will be this Link.
     * The second argument will be the old GraphObject port.
     * The third argument will be the new GraphObject port.
     *
     * If the value is a function, that function must not modify which nodes or links this link connects with.
     * The "from" node and/or port has already been changed -- trying to change it again may produce undefined behavior.
     *
     * The default value is null -- no function is called.
     */
    fromPortChanged: ((thisLink: Link, oldPort: GraphObject, newPort: GraphObject) => void) | null;
    /**
     * Gets or sets the Node that this link goes to.
     * The #toPortId specifies which port the link goes to.
     * The default value is null -- this link is not going to any node.
     *
     * A template should not be connected with any node.
     */
    toNode: Node | null;
    /**
     * Gets or sets the identifier of the port that this link goes to.
     * The default value is the empty string.
     */
    toPortId: string;
    /**
     * This read-only property returns a GraphObject that is the "to" port that this link is connected to.
     * The #toNode provides the node that the link is going to.
     * The #toPortId provides the identifier for which port this link is going to;
     * you can set that property in order to change the value of this property.
     * This method may return null.
     */
    readonly toPort: GraphObject | null;
    /**
     * Gets or sets the function that is called after this Link changes which Node or port it connects to.
     * The first argument will be this Link.
     * The second argument will be the old GraphObject port.
     * The third argument will be the new GraphObject port.
     *
     * If the value is a function, that function must not modify which nodes or links this link connects with.
     * The "to" node and/or port has already been changed -- trying to change it again may produce undefined behavior.
     *
     * The default value is null -- no function is called.
     */
    toPortChanged: ((thisLink: Link, oldPort: GraphObject, newPort: GraphObject) => void) | null;
    /**
     * Gets or sets where this link should connect at the #fromPort.
     * The default value is Spot.Default, meaning that the value
     * actually comes from the GraphObject#fromSpot property of the #fromPort.
     *
     * This property is set by some layouts, if ForceDirectedLayout#setsPortSpots or
     * LayeredDigraphLayout#setsPortSpots or TreeLayout#setsPortSpot or
     * TreeLayout#setsChildPortSpot is true.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     * @see #toSpot
     * @see GraphObject#fromSpot
     * @see #computeSpot
     */
    fromSpot: Spot;
    /**
     * Gets or sets the length of the first segment, when the computed "from spot" at the #fromPort is not Spot.None.
     * The default value is NaN, meaning that the value
     * actually comes from the GraphObject#fromEndSegmentLength property of the #fromPort.
     * This value also limits how short the #fromShortLength may be drawn.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     * @see #toEndSegmentLength
     * @see #computeEndSegmentLength
     * @see GraphObject#fromEndSegmentLength
     * @see #fromShortLength
     */
    fromEndSegmentLength: number;
    /**
     * Gets or sets how far the end segment stops short of the actual port.
     * Positive values are limited by the #fromEndSegmentLength or GraphObject#fromEndSegmentLength.
     * Negative values cause the link to extend into the port.
     * The default value is NaN -- the value actually comes from the
     * GraphObject#fromShortLength property of the #fromPort.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     * If you want to move an arrowhead away from the start of the link route,
     * set the arrowhead Shape's GraphObject#segmentOffset.
     * @see #toShortLength
     * @see GraphObject#fromShortLength
     * @see #fromEndSegmentLength
     */
    fromShortLength: number;
    /**
     * Gets or sets where this link should connect at the #toPort.
     * The default value is Spot.Default, meaning that the value
     * actually comes from the GraphObject#toSpot property of the #toPort.
     *
     * This property is set by some layouts, if ForceDirectedLayout#setsPortSpots or
     * LayeredDigraphLayout#setsPortSpots or TreeLayout#setsPortSpot or
     * TreeLayout#setsChildPortSpot is true.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     * @see #fromSpot
     * @see GraphObject#toSpot
     * @see #computeSpot
     */
    toSpot: Spot;
    /**
     * Gets or sets the length of the last segment.
     * The default value is NaN, meaning that the value
     * actually comes from the GraphObject#toEndSegmentLength property of the #toPort.
     * This value also limits how short the #toShortLength may be drawn.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     * @see #fromEndSegmentLength
     * @see #computeEndSegmentLength
     * @see GraphObject#toEndSegmentLength
     * @see #toShortLength
     */
    toEndSegmentLength: number;
    /**
     * Gets or sets how far the end segment stops short of the actual port.
     * Positive values are limited by the #toEndSegmentLength or GraphObject#toEndSegmentLength.
     * Negative values cause the link to extend into the port.
     * The default value is NaN -- the value actually comes from the
     * GraphObject#toShortLength property of the #toPort.
     *
     * For examples of how to use this property, see <a href="../../intro/connectionPoints.html">Link Connection Points</a>.
     * If you want to move an arrowhead away from the end of the link route,
     * set the arrowhead Shape's GraphObject#segmentOffset.
     * @see #fromShortLength
     * @see GraphObject#toShortLength
     * @see #toEndSegmentLength
     */
    toShortLength: number;
    /**
     * Given a Node, return the node at the other end of this link.
     * @param {Node} node
     * @return {Node} This may return the same node, if the link is reflexive.
     */
    getOtherNode(node: Node): Node | null;
    /**
     * Given a GraphObject that is a "port", return the port at the other end of this link.
     * @param {GraphObject} port
     * @return {GraphObject} This may return the same object, if the link is reflexive.
     */
    getOtherPort(port: GraphObject): GraphObject | null;
    /**
     * This read-only property is true when this Link has any label Nodes, Nodes that are owned by this Link
     * and are arranged along its path in the same manner as elements of the Link Panel.
     * @see #labelNodes
     * @since 1.1
     */
    readonly isLabeledLink: boolean;
    /**
     * This read-only property returns an iterator over the Nodes that act as labels on this Link.
     * Setting Node#labeledLink to refer to this Link
     * will add that Node to this collection.
     *
     * Do not confuse these Nodes with the GraphObjects that
     * are part of this Link Panel.
     * Those objects can also be used as "labels", but not as objects to which
     * there can be link connections, because those panel elements cannot be Nodes.
     * Those panel objects can be used for other purposes too, such as arrowheads.
     *
     * A template should not have any label nodes.
     * @see #isLabeledLink
     */
    readonly labelNodes: Iterator<Node>;
    /**
     * Move this link to a new position.
     * This also shifts all of the Points in the route accordingly.
     * This also moves any #labelNodes.
     * @param {Point} newpos a new Point in document coordinates.
     * @param {boolean=} useLocation true if you want to set the #location instead of the position. False by default.
     */
    move(newpos: Point, useLocation?: boolean): void;
    /**
     * Gets or sets whether the user may reconnect an existing link at the "from" end.
     * This affects the behavior of the RelinkingTool.
     *
     * The initial value is false.
     */
    relinkableFrom: boolean;
    /**
     * Gets or sets whether the user may reconnect an existing link at the "to" end.
     * This affects the behavior of the RelinkingTool.
     *
     * The initial value is false.
     */
    relinkableTo: boolean;
    /**
     * This predicate returns true if #relinkableFrom is true,
     * if the layer's Layer#allowRelink is true, and
     * if the diagram's Diagram#allowRelink is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may relink the From end of this link.
     */
    canRelinkFrom(): boolean;
    /**
     * This predicate returns true if #relinkableTo is true,
     * if the layer's Layer#allowRelink is true, and
     * if the diagram's Diagram#allowRelink is true.
     *
     * This does not check Diagram#isReadOnly or Model#isReadOnly,
     * but commands and tools should check those properties.
     * @expose
     * @return {boolean} true if the user may relink the To end of this link.
     */
    canRelinkTo(): boolean;
    /**
     * Gets or sets whether the user may change the number of segments in this Link,
     * if the link has straight segments.
     * This affects the behavior of the LinkReshapingTool when #curve is not Link.Bezier.
     *
     * The initial value is false.
     * @since 1.2
     */
    resegmentable: boolean;
    /**
     * Gets or sets whether this Link is part of the tree for tree operations
     * such as Node#findTreeChildrenNodes or Node#collapseTree.
     *
     * The initial value is true.
     * @since 1.2
     */
    isTreeLink: boolean;
    /**
     * This read-only property returns the main Shape representing the path of this Link.
     *
     * The value depends on there being an element in this Link that is
     * a Shape and that has GraphObject#isPanelMain set to true.
     * This shape is what displays the stroke going from one node to another node.
     *
     * This path shape's geometry is automatically generated based on the #points
     * of this link's route, depending on other properties such as #curve.
     * Note that the points are in document coordinates, but the path's Shape#geometry
     * will have points that are in local Shape coordinates.
     * This Shape's GraphObject#scale should be one and GraphObject#angle should be zero.
     *
     * If there is no Shape with GraphObject#isPanelMain set to true,
     * the Link will use the first Shape for its path.
     */
    readonly path: Shape | null;
    /**
     * Undocumented
     */
    readonly routeBounds: Rect;
    /**
     * This read-only property returns the point at the middle of the path, in document coordinates.
     */
    readonly midPoint: Point;
    /**
     * This read-only property returns the angle of the path at the #midPoint.
     */
    readonly midAngle: number;
    /**
     * Gets or sets the List of Points in the route.
     * All of the Points must be Point#isReal -- no NaN or infinite values --
     * and all of the points are in document coordinates.
     * Although this list may be replaced by setting this property,
     * one must not modify the contents of the List directly.
     *
     * The setter also accepts an Array of alternating X,Y point values [1, 2, 3, 4, 5, 6],
     * or an Array of Objects each of which has (lowercase) 'x' and 'y' properties that are numbers
     * [{"x":1,"y":2}, {"x":3,"y":4}, {"x":5,"y":6}].
     *
     * Ownership of the List and all of its Points that is provided to the setter
     * is transferred to this Link.
     */
    points: List<Point>;
    /**
     * This read-only property returns the number of points in the route.
     */
    readonly pointsCount: number;
    /**
     * Gets a particular point of the route.
     * @param {number} i int  The zero-based index of the desired point.
     * @return {Point} in document coordinates
     */
    getPoint(i: number): Point;
    /**
     * Sets a particular point of the route; this may only be called within an override of #computePoints.
     * @param {number} i int  The zero-based index of the desired point.
     * @param {Point} p  The new point in document coordinates, which should not have infinite or NaN coordinate values, and which must not be modified afterwards.
     * @see #getPoint
     * @see #insertPoint
     * @see #addPoint
     * @see #removePoint
     * @since 1.6
     */
    setPoint(i: number, p: Point): void;
    /**
     * Undocumented
     * @param {number} i int  The zero-based index of the desired point.
     * @param {number} x  The new X, which should not be infinite or NaN, in document coordinates.
     * @param {number} y  The new Y, which should not be infinite or NaN, in document coordinates.
     */
    setPointAt(i: number, x: number, y: number): void;
    /**
     * Insert a point at a particular position in the route, without replacing an existing point; this may only be called within an override of #computePoints.
     * @param {number} i int  The zero-based index of the new point.
     * @param {Point} p  The new point in document coordinates, which should not have infinite or NaN coordinate values, and which must not be modified afterwards.
     * @see #getPoint
     * @see #setPoint
     * @see #addPoint
     * @see #removePoint
     * @since 1.6
     */
    insertPoint(i: number, p: Point): void;
    /**
     * Undocumented
     * @param {number} i int  The zero-based index of the new point.
     * @param {number} x  The new X, which should not be infinite or NaN, in document coordinates.
     * @param {number} y  The new Y, which should not be infinite or NaN, in document coordinates.
     */
    insertPointAt(i: number, x: number, y: number): void;
    /**
     * Add a point at the end of the route; this may only be called within an override of #computePoints.
     * @param {Point} p  The new point in document coordinates, which should not have infinite or NaN coordinate values, and which must not be modified afterwards.
     * @see #getPoint
     * @see #setPoint
     * @see #insertPoint
     * @see #removePoint
     * @since 1.6
     */
    addPoint(p: Point): void;
    /**
     * Undocumented
     * @param {number} x  The new X, which should not be infinite or NaN, in document coordinates.
     * @param {number} y  The new Y, which should not be infinite or NaN, in document coordinates.
     */
    addPointAt(x: number, y: number): void;
    /**
     * Remove a particular point from the route; this may only be called within an override of #computePoints.
     * @param {number} i int  The zero-based index of the point to extract.
     * @see #getPoint
     * @see #setPoint
     * @see #insertPoint
     * @see #addPoint
     * @see #clearPoints
     * @since 1.6
     */
    removePoint(i: number): void;
    /**
     * Remove all of the points from this link's route; this may only be called within an override of #computePoints.
     * @see #getPoint
     * @see #setPoint
     * @see #insertPoint
     * @see #addPoint
     * @see #removePoint
     * @since 1.6
     */
    clearPoints(): void;
    /**
     * Allow calls to #setPoint, #addPoint, #insertPoint,
     * #removePoint, and #clearPoints.
     * You must call #commitRoute when you are done modifying the route.
     */
    startRoute(): void;
    /**
     * Call this method after a call to #startRoute and calls to methods that modify the route.
     */
    commitRoute(): void;
    /**
     * Call this method instead of #commitRoute if you need to cancel changes to the route.
     */
    rollbackRoute(): void;
    /**
     * Declare that the route (the #points) of this Link need to be recomputed soon.
     * This causes #updateRoute to be called, which will call #computePoints
     * to perform the actual determination of the route.
     * @since 1.6
     */
    invalidateRoute(): void;
    /**
     * Undocumented.
     */
    defaultFromPoint: Point;
    /**
     * Undocumented.
     */
    defaultToPoint: Point;
    /**
     * This method recomputes the route if the route is invalid,
     * to make sure the #points are up-to-date.
     * This method calls #computePoints in order to calculate a new route.
     *
     * The route may become invalid if either connected node is moved,
     * or if various properties are modified, such as #routing.
     * Such changes will call #invalidateRoute.
     *
     * It is unusual for you to have to call this method.
     * @since 1.6
     */
    updateRoute(): void;
    /**
     * The code that constructs a new route by modifying the #points.
     * It is only called by #updateRoute, when needed.
     * All route points are always in document coordinates.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     *
     * Overrides of this method may call a number of "protected" methods in order to build the route:
     * #clearPoints, #addPoint, #insertPoint, #removePoint, #setPoint.
     * Overrides of this method must not set #points.
     *
     * Other methods that are commonly called and may be overridden are:
     * #computeEndSegmentLength, #computeSpot, #computeOtherPoint,
     * #computeCurve, #computeCurviness, #hasCurviness,
     * #computeThickness,
     * #makeGeometry.
     * @expose
     * @return {boolean} true if it computed a route of points
     * @since 1.6
     */
    computePoints(): boolean;
    /**
     * Compute the intersection point in document coordinates for the edge of a particular port GraphObject, given a point,
     * when no particular spot or side has been specified.
     * @expose
     * @param {Node} node
     * @param {GraphObject} port the GraphObject representing a port on the node.
     * @param {Point} focus the point in document coordinates to/from which the link should point,
     *   normally the center of the port.
     * @param {Point} p often this point is far away from the node, to give a general direction,
     *   particularly an orthogonal one.
     * @param {boolean} from true if the link is coming out of the port; false if going to the port.
     * @param {Point=} result an optional Point that is modified and returned; otherwise it allocates and returns a new Point
     * @return {Point} the point in document coordinates of the intersection point on the edge of the port.
     * @since 1.2
     */
    getLinkPointFromPoint(node: Node | null, port: GraphObject | null, focus: Point, p: Point, from: boolean, result?: Point): Point;
    /**
     * Compute the point on a node/port in document coordinates at which the route of a link should end.
     * @expose
     * @param {Node} node
     * @param {GraphObject} port the GraphObject representing a port on the node.
     * @param {Spot} spot a Spot value describing where the link should connect.
     * @param {boolean} from true if the link is coming out of the port; false if going to the port.
     * @param {boolean} ortho whether the link should have orthogonal segments.
     * @param {Node} othernode the node at the other end of the link.
     * @param {GraphObject} otherport the GraphObject port at the other end of the link.
     * @param {Point=} result an optional Point that is modified and returned; otherwise it allocates and returns a new Point
     * @return {Point} in document coordinates.
     * @since 1.2
     */
    getLinkPoint(node: Node | null, port: GraphObject | null, spot: Spot, from: boolean, ortho: boolean, othernode: Node | null, otherport: GraphObject | null, result?: Point): Point;
    /**
     * Compute the direction in which a link should go from a given connection point.
     * @expose
     * @param {Node} node
     * @param {GraphObject} port the GraphObject representing a port on the node.
     * @param {Point} linkpoint the connection point, in document coordinates.
     * @param {Spot} spot a Spot value describing where the link should connect.
     * @param {boolean} from true if the link is coming out of the port; false if going to the port.
     * @param {boolean} ortho whether the link should have orthogonal segments.
     * @param {Node} othernode the node at the other end of the link.
     * @param {GraphObject} otherport the GraphObject port at the other end of the link.
     * @return {number} the absolute angle, in degrees.
     * @since 1.2
     */
    getLinkDirection(node: Node | null, port: GraphObject | null, linkpoint: Point, spot: Spot, from: boolean, ortho: boolean, othernode: Node | null, otherport: GraphObject | null): number;
    /**
     * Get the length of the end segment in document coordinates, typically a short distance, in document units.
     * For spot values that are Spot#isSide, this returns a computed value.
     * Depending on the `from` argument, this will return #fromEndSegmentLength or #toEndSegmentLength.
     * If the value is `NaN`, this will return the #fromPort's GraphObject#fromEndSegmentLength
     * or the #toPort's GraphObject#toEndSegmentLength.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} node
     * @param {GraphObject} port the GraphObject representing a port on the node.
     * @param {Spot} spot a Spot value describing where the link should connect.
     * @param {boolean} from true if the link is coming out of the port; false if going to the port.
     * @return {number} a distance in document coordinates; must be a real number, not NaN or infinity.
     * @since 1.6
     */
    computeEndSegmentLength(node: Node | null, port: GraphObject | null, spot: Spot, from: boolean): number;
    /**
     * Get the Spot that describes how the end of the link should connect with the port.
     * Depending on the `from` argument, this will return #fromSpot or #toSpot.
     * If the value is Spot#isDefault, this will return the #fromPort's GraphObject#fromSpot
     * or the #toPort's GraphObject#toSpot.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {boolean} from
     * @param {GraphObject=} port
     * @return {Spot} must not be Spot.Default, but may be Spot.None
     * @since 1.6
     */
    computeSpot(from: boolean, port?: GraphObject | null): Spot;
    /**
     * Find the approximate point of the other end of the link in document coordinates.
     * This is useful when computing the connection point when there is no specific spot, to have an idea of which general direction the link should be going.
     * By default this will return the center of the other port.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Node} othernode
     * @param {GraphObject} otherport
     * @return {Point} approximately where the other end of this link might end, in document coordinates
     * @since 1.6
     */
    computeOtherPoint(othernode: Node, otherport: GraphObject): Point;
    /**
     * Undocumented
     * @expose
     * @param {boolean} from
     * @return {number} must be a real number, not NaN or infinity.
     */
    protected computeShortLength(from: boolean): number;
    /**
     * This read-only property is true if #routing is a value that implies that the points
     * of the route should be orthogonal, such that each point shares a common X or a common Y value
     * with the immediately previous and next points.
     * This property is completely dependent on the #routing property.
     * Values of Link.Orthogonal and Link.AvoidsNodes causes this property to be true.
     *
     * The points in the route might not actually form an orthogonal route, but when the route is computed
     * the intent is to maintain orthogonality.
     * @since 1.2
     */
    readonly isOrthogonal: boolean;
    /**
     * This read-only property is true when the routing tries to be smart about not overlapping other parts.
     */
    readonly isAvoiding: boolean;
    /**
     * Returns the #curve, unless this link is supposed to pretend to be curved, as with reflexive links.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {EnumValue}
     * @since 1.6
     */
    computeCurve(): EnumValue;
    /**
     * Undocumented
     * @expose
     * @return {number} must be a real number, not NaN or infinity
     */
    protected computeCorner(): number;
    /**
     * Undocumented
     * @expose
     * @return {GraphObject}
     */
    findMidLabel(): GraphObject | null;
    /**
     * Returns the expected spacing between this link and others that connect this link's fromPort and toPort.
     * This calls #computeThickness and also takes any "mid label"'s breadth into account.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {number} must be a real number, not NaN or infinity
     * @see #computeCurviness
     * @since 1.6
     */
    computeSpacing(): number;
    /**
     * Undocumented
     * @expose
     * @param {Array.<Link>} links
     * @param {boolean} reroute
     */
    arrangeBundledLinks(links: Array<Link>, reroute: boolean): void;
    /**
     * Returns the #curviness, if it's a number,
     * or else a computed value based on how many links connect this pair of nodes/ports,
     * by calling #computeSpacing on each link.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {number} must be a real number, not NaN or infinity
     * @see #computeSpacing
     * @since 1.6
     */
    computeCurviness(): number;
    /**
     * Returns the thickness of this link.
     * By default it uses the strokeWidth of the main element, assuming it's a Shape.
     * This is called by #computeSpacing.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {number} a non-negative real number
     * @since 1.6
     */
    computeThickness(): number;
    /**
     * Returns true if an extra or a different point is needed based on #curviness.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {boolean}
     * @since 1.6
     */
    hasCurviness(): boolean;
    /**
     * This method is called by #computePoints when the link is orthogonal
     * and at least one port has a link spot that is not Spot#isNoSpot.
     *
     * #computePoints is responsible for adding the first two
     * and the last two points of the stroke, including the startFrom and endTo values.
     * This method is responsible for adding any additional points in the middle of the stroke.
     * This method calls #computeMidOrthoPosition to determine the
     * distance of the middle segment between the two ports.
     * It also tries to avoid the source node and the destination node.
     * When the #routing property is not Normal,
     * this method uses another, more computationally expensive,
     * method for determining the proper path of the link, which may have many segments.
     * @expose
     * @param {Point} startFrom this Point will already have been added to the stroke by #computePoints before calling this method.
     * @param {number} fromDir Normally 0, 90, 180, or 270 degrees.
     * @param {Point} endTo #computePoints will add this Point after calling this method.
     * @param {number} toDir Normally 0, 90, 180, or 270 degrees.
     * @param {Node} fromnode the Node that the link is coming from.
     * @param {Node} tonode the Node that the link is going to.
     */
    addOrthoPoints(startFrom: Point, fromDir: number, endTo: Point, toDir: number, fromnode: Node, tonode: Node): void;
    /**
     * Undocumented
     * @expose
     * @param {number} fromX The first point's X coordinate.
     * @param {number} fromY The first point's Y coordinate.
     * @param {number} toX The last point's X coordinate.
     * @param {number} toY The last point's Y coordinate.
     * @param {boolean} vertical Whether the mid-position is along the vertical axis or horizontal axis.
     * @return {number} must be a real number, not NaN or infinity
     */
    protected computeMidOrthoPosition(fromX: number, fromY: number, toX: number, toY: number, vertical: boolean): number;
    /**
     * Find the index of the segment that is closest to a given point.
     * This assume the route only has straight line segments.
     * It ignores any jump-overs or jump-gaps.
     * @param {Point} p the Point, in document coordinates.
     * @return {number} int the index of the segment, from zero to the number of points minus 2.
     */
    findClosestSegment(p: Point): number;
    /**
     * This read-only property returns the Geometry that is used by the #path, the link Shape based on the route points.
     *
     * This geometry is automatically generated using the route points and other properties
     * such as the #curve.
     * The points of the Geometry are in local coordinates, whereas the #points of the link route are in document coordinates.
     *
     * More than one Shape may share this geometry as its Shape#geometry,
     * if there is more than one shape element in the Link with GraphObject#isPanelMain set to true.
     */
    readonly geometry: Geometry;
    /**
     * Produce a Geometry given the points of this route,
     * depending on the value of #curve and #corner and perhaps other properties.
     * The points of the Geometry are in local coordinates, whereas the #points of the link route are in document coordinates.
     *
     * This method may be overridden.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {Geometry}
     * @since 1.6
     */
    makeGeometry(): Geometry;
    /**
     * Undocumented
     */
    readonly firstPickIndex: number;
    /**
     * Undocumented
     */
    readonly lastPickIndex: number;
    /**
     * Undocumented
     * @param {Rect} prevbounds
     */
    invalidateOtherJumpOvers(prevbounds: Rect): void;
    /**
     * Gets or sets how the route is computed, including whether it uses the points of its old route to determine the new route.
     * The value must be one of Link.None|None, Link.End|End, Link.Scale|Scale, or Link.Stretch|Stretch.
     *
     * The default value is Link.None|None -- the route is completely recalculated each time.
     */
    adjusting: EnumValue;
    /**
     * Gets or sets how rounded the corners are for adjacent line segments when the #curve
     * is Link.None|None, Link.JumpGap|JumpGap, or Link.JumpOver|JumpOver and
     * the two line segments are orthogonal to each other.
     *
     * The default value is zero -- there is no curve at a corner.
     */
    corner: number;
    /**
     * Gets or sets the way the path is generated from the route's points.
     * The value must be one of Link.None|None, Link.Bezier|Bezier, Link.JumpGap|JumpGap, or Link.JumpOver|JumpOver.
     *
     * Setting this property to Link.JumpOver|JumpOver or Link.JumpGap|JumpGap requires the Diagram to do
     * considerable computation when calculating Link routes. Consider not using
     * a Jump... value with Diagrams that contain large numbers of Links
     * if you are targeting slow devices.
     *
     * The default value is Link.None|None -- each link segment is a straight line.
     * @see #computeCurve
     */
    curve: EnumValue;
    /**
     * Gets or sets how far the control points are offset when the #curve is Link.Bezier|Bezier
     * or when there are multiple links between the same two ports.
     *
     * The default value is NaN -- the actual curviness is computed based on how many links connect the same pair of ports.
     * @see #computeCurviness
     */
    curviness: number;
    /**
     * Gets or sets whether the link's path tries to avoid other nodes.
     * The value must be one of Link.Normal|Normal, Link.Orthogonal|Orthogonal, or Link.AvoidsNodes|AvoidsNodes.
     *
     * Setting this property to Link.AvoidsNodes|AvoidsNodes requires the Diagram to do
     * considerable computation when calculating Link routes. Consider not using
     * the Link.AvoidsNodes|AvoidsNodes with Diagrams that contain large numbers of Nodes and Links
     * if you are targeting slow devices.
     * When using Link.AvoidsNodes|AvoidsNodes, dragging performance can be improved by setting DraggingTool#isComplexRoutingRealtime to false.
     *
     * The default value is Link.Normal|Normal -- the route is primarily a single straight segment,
     * with possible short end segments when connecting with ports that have a "spot" value
     * as the #fromSpot or GraphObject#fromSpot or #toSpot or GraphObject#toSpot.
     */
    routing: EnumValue;
    /**
     * Gets or sets how far the control points are from the points of the route
     * when #routing is Link.Orthogonal|Orthogonal and #curve is Link.Bezier|Bezier.
     *
     * The default value is 0.5.
     * Values of this property typically fall in the range of 0.0 to 1.0.
     * A value of 0.0 indicates that the control points of the curve are at the end points,
     * which will result in straight line segments.
     * A value of 1.0 indicates that the control points are one-third of the link's
     * length away from the end point along the direction of the link,
     * and the same distance away in a perpendicular direction.
     * The distance scales linearly with the value of this property,
     * even with negative values and values greater than 1.0.
     */
    smoothness: number;
    /**
     * This read-only property returns the Links's Model data key if it is in a Diagram and is backed by Model data.
     * Otherwise this returns undefined.
     *
     * If the link data is in a GraphLinksModel, the GraphLinksModel#linkKeyProperty must not be an empty string.
     * @since 1.8
     * @see GraphLinksModel#getKeyForLinkData
     */
    readonly key: Key;
}
/**
 * This is the base class for all of the predefined diagram layout implementations.
 * They only arrange Parts (primarily Nodes and Links) in a Diagram,
 * not to GraphObjects in Panels (i.e. panel layout).
 *
 * The layout classes include TreeLayout, ForceDirectedLayout,
 * LayeredDigraphLayout, CircularLayout, and GridLayout.
 * This base class is not abstract -- in fact an instance of this base class is the default
 * value for Diagram#layout and for Group#layout.
 *
 * An instance of a Layout class will be the value of Diagram#layout.
 * That layout positions the graph of top-level nodes and links.
 * Nodes and links that belong to a Group are laid out by that group's Group#layout.
 * The Diagram will automatically perform all nested group layouts before laying out the whole diagram.
 *
 * If you have position information for all of the nodes when you load a model,
 * you will typically have data bound the Part#location to some property on your node data.
 * In order to avoid an initial layout causing those saved node positions to be discarded,
 * you can either not set the Diagram#layout to a predefined layout or you can
 * set #isInitial to false.
 *
 * Because performing layouts can be expensive in space and time, automatic layouts
 * are performed only on "invalid" layouts, and only well after a layout has been invalidated.
 * This state is held by the #isValidLayout property.
 * Many standard operations, such as adding or removing nodes or links, will cause
 * the layout that is responsible for positioning those nodes or routing those links
 * to be invalidated.
 * Such invalidation is performed by calling #invalidateLayout,
 * which not only clears the #isValidLayout state but also requests that the diagram
 * do an automatic layout soon.
 * You can avoid such invalidations by setting #isOngoing to false.
 *
 * Layouts will ignore parts that have Part#isLayoutPositioned set to false
 * or parts that are not GraphObject#visible.
 * Layouts will also ignore parts that are in layers that are Layer#isTemporary.
 *
 * Various operations on Parts will cause the responsible Layout to be invalidated.
 * This includes adding or removing parts, changing their visibility, and changing their size.
 * You can disable such automatic layout invalidations by setting Part#layoutConditions
 * to the combination of Part flags named "Layout..." that you want.
 *
 * But operations on parts are not the only way in which layouts become invalidated.
 * Setting most properties on the layouts, thereby changing their behavior, will invalidate that layout.
 * Replacing the Diagram#layout or Group#layout will automatically invalidate the new layout.
 * If #isViewportSized is true, when a diagram's Diagram#viewportBounds changes size,
 * the Diagram#layout is invalidated.
 * (This is normally only true for GridLayouts when its GridLayout#wrappingWidth is `NaN`.
 * Most layouts do not care about the size of the viewport.)
 *
 * You can also explicitly call Diagram#layoutDiagram, which can invalidate all layouts and
 * then perform them all.
 * But we recommend that you avoid doing so, to allow the normal updating process perform layouts as needed.
 *
 * If an automatic layout is the first time that a layout has been performed for the model,
 * the diagram first raises the DiagramEvent named "InitialLayoutCompleted".
 * Whenever a Diagram finishes an automatic layout, it raises the DiagramEvent named "LayoutCompleted".
 *
 * It is also possible to call #doLayout explicitly, but this is uncommon and only used with
 * instances of Layout that are not the Diagram#layout or Group#layout.
 * It should only be needed when you want to layout a collection of nodes and links that is not
 * the normal graph of top-level parts of a Diagram or a subgraph of a Group.
 *
 * More complicated layouts make use of a separate LayoutNetwork, consisting of LayoutVertexes and LayoutEdges,
 * that normally holds a graph that is isomorphic to the graph consisting of Nodes and Links in the Diagram or Group.
 * The implementation of #doLayout will call #makeNetwork and remember the result as the #network.
 * #makeNetwork will call #createNetwork and initialize it by adding new instances of LayoutVertexes and LayoutEdges
 * corresponding to the given collection of Nodes and Links.
 *
 * When #doLayout is finished with its work it will call #updateParts, which will call #commitLayout
 * to set new node locations and route links. It then normally discards the #network.
 *
 * The LayoutVertex and LayoutEdge instances allow the layout to work with more information about each Node and Link without
 * actually modifying those Nodes and Links until #commitLayout is called to actually set the Node locations and route the Links.
 * The use of a LayoutNetwork also allows the Layout to work with a graph that is not isomorphic to the given collection of Nodes and Links.
 * This is useful when needing to use dummy vertexes and/or edges to achieve certain layout behaviors,
 * or when one wants to ignore certain vertexes or edges, without actually modifying or adding or removing the diagram's nodes or links.
 *
 * An instance of this base class provides a rudimentary default layout that will position
 * all of the parts that have no position (i.e. the Part#location is (NaN,NaN).
 * Parts that already have a position are ignored.
 * This primitive layout class does not make use of a LayoutNetwork because
 * it ignores all links.
 *
 * To implement your own custom layouts, you can inherit from either this class or
 * from one of the other predefined layout classes.
 * If you inherit from this base class, you will want to override the #doLayout method.
 * You can call the Part#move method to re-position a part, including whole groups.
 * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call a base method.
 * @category Layout
 */
export class Layout {
    /**
     * Create a minimal layout that only positions Nodes that do not have a location.
     */
    constructor();
    /**
     * Copies properties from this object to the given object, which is of the same class.
     * This is called by #copy and should be overridden for each class that adds properties.
     * There are examples of such overrides in the samples.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {?} copy
     * @since 1.6
     */
    protected cloneProtected(copy: this): void;
    /**
     * Creates a copy of this Layout and returns it.
     * When a Group is copied that has a Group#layout, the Layout must also be copied.
     * This calls #cloneProtected on a newly constructed Layout.
     * @expose
     * @return {Layout}
     */
    copy(): this;
    /**
     * Gets the Diagram that owns this layout, if it is the value of Diagram#layout.
     *
     * If this property and #group are non-null, the Group should be in this Diagram.
     * @see #group
     */
    diagram: Diagram | null;
    /**
     * Gets the Group that uses this layout, if it is the value of a group's Group#layout.
     *
     * If this property is set to a Group, the #diagram is automatically set to be the Group's Diagram.
     * @see #diagram
     */
    group: Group | null;
    /**
     * Gets or sets whether this layout can be invalidated by #invalidateLayout.
     * Set this to false to prevent actions such as adding or removing Parts from invalidating this layout.
     * The default value is true.
     * Setting this property does not invalidate this layout.
     *
     * If you set both #isInitial and #isOngoing to false,
     * there will be no automatic layout invalidation, because #invalidateLayout
     * will not set #isValidLayout to false.
     * To get your nodes to appear, you will need to explicitly set or data-bind their Part#location
     * or GraphObject#position to real Point values, because automatic layout will not assign any positions.
     *
     * Another way of controlling when layouts are invalidated is by setting
     * Part#isLayoutPositioned or Part#layoutConditions.
     */
    isOngoing: boolean;
    /**
     * Gets or sets whether this layout is performed on an initial layout.
     * The default value is true.
     * Setting this property to false causes #isValidLayout to be set to true
     * so that the diagram does not perform this layout.
     *
     * If you set both #isInitial and #isOngoing to false,
     * there will be no automatic layout invalidation, because #invalidateLayout
     * will not set #isValidLayout to false.
     * To get your nodes to appear, you will need to explicitly set or data-bind their Part#location
     * or GraphObject#position to real Point values, because automatic layout will not assign any positions.
     *
     * Another way of controlling when layouts are invalidated is by setting
     * Part#isLayoutPositioned or Part#layoutConditions.
     */
    isInitial: boolean;
    /**
     * Gets or sets whether this layout depends on the Diagram#viewportBounds's size.
     * If set to true, the layout will invalidate when the Diagram's viewport changes size.
     * This only applies to diagram layouts, not to group layouts,
     * and only when Diagram#autoScale is set to Diagram.None.
     * The default value is false.
     * Setting this property to true will invalidate this layout.
     */
    isViewportSized: boolean;
    /**
     * Gets or sets whether this layout routes Links.
     * The default value is true.
     * When false, this layout will not explicitly set the Link#points,
     * and the default routing of each individual Link will take place after the Nodes are moved by #commitLayout.
     * Setting this property does not invalidate this layout.
     *
     * Some layouts ignore links, in which case this property is ignored.
     * @since 1.1
     */
    isRouting: boolean;
    /**
     * Gets or sets whether this layout be performed in real-time, before the end of a transaction.
     * All layouts that are invalidated will be performed at the end of a transaction.
     * The default value is null.
     * A null value is treated as true for a Diagram#layout but false for a Group#layout.
     * Setting this property does not invalidate this layout.
     * @since 1.2
     */
    isRealtime: boolean | null;
    /**
     * Gets or sets whether this layout needs to be performed again (if false).
     * Instead of setting this property directly, it is normal to set it to false by calling #invalidateLayout,
     * since that also requests performing a layout in the near future.
     */
    isValidLayout: boolean;
    /**
     * If #isOngoing is true and if an initial layout has not yet been performed,
     * set the #isValidLayout property to false, and ask to perform another layout in the near future.
     * If #isInitial is true, this layout is invalidated only when the Diagram#model is replaced,
     * not under the normal circumstances such as when parts are added or removed or
     * due to other calls to Layout#invalidateLayout.
     *
     * If you set both #isInitial and #isOngoing to false,
     * there will be no automatic layout invalidation, because this method
     * will not set #isValidLayout to false.
     * However you can still set #isValidLayout explicitly.
     *
     * This is typically called when a layout property value has changed,
     * or when a Part is added or removed or changes visibility, if Part#layoutConditions includes the pertinent flags.
     */
    invalidateLayout(): void;
    /**
     * Gets or sets the LayoutNetwork used by this Layout, if any.
     * The default value is null.
     * Setting this property does not invalidate this layout.
     * Not all kinds of layout make use of a LayoutNetwork.
     * Call #createNetwork or #makeNetwork to create a network.
     */
    network: LayoutNetwork | null;
    /**
     * Create a new LayoutNetwork of LayoutVertexes and LayoutEdges.
     * This may be overridden in Layout subclasses to create instances of subclasses of
     * LayoutNetwork.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {LayoutNetwork} a new LayoutNetwork.
     */
    createNetwork(): LayoutNetwork;
    /**
     * Create and initialize a LayoutNetwork with the given nodes and links.
     * This should be called by #doLayout when this layout uses a #network.
     * This method calls #createNetwork to allocate the network.
     * This may be overridden in Layout subclasses to customize the initialization.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Diagram|Group|Iterable.<Part>} coll A Diagram or a Group or a collection of Parts.
     * @return {LayoutNetwork} normally the value of a call to #createNetwork initialized by vertexes and edges corresponding to the *coll* argument.
     */
    makeNetwork(coll: Diagram | Group | Iterable<Part>): LayoutNetwork;
    /**
     * When using a LayoutNetwork, update the "physical" node positionings and link routings.
     * This should be called by #doLayout when this layout uses a #network.
     * This calls #commitLayout to actually set Node positions and route Links.
     * This performs the changes within a transaction.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    updateParts(): void;
    /**
     * When using a LayoutNetwork, commit changes to the diagram
     * by setting Node positions and by routing the Links.
     * This is called by #updateParts within a transaction.
     *
     * You should not call this method -- it is a "protected virtual" method.
     * This may be overridden by subclasses of Layout.
     * By default this method is implemented as follows:
     * ```js
     * protected commitLayout() {
     *   if (this.network === null) return;
     *   var vit = this.network.vertexes.iterator;
     *   while (vit.next()) {
     *     var vert = vit.value;
     *     vert.commit();
     *   }
     *   if (this.isRouting) {
     *     var eit = this.network.edges.iterator;
     *     while (eit.next()) {
     *       var edge = eit.value;
     *       edge.commit();
     *     }
     *   }
     * }
     * ```
     *
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    protected commitLayout(): void;
    /**
     * Position all of the nodes that do not have an assigned Part#location
     * in the manner of a simple rectangular array.
     * The default implementation ignores all Groups and Links;
     * many subclasses of Layout ignore all instances of Parts
     * that are not Nodes or Links.
     *
     * You can override this method to do whatever node positioning and link routing
     * that you wish.
     *
     * When the layout makes use of a LayoutNetwork,
     * this method should call #makeNetwork and #updateParts.
     * The #updateParts method will call #commitLayout within a transaction.
     * The outline of such an override should be like:
     * ```js
     * public doLayout(coll) {
     *   if (this.network === null) this.network = this.makeNetwork(coll);
     *   // assign LayoutVertex.bounds to all vertexes in the network:
     *   var vit = this.network.vertexes.iterator;
     *   while (vit.next()) {
     *     var v = vit.value;
     *     v.centerX = ...
     *     v.centerY = ...
     *   }
     *   this.updateParts();
     *   this.network = null;
     * }
     * ```
     * Override #commitLayout to actually position nodes and route links.
     *
     * When the layout does <em>not</em> make use of a LayoutNetwork,
     * this method should make direct changes to Nodes and Links
     * within a transaction, and it should call #collectParts on the argument *coll* in order
     * to get the proper set of Nodes and Links to work on.
     * Examples are provided in the Extensions directory.
     *
     * If this Layout belongs to a Diagram, the argument must be either the same Diagram or must denote Parts that belong to the Diagram.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @param {Diagram|Group|Iterable.<Part>} coll A Diagram or a Group or a collection of Parts.
     */
    doLayout(coll: Diagram | Group | Iterable<Part>): void;
    /**
     * This method is called by layouts to determine the size and initial position of the nodes that it is laying out.
     * Normally this just returns the part's GraphObject#actualBounds.
     * However, if #boundsComputation has been set to a function, that function will be called
     * in order to return the bounds of the given Part in document coordinates that the layout should pretend it has.
     * @param {Part} part  the Part being laid out
     * @param {Rect=} rect  an optional Rect that will be modified and returned
     * @return {Rect} a Rect in document coordinates
     * @since 2.0
     */
    getLayoutBounds(part: Part, rect?: Rect): Rect;
    /**
     * Gets or sets a function that determines the initial size and position in document coordinates of a LayoutVertex corresponding to a Node.
     * This function is called by #getLayoutBounds.
     * The default value for this property is null, in which case the GraphObject#actualBounds of the Node is used.
     * Setting this property to a new value invalidates this layout.
     *
     * The non-null value must be a function that takes 3 arguments.
     * The first argument will be the Part whose bounds the Layout should use.
     * The second argument will be this Layout.
     * The third argument will be a Rect that must be modified and returned
     * The return value must be in document coordinates.  You may find it convenient to call GraphObject#getDocumentBounds
     * to get the bounds in document coordinates of an object within the node.
     * @since 2.0
     */
    boundsComputation: ((part: Part, lay: Layout, rect: Rect) => Rect) | null;
    /**
     * A convenient way of converting the Diagram|Group|Iterable argument to doLayout to an actual collection of eligible Parts.
     * The resulting Set will not include any Nodes or Links for which Part#canLayout is false.
     * If the argument includes a Group for which Group#layout is null, the resulting Set
     * will include the member parts of that group rather than that group itself.
     * You will not need to call collectParts if you call #makeNetwork,
     * because that method does effectively the same thing when building the LayoutNetwork.
     *
     * Typical usage:
     * ```js
     * public doLayout(coll) {
     *   // COLL might be a Diagram or a Group or some Iterable<Part>
     *   var it = this.collectParts(coll).iterator;
     *   while (it.next()) {
     *     var node = it.value;
     *     if (node instanceof go.Node) {
     *       . . . position the node . . .
     *     }
     *   }
     * }
     * ```
     * @param {Diagram|Group|Iterable.<Part>} coll
     * @return {Set.<Part>}
     * @since 1.7
     */
    protected collectParts(coll: Diagram | Group | Iterable<Part>): Set<Part>;
    /**
     * Gets or sets the top-left point for where the graph should be positioned when laid out.
     * The default value for this property is the Point(0, 0).
     * Setting this property to a new value invalidates this layout.
     * This property is likely to be set by many Layouts that belong to a Group when the layout is performed.
     */
    arrangementOrigin: Point;
    /**
     * Compute the desired value of #arrangementOrigin if this Layout is being performed for a Group.
     * This is typically called near the beginning of the implementation of #doLayout:
     * `this.arrangementOrigin = this.initialOrigin(this.arrangementOrigin);`
     * if the layout wants to respect the pre-layout location of the Group when deciding where to position its member nodes.
     * @expose
     * @param {Point} origin
     * @return {Point}
     */
    protected initialOrigin(origin: Point): Point;
}
/**
 * This provides an abstract view of a diagram as a
 * network (graph) of vertexes and directed edges.
 * The network contains vertexes and edges corresponding to Nodes and Links.
 *
 * This class provides a framework for manipulating the
 * state of nodes and links without modifying the structure of the diagram.
 * Having a separate representation also permits adding or removing vertexes or edges
 * from the network so that the graph that is laid out is not isomorphic to the diagram's graph.
 *
 * For each kind of layout that uses a LayoutNetwork there is a subclass of LayoutVertex and a subclass of LayoutEdge:
 *   - CircularLayout: CircularVertex and CircularEdge
 *   - ForceDirectedLayout: ForceDirectedVertex and ForceDirectedEdge
 *   - LayeredDigraphLayout: LayeredDigraphVertex and LayeredDigraphEdge
 *   - TreeLayout: TreeVertex and TreeEdge
 *
 * Modifying a LayoutNetwork or a LayoutVertex or a LayoutEdge does not
 * invalidate the Layout or raise any changed events.
 * @unrestricted
 * @category Layout
 */
export class LayoutNetwork {
    /**
     * This constructs an empty network.
     * It is commonplace to call #addParts to build up the network
     * to have the same structure as a Diagram's or Group's nodes and links.
     */
    constructor(layout: Layout);
    /**
     * Gets the Layout that uses this network
     * of LayoutVertexes and LayoutEdges.
     * The default value is null.
     * It is set automatically for you when this setting Layout#network.
     */
    layout: Layout;
    /**
     * Gets a collection of all of the LayoutVertexes in this network.
     * Do not modify this collection directly.
     * Instead, call #addVertex, #deleteVertex, #addNode, or #deleteNode.
     */
    readonly vertexes: Set<LayoutVertex>;
    /**
     * Gets a collection of all of the LayoutEdges in this network.
     * Do not modify this collection directly.
     * Instead, call #addEdge, #deleteEdge, #addLink, or #deleteLink.
     */
    readonly edges: Set<LayoutEdge>;
    /**
     * Allocate a new instance of LayoutVertex.
     * This is overridden in LayoutNetwork subclasses to create instances of subclasses of LayoutVertex.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {LayoutVertex}
     */
    createVertex(): LayoutVertex;
    /**
     * Allocate a new instance of LayoutEdge.
     * This is overridden in LayoutNetwork subclasses to create instances of subclasses of LayoutEdge.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     * @return {LayoutEdge}
     */
    createEdge(): LayoutEdge;
    /**
     * Creates a network of LayoutVertexes and LayoutEdges
     * corresponding to the given Nodes and Links.
     * This is usually more convenient than repeatedly calling #addNode and #addLink.
     * This method ignores all instances of Parts that are not Nodes or Links.
     * This will recurse through Groups that have no Group#layout.
     * @param {Iterable.<Part>} parts A collection of Nodes or Links.
     * @param {boolean=} toplevelonly whether to skip Parts in the given collection that are contained by Groups; default is false
     * @param {function(Part):boolean|null=} pred optional predicate to apply to each Part --
     *        if it returns false do not include Vertex or Edge in the network;
     *        default ignores link label nodes or links connecting with them
     */
    addParts(parts: Iterable<Part>, toplevelonly?: boolean, pred?: ((a: Part) => boolean) | null): void;
    /**
     * Adds a LayoutVertex to the network.
     * @param {LayoutVertex} vertex
     */
    addVertex(vertex: LayoutVertex): void;
    /**
     * This convenience method makes sure there is a LayoutVertex
     * in this network corresponding to a Node.
     * @param {Node} node
     * @return {LayoutVertex} a vertex in this network.
     */
    addNode(node: Node): LayoutVertex;
    /**
     * Removes a LayoutVertex from the network.
     * This function also deletes all edges to or from the vertex.
     * @param {LayoutVertex} vertex
     */
    deleteVertex(vertex: LayoutVertex): void;
    /**
     * This convenience method deletes any LayoutVertex
     * corresponding to a Node.
     * @param {Node} node
     */
    deleteNode(node: Node): void;
    /**
     * Returns the LayoutVertex that was constructed for the Node.
     * @param {Node} node
     * @return {LayoutVertex} a vertex in this network, or null if no such vertex exists.
     */
    findVertex(node: Node): LayoutVertex | null;
    /**
     * Adds a LayoutEdge to the network.
     * Although this method is provided for completeness,
     * #linkVertexes provides a more efficient manner of linking
     * vertexes already in the network.
     * @param {LayoutEdge} edge
     */
    addEdge(edge: LayoutEdge): void;
    /**
     * This convenience method makes sure there is a LayoutEdge
     * in this network corresponding to a Link.
     * If #findEdge returns null, this method creates a new LayoutEdge,
     * makes sure the Link#fromNode and Link#toNode have
     * corresponding LayoutVertexes in the network, and adds the edge itself
     * to the network.
     * @param {Link} link
     * @return {LayoutEdge} an edge in this network.
     */
    addLink(link: Link): LayoutEdge;
    /**
     * Removes a LayoutEdge from the network.
     * @param {LayoutEdge} edge
     */
    deleteEdge(edge: LayoutEdge): void;
    /**
     * This convenience method deletes from this network any LayoutEdge
     * corresponding to a Link.
     * @param {Link} link
     */
    deleteLink(link: Link): void;
    /**
     * Returns the LayoutEdge that was constructed for the Link.
     * @param {Link} link
     * @return {LayoutEdge} an edge in this network, or null if no such edge exists.
     */
    findEdge(link: Link): LayoutEdge | null;
    /**
     * Links two vertexes already in the network and returns the created LayoutEdge.
     * @param {LayoutVertex} fromVertex
     * @param {LayoutVertex} toVertex
     * @param {Link} link
     * @return {LayoutEdge} an edge in this network, or null if both vertexes don't exist.
     */
    linkVertexes(fromVertex: LayoutVertex, toVertex: LayoutVertex, link: Link | null): LayoutEdge;
    /**
     * Reverses the direction of a LayoutEdge in the network.
     * @param {LayoutEdge} edge
     */
    reverseEdge(edge: LayoutEdge): void;
    /**
     * Deletes all LayoutEdges whose "to vertex" and "from vertex" are the same vertex.
     */
    deleteSelfEdges(): void;
    /**
     * Deletes all vertexes and edges that have no Part associated with them.
     * @expose
     */
    deleteArtificialVertexes(): void;
    /**
     * Modify this network by splitting it up into separate subnetworks,
     * each of which has all of its vertexes connected to each other, but not
     * to any vertexes in any other subnetworks.
     *
     * This method will first delete from this network all artificial vertexes and
     * all edges that do not connect two different vertexes.
     * Afterwards, this original network may be empty or may contain all of the
     * singleton vertexes, each of which had no edges connecting it to any other vertexes.
     * @return {List.<LayoutNetwork>} a collection of LayoutNetworks,
     * sorted in order of decreasing vertex count.
     */
    splitIntoSubNetworks(): List<LayoutNetwork>;
    /**
     * Retrieve all of the Nodes and Links from the
     * LayoutVertexes and LayoutEdges that are in this network.
     * @return {Set.<Part>} A collection of Parts.
     * @since 1.1
     */
    findAllParts(): Set<Part>;
}
/**
 * A vertex represents a Node in a LayoutNetwork,
 * along with its #bounds and #focus and collections of LayoutEdges that come into and go out of the vertex.
 * The #node property may be null for vertexes that represent "dummy" nodes,
 * when the layout wants to work with a network that is not isomorphic with the collection of Nodes and Links being laid out.
 *
 * This holds layout-specific data for the node.
 * For each kind of layout that uses a LayoutNetwork there is a subclass of LayoutVertex and a subclass of LayoutEdge:
 *   - CircularLayout: CircularVertex and CircularEdge
 *   - ForceDirectedLayout: ForceDirectedVertex and ForceDirectedEdge
 *   - LayeredDigraphLayout: LayeredDigraphVertex and LayeredDigraphEdge
 *   - TreeLayout: TreeVertex and TreeEdge
 *
 * Modifying a LayoutNetwork or a LayoutVertex or a LayoutEdge does not
 * invalidate the Layout or raise any changed events.
 * @unrestricted
 * @category Layout
 */
export class LayoutVertex {
    /**
     * This constructs a vertex that does not know about any Node.
     */
    constructor(network: LayoutNetwork);
    /**
     * Undocumented
     */
    readonly sourceEdgesArrayAccess: Array<LayoutEdge>;
    /**
     * Undocumented
     */
    readonly destinationEdgesArrayAccess: Array<LayoutEdge>;
    /**
     * Gets or sets the data object in the model that should be associated with this vertex, if any.
     * The value will be non-null if this vertex is virtualized,
     * not represented by a real Node in the Diagram.
     * Each data object should have a 'bounds' property that is a Rect
     * or any JavaScript Object with 'x', 'y', 'width', and 'height' properties that are real numbers,
     * and non-negative numbers for the width and the height.
     * Setting this data property will also set the #bounds and #focus properties.
     * @since 1.6
     */
    data: ObjectData | null;
    /**
     * Gets or sets the Node associated with this vertex, if any.
     * The value may be null if this vertex is a "dummy" vertex,
     * not represented by a real Node in the Diagram.
     *
     * This property setter also sets the #bounds and #focus properties
     * according to the value of the Node's actualBounds and its Part#locationObject's center point.
     */
    node: Node | null;
    /**
     * Gets or sets the bounds of this vertex, in document coordinates.
     * Setting this property does not directly modify the position of any Node,
     * but it will modify the value of #centerX and #centerY,
     * as well as the subsidiary convenience properties #x, #y, #width, and #height.
     */
    bounds: Rect;
    /**
     * Gets or sets the offset of the #focusX and #focusY
     * from the #bounds position.
     * Setting this property does not directly modify the position of any Node,
     * but it does affect the value of #centerX and #centerY.
     *
     * The "focus" is normally the center of the Node's Part#locationObject.
     */
    focus: Point;
    /**
     * Gets or sets the center Point#x of this vertex, in document coordinates.
     * This property depends on both #bounds and #focus.
     * Setting this property does not modify the position of any Node,
     * but it will modify the value of #bounds.
     */
    centerX: number;
    /**
     * Gets or sets the center Point#y of this vertex, in document coordinates.
     * This property depends on both #bounds and #focus.
     * Setting this property does not modify the position of any Node,
     * but it will modify the value of #bounds.
     */
    centerY: number;
    /**
     * Gets or sets the relative X position of the "center" point, the focus.
     */
    focusX: number;
    /**
     * Gets or sets the relative Y position of the "center" point, the focus.
     */
    focusY: number;
    /**
     * Gets or sets the left point of this vertex.
     */
    x: number;
    /**
     * Gets or sets the top point of this vertex.
     */
    y: number;
    /**
     * Gets or sets the width of this vertex.
     * This is always the horizontal distance reserved for the node.
     */
    width: number;
    /**
     * Gets or sets the height of this vertex.
     * This is always the vertical distance reserved for the node.
     */
    height: number;
    /**
     * Moves the Node corresponding to this vertex
     * so that its position is at the current #bounds point.
     *
     * To make the most common cases look right, the Node's Part#locationObject
     * is centered.  Thus iconic nodes will have the center of the icon be positioned
     * according to the center of this vertex, ignoring any labels.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    commit(): void;
    /**
     * Adds a LayoutEdge to the list of predecessors
     * (the edge will be coming into this vertex).
     * Calling this method does not modify the LayoutEdge#toVertex of the LayoutEdge.
     * @param {LayoutEdge} edge
     */
    addSourceEdge(edge: LayoutEdge): void;
    /**
     * Deletes a LayoutEdge from the list of predecessors
     * (the edge was coming into this vertex).
     * Calling this method does not modify the LayoutEdge#toVertex of the LayoutEdge.
     * @param {LayoutEdge} edge
     */
    deleteSourceEdge(edge: LayoutEdge): void;
    /**
     * Adds a LayoutEdge to the list of successors
     * (the edge will be going out from this vertex).
     * Calling this method does not modify the LayoutEdge#fromVertex of the LayoutEdge.
     * @param {LayoutEdge} edge
     */
    addDestinationEdge(edge: LayoutEdge): void;
    /**
     * Deletes a LayoutEdge from the list of successors
     * (the edge was going out from this vertex).
     * Calling this method does not modify the LayoutEdge#fromVertex of the LayoutEdge.
     * @param {LayoutEdge} edge
     */
    deleteDestinationEdge(edge: LayoutEdge): void;
    /**
     * Gets the LayoutNetwork that owns this vertex.
     * The default value is null.
     * It is set automatically for you in LayoutNetwork#createVertex.
     */
    network: LayoutNetwork;
    /**
     * This read-only property returns an iterator for all of the vertexes that are connected with edges coming into this vertex.
     */
    readonly sourceVertexes: Iterator<LayoutVertex>;
    /**
     * This read-only property returns an iterator for all of the vertexes that are connected with edges going out of this vertex.
     */
    readonly destinationVertexes: Iterator<LayoutVertex>;
    /**
     * This read-only property returns an iterator for all of the vertexes that are connected in either direction with this vertex.
     */
    readonly vertexes: Iterator<LayoutVertex>;
    /**
     * This read-only property returns an iterator for all of the edges that come into this vertex.
     */
    readonly sourceEdges: Iterator<LayoutEdge>;
    /**
     * This read-only property returns an iterator for all of the edges that go out of this vertex.
     */
    readonly destinationEdges: Iterator<LayoutEdge>;
    /**
     * This read-only property returns an iterator for all of the edges that are connected with this vertex in either direction.
     */
    readonly edges: Iterator<LayoutEdge>;
    /**
     * This read-only property returns the total number of edges that are connected with this vertex in either direction.
     */
    readonly edgesCount: number;
    /**
     * This static function is used to compare the Part#text values of the #nodes of
     * the argument LayoutVertexes.
     * This just does a simple case-sensitive string comparison.
     * Alternatively you may wish to use LayoutVertex.smartComparer or write your own comparison function.
     * @param {LayoutVertex} m
     * @param {LayoutVertex} n
     * @return {number} int -1, 0, or 1, depending on whether the comparison between M and N is less than, equal, or greater than.
     */
    static standardComparer(m: LayoutVertex, n: LayoutVertex): number;
    /**
     * This static function is used to compare the Part#text values of the #nodes of
     * the argument LayoutVertexes.
     * If the text ends in a number, the values are compared as numbers rather than lexicographically.
     * Naturally this comparison is significantly slower than LayoutVertex.standardComparer.
     * @param {LayoutVertex} m
     * @param {LayoutVertex} n
     * @return {number} int -1, 0, or 1, depending on whether the comparison between M and N is less than, equal, or greater than.
     */
    static smartComparer(m: LayoutVertex, n: LayoutVertex): number;
}
/**
 * An edge represents a Link in a LayoutNetwork,
 * along with its #fromVertex and #toVertex.
 * The #link property may be null for edges that represent "dummy" links,
 * when the layout wants to work with a network that is not isomorphic with the collection of Nodes and Links being laid out.
 *
 * It holds layout-specific data for the link.
 * For each kind of layout that uses a LayoutNetwork there is a subclass of LayoutVertex and a subclass of LayoutEdge:
 *   - CircularLayout: CircularVertex and CircularEdge
 *   - ForceDirectedLayout: ForceDirectedVertex and ForceDirectedEdge
 *   - LayeredDigraphLayout: LayeredDigraphVertex and LayeredDigraphEdge
 *   - TreeLayout: TreeVertex and TreeEdge
 *
 * Modifying a LayoutNetwork or a LayoutVertex or a LayoutEdge does not
 * invalidate the Layout or raise any changed events.
 * @unrestricted
 * @category Layout
 */
export class LayoutEdge {
    /**
     * This constructs an edge that does not know about any Link.
     */
    constructor(network: LayoutNetwork);
    /**
     * Commits the route of this edge to the corresponding Link, if any.
     * By default this does nothing.
     * Please read the Introduction page on <a href="../../intro/extensions.html">Extensions</a> for how to override methods and how to call this base method.
     * @expose
     */
    commit(): void;
    /**
     * Returns the edge's vertex at the other of this edge from the given vertex.
     * @param {LayoutVertex} v
     */
    getOtherVertex(v: LayoutVertex): LayoutVertex | null;
    /**
     * Gets the LayoutNetwork that owns this edge.
     * The default value is null.
     * It is set automatically for you in LayoutNetwork#createEdge.
     */
    network: LayoutNetwork;
    /**
     * For virtualized layouts working on model data instead of a Link.
     * @since 1.6
     */
    data: ObjectData | null;
    /**
     * Gets or sets the Link associated with this edge, if any.
     * The value may be null if this edge is a "dummy" edge,
     * not represented by a real Link in the Diagram.
     */
    link: Link | null;
    /**
     * Gets or sets the LayoutVertex that this edge comes from.
     * Setting this property does not change any LayoutVertex#destinationEdges collection.
     */
    fromVertex: LayoutVertex | null;
    /**
     * Gets or sets the LayoutVertex that this edge goes to.
     * Setting this property does not change any LayoutVertex#sourceEdges collection.
     */
    toVertex: LayoutVertex | null;
}
/**
 * This simple layout places all of the Parts in a grid-like arrangement, ordered, spaced apart,
 * and wrapping as needed.  It ignores any Links connecting the Nodes being laid out.
 * There are many samples that use GridLayout.
 * Every Palette uses a GridLayout by default.
 * <p class="box">
 * If you want to experiment interactively with most of the properties, try the <a href="../../samples/gLayout.html">Grid Layout</a> sample.
 * See samples that make use of GridLayout in the <a href="../../samples/index.html#gridlayout">samples index</a>.
 *
 * By default this layout will sort all of the Parts alphabetically
 * (comparing Part#text values, respecting case)
 * and position them left-to-right, separated by #spacing`.width`,
 * until they do not fit in the current row.
 * At that time it starts a new row, separated from the previous row by #spacing`.height`.
 * There is a uniform cell size equal to the maximum Part width (plus spacing width)
 * and the maximum part height (plus spacing height).
 * At least one part is placed in each row, even if the part by itself is wider than the wrapping width.
 *
 * You can specify values for the #cellSize `width` and `height`.
 *